//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially
using System;
using System.Collections.Generic;
using System.Text;

namespace NetworkMiner.Packets {

    //User Datagram Protocol
    class UdpPacket : AbstractPacket {

        private ushort sourcePort;
        private ushort destinationPort;
        private ushort length;
        private ushort checksum;

        internal ushort SourcePort { get { return sourcePort; } }
        internal ushort DestinationPort { get { return destinationPort; } }


        internal UdpPacket(Frame parentFrame, int packetStartIndex, int packetEndIndex) : base(parentFrame, packetStartIndex, packetEndIndex, "UDP") {
            
            this.sourcePort=ByteConverter.ToUInt16(parentFrame.Data, packetStartIndex);
            this.Attributes.Add("Source Port", sourcePort.ToString());
            this.destinationPort=ByteConverter.ToUInt16(parentFrame.Data, packetStartIndex+2);
            this.Attributes.Add("Destination Port", destinationPort.ToString());

            this.length=ByteConverter.ToUInt16(parentFrame.Data, packetStartIndex+4);
            if(length!=packetEndIndex-packetStartIndex+1) {
                parentFrame.Errors.Add(new Frame.Error(parentFrame, packetStartIndex+4, PacketStartIndex+5, "UDP defined length ("+length+") differs from actual length ("+(packetEndIndex-packetStartIndex+1)+")"));
                //use the smallest value..
                if(packetEndIndex>packetStartIndex+length-1)
                    packetEndIndex=packetStartIndex+length-1;
            }
            this.checksum=ByteConverter.ToUInt16(parentFrame.Data, packetStartIndex+6);
        }
        /*
        internal override List<Packet>  GetSubPackets(){
            List<Packet> subPackets=new List<Packet>();
            if(PacketStartIndex+8<PacketEndIndex) {
                if(destinationPort==53 || sourcePort==53) {//DNS
                    try {
                        DnsPacket dnsPacket=new DnsPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                        subPackets.Add(dnsPacket);
                        subPackets.AddRange(dnsPacket.GetSubPackets());
                    }
                    catch(Exception e) {
                        RawPacket rawPacket=new RawPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                        subPackets.Add(rawPacket);
                        subPackets.AddRange(rawPacket.GetSubPackets());
                    }
                }
                else if(destinationPort==137 || sourcePort==137) {
                    try {
                        NetBiosNsPacket netBiosPacket=new NetBiosNsPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                        subPackets.Add(netBiosPacket);
                        subPackets.AddRange(netBiosPacket.GetSubPackets());
                    }
                    catch(Exception e) {
                        RawPacket rawPacket=new RawPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                        subPackets.Add(rawPacket);
                        subPackets.AddRange(rawPacket.GetSubPackets());
                    }
                }
                else {
                    RawPacket rawPacket=new RawPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                    subPackets.Add(rawPacket);
                    subPackets.AddRange(rawPacket.GetSubPackets());
                }
            }
            return subPackets;
        }
        */
        internal override IEnumerable<AbstractPacket> GetSubPackets() {
            //List<Packet> subPackets=new List<Packet>();
            if(PacketStartIndex+8<PacketEndIndex) {
                AbstractPacket packet;
                if(destinationPort==53 || sourcePort==53) {//DNS
                    try {
                        packet=new DnsPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                    }
                    catch(Exception e) {
                        ParentFrame.Errors.Add(new Frame.Error(ParentFrame, PacketStartIndex+8, PacketEndIndex, "Cannot parse DNS packet ("+e.Message+")"));
                        packet=new RawPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                    }
                }
                else if(destinationPort==137 || sourcePort==137) {
                    try {
                        packet=new NetBiosNameServicePacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                    }
                    catch(Exception e) {
                        ParentFrame.Errors.Add(new Frame.Error(ParentFrame, PacketStartIndex+8, PacketEndIndex, "Cannot parse NetBiosNameServicePacket packet ("+e.Message+")"));
                        packet=new RawPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                    }
                }
                else if(destinationPort==138 || sourcePort==138) {
                    try {
                        packet=new NetBiosDatagramServicePacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                    }
                    catch(Exception e) {
                        ParentFrame.Errors.Add(new Frame.Error(ParentFrame, PacketStartIndex+8, PacketEndIndex, "Cannot parse NetBiosDatagramServicePacket packet ("+e.Message+")"));
                        packet=new RawPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                    }
                }
                else if(destinationPort==1900 || sourcePort==1900) {
                    try {
                        packet=new UpnpPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                    }
                    catch(Exception e) {
                        ParentFrame.Errors.Add(new Frame.Error(ParentFrame, PacketStartIndex+8, PacketEndIndex, "Cannot parse UPnP packet ("+e.Message+")"));
                        packet=new RawPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                    }
                }
                else {
                    packet=new RawPacket(ParentFrame, PacketStartIndex+8, PacketEndIndex);
                }
                yield return packet;
                foreach(AbstractPacket subPacket in packet.GetSubPackets())
                    yield return subPacket;
            }
            //return subPackets;
        }
    }
}
