//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially
using System;
using System.Collections.Generic;
using System.Text;

namespace NetworkMiner.Packets {

    //http://ist.marshall.edu/ist362/tcp.html
    //ftp://ftp.rfc-editor.org/in-notes/rfc1323.txt
    //http://tools.ietf.org/html/rfc793
    //http://en.wikipedia.org/wiki/Transmission_Control_Protocol
    //http://support.microsoft.com/kb/224829
    class TcpPacket : AbstractPacket {

        //http://www.it.lut.fi/kurssit/06-07/Ti5312500/luennot/Luento10.pdf
        internal enum OptionKinds : byte {
            EndOfOptionList=0x00,
            NoOperation=0x01,
            MaximumSegmentSize=0x02,
            WindowScaleFactor=0x03,
            SackPermitted=0x04,
            Sack=0x05,
            Echo=0x06,//obsolete
            EchoReply=0x07,//obsolete
            Timestamp=0x08
        }

        internal class Flags{
            byte flagData;

            internal bool CongestionWindowReduced{get{return (flagData&0x80)==0x80;}}
            internal bool ECNEcho{get{return (flagData&0x40)==0x40;}}
            internal bool UrgentPointer{get{return (flagData&0x20)==0x20;}}
            internal bool Acknowledgement { get { return (flagData&0x10)==0x10; } }//needed for OS fingerprinting
            internal bool Push{get{return (flagData&0x08)==0x08;}}
            internal bool Reset{get{return (flagData&0x04)==0x04;}}
            internal bool Synchronize { get { return (flagData&0x02)==0x02; } }//needed for OS fingerprinting
            internal bool Fin{get{return (flagData&0x01)==0x01;}}

            internal Flags(byte data){
                this.flagData=data;

            }
        }

        //public byte[] SourceMACAddress { get { return this.sourceMAC;}}
        //public byte[] DestinationMACAddress {get {return this.destinationMAC;}}
        private ushort sourcePort;
        private ushort destinationPort;
        private uint sequenceNumber;
        private uint acknowledgmentNumber;
        private byte dataOffsetByteCount;
        //I'll skipp "Reserved"
        private Flags flags;
        private ushort windowSize;
        private ushort checksum;
        //I'll skipp "Urgent"
        private List<KeyValuePair<OptionKinds, byte[]>> optionList;
        //private int tcpDataStartIndex;

        internal ushort SourcePort { get { return sourcePort; } }
        internal ushort DestinationPort { get { return destinationPort; } }
        internal uint SequenceNumber { get { return sequenceNumber; } }
        internal byte DataOffsetByteCount { get { return dataOffsetByteCount; } }
        internal ushort WindowSize { get { return this.windowSize; } }//needed for OS fingerprinting
        internal List<KeyValuePair<OptionKinds, byte[]>> OptionList { get { return optionList; } }//needed for OS fingerprinting
        internal Flags FlagBits { get { return flags; } }//needed for OS fingerprinting

        internal int PayloadDataLength { get { return PacketEndIndex-PacketStartIndex-dataOffsetByteCount+1; } }

        internal TcpPacket(Frame parentFrame, int packetStartIndex, int packetEndIndex) : base(parentFrame, packetStartIndex, packetEndIndex, "TCP") {
            
            this.sourcePort=ByteConverter.ToUInt16(parentFrame.Data, packetStartIndex);
            this.Attributes.Add("Source Port", sourcePort.ToString());
            this.destinationPort=ByteConverter.ToUInt16(parentFrame.Data, packetStartIndex+2);
            this.Attributes.Add("Destination Port", destinationPort.ToString());
            this.sequenceNumber=ByteConverter.ToUInt32(parentFrame.Data, packetStartIndex+4);
            this.acknowledgmentNumber=ByteConverter.ToUInt32(parentFrame.Data, packetStartIndex+8);
            this.dataOffsetByteCount=(byte)(4*(parentFrame.Data[packetStartIndex+12]>>4));
            if(dataOffsetByteCount<20)
                parentFrame.Errors.Add(new Frame.Error(parentFrame, packetStartIndex+12, packetStartIndex+12, "Too small defined TCP Data Offset : "+parentFrame.Data[packetStartIndex+12]));
            if(dataOffsetByteCount>60)
                parentFrame.Errors.Add(new Frame.Error(parentFrame, packetStartIndex+12, packetStartIndex+12, "Too large defined TCP Data Offset : "+parentFrame.Data[packetStartIndex+12]));
            this.flags=new Flags(parentFrame.Data[packetStartIndex+13]);
            this.windowSize=ByteConverter.ToUInt16(parentFrame.Data, packetStartIndex+14);
            this.checksum=ByteConverter.ToUInt16(parentFrame.Data, packetStartIndex+16);
            //kolla checksumman!?
            //kolla options
            if(dataOffsetByteCount>20)
                optionList=GetOptionList(packetStartIndex+20);
            else
                optionList=new List<KeyValuePair<OptionKinds,byte[]>>();

        }
        /*
        internal override List<Packet> GetSubPackets() {
            List<Packet> subPackets=new List<Packet>();
            if(PacketStartIndex+dataOffsetByteCount<PacketEndIndex) {
                RawPacket rawPacket=new RawPacket(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex);
                //ParentFrame.Packets.Add(rawPacket.PacketStartIndex+internetHeaderLength, rawPacket);
                subPackets.Add(rawPacket);
                subPackets.AddRange(rawPacket.GetSubPackets());
            }
            return subPackets;
        }
        */
        internal override IEnumerable<AbstractPacket> GetSubPackets() {
            if(PacketStartIndex+dataOffsetByteCount<PacketEndIndex) {
                AbstractPacket packet;
                if(sourcePort==80 || destinationPort==80 || sourcePort==8080 || destinationPort==8080)
                    try {
                        packet=new HttpPacket(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex);
                    }
                    catch(Exception e) {
                        //ParentFrame.Errors.Add(new Frame.Error(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex, "Cannot parse HTTP packet ("+e.Message+")"));
                        packet=new RawPacket(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex);
                    }
                else if(sourcePort==137 || destinationPort==137) {
                    try {
                        packet=new NetBiosNameServicePacket(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex);
                    }
                    catch(Exception e) {
                        packet=new RawPacket(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex);
                    }
                }
                else if(sourcePort==139 || destinationPort==139)
                    try {
                        packet=new NetBiosSessionService(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex, false);
                    }
                    catch(Exception e) {
                        //ParentFrame.Errors.Add(new Frame.Error(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex, "Cannot parse NetBiosSessionService packet ("+e.Message+")"));
                        packet=new RawPacket(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex);
                    }
                else if(sourcePort==445 || destinationPort==445)
                    try {
                        packet=new NetBiosSessionService(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex, true);
                    }
                    catch(Exception e) {
                        //ParentFrame.Errors.Add(new Frame.Error(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex, "Cannot parse NetBiosSessionService packet ("+e.Message+")"));
                        packet=new RawPacket(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex);
                    }
                else
                    packet=new RawPacket(ParentFrame, PacketStartIndex+dataOffsetByteCount, PacketEndIndex);

                yield return packet;
                foreach(AbstractPacket subPacket in packet.GetSubPackets())
                    yield return subPacket;
                //return rawPacket.GetSubPackets();
            }
        }

        internal byte[] GetTcpPacketPayloadData() {
            byte[] data=new byte[PacketEndIndex-PacketStartIndex-dataOffsetByteCount+1];
            for(int i=0; i<data.Length; i++)
                data[i]=base.ParentFrame.Data[PacketStartIndex+dataOffsetByteCount+i];
            return data;
        }


        /*
        public override string ToString() {
            StringBuilder sbDataReceived=new StringBuilder();
            char c;
            for(int byteCounter=54; byteCounter+EthernetFrameOffset<data.Length && byteCounter<256; byteCounter++) {
                c=(char)data[EthernetFrameOffset+byteCounter];
                if(Char.IsLetterOrDigit(c) || Char.IsSymbol(c) || Char.IsWhiteSpace(c))
                    sbDataReceived.Append(c);
                else
                    sbDataReceived.Append(".");
            }
            return sbDataReceived.ToString();
        }
         * */
        private List<KeyValuePair<OptionKinds,byte[]>> GetOptionList(int startIndex){
        //private SortedList<OptionKinds,byte[]> GetOptionList(int startIndex){
            //SortedList<OptionKinds,byte[]> optionList=new SortedList<OptionKinds,byte[]>();
            List<KeyValuePair<OptionKinds, byte[]>> optionList=new List<KeyValuePair<OptionKinds, byte[]>>();
            //System.Collections.Generic.KeyValuePair<OptionKinds,byte[]> options=new KeyValuePair<OptionKinds,byte[]>();
            int i=0;
            while(startIndex+i<this.PacketStartIndex+this.dataOffsetByteCount && this.ParentFrame.Data[startIndex+i]!=(byte)OptionKinds.EndOfOptionList) {
                if(this.ParentFrame.Data[startIndex+i]>8) {
                    ParentFrame.Errors.Add(new Frame.Error(ParentFrame, startIndex+i, startIndex+i, "TCP Option Kind is larger than 8 (it is:"+this.ParentFrame.Data[startIndex+i]+")"));
                    break;
                }
                else{
                    OptionKinds kind=(OptionKinds)this.ParentFrame.Data[startIndex+i];
                    if(kind==OptionKinds.EndOfOptionList) {
                        //if(!optionList.ContainsKey(kind))
                        optionList.Add(new KeyValuePair<OptionKinds,byte[]>(kind, null));
                        i++;
                        break;
                    }
                    else if(kind==OptionKinds.NoOperation) {
                        //if(!optionList.ContainsKey(kind))
                        optionList.Add(new KeyValuePair<OptionKinds,byte[]>(kind, null));
                        i++;
                    }
                    else{
                        byte optionLength=this.ParentFrame.Data[startIndex+i+1];
                        if(optionLength<2) {
                            ParentFrame.Errors.Add(new Frame.Error(ParentFrame, startIndex+i+1, startIndex+i+1, "TCP Option Length ("+optionLength+") is shorter than 2"));
                            optionLength=2;
                        }
                        else if(startIndex+i+optionLength>this.PacketStartIndex+this.dataOffsetByteCount) {
                            ParentFrame.Errors.Add(new Frame.Error(ParentFrame, startIndex+i+1, startIndex+i+1, "TCP Option Length ("+optionLength+") makes option end outside TCP Data Offset ("+this.dataOffsetByteCount+")"));
                            optionLength=(byte)(this.PacketStartIndex+this.dataOffsetByteCount-startIndex-i);
                        }
                        else if(optionLength>44) {
                            ParentFrame.Errors.Add(new Frame.Error(ParentFrame, startIndex+i+1, startIndex+i+1, "TCP Option Length ("+optionLength+") is longer than 44"));
                            optionLength=44;
                        }
                        byte[] optionData=new byte[optionLength-2];
                        Array.Copy(this.ParentFrame.Data, startIndex+i+2, optionData, 0, optionLength-2);
                        //if(!optionList.ContainsKey(kind))
                        optionList.Add(new KeyValuePair<OptionKinds,byte[]>(kind, optionData));

                        i+=optionLength;
                    }
                   
                }
                
            }
            return optionList;
        }



    }
}
