//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially
using System;
using System.Collections.Generic;
using System.Text;

namespace NetworkMiner.Packets {

    //http://en.wikipedia.org/wiki/Http
    //http://tools.ietf.org/html/rfc2616
    //http://tools.ietf.org/html/rfc2617 (HTTP Authentication)
    class HttpPacket : AbstractPacket{
        internal enum RequestMethods { GET, HEAD, POST, PUT, DELETE, TRACE, OPTIONS, CONNECT, none }
        internal enum ContentEncodings { Gzip, Compress, Deflate, Identity }//Identity is default

        //general variables
        private bool messageTypeIsRequest;
        private List<string> headerFields;
        private byte[] messageBody;//the content

        //request variables
        private RequestMethods requestMethod;//this one is "none" unless the message is a request message
        private string requestedHost;
        private string requestedFileName;
        private string userAgentBanner;//client or requester web browser or spider. See: http://en.wikipedia.org/wiki/User_agent

        //reply variables
        private string serverBanner;//server or reply web server. See: http://www.blackhat.com/presentations/bh-asia-02/bh-asia-02-grossman.pdf or http://www.blackhat.com/presentations/bh-usa-03/bh-us-03-shah/bh-us-03-shah.ppt
        private string contentType;
        private int contentLength;//defined in # bytes for "Content-Lenght" as in RFC2616
        private string contentEncoding;//this could be for example GZIP, see: http://www.faqs.org/rfcs/rfc1952.html
        private string transferEncoding;//if encoding is "chunked" (such as for www.ripe.net) I will have to deal with: http://tools.ietf.org/html/rfc2616#section-3.6.1 Why can't people simply rely on TCP sequencing???
        private string wwwAuthenticateBasicRealm;
        private string authorizationCredentialsUsername;
        private string authorizationCredentailsPassword;

        internal bool MessageTypeIsRequest { get { return this.messageTypeIsRequest; } }
        internal RequestMethods RequestMethod { get { return this.requestMethod; } }
        internal string RequestedHost { get { return requestedHost; } }
        internal string RequestedFileName { get { return this.requestedFileName; } }
        internal string UserAgentBanner { get { return this.userAgentBanner; } }
        internal string ServerBanner { get { return this.serverBanner; } }
        internal string ContentType { get { return this.contentType; } }
        internal int ContentLength { get { return this.contentLength; } }
        internal string ContentEncoding { get { return this.contentEncoding; } }
        internal string TransferEncoding { get { return this.transferEncoding; } }
        internal string WwwAuthenticateBasicRealm { get { return this.wwwAuthenticateBasicRealm; } }
        internal string AuthorizationCredentialsUsername { get { return this.authorizationCredentialsUsername; } }
        internal string AuthorizationCredentialsPassword { get { return this.authorizationCredentailsPassword; } }

        internal byte[] MessageBody { get { return this.messageBody; } }

        internal HttpPacket(Frame parentFrame, int packetStartIndex, int packetEndIndex)
            : base(parentFrame, packetStartIndex, packetEndIndex, "HTTP") {
            /*
             * 
             *         generic-message = start-line
             *             *(message-header CRLF)
             *             CRLF
             *             [ message-body ]
             * 
             *         start-line      = Request-Line | Status-Line
             * 
             * */
            this.headerFields=new List<string>();
            this.requestedHost=null;
            this.requestedFileName=null;
            this.userAgentBanner=null;
            this.serverBanner=null;
            this.contentType=null;
            this.contentLength=-1;//instead of null
            this.contentEncoding=null;
            this.transferEncoding=null;
            this.wwwAuthenticateBasicRealm=null;
            this.authorizationCredentialsUsername=null;
            this.authorizationCredentailsPassword=null;


            int dataIndex=packetStartIndex;

            //a start-line
            string startLine=ByteConverter.ReadLine(parentFrame.Data, ref dataIndex);
            if(startLine.Length>255)
                throw new Exception("HTTP start line is longer than 255 bytes. Probably a false HTTP positive");

            if(dataIndex>packetEndIndex)
                throw new Exception("HTTP start line ends after packet end...");

            if(startLine.StartsWith("GET")) {
                this.messageTypeIsRequest=true;
                this.requestMethod=RequestMethods.GET;
            }
            else if(startLine.StartsWith("HEAD")) {
                this.messageTypeIsRequest=true;
                this.requestMethod=RequestMethods.HEAD;
            }
            else if(startLine.StartsWith("POST")) {
                this.messageTypeIsRequest=true;
                this.requestMethod=RequestMethods.POST;
            }
            else if(startLine.StartsWith("PUT")) {
                this.messageTypeIsRequest=true;
                this.requestMethod=RequestMethods.PUT;
            }
            else if(startLine.StartsWith("DELETE")) {
                this.messageTypeIsRequest=true;
                this.requestMethod=RequestMethods.DELETE;
            }
            else if(startLine.StartsWith("TRACE")) {
                this.messageTypeIsRequest=true;
                this.requestMethod=RequestMethods.TRACE;
            }
            else if(startLine.StartsWith("OPTIONS")) {
                this.messageTypeIsRequest=true;
                this.requestMethod=RequestMethods.OPTIONS;
            }
            else if(startLine.StartsWith("CONNECT")) {
                this.messageTypeIsRequest=true;
                this.requestMethod=RequestMethods.CONNECT;
            }
            else if(startLine.StartsWith("HTTP")) {
                this.messageTypeIsRequest=false;
                this.requestMethod=RequestMethods.none;
            }
            else
                throw new Exception("Incorrect HTTP Message Type or Request Method");
         
            //zero or more header fields (also known as "headers")
            for(string headerLine=ByteConverter.ReadLine(parentFrame.Data, ref dataIndex); headerLine.Length>0;headerLine=ByteConverter.ReadLine(parentFrame.Data, ref dataIndex))
                this.headerFields.Add(headerLine);
            //an empty line (i.e., a line with nothing preceding the CRLF)

            //possibly a message-body
            if(dataIndex<packetEndIndex){//we have a body!
                this.messageBody=new byte[packetEndIndex-dataIndex+1];
                for(int i=0; i<this.messageBody.Length; i++)
                    this.messageBody[i]=parentFrame.Data[dataIndex+i];
            }
            else
                this.messageBody=null;


            //now extract some interresting information from the packet
            if(this.messageTypeIsRequest && this.requestMethod==RequestMethods.GET) {
                string fileURI=startLine.Substring(4, startLine.Length-4);
                if(fileURI.Contains(" HTTP")){
                    fileURI=fileURI.Substring(0, fileURI.IndexOf(" HTTP"));
                }
                if(fileURI.Length>0){//If it is the index-file the URI will be just "/"
                    this.requestedFileName=fileURI;
                }



            }

            //look through headers
            foreach(string headerField in headerFields)
                if(headerField.StartsWith("Host: ")) {//look for the host
                    this.requestedHost=headerField.Substring(6);
                    base.Attributes.Add("Requested Host", headerField.Substring(6));
                }
                else if(headerField.StartsWith("User-Agent: ")) {
                    this.userAgentBanner=headerField.Substring(12);
                    base.Attributes.Add("User-Agent", this.userAgentBanner=headerField.Substring(12));
                }
                else if(headerField.StartsWith("Server: ")) {
                    this.serverBanner=headerField.Substring(8);
                    this.Attributes.Add("Server banner", this.serverBanner=headerField.Substring(8));
                }
                else if(headerField.StartsWith("Content-Type: "))
                    this.contentType=headerField.Substring(14);
                else if(headerField.StartsWith("Content-Length: "))
                    this.contentLength=Convert.ToInt32(headerField.Substring(16));
                else if(headerField.StartsWith("Content-Encoding: "))
                    this.contentEncoding=headerField.Substring(18);
                else if(headerField.StartsWith("Transfer-Encoding: "))
                    this.transferEncoding=headerField.Substring(19);
                else if(headerField.StartsWith("WWW-Authenticate: Basic realm="))
                    this.wwwAuthenticateBasicRealm=headerField.Substring(31, headerField.Length-32);
                else if(headerField.StartsWith("Authorization: Basic ")) {
                    try {
                        string base64string=headerField.Substring(21);
                        Byte[] bArray=Convert.FromBase64String(base64string);
                        StringBuilder sb=new StringBuilder(bArray.Length);
                        foreach(byte b in bArray)
                            sb.Append((char)b);
                        //string s=System.Text.Encoding.Unicode.GetString(bArray);
                        string s=sb.ToString();
                        if(s.Contains(":")) {
                            this.authorizationCredentialsUsername=s.Substring(0, s.IndexOf(':'));
                            if(s.IndexOf(':')+1<s.Length)
                                this.authorizationCredentailsPassword=s.Substring(s.IndexOf(':')+1);
                            else
                                this.authorizationCredentailsPassword="";
                        }
                    }
                    catch(Exception e) {
                        parentFrame.Errors.Add(new Frame.Error(parentFrame, PacketStartIndex, packetEndIndex, "Cannot parse credentials in HTTP Authorization ("+e.Message+")"));
                    }
                }
                
                        


        }

        internal override IEnumerable<AbstractPacket> GetSubPackets() {
            //Do nothing, no known sub packets... well or can I define HTTP data chunks as a sub packet?
            yield break;
        }
    }
}
