//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Text;

namespace NetworkMiner.Packets {
    /// <summary>
    /// This class shall be implemented by all packet classes such as PacketIP, PacketEthernet, PacketTCP etc.
    /// </summary>
    public abstract class AbstractPacket {
        private Frame parentFrame;
        private string packetTypeDescription;
        private int packetStartIndex;//first byte position
        private int packetEndIndex;//last byte position
        private NameValueCollection attributes;

        internal Frame ParentFrame { get { return parentFrame; } }
        internal string PacketTypeDescription { get { return packetTypeDescription; } }
        internal int PacketStartIndex { get { return packetStartIndex; } }
        internal int PacketEndIndex {
            get { return packetEndIndex; }
            set { if(value<=packetEndIndex && value>=packetStartIndex) packetEndIndex=value; }//endIndex can only be lowered
        }
        internal int PacketByteCount { get { return packetEndIndex-packetStartIndex+1; } }
        internal NameValueCollection Attributes { get { return attributes; } }
        //internal abstract System.Windows.Forms.TreeNode {

        internal AbstractPacket(Frame parentFrame, int packetStartIndex, int packetEndIndex, string packetTypeDescription) {
            this.parentFrame=parentFrame;
            this.packetStartIndex=packetStartIndex;
            this.packetEndIndex=packetEndIndex;
            this.packetTypeDescription=packetTypeDescription;
            this.attributes=new NameValueCollection();
            if(packetStartIndex>packetEndIndex) {
                string errorMsg="PacketStartIndex ("+packetStartIndex+") > PacketEndIndex ("+packetEndIndex+")";
                parentFrame.Errors.Add(new Frame.Error(parentFrame, packetEndIndex, packetEndIndex, errorMsg));
                throw new Exception(errorMsg);
            }
        }

        /// <summary>
        /// Creates a byte array containing everything from the start to the end of the packet (including payload).
        /// </summary>
        /// <returns></returns>
        internal byte[] GetPacketData() {
            byte[] data=new byte[this.PacketByteCount];
            for(int i=0; i<data.Length; i++)
                data[i]=parentFrame.Data[this.packetStartIndex+i];
            return data;
        }

        //internal abstract List<Packet> GetSubPackets();
        internal abstract IEnumerable<AbstractPacket> GetSubPackets();
        
    }
}
