//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially
using System;
using System.Net;
using System.Collections.Generic;
using System.Text;

namespace NetworkMiner {

    internal delegate void NewNetworkHostHandler(NetworkHost host);

    class PacketHandler {
        private NetworkHostList networkHostList;
        private int nFramesReceived, nBytesReceived;
        private NetworkMinerForm parentForm;
        private LatestFramesQueue receivedFramesQueue;
        private List<Fingerprints.IOsFingerprintCollection> osFingerprintCollectionList;
        private CleartextDictionary.WordDictionary dictionary;
        private Dictionary<int, NetworkSession> networkSessionDictionary;
        private FileTransfer.FileStreamAssemblerPool fileStreamAssemblerPool;
        private List<FileTransfer.ReconstructedFile> reconstructedFileList;
        private SortedList<NetworkCredential, NetworkCredential> credentialList;

        internal CleartextDictionary.WordDictionary Dictionary { set { this.dictionary=value; } }
        internal List<Fingerprints.IOsFingerprintCollection> OsFingerprintCollectionList { get { return this.osFingerprintCollectionList; } }
        internal ICollection<NetworkHost> DetectedHosts { get { return networkHostList.Hosts; } }
        internal NetworkMinerForm ParentForm{get{return this.parentForm;}}

        internal void ResetCapturedData(){
            this.networkHostList.Clear();
            nFramesReceived=0;
            nBytesReceived=0;
            receivedFramesQueue.Clear();
            this.fileStreamAssemblerPool.ClearAll();
            this.reconstructedFileList.Clear();
            this.credentialList.Clear();
        }

        internal PacketHandler(NetworkMinerForm parentForm) : this(parentForm, new Fingerprints.IOsFingerprintCollection[0]){
        }

        internal PacketHandler(NetworkMinerForm parentForm, IEnumerable<Fingerprints.IOsFingerprintCollection> osFingerprintCollectionEnumeration) {
            this.parentForm=parentForm;
            this.networkHostList=new NetworkHostList();
            this.nFramesReceived=0;
            this.nBytesReceived=0;
            this.receivedFramesQueue=new LatestFramesQueue(256);
            this.dictionary=new CleartextDictionary.WordDictionary();
            this.osFingerprintCollectionList=new List<NetworkMiner.Fingerprints.IOsFingerprintCollection>(osFingerprintCollectionEnumeration);
            this.networkSessionDictionary=new Dictionary<int, NetworkSession>();
            this.fileStreamAssemblerPool=new FileTransfer.FileStreamAssemblerPool(this, 100);
            this.reconstructedFileList=new List<NetworkMiner.FileTransfer.ReconstructedFile>();
            this.credentialList=new SortedList<NetworkCredential, NetworkCredential>();
        }


        private IEnumerable<string> GetCleartextWords(Packets.AbstractPacket packet) {
            StringBuilder sb=null;//new StringBuilder();
            for(int i=packet.PacketStartIndex; i<=packet.PacketEndIndex; i++) {
                if(dictionary.IsLetter(packet.ParentFrame.Data[i])) {
                    if(sb==null)
                        sb=new StringBuilder(Convert.ToString((char)packet.ParentFrame.Data[i]));
                    else
                        sb.Append((char)packet.ParentFrame.Data[i]);
                }
                else {
                    if(sb!=null) {
                        if(dictionary.HasWord(sb.ToString()))
                            yield return sb.ToString();
                        sb=null;
                    }
                }
            }
        }


        /// <summary>
        /// Callback method to receive packets from a sniffer
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="packet"></param>
        internal void SnifferPacketReceived(object sender, NetworkWrapper.PacketReceivedEventArgs packet) {
            Frame receivedFrame=null;
            if(packet.PacketType==NetworkWrapper.PacketReceivedEventArgs.PacketTypes.Ethernet2Packet) {
                receivedFrame=new Frame(packet.Timestamp, packet.Data, typeof(Packets.Ethernet2Packet), nFramesReceived);
            }
            else if(packet.PacketType==NetworkWrapper.PacketReceivedEventArgs.PacketTypes.IPv4Packet) {
                receivedFrame=new Frame(packet.Timestamp, packet.Data, typeof(Packets.IPv4Packet), nFramesReceived);
            }
            
            if(receivedFrame!=null) {
                this.nFramesReceived++;
                parentForm.ShowReceivedFrame(receivedFrame);
                foreach(Frame.Error error in receivedFrame.Errors)
                    parentForm.ShowError("Error: "+error.ToString()+" (frame nr: "+nFramesReceived+")");

                //this.SetLabelText(this.nFramesLabel, nFramesReceived.ToString());
                this.nBytesReceived+=receivedFrame.Data.Length;
                //this.SetLabelText(this.nBytesLabel, nBytesReceived.ToString("n0")+" B");

                receivedFramesQueue.Enqueue(receivedFrame);
                //StringBuilder logText=new StringBuilder("\r\n");

                Packets.Ethernet2Packet ethernet2Packet=null;
                Packets.ArpPacket arpPacket=null;
                Packets.IPv4Packet ipv4Packet=null;
                Packets.TcpPacket tcpPacket=null;
                Packets.UdpPacket udpPacket=null;
                Packets.DnsPacket dnsPacket=null;
                Packets.RawPacket rawPacket=null;
                Packets.NetBiosNameServicePacket netBiosNameServicePacket=null;
                Packets.NetBiosDatagramServicePacket netBiosDatagramSericePacket=null;
                Packets.HttpPacket httpPacket=null;
                Packets.CifsPacket.AbstractSmbCommand smbCommandPacket=null;
                Packets.UpnpPacket upnpPacket=null;
                foreach(Packets.AbstractPacket p in receivedFrame.PacketList.Values) {
                    if(p.GetType()==typeof(Packets.IPv4Packet))
                        ipv4Packet=(Packets.IPv4Packet)p;
                    else if(p.GetType()==typeof(Packets.TcpPacket))
                        tcpPacket=(Packets.TcpPacket)p;
                    else if(p.GetType()==typeof(Packets.UdpPacket))
                        udpPacket=(Packets.UdpPacket)p;
                    else if(p.GetType()==typeof(Packets.Ethernet2Packet))
                        ethernet2Packet=(Packets.Ethernet2Packet)p;
                    else if(p.GetType()==typeof(Packets.ArpPacket))
                        arpPacket=(Packets.ArpPacket)p;
                    else if(p.GetType()==typeof(Packets.DnsPacket))
                        dnsPacket=(Packets.DnsPacket)p;
                    else if(p.GetType()==typeof(Packets.RawPacket))
                        rawPacket=(Packets.RawPacket)p;
                    else if(p.GetType()==typeof(Packets.NetBiosNameServicePacket))
                        netBiosNameServicePacket=(Packets.NetBiosNameServicePacket)p;
                    else if(p.GetType()==typeof(Packets.NetBiosDatagramServicePacket))
                        netBiosDatagramSericePacket=(Packets.NetBiosDatagramServicePacket)p;
                    else if(p.GetType()==typeof(Packets.HttpPacket))
                        httpPacket=(Packets.HttpPacket)p;
                    else if(p.GetType().IsSubclassOf(typeof(Packets.CifsPacket.AbstractSmbCommand)))//denna funkar!!
                        smbCommandPacket=(Packets.CifsPacket.AbstractSmbCommand)p;
                    else if(p.GetType()==typeof(Packets.UpnpPacket))
                        upnpPacket=(Packets.UpnpPacket)p;
          

                }
                if(arpPacket!=null) {
                    ExtractArpData(ethernet2Packet, arpPacket);
                }


                NetworkPacket networkPacket=null;
                if(ipv4Packet!=null) {
                    NetworkHost sourceHost, destinationHost;
                    //source
                    //if(!networkHostList.ContainsKey(ipv4Packet.SourceIPAddress)) {
                    if(networkHostList.ContainsIP(ipv4Packet.SourceIPAddress))
                        sourceHost=networkHostList.GetNetworkHost(ipv4Packet.SourceIPAddress);
                    else {
                        sourceHost=new NetworkHost(ipv4Packet.SourceIPAddress);
                        networkHostList.Add(sourceHost);
                        parentForm.ShowDetectedHost(sourceHost);
                    }
                    if(networkHostList.ContainsIP(ipv4Packet.DestinationIPAddress))
                        destinationHost=networkHostList.GetNetworkHost(ipv4Packet.DestinationIPAddress);
                    else {
                        destinationHost=new NetworkHost(ipv4Packet.DestinationIPAddress);
                        networkHostList.Add(destinationHost);
                        parentForm.ShowDetectedHost(destinationHost);
                    }
                    //we now have sourceHost and destinationHost
                    networkPacket=new NetworkPacket(sourceHost, destinationHost, ipv4Packet);

                    if(ethernet2Packet!=null) {
                        sourceHost.MacAddress=ethernet2Packet.SourceMACAddress;
                        destinationHost.MacAddress=ethernet2Packet.DestinationMACAddress;
                    }

                    if(tcpPacket!=null) {
                        networkPacket.SetTcpData(tcpPacket);

                        if(tcpPacket.FlagBits.Synchronize) {//SYN or SYN+ACK
                            AddHostOperatingSystem(sourceHost, ipv4Packet, tcpPacket);
                        }
                        else {//we might have some interresting information in a TCP session!
                            if(networkPacket.SourceHost.NetworkServiceMetadataList.ContainsKey((ushort)networkPacket.SourceTcpPort))//see if we have a service with outgoing data...
                                networkPacket.SourceHost.NetworkServiceMetadataList[(ushort)networkPacket.SourceTcpPort].OutgoingTraffic.AddTcpPayloadData(tcpPacket.GetTcpPacketPayloadData());
                            if(networkPacket.DestinationHost.NetworkServiceMetadataList.ContainsKey((ushort)networkPacket.DestinationTcpPort))//see if we have a service with incoming data...
                                networkPacket.DestinationHost.NetworkServiceMetadataList[(ushort)networkPacket.DestinationTcpPort].IncomingTraffic.AddTcpPayloadData(tcpPacket.GetTcpPacketPayloadData());

                            if(httpPacket!=null) {//I am assuming that the HTTP packet will be inside a TCP packet
                                if(httpPacket.MessageTypeIsRequest) {
                                    //HTTP request


                                    if(httpPacket.UserAgentBanner!=null && httpPacket.UserAgentBanner.Length>0)
                                        sourceHost.AddHttpUserAgentBanner(httpPacket.UserAgentBanner);
                                    if(httpPacket.RequestedHost!=null && httpPacket.RequestedHost.Length>0)
                                        destinationHost.AddHostName(httpPacket.RequestedHost);
                                    if(httpPacket.AuthorizationCredentialsUsername!=null){
                                        NetworkCredential nc=new NetworkCredential(destinationHost, httpPacket.PacketTypeDescription, httpPacket.AuthorizationCredentialsUsername, httpPacket.AuthorizationCredentialsPassword);
                                        this.AddCredential(nc);
                                    }

                                    
                                    if(httpPacket.RequestMethod==Packets.HttpPacket.RequestMethods.GET) {

                                        //file transfer stuff
                                        string fileUri=httpPacket.RequestedFileName;
                                        if(fileUri.Contains("?"))
                                            fileUri=fileUri.Substring(0, fileUri.IndexOf('?'));

                                        char[] separators={ '/' };
                                        string[] uriParts=fileUri.Split(separators);
                                        string filename;
                                        string fileLocation="";

                                        if(fileUri.EndsWith("/")){
                                            filename="index.html";
                                            for(int i=0; i<uriParts.Length; i++)
                                                if(uriParts[i].Length>0)
                                                    fileLocation+="/"+uriParts[i];
                                        }
                                        else {
                                            filename=uriParts[uriParts.Length-1];
                                            for(int i=0; i<uriParts.Length-1; i++)
                                                if(uriParts[i].Length>0)
                                                    fileLocation+="/"+uriParts[i];
                                        }

                                        //I will have to switch source and destination host here since this is only the request, not the actual file transfer!
                                        try {
                                            //FileTransfer.FileStreamAssembler assembler=new FileTransfer.FileStreamAssembler(this.fileStreamAssemblerPool, destinationHost, tcpPacket.DestinationPort, sourceHost, tcpPacket.SourcePort, tcpPacket!=null, FileTransfer.FileStreamTypes.HttpGetNormal, filename, fileLocation, httpPacket.ContentLength, httpPacket.RequestedFileName);
                                            FileTransfer.FileStreamAssembler assembler=new FileTransfer.FileStreamAssembler(this.fileStreamAssemblerPool, destinationHost, tcpPacket.DestinationPort, sourceHost, tcpPacket.SourcePort, tcpPacket!=null, FileTransfer.FileStreamTypes.HttpGetNormal, filename, fileLocation, httpPacket.RequestedFileName);
                                            fileStreamAssemblerPool.Add(assembler);
                                        }
                                        catch(Exception e) {
                                            this.parentForm.ShowError("Error creating assembler for HTTP file transfer: "+e.Message+" (frame nr: "+nFramesReceived+")");
                                            
                                        }
                                    }
                                }
                                else {//reply
                                    if(httpPacket.ServerBanner!=null && httpPacket.ServerBanner.Length>0)
                                        sourceHost.AddHttpServerBanner(httpPacket.ServerBanner, tcpPacket.SourcePort);
                                    if(httpPacket.WwwAuthenticateBasicRealm!=null && httpPacket.WwwAuthenticateBasicRealm.Length>0)
                                        sourceHost.AddHostName(httpPacket.WwwAuthenticateBasicRealm);
                                    if(fileStreamAssemblerPool.ContainsAssembler(sourceHost, tcpPacket.SourcePort, destinationHost, tcpPacket.DestinationPort, true)){
                                        FileTransfer.FileStreamAssembler assembler=fileStreamAssemblerPool.GetAssembler(sourceHost, tcpPacket.SourcePort, destinationHost, tcpPacket.DestinationPort, true);
                                        if(httpPacket.ContentLength>0) {
                                            assembler.FileContentLength=httpPacket.ContentLength;
                                            assembler.FileSegmentRemainingBytes=httpPacket.ContentLength;//we get the whole file in one segment (one serie of TCP packets)
                                        }
                                        //see if the filename is missing its extension
                                        if((!assembler.Filename.Contains(".") || assembler.Filename.EndsWith(".php") || assembler.Filename.EndsWith(".asp") || assembler.Filename.EndsWith(".aspx")) && httpPacket.ContentType!=null && httpPacket.ContentType.Contains("/") && httpPacket.ContentType.IndexOf('/')<httpPacket.ContentType.Length-1) {
                                        //if(httpPacket.ContentType!=null && httpPacket.ContentType.Contains("/") && httpPacket.ContentType.IndexOf('/')<httpPacket.ContentType.Length-1) {
                                            string extension=httpPacket.ContentType.Substring(httpPacket.ContentType.IndexOf('/')+1);
                                            if(extension.Contains(";"))
                                                extension=extension.Substring(0, extension.IndexOf(";"));
                                            if(extension.Length>0 && !assembler.Filename.EndsWith("."+extension))
                                                assembler.Filename=assembler.Filename+"."+extension;
                                        }
                                        if(httpPacket.TransferEncoding=="chunked")
                                            assembler.FileStreamType=FileTransfer.FileStreamTypes.HttpGetChunked;
                                        if(httpPacket.ContentEncoding!=null && httpPacket.ContentEncoding.Length>0)
                                            if(httpPacket.ContentEncoding.Equals("gzip"))//I'll only care aboute gzip for now
                                                assembler.ContentEncoding=Packets.HttpPacket.ContentEncodings.Gzip;
                                        if(httpPacket.MessageBody!=null && httpPacket.MessageBody.Length>0)
                                            if(assembler.FileStreamType==FileTransfer.FileStreamTypes.HttpGetChunked || httpPacket.MessageBody.Length<=assembler.FileSegmentRemainingBytes)
                                                assembler.AddData(httpPacket.MessageBody, tcpPacket.SequenceNumber);
                                    }

                                }
                            }//end of HTTP
                            else if(smbCommandPacket!=null) {//we have an SMB command!
                                if(smbCommandPacket.GetType()==typeof(Packets.CifsPacket.NegotiateProtocolRequest)) {
                                    Packets.CifsPacket.NegotiateProtocolRequest request=(Packets.CifsPacket.NegotiateProtocolRequest)smbCommandPacket;
                                    sourceHost.AcceptedSmbDialectsList=request.DialectList;
                                }
                                else if(smbCommandPacket.GetType()==typeof(Packets.CifsPacket.NegotiateProtocolResponse)) {
                                    Packets.CifsPacket.NegotiateProtocolResponse reply=(Packets.CifsPacket.NegotiateProtocolResponse)smbCommandPacket;
                                    if(destinationHost.AcceptedSmbDialectsList!=null && destinationHost.AcceptedSmbDialectsList.Count>reply.DialectIndex)
                                        sourceHost.PreferredSmbDialect=destinationHost.AcceptedSmbDialectsList[reply.DialectIndex];
                                        //sourceHost.ExtraDetailsList.Add("Preferred SMB dialect", destinationHost.AcceptedSmbDialectsList[reply.DialectIndex]);
                                }
                                else if(smbCommandPacket.GetType()==typeof(Packets.CifsPacket.SetupAndXRequest)) {
                                    Packets.CifsPacket.SetupAndXRequest request=(Packets.CifsPacket.SetupAndXRequest)smbCommandPacket;
                                    if(request.NativeLanManager!=null && request.NativeLanManager.Length>0) {
                                        if(sourceHost.ExtraDetailsList.ContainsKey("SMB Native LAN Manager"))
                                            sourceHost.ExtraDetailsList["SMB Native LAN Manager"]=request.NativeLanManager;
                                        else
                                            sourceHost.ExtraDetailsList.Add("SMB Native LAN Manager", request.NativeLanManager);
                                    }
                                    
                                    if(request.NativeOs!=null && request.NativeOs.Length>0) {
                                        if(sourceHost.ExtraDetailsList.ContainsKey("SMB Native OS"))
                                            sourceHost.ExtraDetailsList["SMB Native OS"]=request.NativeOs;
                                        else
                                            sourceHost.ExtraDetailsList.Add("SMB Native OS", request.NativeOs);
                                    }
                                }
                                else if(smbCommandPacket.GetType()==typeof(Packets.CifsPacket.SetupAndXResponse)) {
                                    Packets.CifsPacket.SetupAndXResponse response=(Packets.CifsPacket.SetupAndXResponse)smbCommandPacket;
                                    if(response.NativeLanManager!=null && response.NativeLanManager.Length>0) {
                                        if(sourceHost.ExtraDetailsList.ContainsKey("SMB Native LAN Manager"))
                                            sourceHost.ExtraDetailsList["SMB Native LAN Manager"]=response.NativeLanManager;
                                        else
                                            sourceHost.ExtraDetailsList.Add("SMB Native LAN Manager", response.NativeLanManager);
                                    }

                                    if(response.NativeOs!=null && response.NativeOs.Length>0) {
                                        if(sourceHost.ExtraDetailsList.ContainsKey("SMB Native OS"))
                                            sourceHost.ExtraDetailsList["SMB Native OS"]=response.NativeOs;
                                        else
                                            sourceHost.ExtraDetailsList.Add("SMB Native OS", response.NativeOs);
                                    }
                                }
                                else if(smbCommandPacket.GetType()==typeof(Packets.CifsPacket.NTCreateAndXRequest)) {
                                    Packets.CifsPacket.NTCreateAndXRequest request=(Packets.CifsPacket.NTCreateAndXRequest)smbCommandPacket;
                                    string filename, filePath;

                                    if(request.Filename.EndsWith("\0"))
                                        filename=request.Filename.Remove(request.Filename.Length-1);
                                    else
                                        filename=request.Filename;
                                    if(filename.Contains("\\")) {
                                        filePath=filename.Substring(0, filename.LastIndexOf("\\"));
                                        filename=filename.Substring(filename.LastIndexOf("\\")+1);
                                    }
                                    else
                                        filePath="\\";

                                    try {
                                        FileTransfer.FileStreamAssembler assembler=new FileTransfer.FileStreamAssembler(this.fileStreamAssemblerPool, destinationHost, tcpPacket.DestinationPort, sourceHost, tcpPacket.SourcePort, tcpPacket!=null, FileTransfer.FileStreamTypes.SMB, filename, filePath, request.Filename);
                                        if(fileStreamAssemblerPool.ContainsAssembler(assembler))
                                            fileStreamAssemblerPool.Remove(assembler, true);
                                        fileStreamAssemblerPool.Add(assembler);
                                    }
                                    catch(Exception e) {
                                        this.parentForm.ShowError("Error creating assembler for SMB file transfer: "+e.Message+" (frame nr: "+nFramesReceived+")");

                                    }
                                }
                                else if(!smbCommandPacket.ParentCifsPacket.FlagsResponse && fileStreamAssemblerPool.ContainsAssembler(destinationHost, tcpPacket.DestinationPort, sourceHost, tcpPacket.SourcePort, true)) {
                                    //Request
                                    FileTransfer.FileStreamAssembler assembler=fileStreamAssemblerPool.GetAssembler(destinationHost, tcpPacket.DestinationPort, sourceHost, tcpPacket.SourcePort, true);

                                    if(smbCommandPacket.GetType()==typeof(Packets.CifsPacket.CloseRequest)) {
                                        fileStreamAssemblerPool.Remove(assembler, true);
                                    }
                                }
                                else if(smbCommandPacket.ParentCifsPacket.FlagsResponse && fileStreamAssemblerPool.ContainsAssembler(sourceHost, tcpPacket.SourcePort, destinationHost, tcpPacket.DestinationPort, true)) {
                                    //Response
                                    FileTransfer.FileStreamAssembler assembler=fileStreamAssemblerPool.GetAssembler(sourceHost, tcpPacket.SourcePort, destinationHost, tcpPacket.DestinationPort, true);

                                    if(smbCommandPacket.GetType()==typeof(Packets.CifsPacket.NTCreateAndXResponse)) {
                                        Packets.CifsPacket.NTCreateAndXResponse response=(Packets.CifsPacket.NTCreateAndXResponse)smbCommandPacket;
                                        int fileLength=(int)response.EndOfFile;//yes, I know I will not be able to store big files now... but an int as length is really enough!
                                        if(fileLength<=0)
                                            fileStreamAssemblerPool.Remove(assembler, true);
                                        else
                                            assembler.FileContentLength=fileLength;
                                    }
                                    else if(smbCommandPacket.GetType()==typeof(Packets.CifsPacket.ReadAndXResponse)) {
                                        Packets.CifsPacket.ReadAndXResponse response=(Packets.CifsPacket.ReadAndXResponse)smbCommandPacket;
                                        assembler.FileSegmentRemainingBytes+=response.DataLength;//setting this one so that it ca receive more bytes
                                        assembler.AddData(response.GetFileData(), tcpPacket.SequenceNumber);
                                        //response.DataOffset
                                    }

                                }

                            }//end of SMB/CIFS command
                            else if(fileStreamAssemblerPool.ContainsAssembler(sourceHost, tcpPacket.SourcePort, destinationHost, tcpPacket.DestinationPort, true)) {
                                //this could be any type of TCP packet... but probably part of a file transfer...
                                FileTransfer.FileStreamAssembler assembler=fileStreamAssemblerPool.GetAssembler(sourceHost, tcpPacket.SourcePort, destinationHost, tcpPacket.DestinationPort, true);
                                if(assembler.FileStreamType==FileTransfer.FileStreamTypes.HttpGetChunked || assembler.FileSegmentRemainingBytes>=tcpPacket.PayloadDataLength)
                                    assembler.AddData(tcpPacket);
                                //I dont need to look for Push flag and call assembler.FinishAssembling() since that can be done in the AddData function
                            }


                        }//end TCP session

                    }
                    else if(udpPacket!=null) {
                        networkPacket.SetUdpData(udpPacket);
                    }

                    sourceHost.AddTtl(ipv4Packet.TimeToLive);
                    if(sourceHost.TtlDistance==byte.MaxValue) {//maxValue=default if no TtlDistance exists
                        foreach(Fingerprints.IOsFingerprintCollection fingerprinter in this.osFingerprintCollectionList)
                            sourceHost.AddProbableTtlDistance(fingerprinter.GetTtlDistance(ipv4Packet.TimeToLive));
                    }

                    if(dnsPacket!=null) {
                        //ExtractDnsData(dnsPacket);
                        if(dnsPacket.Flags.Response) {
                            System.Collections.Specialized.NameValueCollection cNamePointers=new System.Collections.Specialized.NameValueCollection();
                            foreach(Packets.DnsPacket.ResourceRecord r in dnsPacket.AnswerRecords) {
                                if(r.IP!=null) {
                                    if(!networkHostList.ContainsIP(r.IP)) {
                                        NetworkHost host=new NetworkHost(r.IP);
                                        host.AddHostName(r.DNS);
                                        networkHostList.Add(host);
                                        parentForm.ShowDetectedHost(host);
                                    }
                                    else
                                        networkHostList.GetNetworkHost(r.IP).AddHostName(r.DNS);
                                    if(cNamePointers[r.DNS]!=null)
                                        networkHostList.GetNetworkHost(r.IP).AddHostName(cNamePointers[r.DNS]);
                                }
                                else if(r.Type==(ushort)Packets.DnsPacket.RRTypes.CNAME) {
                                    cNamePointers.Add(r.PrimaryName, r.DNS);
                                }
                            }
                        }
                        else {//DNS request
                            if(dnsPacket.QueriedDnsName!=null && dnsPacket.QueriedDnsName.Length>0)
                                sourceHost.AddQueriedDnsName(dnsPacket.QueriedDnsName);
                        }
                    }
                    else if(netBiosDatagramSericePacket!=null) {
                        if(netBiosDatagramSericePacket.SourceNetBiosName!=null && netBiosDatagramSericePacket.SourceNetBiosName.Length>0)
                            sourceHost.AddHostName(netBiosDatagramSericePacket.SourceNetBiosName);
                    }
                    else if(netBiosNameServicePacket!=null) {
                        //now check for some interesting details in the packet
                        if(netBiosNameServicePacket.QueriedNetBiosName!=null)
                            sourceHost.AddQueriedNetBiosName(netBiosNameServicePacket.QueriedNetBiosName);
                        if(netBiosNameServicePacket.AnsweredNetBiosName!=null) {
                            if(networkHostList.ContainsIP(netBiosNameServicePacket.AnsweredIpAddress))
                                networkHostList.GetNetworkHost(netBiosNameServicePacket.AnsweredIpAddress).AddHostName(netBiosNameServicePacket.AnsweredNetBiosName);
                        }
                    }
                    else if(upnpPacket!=null) {
                        if(upnpPacket.FieldList.Count>0) {
                            if(sourceHost.UniversalPlugAndPlayFieldList==null)
                                sourceHost.UniversalPlugAndPlayFieldList=new SortedList<string, string>();
                            foreach(string field in upnpPacket.FieldList)
                                if(!sourceHost.UniversalPlugAndPlayFieldList.ContainsKey(field))
                                    sourceHost.UniversalPlugAndPlayFieldList.Add(field, field);
                        }
                    }

                }//end of IP packet if clause
                if(rawPacket!=null) {
                    SetPacketPayload(networkPacket, rawPacket);
                }
                if(networkPacket!=null) {
                    networkPacket.SourceHost.SentPackets.Add(networkPacket);
                    networkPacket.DestinationHost.ReceivedPackets.Add(networkPacket);

                    //get NetworkSession object
                    NetworkSession networkSession=GetNetworkSession(networkPacket);
                    
                    if(networkSession!=null){
                        bool synAckPreviouslyReceived=networkSession.SynAckPacketReceived;
                        //add packet to session
                        if(networkSession.TryAddPacket(networkPacket)) {
                            if(!synAckPreviouslyReceived && networkSession.SynAckPacketReceived) {//see if there has been created a new session
                                //add session to NetworkHost objects
                                networkSession.ServerHost.IncomingSessionList.Add(networkSession);
                                networkSession.ClientHost.OutgoingSessionList.Add(networkSession);
                            }
                        }
                    }
                }

            }

        }

        private NetworkSession GetNetworkSession(NetworkPacket packet) {
            if(packet.SourceTcpPort!=null && packet.DestinationTcpPort!=null) {//TCP packet
                if(packet.TcpSynFlag) {
                    if(!packet.TcpSynAckFlag) {//first SYN packet
                        NetworkSession session=new NetworkSession(packet);
                        int sessionHash=session.GetHashCode();
                        if(this.networkSessionDictionary.ContainsKey(sessionHash))//earlier session with the same IP's and port numbers
                            this.networkSessionDictionary[sessionHash]=session;//replace the old with the new in the dictionary only!
                        else
                            this.networkSessionDictionary.Add(session.GetHashCode(), session);
                        return session;
                    }
                    else {//SYN+ACK packet (server -> client)
                        NetworkHost clientHost=packet.DestinationHost;
                        NetworkHost serverHost=packet.SourceHost;
                        ushort clientTcpPort=(ushort)packet.DestinationTcpPort;
                        ushort serverTcpPort=(ushort)packet.SourceTcpPort;

                        int key=NetworkSession.GetHashCode(clientHost, serverHost, clientTcpPort, serverTcpPort);
                        if(this.networkSessionDictionary.ContainsKey(key)) {
                            NetworkSession session=networkSessionDictionary[key];
                            if(session.SynPacketReceived && !session.SynAckPacketReceived)//we now have an established session
                                return session;
                            else
                                return null;
                        }
                        else//stray ACK packet
                            return null;
                    }
                }
                else {//No SYN or ACK packet. There should be an active session


                    int clientToServerKey=NetworkSession.GetHashCode(packet.SourceHost, packet.DestinationHost, (ushort)packet.SourceTcpPort, (ushort)packet.DestinationTcpPort);
                    int serverToClientKey=NetworkSession.GetHashCode(packet.DestinationHost, packet.SourceHost, (ushort)packet.DestinationTcpPort, (ushort)packet.SourceTcpPort);

                    if(this.networkSessionDictionary.ContainsKey(clientToServerKey))//see if packet is client to server
                        return this.networkSessionDictionary[clientToServerKey];
                    else if(this.networkSessionDictionary.ContainsKey(serverToClientKey))//see if packet is server to client
                        return this.networkSessionDictionary[serverToClientKey];
                    else//no such session....
                        return null;
                }
            }
            else//no TCP packet
                return null;
        }

        internal void AddReconstructedFile(FileTransfer.ReconstructedFile file) {
            this.reconstructedFileList.Add(file);
            parentForm.ShowReconstructedFile(file);
        }
        internal void AddCredential(NetworkCredential credential) {
            if(!credentialList.ContainsKey(credential)) {
                this.credentialList.Add(credential, credential);
                parentForm.ShowCredential(credential, credentialList.Count);
            }
        }
        private void AddHostOperatingSystem(NetworkHost host, Packets.IPv4Packet ipv4Packet, Packets.TcpPacket tcpPacket) {
            if(tcpPacket.FlagBits.Synchronize) {
                foreach(Fingerprints.IOsFingerprintCollection fingerprinter in this.osFingerprintCollectionList) {
                    string[] operatingSystems=fingerprinter.GetOperatingSystems(ipv4Packet, tcpPacket);
                    if(operatingSystems!=null && operatingSystems.Length>0) {
                        foreach(string os in operatingSystems)
                            host.AddProbableOs(os, 1.0/operatingSystems.Length);
                            //host.AddProbableOs(os+" ("+fingerprinter.ToString()+")", 1.0/operatingSystems.Length);//tis one is just for fingerpinting debug
                        host.AddProbableTtlDistance(fingerprinter.GetTtlDistance(ipv4Packet, tcpPacket));
                    }
                }
                //string[] operatingSystems=osFingerprintCollection.GetOperatingSystems(ipv4Packet, tcpPacket);
                //Fingerprints.EttarcapOsFingerprintCollection.OsFingerprintResult osFingerprintResult=osFingerprintCollection.GetOperatingSystem(ipv4Packet, tcpPacket);
                
            }
        }

        private void ExtractArpData(Packets.Ethernet2Packet ethernet2Packet, Packets.ArpPacket arpPacket) {
            if(arpPacket.SenderIPAddress!=null && ethernet2Packet!=null) {
                if(arpPacket.SenderHardwareAddress.Equals(ethernet2Packet.SourceMACAddress)) {
                    NetworkHost host=null;
                    if(!this.networkHostList.ContainsIP(arpPacket.SenderIPAddress)) {
                        host=new NetworkHost(arpPacket.SenderIPAddress);
                        host.MacAddress=arpPacket.SenderHardwareAddress;
                        networkHostList.Add(host);
                        parentForm.ShowDetectedHost(host);
                    }
                    if(host!=null)
                        host.AddQueriedIP(arpPacket.TargetIPAddress);

                }
                else {
                    parentForm.ShowError(
                        "Different source MAC addresses in Ethernet and ARP packet: "+
                                "Ethernet MAC="+ethernet2Packet.SourceMACAddress+
                                ", ARP MAC="+arpPacket.SenderHardwareAddress+
                                ", ARP IP="+arpPacket.SenderIPAddress);
                }

            }
        }

        /*private void ExtractDnsData(Packets.DnsPacket dnsPacket) {

        }*/

        private void SetPacketPayload(NetworkPacket networkPacket, Packets.AbstractPacket payloadPacket) {
            //check if there are words in the packet

            //see if the packet had some cleartext words just by accident
            int wordCharCount=0;
            foreach(string word in GetCleartextWords(payloadPacket))
                wordCharCount+=word.Length;
            if(networkPacket!=null)
                networkPacket.SetPayload(payloadPacket.PacketByteCount, wordCharCount);
            parentForm.ShowCleartextWords(GetCleartextWords(payloadPacket), wordCharCount, payloadPacket.PacketByteCount);
        }


    }
}
