//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially

using System;
using System.Collections.Generic;
using System.Text;

namespace NetworkMiner {
    internal class NetworkSession : IComparable {//only TCP sessions
        private NetworkHost clientHost, serverHost;
        private ushort clientTcpPort, serverTcpPort;
        private NetworkPacketList clientToServerPacketList, serverToClientPacketList;
        
        private uint serverTcpInitialSequenceNumber;
        private uint serverTcpLargestSequenceNumber;
        private uint clientTcpInitialSequenceNumber;
        private uint clientTcpLargestSequenceNumber;

        private DateTime synPacketTimestamp;
        private DateTime latestPacketTimestamp;

        private bool synPacketReceived;
        private bool synAckPacketReceived;

        //private Fingerprints.NetworkMinerSessionFingerprint sessionFingerprint;

        internal NetworkHost ClientHost { get { return this.clientHost; } }
        internal NetworkHost ServerHost { get { return this.serverHost; } }
        internal ushort ClientTcpPort { get { return this.clientTcpPort; } }
        internal ushort ServerTcpPort { get { return this.serverTcpPort; } }
        internal NetworkPacketList ClientToServerPacketList { get { return this.clientToServerPacketList; } }
        internal NetworkPacketList ServerToClientPacketList { get { return this.serverToClientPacketList; } }
        internal DateTime SessionStartTimestamp { get { return this.synPacketTimestamp; } }
        internal DateTime SessionEndTimestamp { get { return this.latestPacketTimestamp; } }
        internal bool SynPacketReceived { get { return this.synPacketReceived; } }
        internal bool SynAckPacketReceived { get { return this.synAckPacketReceived; } }

        /// <summary>
        /// Creates a new NetworkSession but does not add the SYN packet to the list.
        /// Use TryAddPacket(NetworkPacket packet) to add the SYN packet.
        /// </summary>
        /// <param name="synPacket">SYN packet from client to server</param>
        internal NetworkSession(NetworkPacket synPacket) {
            this.synAckPacketReceived=false;
            this.synPacketReceived=false;

            if(synPacket.SourceTcpPort==null)
                throw new Exception("No TCP source port in SYN packet");
            if(synPacket.DestinationTcpPort==null)
                throw new Exception("No TCP destination port in SYN packet");
            this.clientHost=synPacket.SourceHost;
            this.serverHost=synPacket.DestinationHost;
            this.clientTcpPort=(ushort)synPacket.SourceTcpPort;
            this.serverTcpPort=(ushort)synPacket.DestinationTcpPort;
            
            this.synPacketTimestamp=synPacket.Timestamp;

            this.clientToServerPacketList=new NetworkPacketList();
            this.serverToClientPacketList=new NetworkPacketList();
            //this.sessionFingerprint=new NetworkMiner.Fingerprints.NetworkServiceMetadata();
            
        }

        /*
        internal NetworkSession(NetworkPacket synPacket, NetworkPacket synAckPacket) {
            this.synAckPacketReceived=false;
            this.synPacketReceived=false;

            if(synPacket.SourceHost!=synAckPacket.DestinationHost)
                throw new Exception("SYN and SYN+ACK packet IPs don't match!");
            if(synPacket.DestinationHost!=synAckPacket.SourceHost)
                throw new Exception("SYN and SYN+ACK packet IPs don't match!");

            if(synAckPacket.SourceTcpPort==null)
                throw new Exception("No TCP port in SYN ACK packet");
            if(synPacket.SourceTcpPort!=synAckPacket.DestinationTcpPort)
                throw new Exception("SYN and SYN+ACK packet TCP ports don't match!");
            if(synPacket.DestinationTcpPort!=synAckPacket.SourceTcpPort)
                throw new Exception("SYN and SYN+ACK packet TCP ports don't match!");
            
            if(synPacket.SourceTcpPort==null)
                throw new Exception("No TCP source port in SYN packet");
            if(synPacket.DestinationTcpPort==null)
                throw new Exception("No TCP destination port in SYN packet");
            this.clientHost=synPacket.SourceHost;
            this.serverHost=synPacket.DestinationHost;
            this.clientTcpPort=(ushort)synPacket.SourceTcpPort;
            this.serverTcpPort=(ushort)synPacket.DestinationTcpPort;

            this.clientToServerPacketList=new NetworkPacketList();
            this.serverToClientPacketList=new NetworkPacketList();
        }*/


        public static int GetHashCode(NetworkHost clientHost, NetworkHost serverHost, ushort clientTcpPort, ushort serverTcpPort) {
            int cHash=clientHost.IPAddress.GetHashCode()^clientTcpPort;
            int sHash=serverHost.IPAddress.GetHashCode()^serverTcpPort;
            return cHash^(sHash<<16)^(sHash>>16);
        }

        public override int GetHashCode() {
            return GetHashCode(this.clientHost, this.serverHost, this.clientTcpPort, this.serverTcpPort);
            /*
            int cHash=this.clientHost.IPAddress.GetHashCode()^this.clientTcpPort;
            int sHash=this.serverHost.IPAddress.GetHashCode()^this.serverTcpPort;
            return cHash^(sHash<<16)^(sHash>>16);*/
        }
        public override string ToString() {
            return "Server: "+this.serverHost.ToString()+" TCP "+this.serverTcpPort+" ("+this.ServerToClientPacketList.Count+" packets sent), Client: "+this.clientHost.ToString()+" TCP "+this.ClientTcpPort+" ("+this.clientToServerPacketList.Count+" packets sent), Session start: "+this.synPacketTimestamp+", Session end: "+this.latestPacketTimestamp;
        }

        internal bool TryAddPacket(NetworkPacket packet) {
            if(!this.synPacketReceived){
                if(packet.TcpSynFlag)
                    this.synPacketReceived=true;
                else
                    return false;
            }
            else if(!this.synAckPacketReceived) {
                if(packet.TcpSynAckFlag)
                    this.synAckPacketReceived=true;
                else
                    return false;
            }

            if(packet.SourceHost==this.clientHost){//client -> server
                if(packet.DestinationHost!=serverHost)
                    return false;
                if(packet.SourceTcpPort!=clientTcpPort)
                    return false;
                if(packet.DestinationTcpPort!=serverTcpPort)
                    return false;
                //the packet is now trusted to be OK
                this.clientToServerPacketList.Add(packet);
                this.latestPacketTimestamp=packet.Timestamp;
            }
            else if(packet.SourceHost==this.serverHost){//server -> client
                if(packet.DestinationHost!=clientHost)
                    return false;
                if(packet.SourceTcpPort!=serverTcpPort)
                    return false;
                if(packet.DestinationTcpPort!=clientTcpPort)
                    return false;
                //the packet is now trusted to be OK
                this.serverToClientPacketList.Add(packet);
                this.latestPacketTimestamp=packet.Timestamp;
            }
            else//unknown direction
                return false;

            //If we've come this far the packet should be allright for the networkSession
            if(this.synPacketReceived && this.synAckPacketReceived && !this.ServerHost.NetworkServiceMetadataList.ContainsKey(this.serverTcpPort))
                this.ServerHost.NetworkServiceMetadataList.Add(this.serverTcpPort, new NetworkServiceMetadata(this.ServerHost, this.serverTcpPort));
            //unfortunently I don't have the full content TCP data available here so I'll have to add the TCP content to the NetworkServiceMetadata somewhere else...
            //Hint: it's being done in NetworkPacket / SetTcpData(Packets.TcpPacket tcpPacket)
            return true;
        }

        /*
        //This function shall not store the TCP packet, instead it should mine information from the TCP packet!
        internal void AddTcpData(Packets.TcpPacket tcpPacket){
            this.sessionFingerprint.AddFingerprintData(tcpPacket);
            //is there maybe something else smart I could do with full TCP packet data?
        }
        */

        #region IComparable Members

        public int CompareTo(NetworkSession session) {
            if(this.clientHost.CompareTo(session.clientHost)!=0)
                return this.clientHost.CompareTo(session.clientHost);
            else if(this.serverHost.CompareTo(session.serverHost)!=0)
                return this.serverHost.CompareTo(session.serverHost);
            else if(this.clientTcpPort!=session.clientTcpPort)
                return this.clientTcpPort-session.clientTcpPort;
            else if(this.serverTcpPort!=session.serverTcpPort)
                return this.serverTcpPort-session.serverTcpPort;
            else if(this.SessionStartTimestamp.CompareTo(session.SessionStartTimestamp)!=0)
                return this.SessionStartTimestamp.CompareTo(session.SessionStartTimestamp);
            else
                return 0;
        }
        public int CompareTo(object obj) {
            NetworkSession s=(NetworkSession)obj;
            return CompareTo(s);
        }

        #endregion
    }
}
