//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
//using MyClasses;

namespace NetworkMiner {
    public partial class NetworkMinerForm : Form {



        private int nFilesReceived;

        private NetworkWrapper.ISniffer sniffer;

        //private Fingerprints.MacCollection macCollection;//I'm only using this in NetworkHostTreeNode
        private ImageList imageList;
        //private Fingerprints.OsFingerprintCollection osFingerprintCollection;

        private PacketHandler packetHandler;

        delegate void AppendTextCallback(string text);//used for EventLog and Callback
        delegate void GenericStringCallback(string text);
        delegate void AppendFrameToTreeViewCallback(Frame frame);
        delegate void AppendNetworkHostToTreeViewCallback(NetworkHost networkHost);
        delegate void SetControlTextCallback(Control control, string text);
        delegate void AppendFileToFileList(FileTransfer.ReconstructedFile file);
        delegate void AppendCredentialToCredentialList(NetworkCredential credential, int credentialCount);

        private CleartextDictionary.WordDictionary dictionary;


        public NetworkMinerForm() {

            InitializeComponent();
            this.imageList=new ImageList();
            imageList.ImageSize=new Size(64, 64);
            this.imagesListView.LargeImageList=imageList;

            List<Fingerprints.IOsFingerprintCollection> osDetectors=new List<NetworkMiner.Fingerprints.IOsFingerprintCollection>();
            osDetectors.Add(new NetworkMiner.Fingerprints.EttarcapOsFingerprintCollection("etter.finger.os"));//, NetworkMiner.Fingerprints.EttarcapOsFingerprintCollection.OsFingerprintFileFormat.Ettercap)
            osDetectors.Add(new NetworkMiner.Fingerprints.P0fOsFingerprintCollection("p0f.fp", "p0fa.fp"));
            
            this.packetHandler=new PacketHandler(this, osDetectors);

            this.nFilesReceived=0;
            
            List<NetworkWrapper.IAdapter> networkAdapters=new List<NetworkWrapper.IAdapter>();
            networkAdapters.Add(new NetworkWrapper.NullAdapter());
            //Get the WinPcap adapters
            try {
                networkAdapters.AddRange(NetworkWrapper.WinPCapAdapter.GetAdapters());
            }
            catch(Exception ex) {
                MessageBox.Show("Unable to load WinPcap adapter (available at www.winpcap.org)\nApplication will run using only Raw Socket connections\n\n"+ex.Message, "NetworkMiner", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            //get all SocketAdapters
            networkAdapters.AddRange(NetworkWrapper.SocketAdapter.GetAdapters());
            
            this.networkAdaptersComboBox.DataSource=networkAdapters;

            this.framesTreeView.Nodes.Clear();


            this.networkHostTreeView.ImageList=new ImageList();
            //AddImage(networkHostTreeView, "white", "white.gif");
            AddImage(networkHostTreeView, "white", "white.jpg");//first is default
            AddImage(networkHostTreeView, "iana", "iana.jpg");
            AddImage(networkHostTreeView, "computer", "computer.jpg");
            AddImage(networkHostTreeView, "multicast", "multicast.jpg");
            AddImage(networkHostTreeView, "broadcast", "broadcast.jpg");

            AddImage(networkHostTreeView, "windows", "windows.jpg");
            AddImage(networkHostTreeView, "macos", "macos.jpg");
            //AddImage(networkHostTreeView, "unix", "unix.gif");
            AddImage(networkHostTreeView, "unix", "unix.jpg");
            AddImage(networkHostTreeView, "linux", "linux.jpg");
            AddImage(networkHostTreeView, "freebsd", "freebsd.jpg");
            AddImage(networkHostTreeView, "netbsd", "netbsd.jpg");
            AddImage(networkHostTreeView, "solaris", "solaris.jpg");

            AddImage(networkHostTreeView, "sent", "arrow_sent.jpg");
            AddImage(networkHostTreeView, "received", "arrow_received.jpg");
            AddImage(networkHostTreeView, "incoming", "arrow_incoming.jpg");
            AddImage(networkHostTreeView, "outgoing", "arrow_outgoing.jpg");

            AddImage(networkHostTreeView, "nic", "network_card.jpg");
            AddImage(networkHostTreeView, "details", "details.gif");

            this.networkHostTreeView.BeforeExpand+=new TreeViewCancelEventHandler(NetworkHostTreeView_BeforeExpand);
            //this.framesTreeView.BeforeExpand+=new TreeViewCancelEventHandler(TreeView_BeforeExpand);//test
            
            if(this.networkAdaptersComboBox.SelectedValue.GetType().Equals(typeof(NetworkWrapper.WinPCapAdapter)))
                this.sniffer=new NetworkWrapper.WinPCapSniffer((NetworkWrapper.WinPCapAdapter)this.networkAdaptersComboBox.SelectedValue);
            else if(this.networkAdaptersComboBox.SelectedValue.GetType().Equals(typeof(NetworkWrapper.SocketAdapter)))
                this.sniffer=new NetworkWrapper.SocketSniffer((NetworkWrapper.SocketAdapter)this.networkAdaptersComboBox.SelectedValue);
            else if(this.networkAdaptersComboBox.SelectedValue.GetType().Equals(typeof(NetworkWrapper.NullAdapter)))
                this.sniffer=null;
            else
                throw new Exception(""+this.networkAdaptersComboBox.SelectedValue.GetType().ToString());

            NetworkWrapper.WinPCapSniffer.PacketReceived+=new NetworkWrapper.PacketReceivedHandler(packetHandler.SnifferPacketReceived);
            NetworkWrapper.SocketSniffer.PacketReceived+=new NetworkWrapper.PacketReceivedHandler(packetHandler.SnifferPacketReceived);

            this.dictionary=new NetworkMiner.CleartextDictionary.WordDictionary();
            this.loadCleartextDictionary("all-words.txt");

            this.openFileDialog1.Filter="Pcap files (*.pcap, *.cap)|*.pcap;*.cap|NetworkMiner files (*.nmine)|*.nmine|All files (*.*)|*.*";
            this.openFileDialog1.FileName="";
        }

        private void AddImage(TreeView treeView, string key, string imageFileName) {
            
            treeView.ImageList.Images.Add(key, Image.FromFile(System.IO.Path.GetDirectoryName(System.Windows.Forms.Application.ExecutablePath)+"\\images\\"+imageFileName));
        }



        private void AddFrameToTreeView(Frame frame) {
            TreeNode frameNode=new TreeNode(frame.ToString());
            foreach(Packets.AbstractPacket p in frame.PacketList.Values) {
                TreeNode packetNode=new TreeNode(p.PacketTypeDescription+" ["+p.PacketStartIndex+"-"+p.PacketEndIndex+"]");
                foreach(string attributeKey in p.Attributes.AllKeys)
                    packetNode.Nodes.Add(attributeKey+" = "+p.Attributes[attributeKey]);
                frameNode.Nodes.Add(packetNode);
            }
            this.framesTreeView.Nodes.Add(frameNode);
        }
        private void AddNetworkHostToTreeView(NetworkHost networkHost) {
            NetworkHostTreeNode treeNode=new NetworkHostTreeNode(networkHost);
            this.networkHostTreeView.Nodes.Add(treeNode);
            //this.nHostsLabel.Text=this.packetHandler.DetectedHosts.Count.ToString();
            this.SetControlText(this.tabPageDetectedHosts, "Detected Hosts ("+this.packetHandler.DetectedHosts.Count+")");
        }

        private void AddFileToFileList(FileTransfer.ReconstructedFile file) {

            ListViewItem item=new ListViewItem(
                new string[] {
                    file.FilePath,
                    file.SourceHost.ToString(),
                    file.SourcePortString,
                    file.DestinationHost.ToString(),
                    file.DestinationPortString,
                    file.FileStreamType.ToString(),
                    file.Filename,
                    file.FileSizeString,
                    file.Details
                });
            item.ToolTipText=item.Text;

            this.filesListView.Items.Add(item);

            this.nFilesReceived++;
            this.SetControlText(this.tabPageFiles, "Files ("+this.nFilesReceived+")");

            try {
                if(file.IsImage())
                    AddImageToImageList(file, new Bitmap(file.FilePath));
                else if(file.IsIcon())
                    AddImageToImageList(file, new Icon(file.FilePath).ToBitmap());
            }
            catch(Exception e) {
                this.ShowError("Error: Exception when loading image \""+file.Filename+"\". "+e.Message);
            }

        }
        private void AddCredentialToCredentialList(NetworkCredential credential, int credentialCount) {
            string validCredential="Unknown";
            if(credential.IsProvenValid)
                validCredential="Yes";
            ListViewItem item=new ListViewItem(
                new string[] {
                    credential.Host.ToString(),
                    credential.ProtocolString,
                    credential.Username,
                    credential.Password,
                    validCredential
                });
            item.ToolTipText=item.Text;
            this.credentialsListView.Items.Add(item);
            this.SetControlText(this.tabPageCredentials, "Credentials ("+credentialCount+")");
        }

        private void AddImageToImageList(FileTransfer.ReconstructedFile file, Bitmap bitmapImage) {
            this.imageList.Images.Add(new Bitmap(bitmapImage));//I do the new in order to release the file handle for the original file
            ListViewItem item=this.imagesListView.Items.Add(file.Filename+"\n"+bitmapImage.Width+"x"+bitmapImage.Height+", "+file.FileSizeString, imageList.Images.Count-1);
            item.ToolTipText="Source: "+file.SourceHost+"\nDestination: "+file.DestinationHost+"\nReconstructed file path: "+file.FilePath;

            this.SetControlText(this.tabPageImages, "Images ("+this.imageList.Images.Count+")");
        }

        //test
        void NetworkHostTreeView_BeforeExpand(object sender, TreeViewCancelEventArgs e) {
            //try {
            if(e.Node is IBeforeExpand) {
                IBeforeExpand expandable=(IBeforeExpand)e.Node;
                expandable.BeforeExpand();
            }
            
        }

        //see: ms-help://MS.VSCC.v80/MS.MSDN.v80/MS.VisualStudio.v80.en/dv_fxmclictl/html/138f38b6-1099-4fd5-910c-390b41cbad35.htm
        //or: http://www.osix.net/modules/article/?id=832
        internal void ShowReceivedFrame(Frame frame) {

            //this.SetControlText(this.nFramesLabel, frame.FrameNumber.ToString());
            this.SetControlText(this.tabPageReceivedFrames, "Received Frames ("+frame.FrameNumber+")");

            //this.SetControlText(this.nBytesLabel, frame.Data.Length.ToString("n0")+" B");
            foreach(Frame.Error error in frame.Errors) {
                ShowError(error.ToString()+" (frame nr: "+frame.FrameNumber+")");
            }

            int maxFramesToShow=100;//this should maybe be a changeable class variable...
            if(this.framesTreeView.Nodes.Count < maxFramesToShow) {
                AppendFrameToTreeViewCallback treeViewCallback = new AppendFrameToTreeViewCallback(AddFrameToTreeView);
                this.Invoke(treeViewCallback, frame);
            }
        }

        internal void ShowCleartextWords(IEnumerable<string> words, int wordCharCount, int totalByteCount) {
            if(this.showDetectedCleartextCheckBox.Checked && wordCharCount*10 > totalByteCount) {
                StringBuilder sb=new StringBuilder();
                foreach(string word in words) {
                    sb.Append(word);
                    sb.Append(" ");
                }
                AppendTextCallback cleartextCallback = new AppendTextCallback(this.cleartextTextBox.AppendText);
                this.Invoke(cleartextCallback, sb.ToString());
            }
        }

        internal void ShowReconstructedFile(FileTransfer.ReconstructedFile file) {
            AppendFileToFileList newFileCallback=new AppendFileToFileList(AddFileToFileList);
            this.Invoke(newFileCallback, file);
        }
        internal void ShowCredential(NetworkCredential credential, int credentialCount) {
            AppendCredentialToCredentialList newCredentialCallback=new AppendCredentialToCredentialList(AddCredentialToCredentialList);
            //do I have to make an object[] here???
            this.Invoke(newCredentialCallback, credential, credentialCount);
        }
        internal void ShowDetectedHost(NetworkHost host) {
            AppendNetworkHostToTreeViewCallback newHostCallback=new AppendNetworkHostToTreeViewCallback(AddNetworkHostToTreeView);
            this.Invoke(newHostCallback, host);
        }
        internal void ShowError(string errorText) {
            AppendTextCallback logCallback = new AppendTextCallback(this.eventLog.AppendText);
            this.Invoke(logCallback, "\r\n["+DateTime.Now.ToString()+"] Error : "+errorText);
        }

        //http://www.codeproject.com/csharp/begininvoke.asp?df=100&forumid=178776&exp=0&select=1519433
        private void SetControlText(Control c, string text) {
            if(this.InvokeRequired) {
                object[] args=new object[2];
                args[0]=c;
                args[1]=text;
                this.BeginInvoke(new SetControlTextCallback(SetControlText), args);
            }
            else {
                c.Text=text;
            }
        }

        private void startButton_Click(object sender, EventArgs e) {
            networkAdaptersComboBox.Enabled=false;
            startButton.Enabled=false;
            sniffer.StartSniffing();
            

        }



        private void stopButton_Click(object sender, EventArgs e) {
            if(sniffer!=null)
                sniffer.StopSniffing();
            networkAdaptersComboBox.Enabled=true;
            //if(networkAdaptersComboBox
            if(!this.networkAdaptersComboBox.SelectedValue.GetType().Equals(typeof(NetworkWrapper.NullAdapter)))
                startButton.Enabled=true;
            //eventLog.Clear();
        }

        private void button3_Click(object sender, EventArgs e) {
            eventLog.Clear();
        }

        private void networkAdaptersComboBox_SelectedIndexChanged(object sender, EventArgs e) {
            if(this.networkAdaptersComboBox.SelectedValue.GetType().Equals(typeof(NetworkWrapper.WinPCapAdapter))) {
                this.sniffer=new NetworkWrapper.WinPCapSniffer((NetworkWrapper.WinPCapAdapter)this.networkAdaptersComboBox.SelectedValue);
                this.startButton.Enabled=true;
            }
            else if(this.networkAdaptersComboBox.SelectedValue.GetType().Equals(typeof(NetworkWrapper.SocketAdapter))) {
                try {
                    this.sniffer=new NetworkWrapper.SocketSniffer((NetworkWrapper.SocketAdapter)this.networkAdaptersComboBox.SelectedValue);
                    this.startButton.Enabled=true;
                }
                catch(System.Net.Sockets.SocketException ex) {
                    MessageBox.Show("Administrator rights are needed in order to use Socket connections \n"+ex.Message, "NetworkMiner", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this.networkAdaptersComboBox.SelectedIndex=0;
                    this.startButton.Enabled=false;
                }
            }
            else if(this.networkAdaptersComboBox.SelectedValue.GetType().Equals(typeof(NetworkWrapper.NullAdapter))) {
                this.startButton.Enabled=false;
            }
            else{
                throw new Exception(""+this.networkAdaptersComboBox.SelectedValue.GetType().ToString());
            }
            
        }

        private void button1_Click(object sender, EventArgs e) {
            this.framesTreeView.Nodes.Clear();
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e) {
            if(sniffer!=null)
                sniffer.StopSniffing();
            this.Close();
        }

        private void loadCleartextDictionaryToolStripMenuItem_Click(object sender, EventArgs e) {
            string dictionaryFile="all-words.txt";
            loadCleartextDictionary(dictionaryFile);
        }

        private void loadCleartextDictionary(string dictionaryFile) {
            if(this.InvokeRequired) {
                GenericStringCallback callback=new GenericStringCallback(loadCleartextDictionary);
                this.Invoke(callback,dictionaryFile);
            }
            else {
                CleartextDictionary.WordDictionary d=new NetworkMiner.CleartextDictionary.WordDictionary();
                d.LoadDictionaryFile(System.IO.Path.GetDirectoryName(System.Windows.Forms.Application.ExecutablePath)+"\\"+dictionaryFile);
                packetHandler.Dictionary=d;
                this.showDetectedCleartextCheckBox.Enabled=true;
                this.dictionaryNameLabel.Text=dictionaryFile;
            }
        }

        private void NetworkMinerForm_Load(object sender, EventArgs e) {

            //this.reportViewer1.RefreshReport();
        }

        private void crystalReportViewer1_Load(object sender, EventArgs e) {

        }

        private void button1_Click_1(object sender, EventArgs e) {
            this.cleartextTextBox.Clear();
        }

        private void detectedHostsTreeRebuildButton_Click(object sender, EventArgs e) {
            this.networkHostTreeView.Nodes.Clear();
            //there might be errors if a host is added to the list during the period when this for-loop runs
            try {
                foreach(NetworkHost host in this.packetHandler.DetectedHosts)
                    AddNetworkHostToTreeView(host);
            }
            catch {
                this.networkHostTreeView.Nodes.Clear();//just to show that something went wrong...
            }

        }

        private void reportViewer1_Load(object sender, EventArgs e) {
            
        }

        private void openToolStripMenuItem_Click(object sender, EventArgs e) {
            this.openFileDialog1.ShowDialog();
        }

        private void openFileDialog1_FileOk(object sender, CancelEventArgs e) {
            if(this.openFileDialog1.FileName.EndsWith(".pcap")) {
                //try {
                    PcapFileHandler.PcapFileReader pcapReader=new PcapFileHandler.PcapFileReader(this.openFileDialog1.FileName);

                    foreach(PcapFileHandler.PcapFileReader.PcapPacket pcapPacket in pcapReader.PacketEnumerator()) {
                        packetHandler.SnifferPacketReceived(this, new NetworkWrapper.PacketReceivedEventArgs(pcapPacket.Data, pcapPacket.Timestamp, NetworkWrapper.PacketReceivedEventArgs.PacketTypes.Ethernet2Packet));
                    }
               // }
                /*catch(Exception ex) {
                    MessageBox.Show(ex.Message, "NetworkMiner", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }*/
            }
            detectedHostsTreeRebuildButton_Click(sender, e);
        }

        private void resetCapturedDataToolStripMenuItem_Click(object sender, EventArgs e) {
            //this.packetHandler=new PacketHandler(this, packetHandler.OsFingerprintCollectionList);
            this.networkHostTreeView.Nodes.Clear();
            this.framesTreeView.Nodes.Clear();
            this.imagesListView.Items.Clear();
            this.imageList.Images.Clear();
            this.filesListView.Items.Clear();
            this.credentialsListView.Clear();
            this.cleartextTextBox.Clear();
            this.eventLog.Clear();

            this.SetControlText(this.tabPageDetectedHosts, "Detected Hosts");
            this.SetControlText(this.tabPageReceivedFrames, "Received Frames");
            this.SetControlText(this.tabPageFiles, "Files");
            this.SetControlText(this.tabPageImages, "Images");
            this.SetControlText(this.tabPageCredentials, "Credentials");

            this.packetHandler.ResetCapturedData();
            //this.nFramesReceived=0;
            //this.nBytesReceived=0;
            this.nFilesReceived=0;
        }

        private void aboutNetworkMinerToolStripMenuItem_Click(object sender, EventArgs e) {
            string text="NetworkMiner is an open source passive network sniffer/packet capturing tool for Windows with a great user interface. It can detect OS's, sessions, hostnames, open ports etc. without putting any traffic on the network. NetworkMiner can also parse PCAP files for off-line analysis.\n\nAuthor: Erik Hjelmvik, <hjelmvik@users.sourceforge.net> or <erik.hjelmvik@gmail.com>\nWebsite: http://sourceforge.net/projects/networkminer/\n\nBig cred to the creators of p0f and Ettercap for their passice OS fingerprint databases";
            MessageBox.Show(text, "About NetworkMiner", MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

    }
}