//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

namespace NetworkMiner {

    interface IBeforeExpand {
        void BeforeExpand();
    }

    class NetworkHostTreeNode : TreeNode, IBeforeExpand{
        private NetworkHost networkHost;

        internal NetworkHostTreeNode(NetworkHost networkHost) {
            

            this.networkHost=networkHost;
            this.Text=networkHost.ToString();
            this.Nodes.Add("dummie node");

            if(networkHost.SentPackets.Count==0)
                this.ForeColor=System.Drawing.Color.Gray;

            if(GetIpImageKey()!=null)
                this.ImageKey=GetIpImageKey();
            else if(GetOsImageKey()!=null)
                this.ImageKey=GetOsImageKey();
            else if(networkHost.SentPackets.Count>0)
                this.ImageKey="computer";
            else
                this.ImageKey="white";

            this.SelectedImageKey=this.ImageKey;


            this.ToolTipText="Sent packets: "+networkHost.SentPackets.Count+"\nReceived packets: "+networkHost.ReceivedPackets;

        }

        /// <summary>
        /// Returns the correct imageKey (based on IP) if one exists, otherwise null
        /// </summary>
        /// <returns></returns>
        private string GetIpImageKey() {
            if(networkHost.IpIsReserved)
                return "iana";
            else if(networkHost.IpIsMulticast)
                return "multicast";
            else if(networkHost.IpIsBroadcast)
                return "broadcast";
            else
                return null;
        }
        private string GetOsImageKey() {
            if(networkHost.OS==NetworkHost.OperatingSystemID.Windows)
                return "windows";
            else if(networkHost.OS==NetworkHost.OperatingSystemID.Linux)
                return "linux";
            else if(networkHost.OS==NetworkHost.OperatingSystemID.MacOS)
                return "macos";
            else if(networkHost.OS==NetworkHost.OperatingSystemID.UNIX)
                return "unix";
            else if(networkHost.OS==NetworkHost.OperatingSystemID.FreeBSD)
                return "freebsd";
            else if(networkHost.OS==NetworkHost.OperatingSystemID.NetBSD)
                return "netbsd";
            else if(networkHost.OS==NetworkHost.OperatingSystemID.Solaris)
                return "solaris";
            else if(networkHost.OS==NetworkHost.OperatingSystemID.Cisco)
                return "cisco";
            else
                return null;
        }

        public void BeforeExpand() {
            this.Nodes.Clear();
            TreeNode ipNode=new TreeNode("IP: "+networkHost.IPAddress.ToString());

            if(networkHost.IpIsReserved)
                ipNode.Text+=" (IANA Reserved)";
            if(networkHost.IpIsMulticast)
                ipNode.Text+=" (Multicast)";
            if(networkHost.IpIsBroadcast)
                ipNode.Text+=" (Broadcast)";
            if(GetIpImageKey()!=null)
                ipNode.ImageKey=GetIpImageKey();
            ipNode.SelectedImageKey=ipNode.ImageKey;
            this.Nodes.Add(ipNode);

            if(networkHost.MacAddress!=null)
                this.Nodes.Add("nic", "MAC: "+networkHost.MacAddress.ToString()+" ("+Fingerprints.MacCollection.GetMacCollection().GetMacVendor(networkHost.MacAddress)+")", "nic", "nic");
            else
                this.Nodes.Add("nic", "MAC: Unknown", "nic", "nic");
            this.Nodes.Add("Hostname: "+networkHost.HostName);
            if(GetOsImageKey()!=null)
                this.Nodes.Add("os", "OS: "+networkHost.OsDetails, GetOsImageKey(), GetOsImageKey());
            else
                this.Nodes.Add("OS: "+networkHost.OsDetails);
            if(networkHost.Ttl>0)
                this.Nodes.Add("TTL: "+networkHost.Ttl+" (distance: "+networkHost.TtlDistance+")");
            else
                this.Nodes.Add("TTL: Unknown");

            this.Nodes.Add(new ServiceListTreeNode(networkHost));

            //add packets
            this.Nodes.Add(new SentReceivedTreeNode(networkHost, true));
            this.Nodes.Add(new SentReceivedTreeNode(networkHost, false));

            //add sessions
            this.Nodes.Add(new SessionListTreeNode(networkHost, true));
            this.Nodes.Add(new SessionListTreeNode(networkHost, false));

            //Detailes
            if(networkHost.HostDetailCollection.Count>0) {
                this.Nodes.Add(new HostDetailListTreeNode(networkHost.HostDetailCollection));
            }

        }

        internal class ServiceListTreeNode : TreeNode, IBeforeExpand {
            private NetworkHost host;
            
            internal ServiceListTreeNode(NetworkHost host) {
                this.host=host;
                {
                    StringBuilder sb=new StringBuilder("Open TCP Ports:");
                    foreach(uint port in host.OpenTcpPorts)
                        sb.Append(" "+port);
                    this.Text=sb.ToString();
                    //this.Nodes.Add(sb.ToString());
                }
                if(host.NetworkServiceMetadataList.Count > 0)
                    this.Nodes.Add("dummie node");//so that it can be expanded
            }
            #region IBeforeExpand Members

            public void BeforeExpand() {
                this.Nodes.Clear();
                //List<NetworkSession> serviceList;
                //I want the services sorted by port, so I'll have to complicate things a bit
                //SortedList<string, TreeNode> sessionServerNodes=new SortedList<string, TreeNode>();
                foreach(NetworkServiceMetadata networkService in host.NetworkServiceMetadataList.Values) {

                    this.Nodes.Add("TCP "+networkService.TcpPort+" - "+
                        "Entropy (in \\ out): "+networkService.IncomingTraffic.CalculateEntropy().ToString("#.00")+" \\ "+networkService.OutgoingTraffic.CalculateEntropy().ToString("#.00")+//wildcard integers and 2 decimals?
                        " Typical data (in \\ out): "+networkService.IncomingTraffic.GetTypicalData()+" \\ "+networkService.OutgoingTraffic.GetTypicalData());

                }

            }

            #endregion
        }

        internal class SessionListTreeNode : TreeNode, IBeforeExpand{
            private  NetworkHost host;
            private bool sessionsAreIncoming;
            internal SessionListTreeNode(NetworkHost host, bool sessionsAreIncoming){
                this.host=host;
                this.sessionsAreIncoming=sessionsAreIncoming;
                if(sessionsAreIncoming){
                    int sessionCount=host.IncomingSessionList.Count;
                    this.Text="Incoming sessions: "+sessionCount;
                    if(sessionCount>0) {
                        this.Nodes.Add("dummie node");
                        this.ImageKey="incoming";
                    }
                }
                else{
                    int sessionCount=host.OutgoingSessionList.Count;
                    this.Text="Outgoing sessions: "+sessionCount;
                    if(sessionCount>0) {
                        this.Nodes.Add("dummie node");
                        this.ImageKey="outgoing";
                    }
                }
                this.SelectedImageKey=this.ImageKey;
            }

            public void BeforeExpand() {
                this.Nodes.Clear();
                List<NetworkSession> sessionList;
                if(sessionsAreIncoming)//host is server
                    sessionList=host.IncomingSessionList;
                else
                    sessionList=host.OutgoingSessionList;
                //I want the session servers sorted by IP and port, so I'll have to complicate things a bit
                SortedList<string, TreeNode> sessionServerNodes=new SortedList<string, TreeNode>();
                foreach(NetworkSession networkSession in sessionList) {
                    byte[] ipBytes=networkSession.ServerHost.IPAddress.GetAddressBytes();
                    string sessionServerKey="";
                    foreach(byte b in ipBytes)
                        sessionServerKey+=b.ToString("X2");
                    sessionServerKey+=networkSession.ServerTcpPort.ToString("X2");
                    string sessionServerString="Server: "+networkSession.ServerHost.ToString()+" TCP "+networkSession.ServerTcpPort;
                    if(!sessionServerNodes.ContainsKey(sessionServerKey))
                        sessionServerNodes.Add(sessionServerKey, new TreeNode(sessionServerString));
                    sessionServerNodes[sessionServerKey].Nodes.Add(networkSession.ToString());
                }
                foreach(TreeNode sessionServerNode in sessionServerNodes.Values)
                    this.Nodes.Add(sessionServerNode);
            }
        }

        internal class SentReceivedTreeNode : TreeNode, IBeforeExpand {
            private NetworkHost host;
            private bool hostIsSender;
            //private TreeNode treeNode;
            internal SentReceivedTreeNode(NetworkHost host, bool hostIsSender) {
                this.host=host;
                this.hostIsSender=hostIsSender;
                if(hostIsSender) {
                    this.Text="Sent: "+host.SentPackets.ToString();
                    if(host.SentPackets.Count>0) {
                        this.Nodes.Add("dummie node");//so that it can be expanded
                        this.ImageKey="sent";
                    }

                }
                else {//host is reciever
                    this.Text="Received: "+host.ReceivedPackets.ToString();
                    if(host.ReceivedPackets.Count>0) {
                        this.Nodes.Add("dummie node");//so that it can be expanded
                        this.ImageKey="received";
                    }
                }
                this.SelectedImageKey=this.ImageKey;
            }

            public void BeforeExpand() {
                //if(sourceHost.SentPackets.Count>0) {
                    this.Nodes.Clear();
                    if(hostIsSender) {
                        foreach(KeyValuePair<NetworkHost, NetworkPacketList> hostList in host.GetSentPacketListsPerDestinationHost()) {
                            this.Nodes.Add(new SubNetworkHostTreeNode(host, hostList.Key, hostList.Value));
                            //this.Nodes.Add("to "+hostList.Key.ToString()+": "+hostList.Value.ToString());
                        }
                    }
                    else {
                        foreach(KeyValuePair<NetworkHost, NetworkPacketList> hostList in host.GetReceivedPacketListsPerSourceHost()) {
                            this.Nodes.Add(new SubNetworkHostTreeNode(hostList.Key, host, hostList.Value));
                            //this.Nodes.Add("from "+hostList.Key.ToString()+": "+hostList.Value.ToString());

                        }
                    }
                //}
            }
            /// <summary>
            /// Holds information regarding sub hosts that the host has communicated with
            /// </summary>
            internal class SubNetworkHostTreeNode : TreeNode, IBeforeExpand {
                private NetworkHost sourceHost, destinationHost;
                private NetworkPacketList packetList;

                internal SubNetworkHostTreeNode(NetworkHost sourceHost, NetworkHost destinationHost, NetworkPacketList packetList) {
                    this.sourceHost=sourceHost;
                    this.destinationHost=destinationHost;
                    this.packetList=packetList;
                    this.Text=sourceHost.ToString()+" -> "+destinationHost.ToString()+" : "+packetList.ToString();
                    if(packetList.Count>0)
                        this.Nodes.Add("dummie node");//so that it can be expanded
                }

                public void BeforeExpand() {
                    this.Nodes.Clear();
                    ICollection<KeyValuePair<ushort[], NetworkPacketList>> tcpPortPairLists=packetList.GetSubsetPerTcpPortPair();
                    foreach(KeyValuePair<ushort[], NetworkPacketList> portPairList in tcpPortPairLists)
                        this.Nodes.Add("TCP: "+portPairList.Key[0]+" -> "+portPairList.Key[1]+" : "+portPairList.Value.ToString());

                    ICollection<KeyValuePair<ushort[], NetworkPacketList>> udpPortPairLists=packetList.GetSubsetPerUdpPortPair();
                    foreach(KeyValuePair<ushort[], NetworkPacketList> portPairList in udpPortPairLists)
                        this.Nodes.Add("UDP: "+portPairList.Key[0]+" -> "+portPairList.Key[1]+" : "+portPairList.Value.ToString());

                }

            }


        }

        internal class HostDetailListTreeNode : TreeNode, IBeforeExpand {
            private System.Collections.Specialized.NameValueCollection details;

            internal HostDetailListTreeNode(System.Collections.Specialized.NameValueCollection details) {
                this.details=details;
                this.Text="Host Details";
                this.ImageKey="details";
                this.Nodes.Add("dummie node");//so that it can be expanded
            }

            #region IBeforeExpand Members

            public void BeforeExpand() {
                this.Nodes.Clear();
                for(int i=0; i<details.Count; i++) {
                    this.Nodes.Add(details.Keys[i]+" : "+details[i]);
                }
            }

            #endregion
        }
        

    }
}
