//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially
using System;
using System.Collections.Generic;
using System.Text;
using System.Net;
using System.Net.NetworkInformation;

namespace NetworkMiner {
    class NetworkHost : IComparable{
        internal enum OperatingSystemID { Windows, Linux, UNIX, FreeBSD, NetBSD, Solaris, MacOS, Cisco, Other, Unknown }

        #region Basic Private Data
        private IPAddress ipAddress;
        private PhysicalAddress macAddress;//the MAC address the host is behind
        private System.Collections.Generic.List<string> hostNameList;//DNS, NetBIOS, HTTP-GET-host-value or similar

        private System.Collections.Generic.List<ushort> openTcpPortList;//I shouldn't really need this since the networkServiceList has almost the same information
        //private System.Collections.Generic.SortedList<ushort, NetworkService> networkServiceList;//All services running on different TCP ports of the host
        

        private System.Collections.Generic.SortedList<byte, int> ttlCount;//TTL values from host packets
        private System.Collections.Generic.SortedList<byte, int> ttlDistanceCount;//the guessed distance to the host
        
        private System.Collections.Generic.SortedList<string, double> operatingSystemCount;
        private NetworkPacketList sentPackets, receivedPackets;

        private List<NetworkSession> incomingSessionList, outgoingSessionList;
        private SortedList<ushort, NetworkServiceMetadata> networkServiceMetadataList;//the key is identified by the TCP port number of the service (on the server)
        #endregion

        #region Extra (detailed) Private Data
        private List<IPAddress> queriedIpList;
        private List<string> queriedNetBiosNameList;
        private List<string> queriedDnsNameList;
        private List<string> httpUserAgentBannerList;
        private List<string> httpServerBannerList;
        private SortedList<string, string> universalPlugAndPlayFieldList;

        private List<string> acceptedSmbDialectsList;
        private string preferredSmbDialect;

        private SortedList<string, string> extraDetailsList;//a simple list to store extra details about the host

        #endregion

        internal PhysicalAddress MacAddress { 
            get { return this.macAddress; }
            set { this.macAddress=value; }
        }
        internal IPAddress IPAddress { get { return this.ipAddress; } }
        internal string HostName {
            get {
                StringBuilder sb=new StringBuilder("");
                foreach(string hostname in hostNameList)
                    sb.Append(hostname+" ");
                return sb.ToString();
            }
        }
        internal ushort[] OpenTcpPorts { get { return openTcpPortList.ToArray();} }

        internal NetworkPacketList SentPackets { get { return sentPackets; } }
        internal NetworkPacketList ReceivedPackets { get { return receivedPackets; } }
        internal List<NetworkSession> IncomingSessionList { get { return incomingSessionList; } }
        internal List<NetworkSession> OutgoingSessionList { get { return outgoingSessionList; } }
        internal SortedList<ushort, NetworkServiceMetadata> NetworkServiceMetadataList { get { return networkServiceMetadataList; } }
        internal SortedList<string, string> UniversalPlugAndPlayFieldList { get { return this.universalPlugAndPlayFieldList; } set { this.universalPlugAndPlayFieldList=value; } }

        internal System.Collections.Specialized.NameValueCollection HostDetailCollection { 
            get{
                System.Collections.Specialized.NameValueCollection details=new System.Collections.Specialized.NameValueCollection();
                if(this.queriedIpList.Count>0) {
                    StringBuilder queriedIPs=new StringBuilder();
                    foreach(IPAddress ip in this.queriedIpList) {
                        details.Add("Queried IP Addresses", ip.ToString());
                    }
                    //details.Add("Queried NetBIOS names", queriedNames.ToString());
                }
                if(this.queriedNetBiosNameList.Count>0) {
                    StringBuilder queriedNames=new StringBuilder();
                    foreach(string name in this.queriedNetBiosNameList) {
                        //queriedNames.Append(name+" ");
                        details.Add("Queried NetBIOS names", name);
                    }
                    //details.Add("Queried NetBIOS names", queriedNames.ToString());
                }
                if(this.queriedDnsNameList.Count>0) {
                    StringBuilder queriedNames=new StringBuilder();
                    foreach(string name in this.queriedDnsNameList) {
                        //queriedNames.Append(name+" ");
                        details.Add("Queried DNS names", name);
                    }
                    //details.Add("Queried DNS names", queriedNames.ToString());
                }
                for(int i=0; i<this.httpUserAgentBannerList.Count; i++)
                    details.Add("Web Browser Banner "+(i+1), httpUserAgentBannerList[i]);
                for(int i=0; i<this.httpServerBannerList.Count; i++)
                    details.Add("Web Server Banner "+(i+1), httpServerBannerList[i]);
                if(this.universalPlugAndPlayFieldList!=null)
                    foreach(string field in this.universalPlugAndPlayFieldList.Values) {
                        if(field.Contains(":")) {
                            details.Add("UPnP field : "+field.Substring(0,field.LastIndexOf(':')),field.Substring(field.LastIndexOf(':')+1));
                        }
                        else
                            details.Add("UPnP field", field);
                    }
                if(this.acceptedSmbDialectsList!=null)
                    foreach(string dialectName in acceptedSmbDialectsList)
                        details.Add("Accepted SMB dialects", dialectName);
                if(this.preferredSmbDialect!=null)
                    details.Add("Preferred SMB dialect", this.preferredSmbDialect);
                foreach(KeyValuePair<string, string> keyValue in extraDetailsList) {
                    details.Add(keyValue.Key, keyValue.Value);
                }
                return details;
                
            }
        }

        internal bool IpIsMulticast {
            get{
                byte[] ip=this.ipAddress.GetAddressBytes();
                if(ip.Length==4){//let's start with IPv4
                    //http://en.wikipedia.org/wiki/Multicast_address
                    if(ip[0]>=224 && ip[0]<=239)
                        return true;
                }
                return false;
            }
        }
        internal bool IpIsBroadcast {//this one is probabilistic, since we don't know the subnet mask
            get {
                if(this.sentPackets.Count==0) {
                    byte[] ip=this.ipAddress.GetAddressBytes();
                    //let's assume we need a subnet mask of 6 bits or more
                    byte mask=0x3f;
                    if((ip[ip.Length-1]&mask)==mask)
                        return true;
                }
                return false;
            }
        }
        internal bool IpIsReserved {//IANA Reserved IP
            get {
                //http://www.iana.org/assignments/ipv4-address-space
                byte[] ip=this.ipAddress.GetAddressBytes();
                if(ip.Length==4) {//let's start with IPv4
                    if(ip[0]<3)
                        return true;
                    if(ip[0]==5)
                        return true;
                    if(ip[0]==7)
                        return true;
                    if(ip[0]==23)
                        return true;
                    if(ip[0]==27)
                        return true;
                    if(ip[0]==31)
                        return true;
                    if(ip[0]==36)
                        return true;
                    if(ip[0]==37)
                        return true;
                    if(ip[0]==39)
                        return true;
                    if(ip[0]==42)
                        return true;
                    if(ip[0]==94)
                        return true;
                    if(ip[0]>=100 && ip[0]<=115)
                        return true;
                    if(ip[0]==127)
                        return true;
                    if(ip[0]>=173 && ip[0]<=187)
                        return true;
                    if(ip[0]==197)
                        return true;
                    if(ip[0]>=240)
                        return true;
                }
                return false;


            }
        }
        internal OperatingSystemID OS{//there is something called System.PlatformID, but it isn't good enough
            get{
                if(this.operatingSystemCount.Count==0)
                    return OperatingSystemID.Unknown;

                System.Collections.Generic.Dictionary<OperatingSystemID, double> myOsCount=new Dictionary<OperatingSystemID,double>();
                foreach(string os in operatingSystemCount.Keys) {
                    if(os.ToLower().Contains("windows")) {
                        if(!myOsCount.ContainsKey(OperatingSystemID.Windows))
                            myOsCount.Add(OperatingSystemID.Windows, operatingSystemCount[os]);
                        else
                            myOsCount[OperatingSystemID.Windows]+=operatingSystemCount[os];
                    }
                    else if(os.ToLower().Contains("linux")) {
                        if(!myOsCount.ContainsKey(OperatingSystemID.Linux))
                            myOsCount.Add(OperatingSystemID.Linux, operatingSystemCount[os]);
                        else
                            myOsCount[OperatingSystemID.Linux]+=operatingSystemCount[os];
                    }
                    else if(os.ToLower().Contains("unix")) {
                        if(!myOsCount.ContainsKey(OperatingSystemID.UNIX))
                            myOsCount.Add(OperatingSystemID.UNIX, operatingSystemCount[os]);
                        else
                            myOsCount[OperatingSystemID.UNIX]+=operatingSystemCount[os];
                    }
                    else if(os.ToLower().Contains("freebsd") || os.ToLower().Contains("free bsd")) {
                        if(!myOsCount.ContainsKey(OperatingSystemID.FreeBSD))
                            myOsCount.Add(OperatingSystemID.FreeBSD, operatingSystemCount[os]);
                        else
                            myOsCount[OperatingSystemID.FreeBSD]+=operatingSystemCount[os];
                    }
                    else if(os.ToLower().Contains("netbsd") || os.ToLower().Contains("net bsd")) {
                        if(!myOsCount.ContainsKey(OperatingSystemID.NetBSD))
                            myOsCount.Add(OperatingSystemID.NetBSD, operatingSystemCount[os]);
                        else
                            myOsCount[OperatingSystemID.NetBSD]+=operatingSystemCount[os];
                    }
                    else if(os.ToLower().Contains("solaris")) {
                        if(!myOsCount.ContainsKey(OperatingSystemID.Solaris))
                            myOsCount.Add(OperatingSystemID.Solaris, operatingSystemCount[os]);
                        else
                            myOsCount[OperatingSystemID.Solaris]+=operatingSystemCount[os];
                    }
                    else if(os.ToLower().Contains("macos") || os.ToLower().Contains("mac os")) {
                        if(!myOsCount.ContainsKey(OperatingSystemID.MacOS))
                            myOsCount.Add(OperatingSystemID.MacOS, operatingSystemCount[os]);
                        else
                            myOsCount[OperatingSystemID.MacOS]+=operatingSystemCount[os];
                    }
                    else if(os.ToLower().Contains("cisco")) {
                        if(!myOsCount.ContainsKey(OperatingSystemID.Cisco))
                            myOsCount.Add(OperatingSystemID.Cisco, operatingSystemCount[os]);
                        else
                            myOsCount[OperatingSystemID.Cisco]+=operatingSystemCount[os];
                    }
                    else {
                        if(!myOsCount.ContainsKey(OperatingSystemID.Other))
                            myOsCount.Add(OperatingSystemID.Other, operatingSystemCount[os]);
                        else
                            myOsCount[OperatingSystemID.Other]+=operatingSystemCount[os];
                    }
                }
                //we now have counted all the OS's and want to get the one with best count
                OperatingSystemID probableOS=OperatingSystemID.Unknown;
                double bestOsCount=0.0;
                foreach(OperatingSystemID os in myOsCount.Keys)
                    if(myOsCount[os]>bestOsCount) {
                        probableOS=os;
                        bestOsCount=myOsCount[os];
                    }
                return probableOS;
            }
        }

        internal string OsDetails {
            get {
                if(operatingSystemCount.Count==0)
                    return "";
                else {
                    StringBuilder osString=new StringBuilder();
                    double totalOsCount=0.0;
                    foreach(string os in operatingSystemCount.Keys) {
                        totalOsCount+=operatingSystemCount[os];
                    }
                    if(totalOsCount==0)//just an extra check to avoid zero division
                        return "";
                    else {
                        foreach(string os in operatingSystemCount.Keys)
                            osString.Append(os+" ("+((double)(operatingSystemCount[os]/totalOsCount)).ToString("p")+") ");
                        return osString.ToString();
                    }
                }
            }
        }
        internal byte Ttl {
            get {
                if(ttlCount.Count==0)
                    return byte.MinValue;
                else {
                    int bestTtlCount=0;
                    byte bestTtl=0x00;
                    foreach(byte ttl in ttlCount.Keys) {
                        int count=ttlCount[ttl];
                        if(count>bestTtlCount) {
                            bestTtl=ttl;
                            bestTtlCount=count;
                        }
                    }
                    return bestTtl;
                }
            }
        }
        internal byte TtlDistance {
            get {
                if(ttlDistanceCount.Count==0)
                    return byte.MaxValue;
                else {
                    int bestTtlDistanceCount=0;
                    byte bestTtlDistance=0x00;
                    foreach(byte ttlDistance in ttlDistanceCount.Keys) {
                        int count=ttlDistanceCount[ttlDistance];
                        if(count>=bestTtlDistanceCount) {//if several are of equal value, I wan the largest TTL Distance
                            bestTtlDistance=ttlDistance;
                            bestTtlDistanceCount=count;
                        }
                    }
                    return bestTtlDistance;
                }
            }
        }
        //internal System.Collections.Specialized.NameValueCollection HostDetailCollection { get { return this.hostDetailCollection; } }
        internal List<string> AcceptedSmbDialectsList { get { return this.acceptedSmbDialectsList; } set { this.acceptedSmbDialectsList=value; } }
        internal string PreferredSmbDialect { get { return this.preferredSmbDialect; } set { this.preferredSmbDialect=value; } }
        internal SortedList<string, string> ExtraDetailsList { get { return this.extraDetailsList; } }

        internal NetworkHost(IPAddress ipAddress) {
            this.ipAddress=ipAddress;
            this.macAddress=null;
            this.ttlCount=new SortedList<byte, int>();
            this.ttlDistanceCount=new SortedList<byte, int>();
            this.operatingSystemCount=new SortedList<string, double>();
            this.hostNameList=new List<string>();
            //this.dnsAddress="";
            this.openTcpPortList=new List<ushort>();
            //this.networkServiceList=new SortedList<ushort, NetworkService>();
            this.networkServiceMetadataList=new SortedList<ushort, NetworkServiceMetadata>();
            //this.openTcpPortArray=new System.Collections.BitArray(1<<16, false);

            this.sentPackets=new NetworkPacketList();
            this.receivedPackets=new NetworkPacketList();
            this.incomingSessionList=new List<NetworkSession>();
            this.outgoingSessionList=new List<NetworkSession>();
            this.queriedIpList=new List<IPAddress>();
            this.queriedNetBiosNameList=new List<string>();
            this.queriedDnsNameList=new List<string>();
            this.httpUserAgentBannerList=new List<string>();
            this.httpServerBannerList=new List<string>();
            this.extraDetailsList=new SortedList<string, string>();

            this.universalPlugAndPlayFieldList=null;//I could just as well set this to null 'cause it is not often used. I'll initialize it when it is needed.
            this.acceptedSmbDialectsList=null;
            this.preferredSmbDialect=null;
        }

        public override int GetHashCode() {
            return ipAddress.GetHashCode();
            //return base.GetHashCode();
        }
        public override string ToString() {
            string str=ipAddress.ToString();
            foreach(string hostname in this.hostNameList)
                str+=" ["+hostname+"]";
            if(this.operatingSystemCount.Count>0)
                str+=" ("+this.OS+")";
            return str;
        }

        internal SortedList<NetworkHost, NetworkPacketList> GetSentPacketListsPerDestinationHost() {
            SortedList<NetworkHost, NetworkPacketList> masterList=new SortedList<NetworkHost, NetworkPacketList>();
            foreach(NetworkPacket p in sentPackets) {
                if(!masterList.ContainsKey(p.DestinationHost)) {
                    masterList.Add(p.DestinationHost, new NetworkPacketList());
                }
                masterList[p.DestinationHost].Add(p);
            }
            return masterList;
        }
        internal SortedList<NetworkHost, NetworkPacketList> GetReceivedPacketListsPerSourceHost() {
            SortedList<NetworkHost, NetworkPacketList> masterList=new SortedList<NetworkHost, NetworkPacketList>();
            foreach(NetworkPacket p in ReceivedPackets) {
                if(!masterList.ContainsKey(p.SourceHost)) {
                    masterList.Add(p.SourceHost, new NetworkPacketList());
                }
                masterList[p.SourceHost].Add(p);
            }
            return masterList;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="operatingSystem"></param>
        /// <param name="probability">A number between 0.0 and 1.0</param>
        internal void AddProbableOs(string operatingSystem, double probability) {

            if(this.operatingSystemCount.ContainsKey(operatingSystem))
                this.operatingSystemCount[operatingSystem]+=probability;
            else
                this.operatingSystemCount.Add(operatingSystem, probability);
        }

        internal void AddTtl(byte ttl) {
            if(this.ttlCount.ContainsKey(ttl))
                this.ttlCount[ttl]++;
            else
                this.ttlCount.Add(ttl, 1);
        }
        internal void AddProbableTtlDistance(byte ttlDistance){
            if(this.ttlDistanceCount.ContainsKey(ttlDistance))
                this.ttlDistanceCount[ttlDistance]++;
            else
                this.ttlDistanceCount.Add(ttlDistance, 1);
        }

        /// <summary>
        /// Adds a host name of some form
        /// </summary>
        /// <param name="hostname">DNS address, NetBIOS name or simiar</param>
        internal void AddHostName(string hostname) {
            if(!this.hostNameList.Contains(hostname))
                this.hostNameList.Add(hostname);
        }
        internal void AddQueriedIP(IPAddress ip) {
            if(!this.queriedIpList.Contains(ip))
                this.queriedIpList.Add(ip);
        }
        internal void AddQueriedNetBiosName(string netBiosName) {
            if(!this.queriedNetBiosNameList.Contains(netBiosName))
                this.queriedNetBiosNameList.Add(netBiosName);
        }
        internal void AddQueriedDnsName(string dnsName) {
            if(!this.queriedDnsNameList.Contains(dnsName))
                this.queriedDnsNameList.Add(dnsName);
        }
        internal void AddHttpUserAgentBanner(string banner) {
            if(!this.httpUserAgentBannerList.Contains(banner))
                this.httpUserAgentBannerList.Add(banner);
        }
        internal void AddHttpServerBanner(string banner, ushort serverTcpPort) {
            if(!this.httpServerBannerList.Contains("TCP "+serverTcpPort+" : "+banner))
                this.httpServerBannerList.Add("TCP "+serverTcpPort+" : "+banner);
        }

        internal void AddOpenTcpPort(ushort port) {
            this.openTcpPortList.Add(port);
            //openTcpPortArray[port]=true;
        }
        internal bool TcpPortIsOpen(ushort port) {
            return this.openTcpPortList.Contains(port);
            //return openTcpPortArray[port];
        }


        #region IComparable Members
        
        public int CompareTo(NetworkHost host) {

            if(this.IPAddress.Equals(host.IPAddress))
                return 0;
            else {
                byte[] localBytes=this.IPAddress.GetAddressBytes();
                byte[] remoteBytes=host.IPAddress.GetAddressBytes();
                for(int i=0; i<localBytes.Length && i<remoteBytes.Length; i++) {
                    if(localBytes[i]!=remoteBytes[i])
                        return localBytes[i]-remoteBytes[i];
                }
                return 0;
            }
        }

        public int CompareTo(object obj) {
            NetworkHost host=(NetworkHost)obj;
            return CompareTo(host);
        }

        #endregion

    }
}
