using System;
using System.Collections.Generic;
using System.Text;

namespace NetworkMiner.FileTransfer {
    class FileStreamAssemblerPool {
        private System.Collections.Generic.SortedList<string, LinkedListNode<FileStreamAssembler>> sortedList;
        private System.Collections.Generic.LinkedList<FileStreamAssembler> linkedList;
        private int maxPoolSize;//maximum number of assemblers that can be kept (to prevent filling memory)
        private PacketHandler packetHandler;

        private bool decompressGzipStreams;

        internal bool DecompressGzipStreams { get { return this.decompressGzipStreams; } }
        internal string FileOutputFolder { get { return System.IO.Path.GetDirectoryName(System.Windows.Forms.Application.ExecutablePath)+"\\"+"assembledFiles"; } }
        internal PacketHandler PacketHandler { get { return packetHandler; } }

        //internal FileStreamAssemblerPool() : base(100) { }
        internal FileStreamAssemblerPool(PacketHandler packetHandler, int maxPoolSize) {
            this.packetHandler = packetHandler;
            sortedList=new SortedList<string, LinkedListNode<FileStreamAssembler>>();
            linkedList=new LinkedList<FileStreamAssembler>();
            this.maxPoolSize=maxPoolSize;//I could make an overload consturctor that takes the poolSize as an argument

            this.decompressGzipStreams=true;//this should be a setting that can be changed in an option-menu.
        }

        private string GetAssemblerId(FileStreamAssembler assembler) {
            return GetAssemblerId(assembler.SourceHost, assembler.SourcePort, assembler.DestinationHost, assembler.DestinationPort, assembler.TcpTransfer);
        }
        private string GetAssemblerId(NetworkHost sourceHost, ushort sourcePort, NetworkHost destinationHost, ushort destinationPort, bool tcpTransfer) {
            return sourceHost.IPAddress.ToString()+sourcePort.ToString()+destinationHost.IPAddress.ToString()+destinationPort.ToString()+tcpTransfer.ToString();
        }

        internal bool ContainsAssembler(FileStreamAssembler assembler) {
            string id=GetAssemblerId(assembler);
            return sortedList.ContainsKey(id);
        }
        internal bool ContainsAssembler(NetworkHost sourceHost, ushort sourcePort, NetworkHost destinationHost, ushort destinationPort, bool tcpTransfer) {
            string id=GetAssemblerId(sourceHost, sourcePort, destinationHost, destinationPort, tcpTransfer);
            return sortedList.ContainsKey(id);
        }
        internal void Remove(FileStreamAssembler assembler, bool closeAssembler) {
            string id=GetAssemblerId(assembler);
            LinkedListNode<FileStreamAssembler> llNode=sortedList[id];//O(log(n))
            linkedList.Remove(llNode);//O(1)
            sortedList.Remove(id);//O(log(n))
            if(closeAssembler)//it should sometimes be closed elsewhere
                assembler.Clear();
        }

        internal FileStreamAssembler GetAssembler(NetworkHost sourceHost, ushort sourcePort, NetworkHost destinationHost, ushort destinationPort, bool tcpTransfer) {
            string id=GetAssemblerId(sourceHost, sourcePort, destinationHost, destinationPort, tcpTransfer);
            //reorder
            LinkedListNode<FileStreamAssembler> llNode=sortedList[id];//O(log(n))
            linkedList.Remove(llNode);//O(1)
            linkedList.AddFirst(llNode);//O(1)
           
            return llNode.Value;
        }

        internal void Add(FileStreamAssembler assembler) {
            LinkedListNode<FileStreamAssembler> llNode=new LinkedListNode<FileStreamAssembler>(assembler);
            linkedList.AddFirst(llNode);//O(1)
            string id=GetAssemblerId(assembler);
            sortedList.Add(id, llNode);//O(log(n))
            //see if there are too many assemblers, if it is: remove one or several...
            while(sortedList.Count>this.maxPoolSize) {
                LinkedListNode<FileStreamAssembler> lastNode=linkedList.Last;//O(1)
                sortedList.Remove(GetAssemblerId(lastNode.Value));//O(log(n))
                linkedList.Remove(lastNode);//O(1)
            }
        }

        //Removes all data and stored files
        internal void ClearAll() {
            foreach(FileStreamAssembler assembler in linkedList)
                assembler.Clear();
            linkedList.Clear();
            sortedList.Clear();

            //remove all files
            foreach(string subDirectory in System.IO.Directory.GetDirectories(this.FileOutputFolder))
                if(subDirectory==this.FileOutputFolder+"\\cache")
                    foreach(string cacheFile in System.IO.Directory.GetFiles(subDirectory))
                        try {
                            System.IO.File.Delete(cacheFile);
                        }
                        catch {
                            packetHandler.ParentForm.ShowError("Error deleting file \""+cacheFile+"\"");
                        }

                else
                    try {
                        System.IO.Directory.Delete(subDirectory, true);
                    }
                    catch(Exception e) {
                        packetHandler.ParentForm.ShowError("Error deleting directory \""+subDirectory+"\"");
                    }
        }
    }
}
