//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially
using System;
using System.Collections.Generic;
using System.Text;

namespace NetworkMiner.FileTransfer {
    

    class FileStreamAssembler {
        //internal enum FileStreamTypes{HttpGetNormal, HttpGetChunked, HttpPost, FTP, SMB}

        private FileStreamAssemblerPool parentPool;
        private NetworkHost sourceHost, destinationHost;
        private ushort sourcePort, destinationPort;
        private bool tcpTransfer;
        private FileStreamTypes fileStreamType;
        private Packets.HttpPacket.ContentEncodings contentEncoding;
        private string filename, fileLocation;
        private int fileContentLength;//in bytes
        private int fileSegmentRemainingBytes;//the length (in bytes) of the current file segment to recieve
        private string details;

        private int assembledByteCount;
        private System.IO.FileStream fileStream;
        private SortedList<uint, byte[]> tcpPacketBufferWindow;

        internal NetworkHost SourceHost { get { return sourceHost; } }
        internal NetworkHost DestinationHost { get { return destinationHost; } }
        internal ushort SourcePort { get { return sourcePort; } }
        internal ushort DestinationPort { get { return destinationPort; } }
        internal string Filename { get { return filename; } set { filename=value; } }
        internal bool TcpTransfer { get { return tcpTransfer; } }
        internal int FileContentLength { get { return fileContentLength; } set { this.fileContentLength=value; } }
        internal int FileSegmentRemainingBytes { get { return fileSegmentRemainingBytes; } set { this.fileSegmentRemainingBytes=value; } }
        internal FileStreamTypes FileStreamType { get { return fileStreamType; } set { this.fileStreamType=value; } }
        internal Packets.HttpPacket.ContentEncodings ContentEncoding {
            get { return this.contentEncoding; }
            set {
                this.contentEncoding=value;
                if(!parentPool.DecompressGzipStreams && !this.filename.EndsWith(".gz"))
                    this.filename=this.filename+".gz";
            }
        }


        internal FileStreamAssembler(FileStreamAssemblerPool parentPool, NetworkHost sourceHost, ushort sourcePort, NetworkHost destinationHost, ushort destinationPort, bool tcpTransfer, FileStreamTypes fileStreamType, string filename, string fileLocation, string details):
           this(parentPool, sourceHost, sourcePort, destinationHost, destinationPort, tcpTransfer, fileStreamType, filename, fileLocation, 0, 0, details) {//shall I maybe have -1 as fileContentLength???
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sourceHost"></param>
        /// <param name="sourcePort"></param>
        /// <param name="destinationHost"></param>
        /// <param name="destinationPort"></param>
        /// <param name="tcpTransfer">True=TCP, False=UDP</param>
        /// <param name="fileStreamType"></param>
        /// <param name="filename">for example "image.gif"</param>
        /// <param name="fileLocation">for example "/images", empty string for root folder</param>
        internal FileStreamAssembler(FileStreamAssemblerPool parentPool, NetworkHost sourceHost, ushort sourcePort, NetworkHost destinationHost, ushort destinationPort, bool tcpTransfer, FileStreamTypes fileStreamType, string filename, string fileLocation, int fileContentLength, int fileSegmentRemainingBytes, string details) {
            this.parentPool=parentPool;
            this.sourceHost=sourceHost;
            this.sourcePort=sourcePort;
            this.destinationHost=destinationHost;
            this.destinationPort=destinationPort;
            this.tcpTransfer=tcpTransfer;
            this.fileStreamType=fileStreamType;
            this.fileContentLength=fileContentLength;//this one can not be set already when the client requests the file...so it has to be changed later
            this.fileSegmentRemainingBytes=fileSegmentRemainingBytes;
            this.details=details;
            this.contentEncoding=NetworkMiner.Packets.HttpPacket.ContentEncodings.Identity;//default

            char[] specialCharacters={':', '*', '?', '"', '<', '>', '|' };
            char[] directorySeparators={ '\\', '/' };


            //Sigh I just hate the limitation on file and folder length.
            //See: http://msdn2.microsoft.com/en-us/library/aa365247.aspx
            //Or: http://blogs.msdn.com/bclteam/archive/2007/02/13/long-paths-in-net-part-1-of-3-kim-hamilton.aspx

            this.filename=System.Web.HttpUtility.UrlDecode(filename);
            while(this.filename.IndexOfAny(specialCharacters)>-1)
                this.filename=this.filename.Remove(this.filename.IndexOfAny(specialCharacters), 1);
            while(this.filename.IndexOfAny(directorySeparators)>-1)
                this.filename=this.filename.Remove(this.filename.IndexOfAny(directorySeparators), 1);
            if(this.filename.Length>32) {//not allowed by Windows to be more than 260 characters
                //I want to make sure I keep the extension when the filename is cut...
                int extensionPosition=this.filename.LastIndexOf('.');
                if(extensionPosition<0 || extensionPosition<=this.filename.Length-20)
                    this.filename=this.filename.Substring(0, 20);
                else//I hope this next line works allright...
                    this.filename=this.filename.Substring(0, 20-this.filename.Length+extensionPosition)+this.filename.Substring(extensionPosition);
            }

            this.fileLocation=System.Web.HttpUtility.UrlDecode(fileLocation);
            this.fileLocation=this.fileLocation.Replace('\\', '/');//I prefer using frontslash
            while(this.fileLocation.IndexOfAny(specialCharacters)>-1)
                this.fileLocation=this.fileLocation.Remove(this.fileLocation.IndexOfAny(specialCharacters), 1);
            if(this.fileLocation.Length>40)//248 characters totally is the maximum allowed
                this.fileLocation=this.fileLocation.Substring(0, 40);

            this.assembledByteCount=0;
            this.tcpPacketBufferWindow=new SortedList<uint, byte[]>();

            fileStream=new System.IO.FileStream(GetFilePath(true), System.IO.FileMode.Create, System.IO.FileAccess.ReadWrite);
        }

        private string GetFilePath(bool tempCachePath) {
            string filePath;
            string protocolString;
            if(fileStreamType==FileStreamTypes.HttpGetNormal || fileStreamType==FileStreamTypes.HttpGetChunked)
                protocolString="HTTP";
            else if(FileStreamType==FileStreamTypes.SMB)
                protocolString="SMB";
            else
                throw new Exception("Not implemented yet");

            string transportString;
            if(tcpTransfer)
                transportString="TCP";
            else
                transportString="UDP";

            if(tempCachePath) {
                filePath="cache/"+sourceHost.IPAddress.ToString()+"_"+transportString+sourcePort.ToString()+" - "+destinationHost.IPAddress.ToString()+"_"+transportString+DestinationPort.ToString()+".txt";
            }
            else {
                filePath=SourceHost.IPAddress.ToString()+"/"+protocolString+" - "+transportString+" "+sourcePort.ToString()+fileLocation+"/"+filename;
            }
            filePath=parentPool.FileOutputFolder+"\\"+filePath;
            /*if(filePath.Length>=248)
                filePath="\\\\?\\"+filePath;*/
            return filePath;
        }

        internal void AddData(Packets.TcpPacket tcpPacket) {
            if(!this.tcpTransfer)
                throw new Exception("No TCP packets accepted, only UDP");
            AddData(tcpPacket.GetTcpPacketPayloadData(), tcpPacket.SequenceNumber);
            //if(this.FileStreamType==FileStreamAssembler.FileStreamTypes.HttpGetChunked && tcpPacket.FlagBits.Push)
            //    this.FinishAssembling();//last packet is normally marked with a Push-flag to push it to the application layer
        }

        internal void AddData(byte[] packetData, uint tcpPacketSequenceNumber){
            if(packetData.Length<=0)
                return;//don't do anything with empty packets...
            if(tcpPacketBufferWindow.ContainsKey(tcpPacketSequenceNumber))
                return;//we allready have the packet (I'm not putting any effort into seeing if they are different and which one is correct)
            if(this.FileStreamType!=FileStreamTypes.HttpGetChunked) {
                if(fileSegmentRemainingBytes<packetData.Length)
                    throw new Exception("Assembler is only expecting data segment length up to "+fileSegmentRemainingBytes+" bytes");
                fileSegmentRemainingBytes-=packetData.Length;
            }
            tcpPacketBufferWindow.Add(tcpPacketSequenceNumber, packetData);
            this.assembledByteCount+=packetData.Length;

            //in order to improve performance I could ofcourse have a separate thread doing all the file operations
            while(tcpPacketBufferWindow.Count>8) {//I'll just simply set the maximum TCP packets stored to 8, this can be adjusted to improve performance
                uint key=tcpPacketBufferWindow.Keys[0];
                fileStream.Write(tcpPacketBufferWindow[key], 0, tcpPacketBufferWindow[key].Length);
                tcpPacketBufferWindow.Remove(key);
            }
            if((this.FileStreamType==FileStreamTypes.HttpGetNormal || this.FileStreamType==FileStreamTypes.SMB) && assembledByteCount>=fileContentLength) {//we have received the whole file
                FinishAssembling();
            }
            else if(this.FileStreamType==FileStreamTypes.HttpGetChunked) {
                byte[] chunkTrailer={ 0x30, 0x0d, 0x0a, 0x0d, 0x0a };//see: RFC 2616 3.6.1 Chunked Transfer Coding
                if(packetData.Length>=chunkTrailer.Length) {
                    bool packetDataHasChunkTrailer=true;
                    for(int i=0; i<chunkTrailer.Length && packetDataHasChunkTrailer; i++)
                        if(packetData[packetData.Length-chunkTrailer.Length+i]!=chunkTrailer[i])
                            packetDataHasChunkTrailer=false;
                    if(packetDataHasChunkTrailer)
                        FinishAssembling();

                }

            }
        }

        internal void FinishAssembling() {
            foreach(byte[] data in tcpPacketBufferWindow.Values)
                fileStream.Write(data, 0, data.Length);
            tcpPacketBufferWindow.Clear();
            parentPool.Remove(this, false);
            //fileStream.Close();
            //Microsoft.Win32.SafeHandles.SafeFileHandle fileHandle=fileStream.SafeFileHandle;
            string destinationPath=GetFilePath(false);
            string directoryName=destinationPath.Substring(0, destinationPath.Length-this.filename.Length);
            if(!System.IO.Directory.Exists(directoryName))
                System.IO.Directory.CreateDirectory(directoryName);
            if(System.IO.File.Exists(destinationPath))
                try {
                    System.IO.File.Delete(destinationPath);
                }
                catch(Exception e) {
                    parentPool.PacketHandler.ParentForm.ShowError("Error deleting file \""+destinationPath+"\" (tried to replace it)");
                }

            
            if(this.fileStreamType==FileStreamTypes.HttpGetChunked || (parentPool.DecompressGzipStreams && this.contentEncoding==Packets.HttpPacket.ContentEncodings.Gzip)) {
                this.fileStream.Position=0;//move to fileStream start since it needs to be read

                if(this.fileStreamType==FileStreamTypes.HttpGetChunked && (parentPool.DecompressGzipStreams && this.contentEncoding==Packets.HttpPacket.ContentEncodings.Gzip)) {
                    DeChunkedDataStream deChunkedStream=new DeChunkedDataStream(fileStream);
                    System.IO.Compression.GZipStream decompressedStream=new System.IO.Compression.GZipStream(deChunkedStream, System.IO.Compression.CompressionMode.Decompress);
                    try{
                        this.WriteStreamToFile(decompressedStream, destinationPath);
                    }
                    catch(Exception e){
                        this.parentPool.PacketHandler.ParentForm.ShowError("Error: Cannot write to file "+destinationPath+" ("+e.Message+")");
                    }
                    deChunkedStream.Close();
                    decompressedStream.Close();
                }
                else if(this.fileStreamType==FileStreamTypes.HttpGetChunked) {
                    DeChunkedDataStream deChunkedStream=new DeChunkedDataStream(fileStream);
                    try{
                        this.WriteStreamToFile(deChunkedStream, destinationPath);
                    }
                    catch(Exception e){
                         this.parentPool.PacketHandler.ParentForm.ShowError("Error: Cannot write to file "+destinationPath+" ("+e.Message+")");
                    }
                    deChunkedStream.Close();
                }
                else {
                    System.IO.Compression.GZipStream decompressedStream=new System.IO.Compression.GZipStream(fileStream, System.IO.Compression.CompressionMode.Decompress);
                    try{
                        this.WriteStreamToFile(decompressedStream, destinationPath);
                    }
                    catch(Exception e){
                         this.parentPool.PacketHandler.ParentForm.ShowError("Error: Cannot write to file "+destinationPath+" ("+e.Message+")");
                    }
                    decompressedStream.Close();
                }

                fileStream.Close();
                System.IO.File.Delete(GetFilePath(true));

            }

            else {
                fileStream.Close();
                try {
                    System.IO.File.Move(GetFilePath(true), destinationPath);
                }
                catch(Exception e) {
                    parentPool.PacketHandler.ParentForm.ShowError("Error moving file \""+GetFilePath(true)+"\" to \""+destinationPath+"\". "+e.Message);
                }
            }
            try {
                ReconstructedFile completedFile=new ReconstructedFile(GetFilePath(false), sourceHost, destinationHost, sourcePort, destinationPort, tcpTransfer, fileStreamType, details);
                parentPool.PacketHandler.AddReconstructedFile(completedFile);
                //parentPool.PacketHandler.ParentForm.ShowReconstructedFile(completedFile);
            }
            catch(Exception e) {
                parentPool.PacketHandler.ParentForm.ShowError("Error creating reconstructed file: "+e.Message);
            }
            //Vad gr jag nu? Kanske lgger till filen if NetworkMinerForm?

            //I would need some sort of event here that can be triggered when the file transfer is finished

        
        }

        
        //this one might generate exceptions thatt needs to be catched further up in the hierarchy!!!
        internal void WriteStreamToFile(System.IO.Stream stream, string destinationPath) {
            System.IO.FileStream outputFile=new System.IO.FileStream(destinationPath, System.IO.FileMode.Create);
            byte[] buffer=new byte[1024];
            while(true) {//I don't like while(true) loops, but they used it at ms-help://MS.VSCC.v80/MS.MSDN.v80/MS.NETDEVFX.v20.en/cpref8/html/T_System_IO_Compression_CompressionMode.htm
                int bytesRead=stream.Read(buffer, 0, buffer.Length);
                if(bytesRead==0)
                    break;
                outputFile.Write(buffer, 0, bytesRead);
            }
            outputFile.Close();
        }

        internal void Clear() {
            this.tcpPacketBufferWindow.Clear();
            fileStream.Close();
            System.IO.File.Delete(GetFilePath(true));
        }

        internal class DeChunkedDataStream : System.IO.Stream {
            //private int position;//should be long
            private System.IO.Stream chunkedStream;
            private int currentChunkSize;
            private int readBytesInCurrentChunk;

            public override bool CanRead {
                get { return true; }
            }

            public override bool CanSeek {
                get { return false; }
            }

            public override bool CanWrite {
                get { return false; }
            }

            public override void Flush() {
                throw new Exception("The method or operation is not implemented.");
            }

            public override long Length {
                get { throw new Exception("The method or operation is not implemented."); }
            }

            public override long Position {
                get {
                    throw new Exception("The method or operation is not implemented.");
                }
                set {
                    throw new Exception("The method or operation is not implemented.");
                }
            }

            public DeChunkedDataStream(System.IO.Stream chunkedStream) {
                this.chunkedStream=chunkedStream;
                //this.position=0;//should be long
                this.currentChunkSize=0;
                this.readBytesInCurrentChunk=0;
            }

            public override int Read(byte[] buffer, int offset, int count) {
                int bytesRead=0;
                if(this.readBytesInCurrentChunk>=this.currentChunkSize){//I need to read a new chunk
                    StringBuilder chunkSizeString=new StringBuilder();//chunk-size as hex-string
                    while(true){
                        int ib=chunkedStream.ReadByte();
                        if(ib<0)//end of stream
                            return 0;

                        byte b=(byte)ib;
                        //if(b==0x20) { }//do nothing, it's just spacer-padding
                        if(b!=0x0d) {
                            char c=(char)b;
                            string hexCharacters="0123456789abcdefABCDEF";

                            if(hexCharacters.Contains(""+c))
                                chunkSizeString.Append((char)b);
                        }
                        else {
                            chunkedStream.ReadByte();//this should be the 0x0a that follows the 0x0d
                            if(chunkSizeString.Length>0)//there are sometimes CRLF before the chunk-size value
                                break;
                        }
                    }
                    if(chunkSizeString.ToString().Length==0)
                        this.currentChunkSize=0;//I'm not sure if this line should really be needed...
                    else
                        this.currentChunkSize=Convert.ToInt32(chunkSizeString.ToString(), 16);
                    this.readBytesInCurrentChunk=0;

                    if(this.currentChunkSize==0)//end of chunk-stream
                        return 0;
                }

                //byte[] tmpBuffer=new byte[buffer.Length];
                bytesRead=this.chunkedStream.Read(buffer, offset, Math.Min(count, this.currentChunkSize-this.readBytesInCurrentChunk));
                this.readBytesInCurrentChunk+=bytesRead;

                //see if I have to start reading the next chunk as well....
                if(bytesRead<count && this.readBytesInCurrentChunk==this.currentChunkSize){
                    //we are at the end of the chunk
                    return bytesRead+this.Read(buffer, offset+bytesRead, count-bytesRead);//nice recursive stuff
                }
                else
                    return bytesRead;
            }

            public override long Seek(long offset, System.IO.SeekOrigin origin) {
                throw new Exception("The method or operation is not implemented.");
            }

            public override void SetLength(long value) {
                throw new Exception("The method or operation is not implemented.");
            }

            public override void Write(byte[] buffer, int offset, int count) {
                throw new Exception("The method or operation is not implemented.");
            }
        }
    }
}
