//  Copyright: Erik Hjelmvik <hjelmvik@users.sourceforge.net>
//
//  NetworkMiner is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License
//
//  Contact Erik Hjelmvik if you wish to use NetworkMiner commersially
using System;
using System.Collections.Generic;
using System.Text;

namespace NetworkMiner {
    /// <summary>
    /// Sadly this class is needed since Big and little endian can't be configured in Syste.ByteConverter
    /// I'll use the same types of functions as in ByteConverter.
    /// </summary>
    /// 
    static class ByteConverter{



        public static ushort ToUInt16(byte[] value, int startIndex) {
            return (ushort)ToUInt32(value, startIndex, 2, false);
        }
        public static ushort ToUInt16(byte[] value, int startIndex, bool reverseByteOrder) {
            return (ushort)ToUInt32(value, startIndex, 2, reverseByteOrder);
        }

        private static uint ToUInt32(byte[] value, int startIndex, int nBytes, bool reverseByteOrder) {
            uint returnValue=0;

            for(int i=0; i<nBytes; i++) {
                returnValue<<=8;
                if(reverseByteOrder)//first byte is smallest value (LSB)
                    returnValue+=(uint)value[startIndex+nBytes-1-i];
                else//first byte is largest value (MSB)
                    returnValue+=(uint)value[startIndex+i];
            }
            return returnValue;
        }

        public static ulong ToUInt64(byte[] value, int startIndex, bool reverseOrder) {
            ulong returnValue=0;

            uint i1=ToUInt32(value, startIndex, 4, reverseOrder);
            uint i2=ToUInt32(value, startIndex+4, 4, reverseOrder);
            if(reverseOrder) {
                returnValue+=i2;
                returnValue<<=32;
                returnValue+=i1;
            }
            else {
                returnValue+=i1;
                returnValue<<=32;
                returnValue+=i2;
            }
            return returnValue;
        }

        public static uint ToUInt32(byte[] value) {
            return ToUInt32(value, 0, value.Length, false);
        }
        public static uint ToUInt32(byte[] value, int startIndex) {
            return ToUInt32(value, startIndex, 4, false);
        }

        public static uint ToUInt32(System.Net.IPAddress ip) {
            byte[] bytes=ip.GetAddressBytes();
            long l=0;
            for(int i=0; i<bytes.Length; i++)
                l=(l<<8)+bytes[i];
            return (uint)l;
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="ushort1">Most significant 2 bytes</param>
        /// <param name="ushort2">Least significant 2 bytes</param>
        /// <returns></returns>
        public static uint ToUInt32(ushort ushort1, ushort ushort2) {
            /*uint i1=(uint)ushort1;
            uint i2=(uint)ushort2;
            i1<<=16;
            return (uint)(i1 & i2);*/
            uint returnValue=(uint)ushort1;
            returnValue<<=16;
            returnValue^=ushort2;
            return returnValue;
            //return (uint)(ushort1<<16 & ushort2);
        }

        /// <summary>
        /// Reads one line from the byte[] in data and returns the line as a string.
        /// A line is defined by a number of char's followed by \r\n
        /// </summary>
        /// <param name="data"></param>
        /// <param name="dataIndex"></param>
        /// <returns></returns>
        public static string ReadLine(byte[] data, ref int dataIndex) {
            int maxStringLength=1024;
            //  \r = 0x0d = carriage return
            //  \n = 0x0a = line feed
            StringBuilder line=new StringBuilder();
            bool carrigeReturnReceived=false;
            bool lineFeedReceived=false;
            int indexOffset=0;
            while(dataIndex+indexOffset<data.Length && indexOffset<maxStringLength && (!carrigeReturnReceived || !lineFeedReceived)) {
                byte b=data[dataIndex+indexOffset];
                if(b==0x0d)
                    carrigeReturnReceived=true;
                else if(b==0x0a)
                    lineFeedReceived=true;
                else
                    line.Append((char)b);
                indexOffset++;
            }
            dataIndex+=indexOffset;
            return line.ToString();

        }

        public static string ReadString(byte[] data, ref int dataIndex, int bytesToRead, bool unicodeData, bool reverseOrder) {
            int i=0;
            StringBuilder sb=new StringBuilder();
            while(i<bytesToRead && dataIndex+i<data.Length) {
                if(unicodeData) {
                    ushort unicodeValue=ByteConverter.ToUInt16(data, dataIndex+i, reverseOrder);
                    sb.Append((char)unicodeValue);
                    i+=2;
                }
                else {
                    sb.Append(data[dataIndex+i]);
                    i++;
                }
            }
            dataIndex+=i;
            return sb.ToString();
        }

        public static string ReadNullTerminatedString(byte[] data, ref int dataIndex) {
            return ReadNullTerminatedString(data, ref dataIndex, false, false);
        }
        public static string ReadNullTerminatedString(byte[] data, ref int dataIndex, bool unicodeData, bool reverseOrder) {
            int maxStringLength=1024;
            StringBuilder returnString =new StringBuilder();

            if(!unicodeData) {
                for(int offset=0; dataIndex+offset<data.Length && offset<maxStringLength; offset++) {
                    byte b=data[dataIndex+offset];
                    if(b==0x00) {
                        dataIndex+=(offset+1);
                        return returnString.ToString();
                    }
                    else {
                        returnString.Append((char)b);
                    }
                }
            }
            else {//unicode
                for(int offset=0; dataIndex+offset+1<data.Length && offset<maxStringLength*2; offset+=2) {
                    ushort b=ByteConverter.ToUInt16(data, dataIndex+offset, reverseOrder);
                    if(b==0x0000) {
                        dataIndex+=(offset+2);
                        return returnString.ToString();
                    }
                    else {
                        returnString.Append((char)b);
                    }
                }
            }
            //we should hopefully not end up here!!!
            //But sometimes implementations just don't use a terminator and instead they just end the whole packet!!!
            //so let's degrade gracefully...
            return returnString.ToString();
            //throw new Exception("Null terminated string is out of bounds or too long!");
        }

    }
}
