/*
  ESP32 I2C Dual I2C Bus Demo
  esp32-dual-i2c.ino
  Demonstrates using both I2C buses in ESP32
  Uses ESP32, 2 OLED displays and temperature/humidity sensor
  
  DroneBot Workshop 2025
  https://dronebotworkshop.com
*/

#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <Adafruit_AHTX0.h>

// OLED Display configuration
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
#define SCREEN_ADDRESS 0x3C  // BOTH displays have the SAME address!

// I2C Bus 0 pins (for temperature display and sensor) - Main I2C
#define I2C0_SDA 4
#define I2C0_SCL 5

// I2C Bus 1 pins (for humidity display) - LP I2C (FIXED pins only!)
#define I2C1_SDA 6
#define I2C1_SCL 7

// Create TwoWire objects for each bus
TwoWire I2C_Bus0 = TwoWire(0);
TwoWire I2C_Bus1 = TwoWire(1);

// Create peripheral objects - NOTE: Both OLEDs use same address!
Adafruit_SSD1306 tempDisplay(SCREEN_WIDTH, SCREEN_HEIGHT, &I2C_Bus0, OLED_RESET);
Adafruit_SSD1306 humidityDisplay(SCREEN_WIDTH, SCREEN_HEIGHT, &I2C_Bus1, OLED_RESET);
Adafruit_AHTX0 aht;

void setup() {
  Serial.begin(115200);

  // Initialize I2C buses with specific pins
  I2C_Bus0.begin(I2C0_SDA, I2C0_SCL, 100000);
  I2C_Bus1.begin(I2C1_SDA, I2C1_SCL, 100000);

  // Initialize temperature display on Bus 0 (same address as humidity display!)
  if (!tempDisplay.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS)) {
    Serial.println(F("Temperature display allocation failed"));
    for (;;)
      ;
  }

  // Initialize humidity display on Bus 1 (same address - but different bus!)
  if (!humidityDisplay.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS)) {
    Serial.println(F("Humidity display allocation failed"));
    for (;;)
      ;
  }

  // Initialize AHT20 sensor on Bus 0 (with temperature display)
  if (!aht.begin(&I2C_Bus0)) {
    Serial.println("Could not find AHT20 sensor on Bus 0!");
    while (1) delay(10);
  }

  // Initialize displays
  setupDisplay(tempDisplay, "Temperature", "Bus 0");
  setupDisplay(humidityDisplay, "Humidity", "Bus 1");

  Serial.println("Multiple I2C Bus Demo initialized");
  Serial.println("Both OLED displays have address 0x3C - but on separate buses!");

  // Scan both buses for verification
  scanAllBuses();
}

void setupDisplay(Adafruit_SSD1306& display, const char* title, const char* bus) {
  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(0, 0);
  display.println(title);
  display.println(bus);
  display.println("Initializing...");
  display.display();
  delay(1000);
}

void loop() {
  // Read sensor data
  sensors_event_t humidity, temp;
  aht.getEvent(&humidity, &temp);

  // Update temperature display (Bus 0)
  updateTemperatureDisplay(temp.temperature);

  // Update humidity display (Bus 1)
  updateHumidityDisplay(humidity.relative_humidity);

  // Print to serial monitor
  Serial.print("Temp: ");
  Serial.print(temp.temperature, 1);
  Serial.print("°C, Humidity: ");
  Serial.print(humidity.relative_humidity, 1);
  Serial.println("%");

  delay(2000);
}

void updateTemperatureDisplay(float temperature) {
  tempDisplay.clearDisplay();
  tempDisplay.setCursor(0, 0);
  tempDisplay.setTextSize(1);
  tempDisplay.println("TEMPERATURE");
  tempDisplay.println("I2C Bus 0");
  tempDisplay.print("Address: 0x");
  tempDisplay.println(SCREEN_ADDRESS, HEX);
  tempDisplay.println();

  tempDisplay.setTextSize(2);
  tempDisplay.print(temperature, 1);
  tempDisplay.println(" C");

  tempDisplay.display();
}

void updateHumidityDisplay(float humidity) {
  humidityDisplay.clearDisplay();
  humidityDisplay.setCursor(0, 0);
  humidityDisplay.setTextSize(1);
  humidityDisplay.println("HUMIDITY");
  humidityDisplay.println("I2C Bus 1");
  humidityDisplay.print("Address: 0x");
  humidityDisplay.println(SCREEN_ADDRESS, HEX);
  humidityDisplay.println();

  humidityDisplay.setTextSize(2);
  humidityDisplay.print(humidity, 1);
  humidityDisplay.println(" %");

  humidityDisplay.display();
}

// Function to scan devices on both buses
void scanAllBuses() {
  Serial.println("Scanning I2C Bus 0:");
  scanI2CBus(I2C_Bus0);

  Serial.println("Scanning I2C Bus 1:");
  scanI2CBus(I2C_Bus1);
}

void scanI2CBus(TwoWire& wire) {
  byte error, address;
  int nDevices = 0;

  for (address = 1; address < 127; address++) {
    wire.beginTransmission(address);
    error = wire.endTransmission();

    if (error == 0) {
      Serial.print("I2C device found at address 0x");
      if (address < 16) Serial.print("0");
      Serial.print(address, HEX);
      Serial.println(" !");
      nDevices++;
    }
  }

  if (nDevices == 0) {
    Serial.println("No I2C devices found");
  }
  Serial.println();
}
