/*
  Project Name
  esp32-aht20-wire.ino
  Demonstrates use of Wire Library for I2C
  Read Temperature & Humidity from AHT20
  AHT20 I2C address: 0x38
  Sequence:
    1. Soft reset
    2. Trigger measurement
    3. Wait for busy flag to clear
    4. Read 6 bytes and convert to values
  
  DroneBot Workshop 2025
  https://dronebotworkshop.com
*/

// Include Wire Library for I2C
#include <Wire.h>

// Define Pins
const uint8_t SDA_PIN = 4;
const uint8_t SCL_PIN = 5;
const uint8_t AHT20_ADDR = 0x38;

void setup() {

  // Start Serial Monitor
  Serial.begin(115200);
  delay(200);

  // Start I2C with custom SDA/SCL pins
  Wire.begin(SDA_PIN, SCL_PIN);

  Serial.println("Initializing AHT20...");

  // 1. Soft reset
  Wire.beginTransmission(AHT20_ADDR);
  Wire.write(0xBA);  // Soft reset command
  Wire.endTransmission();
  delay(20);  // Small delay after reset

  // 2. Initialization / calibration command
  Wire.beginTransmission(AHT20_ADDR);
  Wire.write(0xBE);  // Init command
  Wire.write(0x08);
  Wire.write(0x00);
  Wire.endTransmission();
  delay(10);
}

void loop() {
  // 3. Trigger measurement (temperature + humidity)
  Wire.beginTransmission(AHT20_ADDR);
  Wire.write(0xAC);  // Trigger measurement command
  Wire.write(0x33);
  Wire.write(0x00);
  Wire.endTransmission();

  delay(80);  // Wait typical measurement time (~80ms)

  // 4. Read 6 bytes of data
  Wire.requestFrom(AHT20_ADDR, (uint8_t)6);
  if (Wire.available() == 6) {
    uint8_t data[6];
    for (int i = 0; i < 6; i++) {
      data[i] = Wire.read();
    }

    // Byte0 bit7 = busy flag (should be 0 now)
    if (data[0] & 0x80) {
      Serial.println("Sensor busy, try again");
      delay(500);
      return;
    }

    // Combine humidity bytes (20-bit value)
    uint32_t rawHumidity = ((uint32_t)data[1] << 12) | ((uint32_t)data[2] << 4) | ((uint32_t)data[3] >> 4);

    // Combine temperature bytes (20-bit value)
    uint32_t rawTemp = (((uint32_t)data[3] & 0x0F) << 16) | ((uint32_t)data[4] << 8) | (uint32_t)data[5];

    // Convert to human-readable
    float humidity = (rawHumidity * 100.0) / 1048576.0;  // 2^20 = 1048576
    float temperature = (rawTemp * 200.0 / 1048576.0) - 50.0;

    // Output results
    Serial.print("Humidity: ");
    Serial.print(humidity, 1);
    Serial.print(" %  |  Temp: ");
    Serial.print(temperature, 1);
    Serial.println(" C");
  } else {
    Serial.println("Read error");
  }

  delay(2000);  // Wait before next reading
}
