/*
  ESP32 FRAM Demo
  esp32-fram-demo.ino
  Writes web page to external Flash Memory
  Uses MB85RC256V FRAM module
  Requires Adafruit_FRAM_I2C Library
  Requires Adafruit_BME280 Library

  How it works:
  1. In setup(), it tries to read the last saved temperature from a known
     address in the FRAM. If valid data is found, it reports this
     "recovered" value to the Serial Monitor.
  2. In the loop(), it reads the current temperature from the BME280
     sensor every 200 milliseconds.
  3. It immediately writes this new temperature value to the FRAM.

  The Demo:
  - Let the sketch run for a few seconds.
  - Unplug the ESP32-C6 to simulate a power failure.
  - Plug it back in. Observe the "Recovered last known temperature"
    message in the Serial Monitor, proving the data survived.

  DroneBot Workshop 2025
  https://dronebotworkshop.com
*/

// Include Required Libraries
#include <Wire.h>
#include <Adafruit_FRAM_I2C.h>
#include <Adafruit_Sensor.h>
#include <Adafruit_BME280.h>

// I2C Pin Configuration (change if required)
const int I2C_SDA_PIN = 10;
const int I2C_SCL_PIN = 11;

// Create the FRAM and BME280 objects
Adafruit_FRAM_I2C fram = Adafruit_FRAM_I2C();
Adafruit_BME280 bme;

// We'll store the temperature (a float, which is 4 bytes)
// at the very beginning of the FRAM memory, address 0.
#define TEMP_ADDRESS 0

void setup() {
  Serial.begin(115200);
  while (!Serial) {
    delay(10);
  }
  delay(1000);
  Serial.println("\n--- FRAM Power-Loss-Proof Logger ---");

  // Initialize the I2C bus with our defined pins.
  Wire.begin(I2C_SDA_PIN, I2C_SCL_PIN);

  // Initialize FRAM
  if (!fram.begin()) {
    Serial.println("Could not find a valid FRAM chip. Check wiring!");
    while (1)
      ;
  }
  Serial.println("FRAM chip initialized.");

  // Initialize BME280 at the detected address 0x77
  if (!bme.begin(0x77)) {
    Serial.println("Could not find a valid BME280 sensor. Check wiring!");
    while (1)
      ;
  }
  Serial.println("BME280 sensor initialized.");

  // --- POWER-ON RECOVERY ---
  // Read the 4 bytes starting at TEMP_ADDRESS and interpret them as a float.
  float lastTemp = 0.0;
  fram.read(TEMP_ADDRESS, (uint8_t*)&lastTemp, sizeof(float));

  // isnan() checks for "Not a Number". Uninitialized FRAM often has this
  if (!isnan(lastTemp) && lastTemp > -50.0 && lastTemp < 100.0) {
    Serial.println("-------------------------------------------");
    Serial.print(">> Recovered last known temperature: ");
    Serial.print(lastTemp);
    Serial.println(" C");
    Serial.println("-------------------------------------------");
  } else {
    Serial.println("No valid previous data found in FRAM.");
  }

  // Slight Delay
  delay(1500);

  Serial.println("Starting real-time logging...");
}

void loop() {
  // Read the current temperature from the BME280
  float currentTemp = bme.readTemperature();

  // Check if the read was successful
  if (isnan(currentTemp)) {
    Serial.println("Failed to read from BME sensor!");
    return;
  }

  // Print the current temperature to the serial monitor
  Serial.print("Logging temperature: ");
  Serial.print(currentTemp);
  Serial.println(" C");

  // Write the new temperature value to FRAM, overwriting the old one.
  // This happens on every loop, demonstrating high write endurance.
  fram.write(TEMP_ADDRESS, (uint8_t*)&currentTemp, sizeof(float));

  // Wait a short amount of time before the next reading.
  delay(200);
}
