/*
  ESP32 External EEPROM Demo
  esp32-eeprom-demo.ino
  Demonstrates use of 24LC256 EEPROM with ESP32
  Uses ESP32-S2 Dev Kit
  
  DroneBot Workshop 2025
  https://dronebotworkshop.com
*/

// Include Wire Library for I2C
#include <Wire.h>

// Define constants
const uint8_t EEPROM_ADDR = 0x50;
const uint8_t SDA_PIN = 21;
const uint8_t SCL_PIN = 33;

// EEPROM Write Page Function
void eepromWritePage(uint16_t addr, const uint8_t* data, size_t len) {
  Wire.beginTransmission(EEPROM_ADDR);
  Wire.write((addr >> 8) & 0xFF);
  Wire.write(addr & 0xFF);
  for (size_t i = 0; i < len; i++) Wire.write(data[i]);
  Wire.endTransmission();

  // ACK polling until write cycle completes (~5 ms)
  while (true) {
    Wire.beginTransmission(EEPROM_ADDR);
    uint8_t err = Wire.endTransmission();
    if (err == 0) break;
    delay(1);
  }
}

// EEPROM Read Block Function
void eepromReadBlock(uint16_t addr, uint8_t* data, size_t len) {
  Wire.beginTransmission(EEPROM_ADDR);
  Wire.write((addr >> 8) & 0xFF);
  Wire.write(addr & 0xFF);
  Wire.endTransmission(false);
  Wire.requestFrom(EEPROM_ADDR, (uint8_t)len);
  for (size_t i = 0; i < len && Wire.available(); i++) {
    data[i] = Wire.read();
  }
}

void setup() {
  
  // Start Serial Monitor
  Serial.begin(115200);
  delay(500);
  
  // Start I2C at 400kHz
  Wire.begin(SDA_PIN, SCL_PIN);
  Wire.setClock(400000);

  // Align to page boundary (multiples of 64)
  const uint16_t base = 0x0040;
  
  // Create 32 values to write to EEPROM 
  uint8_t tx[32];
  for (int i = 0; i < 32; i++) tx[i] = i;

  // Write to EEPROM
  Serial.println("Writing 32 bytes (page write)...");
  eepromWritePage(base, tx, sizeof(tx));
  
  // Read from EEPROM
  Serial.println("Reading back 32 bytes:");
  uint8_t rx[32] = { 0 };
  eepromReadBlock(base, rx, sizeof(rx));

  // Cycle through data & print
  for (int i = 0; i < 32; i++) {
    Serial.printf("%02X ", rx[i]);
    if ((i + 1) % 16 == 0) Serial.println();
  }
  Serial.println();
}

void loop() {}
