/*
  SCD41 CO2 Sensor
  scd41.ino
  Periodic 5 s measurements (CO2 ppm, Temp C, RH%)
  Uses Sensirion Scd4x Library
  Uses Seeeduino XIAO ESP32-S3, adjust pins for other ESP32

  DroneBot Workshop 2025
  https://dronebotworkshop.com
*/

// Include required libraries
#include <Wire.h>
#include <SensirionI2cScd4x.h>

// I2C Pins and Parameters
#define SDA_PIN 5  // XIAO D4
#define SCL_PIN 6  // XIAO D5
#define SCD41_ADDR 0x62

// Create object for sensor
SensirionI2cScd4x scd4x;

// Output formatting
const uint8_t HEADER_EVERY = 12;
uint8_t rowCount = 0;

// Ventilation hint function
const char* ventHint(uint16_t co2) {
  if (co2 >= 1500) return "VENTILATE NOW";
  if (co2 >= 1000) return "Add fresh air";
  if (co2 >= 800) return "Okay";
  return "Good";
}

// Header Print Function
void printHeader() {
  Serial.println();
  Serial.println(F("+----------+----------+---------+--------+---------------+"));
  Serial.println(F("| Time (s) | CO2 ppm  | Temp C  | RH  %  | Ventilation   |"));
  Serial.println(F("+----------+----------+---------+--------+---------------+"));
}

void setup() {
  // Start Serial Monitor
  Serial.begin(115200);
  delay(200);

  // Start I2C
  Wire.begin(SDA_PIN, SCL_PIN);
  Wire.setClock(100000);

  // Start SCD41
  scd4x.begin(Wire, SCD41_ADDR);
  scd4x.stopPeriodicMeasurement();
  delay(500);

  // Start Periodic Measurement
  if (scd4x.startPeriodicMeasurement() != 0) {
    Serial.println("Failed to start SCD41 periodic measurement.");
    while (1) delay(10);
  }

  Serial.println(F("SCD41 measuring... first valid sample in ~5–10 s."));
  printHeader();
}

void loop() {
  // Only read when data-ready to avoid I2C NACKs
  bool ready = false;
  if (scd4x.getDataReadyStatus(ready) == 0 && ready) {
    uint16_t co2 = 0;
    float tC = NAN, rh = NAN;
    if (scd4x.readMeasurement(co2, tC, rh) == 0 && co2 != 0) {
      if (rowCount % HEADER_EVERY == 0) printHeader();
      Serial.printf("| %8lu | %8u | %7.2f | %6.1f | %-13s |\n",
                    millis() / 1000UL, co2, tC, rh, ventHint(co2));
      Serial.printf("  Note: CO2=%u ppm → %s\n\n", co2, ventHint(co2));
      rowCount++;
    }
  }
  delay(200);  // poll lightly; new SCD41 sample about every 5 s
}
