/*
  XIAO ESP32-S3 Integrated Air-Quality Logger
  multisensor.ino
  Uses SCD41, ENS160 & AHT21
  True CO2 from SCD41
  VOC/eCO2/AQI from ENS160
  T/RH from AHT21 (also feeds ENS160 compensation)
  
  DroneBot Workshop 2025
  https://dronebotworkshop.com
*/

// Include required libraries
#include <Wire.h>
#include <SensirionI2cScd4x.h>
#include "SparkFun_ENS160.h"
#include <Adafruit_AHTX0.h>

// I2C
#define SDA_PIN     5
#define SCL_PIN     6
#define SCD41_ADDR  0x62

// Create Objects
SensirionI2cScd4x scd4x;
SparkFun_ENS160   ens;
Adafruit_AHTX0    aht;

// Header format
const uint8_t HEADER_EVERY = 12;
uint8_t rowCount = 0;

// Latest values
uint16_t co2_ppm = 0;
float tC_scd = NAN, rh_scd = NAN;
float tC_aht = NAN, rh_aht = NAN;
uint16_t tvoc_ppb = 0, eco2_ppm = 0;
uint8_t  aqi = 0;

// AQI Function
const char* aqiText(uint8_t v){
  switch (v){ case 1:return "Excellent"; case 2:return "Good"; case 3:return "Moderate";
              case 4:return "Poor"; case 5:return "Unhealthy"; default:return "?"; }
}

// Ventilation Hint Function
const char* ventHint(uint16_t co2){
  if (co2 >= 1500) return "VENTILATE NOW";
  if (co2 >= 1000) return "Add fresh air";
  if (co2 >= 800)  return "Okay";
  return "Good";
}

// Formatting
void printFloatOrDash(float v, uint8_t width, uint8_t prec){
  if (isnan(v)) { for (uint8_t i=0;i<width;i++) Serial.print(i==width/2?'-':' '); }
  else { char buf[20]; dtostrf(v, width, prec, buf); Serial.print(buf); }
}

// Print Header Function
void printHeader(){
  Serial.println();
  Serial.println(F("+----------+----------+---------+--------+---------+--------+-----------+-----------+-----+--------------+"));
  Serial.println(F("| Time (s) | CO2 ppm  | T_SCD C | RH_SCD | T_AHT C | RH_AHT | TVOC ppb  | eCO2 ppm  | AQI | AQI (text)   |"));
  Serial.println(F("+----------+----------+---------+--------+---------+--------+-----------+-----------+-----+--------------+"));
}

void setup() {
  // Start Serial Monitor
  Serial.begin(115200);
  delay(200);

  // Start I2C
  Wire.begin(SDA_PIN, SCL_PIN);
  Wire.setClock(400000);

  // ENS160
  if (!ens.begin()) { Serial.println("ENS160 not found (0x53/0x52)."); while (1) delay(100); }
  ens.setOperatingMode(SFE_ENS160_STANDARD);

  // AHT21
  if (!aht.begin()) { Serial.println("AHT21 not found (0x38)."); while (1) delay(100); }

  // SCD41
  scd4x.begin(Wire, SCD41_ADDR);
  scd4x.stopPeriodicMeasurement();
  delay(500);
  scd4x.startPeriodicMeasurement();

  Serial.println(F("Integrated air-quality monitor running... (SCD41 updates ~5 s)"));
  printHeader();
}

void loop() {
  // --- SCD41: read only when ready ---
  bool ready = false;
  if (scd4x.getDataReadyStatus(ready) == 0 && ready) {
    uint16_t co2; float tC; float rh;
    if (scd4x.readMeasurement(co2, tC, rh) == 0 && co2 != 0) {
      co2_ppm = co2; tC_scd = tC; rh_scd = rh;
    }
  }

  // --- AHT21 ---
  sensors_event_t hum, temp;
  if (aht.getEvent(&hum, &temp)) {
    tC_aht = temp.temperature;
    rh_aht = hum.relative_humidity;
  }

  // --- ENS160 (~1 Hz) ---
  if (ens.checkDataStatus()) {
    tvoc_ppb = ens.getTVOC();
    eco2_ppm = ens.getECO2();
    aqi      = ens.getAQI();
  }

  // --- Print once per second ---
  static uint32_t lastPrint = 0;
  if (millis() - lastPrint >= 1000) {
    lastPrint = millis();
    if (rowCount % HEADER_EVERY == 0) printHeader();

    Serial.print("| ");
    Serial.printf("%8lu | %8u | ", millis() / 1000UL, co2_ppm);
    printFloatOrDash(tC_scd, 7, 2); Serial.print(" | ");
    printFloatOrDash(rh_scd, 6, 1); Serial.print(" | ");
    printFloatOrDash(tC_aht, 7, 2); Serial.print(" | ");
    printFloatOrDash(rh_aht, 6, 1); Serial.print(" | ");
    Serial.printf("%9u | %9u |  %u  | %-12s |\n",
                  tvoc_ppb, eco2_ppm, aqi, aqiText(aqi));

    Serial.printf("  Note: CO2=%u ppm → %s  |  AQI=%u (%s); eCO2 is estimated\n\n",
                  co2_ppm, ventHint(co2_ppm), aqi, aqiText(aqi));

    rowCount++;
  }

  delay(50);
}

