/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef USERS_H
#define USERS_H

#include <QDialog>

class QComboBox;
class QDialogButtonBox;
class DTitleLabel;
class QGroupBox;
class QFormLayout;
class QTreeWidget;
class QTreeWidgetItem;
class QAbstractButton;
class QLineEdit;
class QLabel;

#include "dbms.h"
#include "dmdisubwindow.h"

/**
 * @brief The Users class provides an interface for database user management.
 *
 * This class inherits from DMdiSubWindow and allows administrators to:
 * - Create new user accounts.
 * - Manage existing users and hosts.
 * - Grant or revoke global, database-level, and table-level privileges using a tree view.
 * - Set or change passwords and resource limits.
 */
class Users : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the Users management window.
   * @param serverConnection Pointer to the active DBMS connection.
   */
  Users(DBMS *serverConnection);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels, buttons, and tree headers to reflect the current language.
   */
  void retranslateUI();

private:
  DBMS *serverConnection;
  QComboBox *usernameComboBox;
  QComboBox *hostComboBox;
  QDialogButtonBox *buttonBox;
  DTitleLabel *dTitleLabel;
  QGroupBox *loginGroupBox;
  QFormLayout *loginFormLayout;
  QTreeWidget *privilegesListWidget;
  QList<QTreeWidgetItem *> tables;

  /**
   * @brief Enumeration for identifying the type of item in the privileges tree.
   */
  enum PrivilegeType {
    Global = 1001,      /**< Represents the root or global scope. */
    Element,            /**< Represents a database or table object. */
    Privilege,          /**< Represents a specific permission node (e.g., SELECT). */
    GlobalPrivilege     /**< Represents a server-wide permission. */
  };

  /**
   * @brief Helper to construct the full user string.
   * @return A string in the format 'username'@'host'.
   */
  QString getFullUsername();

  QList<QTreeWidgetItem *> objectsToBeGranted;
  QList<QTreeWidgetItem *> selectedObjectsToBeGranted;

  /**
   * @brief Populates the privileges tree widget.
   *
   * Fetches the current grants for the selected user and displays them hierarchically.
   */
  void fillListWidget();

  QGroupBox * newUserGroupBox;
  QFormLayout *newUserFormLayout;
  QLineEdit *newUsername;
  QLineEdit *newUserHost;
  QLineEdit *newUserPassword;
  QLabel *newUserLimitations;

private slots:
  /**
   * @brief Handles item interaction in the privilege tree.
   *
   * Toggles the grant/revoke state when a checkbox or item is clicked.
   * @param item The tree item clicked.
   * @param column The column clicked.
   */
  void itemActivatedSlot(QTreeWidgetItem *item, int column);

  /**
   * @brief Reloads data when the username selection changes.
   * @param text The new username.
   */
  void usernameCurrentTextChanged(const QString &text);

  /**
   * @brief Reloads data when the host selection changes.
   * @param text The new host string.
   */
  void hostComboBoxCurrentTextChanged(const QString &text);

  /**
   * @brief Handles standard dialog buttons (Save, Cancel, Create).
   * @param button The button that was clicked.
   */
  void actionRoleSlot(QAbstractButton *button);
};

/**
 * @brief The DTreeWidgetDialog class provides a dialog with a checkable tree view.
 *
 * This is a helper dialog used to select specific databases or tables when
 * assigning specific privileges in the Users manager.
 */
class DTreeWidgetDialog : public QDialog
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the selection dialog.
   * @param items A list of TreeWidgetItems to populate the view.
   */
  DTreeWidgetDialog(QList<QTreeWidgetItem *> items);

  /**
   * @brief Retrieves the items selected by the user.
   * @return A list of QStringLists representing the selected data hierarchy.
   */
  QList<QStringList> selectedItems();

private:
  QDialogButtonBox *buttonBox;
  QTreeWidget *dTreeWidgetTables;
  QList<QStringList> itemsSelected;
  QList<QTreeWidgetItem *> items;
};


#endif // USERS_H
