/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QVBoxLayout>
#include <QFormLayout>
#include <QGroupBox>
#include <QCheckBox>
#include <QPushButton>
#include <QDialogButtonBox>
#include <QTreeWidget>
#include <QComboBox>
#include <QLineEdit>
#include <QLabel>

#include "users.h"
#include "dtitlelabel.h"
#include "dicon.h"

Users::Users(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
  setWindowIcon(DIcon::DatabaseUser());

  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  dTitleLabel = new DTitleLabel;
  mainVLayout->addWidget(dTitleLabel);

  loginGroupBox = new QGroupBox;
  loginFormLayout = new QFormLayout;

  usernameComboBox = new QComboBox();
  connect(usernameComboBox, SIGNAL(currentTextChanged(QString)), this, SLOT(usernameCurrentTextChanged(QString)));
  hostComboBox = new QComboBox;
  connect(hostComboBox, SIGNAL(currentTextChanged(QString)), this, SLOT(hostComboBoxCurrentTextChanged(QString)));
  usernameComboBox->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  usernameComboBox->setEditable(true);
  usernameComboBox->insertItems(0, this->serverConnection->user()->getUserList());

  hostComboBox->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  hostComboBox->setToolTip(tr("List of aviable hosts"));
  hostComboBox->setStatusTip(hostComboBox->toolTip());
  hostComboBox->setWhatsThis(hostComboBox->toolTip());
  hostComboBox->setEditable(true);
  hostComboBox->insertItems(0, this->serverConnection->user()->getUserList());

  loginFormLayout->addRow(" ", usernameComboBox);
  loginFormLayout->addRow(" ", hostComboBox);
  loginGroupBox->setLayout(loginFormLayout);
  mainVLayout->addWidget(loginGroupBox);

  if (this->serverConnection->user()->isRootAndGrantor()) {
    newUserGroupBox = new QGroupBox;
    newUserFormLayout = new QFormLayout;
    newUsername = new QLineEdit;
    newUserPassword = new QLineEdit;
    newUserPassword->setEchoMode(QLineEdit::Password);
    newUserLimitations = new QLabel;
    newUserHost = new QLineEdit;

    newUserFormLayout->addRow(" ", newUsername);
    newUserFormLayout->addRow(" ", newUserPassword);
    newUserFormLayout->addRow(" ", newUserHost);
    newUserFormLayout->addRow(" ", newUserLimitations);
    newUserGroupBox->setLayout(newUserFormLayout);
    mainVLayout->addWidget(newUserGroupBox);
  }
  privilegesListWidget = new QTreeWidget();
  privilegesListWidget->setHeaderItem(new QTreeWidgetItem(QStringList() << "" << ""));
  connect(privilegesListWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(itemActivatedSlot(QTreeWidgetItem*,int)));
  mainVLayout->addWidget(privilegesListWidget);

  fillListWidget();

  if (this->serverConnection->user()->isRootAndGrantor()) {
    buttonBox = new QDialogButtonBox();
    buttonBox->addButton(tr("Add object"), QDialogButtonBox::ActionRole);
    buttonBox->addButton(tr("Add user"), QDialogButtonBox::ActionRole);
    buttonBox->addButton(tr("Clear list"), QDialogButtonBox::ActionRole);
    connect(buttonBox, SIGNAL(clicked(QAbstractButton*)), this, SLOT(actionRoleSlot(QAbstractButton*)));
    mainVLayout->addWidget(buttonBox);
  }

  QWidget *widMain = new QWidget;
  widMain->setLayout(mainVLayout);
  setWidget(widMain);
  retranslateUI();
}

void Users::retranslateUI()
{
  setWindowTitle(tr("User Administration"));
  setObjectName(windowTitle());
  dTitleLabel->setText(windowTitle());
  dTitleLabel->setToolTip(dTitleLabel->text());
  loginGroupBox->setTitle(tr("Login informaction"));
  loginGroupBox->setToolTip(loginGroupBox->title());
  QLabel *label = qobject_cast<QLabel *>(loginFormLayout->labelForField(usernameComboBox));
  label->setText(tr("Username:"));
  label->setToolTip(label->text());
  label = qobject_cast<QLabel *>(loginFormLayout->labelForField(hostComboBox));
  label->setText(tr("Host:"));
  label->setToolTip(label->text());
  newUserGroupBox->setTitle(tr("User creation"));
  newUserGroupBox->setToolTip(newUserGroupBox->title());
  label = qobject_cast<QLabel *>(newUserFormLayout->labelForField(newUsername));
  label->setText(tr("Username:"));
  label->setToolTip(label->text());
  label = qobject_cast<QLabel *>(newUserFormLayout->labelForField(newUserPassword));
  label->setText(tr("Password:"));
  label->setToolTip(label->text());
  label = qobject_cast<QLabel *>(newUserFormLayout->labelForField(newUserHost));
  label->setText(tr("Host:"));
  label->setToolTip(label->text());
  label = qobject_cast<QLabel *>(newUserFormLayout->labelForField(newUserLimitations));
  label->setText(tr("Limitatios:"));
  label->setToolTip(label->text());
  newUserLimitations->setText(tr("If you need to set options like SSL, authentication plugins, password expiration please use the Query window"));
  privilegesListWidget->headerItem()->setText(1, tr("Context"));
  privilegesListWidget->headerItem()->setText(2, tr("Comment"));
  privilegesListWidget->setToolTip(privilegesListWidget->windowTitle());
  privilegesListWidget->setHeaderLabel(privilegesListWidget->windowTitle());
  privilegesListWidget->expandItem(tables.first());
  privilegesListWidget->resizeColumnToContents(0);
  privilegesListWidget->resizeColumnToContents(1);
//  privilegesListWidget->collapseItem(tables.first());
}

QString Users::getFullUsername()
{
  return "`" + usernameComboBox->currentText() + "`@`" + hostComboBox->currentText() + "`";
}

void Users::fillListWidget()
{
  /*
   * Strangely enough if I replace the name of the variable, I got tons of strange errors.
  */
  tables.clear();
  QTreeWidgetItem *item;
  QTreeWidgetItem *itemChild;
  QList<QStringList> *privileges;
  if (this->serverConnection->user()->isRootAndGrantor()) {
    item = new QTreeWidgetItem((QTreeWidget*)0, QStringList() << tr("Global") << tr("Privilege"), PrivilegeType::Global);
    item->setIcon(0, DIcon::Database());
    tables.append(item);
    privileges = serverConnection->getPrivileges();
    for (int counter = 0; counter < privileges->count(); counter++) {
      itemChild = new QTreeWidgetItem(item, QStringList() << privileges->at(counter).at(0)
                                      << privileges->at(counter).at(1)
                                      << privileges->at(counter).at(2), PrivilegeType::GlobalPrivilege);
      itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      itemChild->setIcon(0, DIcon::DatabaseTable());
      itemChild->setCheckState(0, Qt::Unchecked);
      tables.append(itemChild);
    }
  } else {
    QString privilegeTMP;
    foreach (QString grant, this->serverConnection->user()->getGrants()) {
      if (grant.endsWith("WITH GRANT OPTION")) {
        item = new QTreeWidgetItem((QTreeWidget*)0, QStringList(grant.split(" ON ").at(1).split(" TO ").at(0)), PrivilegeType::Element);
        item->setIcon(0, DIcon::Database());
        tables.append(item);
        privilegeTMP = grant.split(" ON ").at(0);
        foreach (QString priv, privilegeTMP.right(privilegeTMP.length() - 6).split(", ")) {
          itemChild = new QTreeWidgetItem(item, QStringList(priv), PrivilegeType::Privilege);
          itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
          itemChild->setIcon(0, DIcon::DatabaseTable());
          itemChild->setCheckState(0, Qt::Unchecked);
          tables.append(itemChild);
        }
      }
    }
  }
  privilegesListWidget->clear();
  privilegesListWidget->insertTopLevelItems(0, tables);
  privilegesListWidget->resizeColumnToContents(0);
}

void Users::itemActivatedSlot(QTreeWidgetItem *item, int column)
{
  if (!item->parent())
    return;
  QString statement;
  if (item->checkState(column)) {
    statement = "GRANT " + item->text(column)
                + " ON "
                + (item->type() == PrivilegeType::Privilege ? item->parent()->text(column) : "*.*")
                + " TO "
                + getFullUsername();
  } else {
    statement = "REVOKE "
                + item->text(column)
                + " ON "
                + (item->type() == PrivilegeType::Privilege ? item->parent()->text(column) : "*.*")
                + " FROM "
                + getFullUsername();
  }
  qDebug() << statement;
  this->serverConnection->executeQuery(statement);
}

void Users::usernameCurrentTextChanged(const QString &text)
{
  hostComboBox->clear();
  hostComboBox->insertItems(0, this->serverConnection->user()->getUserHostList(text));
}

void Users::hostComboBoxCurrentTextChanged(const QString &text)
{
  Q_UNUSED(text);
  if (!hostComboBox->currentText().isEmpty()) {
    foreach (QTreeWidgetItem *item, tables) {
      if (item->type() == PrivilegeType::Privilege
          || item->type() == PrivilegeType::GlobalPrivilege) {
        if (this->serverConnection->user(getFullUsername())->hasGrant(item->text(0)
                                                                      , item->type() == PrivilegeType::Privilege
                                                                        ? item->parent()->text(0)
                                                                        : "*.*"
                                                                      )) {
          item->setCheckState(0, Qt::Checked);
        } else {
          item->setCheckState(0, Qt::Unchecked);
        }
      }
    }
  }
}

void Users::actionRoleSlot(QAbstractButton *button)
{
  if (button->text() == tr("Add object") && this->serverConnection->user()->isRootAndGrantor()) {
    privilegesListWidget->collapseItem(tables.first());
    QList<QStringList> *objects;
    objects = this->serverConnection->runQuery("SELECT 'DATABASE' AS `ItemType`, CONCAT('`', `SCHEMA_NAME`, '`.*') AS `Object` FROM `information_schema`.`SCHEMATA`"
                                               "UNION SELECT `TABLE_TYPE` AS `ItemType`, CONCAT('`', `TABLE_SCHEMA`, '`.`', `TABLE_NAME`, '`') AS `Object` FROM `information_schema`.`TABLES` WHERE `TABLE_TYPE` NOT IN ('SYSTEM VIEW')"
                                               "UNION SELECT `ROUTINE_TYPE` AS `ItemType`, CONCAT('`', `ROUTINE_SCHEMA`, '`.`', `ROUTINE_NAME`, '`') AS `Object` FROM `information_schema`.`ROUTINES`"
                                               "ORDER BY `Object`");
    objects->takeLast();
    objectsToBeGranted.clear();
    QTreeWidgetItem *item = new QTreeWidgetItem();
    QTreeWidgetItem *itemChild;
    for (int row = 0; row < objects->count(); row++) {
      if (objects->at(row).at(0) == "DATABASE") {
        item = new QTreeWidgetItem((QTreeWidget*)0, QStringList() << objects->at(row).at(1) << objects->at(row).at(0), PrivilegeType::Privilege);
        item->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
        item->setIcon(0, DIcon::DatabaseTable());
        item->setCheckState(0, Qt::Unchecked);
        objectsToBeGranted.append(item);
      } else {
        itemChild = new QTreeWidgetItem(item, QStringList() << objects->at(row).at(1) << objects->at(row).at(0), PrivilegeType::Privilege);
        itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
        itemChild->setIcon(0, DIcon::DatabaseTable());
        itemChild->setCheckState(0, Qt::Unchecked);
        objectsToBeGranted.append(itemChild);
      }
    }
    DTreeWidgetDialog *dTreeWidgetDialog = new DTreeWidgetDialog(objectsToBeGranted);
    if (dTreeWidgetDialog->exec() == QDialog::Accepted) {
      QList<QStringList> *privileges;
      QStringList databasePrivileges;
      QStringList procedurePrivileges;
      QStringList baseTablePrivileges;
      QStringList viewPrivileges;
      QStringList functionPrivileges;
      privileges = serverConnection->getPrivileges();
      //DATABASE, PROCEDURE, BASE TABLE, VIEW, FUNCTION
      for (int counter = 0; counter < privileges->count(); counter++) {
        if (privileges->at(counter).at(1).contains("DATABASE", Qt::CaseInsensitive)) {
          databasePrivileges.append(privileges->at(counter).at(0));
        }
        if (privileges->at(counter).at(1).contains("PROCEDURE", Qt::CaseInsensitive)) {
          procedurePrivileges.append(privileges->at(counter).at(0));
        }
        if (privileges->at(counter).at(1).contains("FUNCTION", Qt::CaseInsensitive)) {
          functionPrivileges.append(privileges->at(counter).at(0));
        }
        if (privileges->at(counter).at(1).contains("TABLE", Qt::CaseInsensitive)) {
          baseTablePrivileges.append(privileges->at(counter).at(0));
        }
        if (privileges->at(counter).at(1).contains("TABLE", Qt::CaseInsensitive)) {
          viewPrivileges.append(privileges->at(counter).at(0));
        }
      }
      selectedObjectsToBeGranted.clear();
      foreach (QList selectedItem, dTreeWidgetDialog->selectedItems()) {
        item = new QTreeWidgetItem((QTreeWidget*)0, QStringList() << selectedItem.at(0) << tr("Privilege"), PrivilegeType::Element);
        //          item->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
        //          item->setIcon(0, DIcon::DatabaseTable());
        //          item->setCheckState(0, Qt::Unchecked);
        selectedObjectsToBeGranted.append(item);
//        qDebug() << item->text(0) << item->text(1);
        if (selectedItem.at(1).contains("DATABASE", Qt::CaseInsensitive)) {
          databasePrivileges << procedurePrivileges << baseTablePrivileges;
          foreach (QString priv, databasePrivileges) {
            itemChild = new QTreeWidgetItem(item, QStringList() << priv << selectedItem.at(1), PrivilegeType::Privilege);
            itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
            itemChild->setIcon(0, DIcon::Database());
            item->setIcon(0, DIcon::Database());
            itemChild->setCheckState(0, Qt::Unchecked);
            selectedObjectsToBeGranted.append(itemChild);
          }
        }
        if (selectedItem.at(1).contains("PROCEDURE", Qt::CaseInsensitive)) {
          foreach (QString priv, procedurePrivileges) {
            itemChild = new QTreeWidgetItem(item, QStringList() << priv << selectedItem.at(1), PrivilegeType::Privilege);
            itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
            itemChild->setIcon(0, DIcon::DatabaseRoutine());
            item->setIcon(0, DIcon::DatabaseRoutine());
            itemChild->setCheckState(0, Qt::Unchecked);
            selectedObjectsToBeGranted.append(itemChild);
          }
        }
        if (selectedItem.at(1).contains("FUNCTION", Qt::CaseInsensitive)) {
          foreach (QString priv, functionPrivileges) {
            itemChild = new QTreeWidgetItem(item, QStringList() << priv << selectedItem.at(1), PrivilegeType::Privilege);
            itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
            itemChild->setIcon(0, DIcon::DatabaseRoutine());
            item->setIcon(0, DIcon::DatabaseRoutine());
            itemChild->setCheckState(0, Qt::Unchecked);
            selectedObjectsToBeGranted.append(itemChild);
          }
        }
        if (selectedItem.at(1).contains("TABLE", Qt::CaseInsensitive)) {
          foreach (QString priv, baseTablePrivileges) {
            itemChild = new QTreeWidgetItem(item, QStringList() << priv << selectedItem.at(1), PrivilegeType::Privilege);
            itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
            itemChild->setIcon(0, DIcon::DatabaseTable());
            item->setIcon(0, DIcon::DatabaseTable());
            itemChild->setCheckState(0, Qt::Unchecked);
            selectedObjectsToBeGranted.append(itemChild);
          }
        }
        if (selectedItem.at(1).contains("VIEW", Qt::CaseInsensitive)) {
          foreach (QString priv, viewPrivileges) {
            itemChild = new QTreeWidgetItem(item, QStringList() << priv << selectedItem.at(1), PrivilegeType::Privilege);
            itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
            itemChild->setIcon(0, DIcon::DatabaseView());
            item->setIcon(0, DIcon::DatabaseView());
            itemChild->setCheckState(0, Qt::Unchecked);
            selectedObjectsToBeGranted.append(itemChild);
          }
        }
      }
      privilegesListWidget->insertTopLevelItems(0, selectedObjectsToBeGranted);
      privilegesListWidget->resizeColumnToContents(0);
    }
  }
  if (button->text() == tr("Clear list")) {
    privilegesListWidget->clear();
    fillListWidget();
  }
  if (button->text() == tr("Add user")) {
    this->serverConnection->executeQuery("CREATE USER `" + newUsername->text() + "`@`" + newUserHost->text() + "` IDENTIFIED BY '" + newUserPassword->text() + "'");
    usernameComboBox->clear();
    usernameComboBox->insertItems(0, this->serverConnection->user()->getUserList());
    privilegesListWidget->clear();
    fillListWidget();
    newUsername->clear();
    newUserHost->clear();
    newUserPassword->clear();
  }
}

DTreeWidgetDialog::DTreeWidgetDialog(QList<QTreeWidgetItem *> items)
{
  this->items = items;
  setWindowTitle(tr("Select objects"));
  setObjectName(windowTitle());
  setWindowIcon(DIcon::DatabaseTable());

  buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel);
  connect(buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
  connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));

  dTreeWidgetTables = new QTreeWidget();
  dTreeWidgetTables->insertTopLevelItems(0, this->items);
  dTreeWidgetTables->headerItem()->setText(0, tr("Object"));
  dTreeWidgetTables->headerItem()->setText(1, tr("Context"));
  dTreeWidgetTables->setToolTip(dTreeWidgetTables->windowTitle());
  dTreeWidgetTables->setHeaderLabel(dTreeWidgetTables->windowTitle());
  dTreeWidgetTables->expandItem(this->items.first());
  dTreeWidgetTables->resizeColumnToContents(0);
  dTreeWidgetTables->resizeColumnToContents(1);

  QVBoxLayout *verticalLayout = new QVBoxLayout;
  verticalLayout->addWidget(new DTitleLabel(windowTitle()));
  verticalLayout->addWidget(dTreeWidgetTables);
  verticalLayout->addWidget(buttonBox);
  setLayout(verticalLayout);
}

QList<QStringList> DTreeWidgetDialog::selectedItems()
{
  itemsSelected.clear();
  foreach (QTreeWidgetItem *item, this->items) {
    if (item->checkState(0) == Qt::Checked) {
      itemsSelected.append(QStringList()  << item->text(0) << item->text(1));
    }
  }
  return itemsSelected;
}
