/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef TEXTEDITOR_H
#define TEXTEDITOR_H

#include <QSystemTrayIcon>
#include <QPrinter>
#include <QWidget>

#include "editortypes.h"
#include "dsettings.h"
//#include "dmdisubwindow.h"

class QCompleter;
class QStandardItemModel;
class QToolBar;
class QSignalMapper;
class QActionGroup;
class Projects;
class DToolTip;
class SubversionedFile;
class QMenuBar;
class FindReplace;
class QSplitter;
class QListWidget;
class QListWidgetItem;
class QLineEdit;
class QLabel;
class QMenu;
class QAction;
class BaseTextEditor;
class QTextBlock;
class DTitleLabel;
class DBMS;
class QTextDocument;

/**
 * @brief The TextEditor class provides a comprehensive code editing widget.
 *
 * This class wraps a `BaseTextEditor` to provide a full-featured text editing experience.
 * It includes:
 * - Syntax highlighting (SQL, Diff, etc.) via EditorType.
 * - Auto-completion for SQL keywords, functions, and database symbols.
 * - Code snippets and templates.
 * - File operations (Open, Save, Export to PDF/HTML).
 * - View controls (Zoom, Word Wrap, Split View).
 * - Version control integration (SVN).
 * - Search and Replace functionality.
 */
class TextEditor : public QWidget
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the TextEditor widget.
   * @param project Pointer to the active project context.
   * @param serverConnection Pointer to the active DBMS connection (for metadata).
   * @param type The type of editor (SQL, Diff, Plain Text, etc.).
   * @param windowCount A unique identifier for the window instance.
   * @param hideTitle Whether to hide the title header widget.
   */
  TextEditor(Projects *project, DBMS *serverConnection, EditorTypes::EditorType type, unsigned int windowCount = 0, bool hideTitle = false);

  /**
   * @brief Checks if the editor can be safely closed.
   *
   * Prompts the user to save changes if the document is modified.
   * @return True if safe to close, false if canceled.
   */
  bool okToClose();
//  void gotoLine(unsigned int lineNumber = 0);

  /**
   * @brief The absolute path to the currently open file.
   */
  QString fileName;

  QAction *reloadFile;
  BaseTextEditor *textEditor;

  /**
   * @brief Moves the cursor to the specified line number.
   * @param line The line number to jump to.
   */
  void gotoLine(unsigned int line);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates menus, tooltips, and actions to match the current language.
   */
  void retranslateUI();

  /**
   * @brief Sets the content of the editor.
   * @param text The new text string.
   */
  void setPlainText(const QString &text);

  /**
   * @brief Retrieves the underlying text document.
   * @return Pointer to the QTextDocument.
   */
  QTextDocument *document();

  /**
   * @brief Retrieves the list of main menus associated with this editor.
   * @return A list of QMenu pointers (File, Edit, View, etc.).
   */
  QList<QAction *> getMenus();

protected:
//  void keyPressEvent(QKeyEvent *event);
//  void mousePressEvent(QMouseEvent *event);
//  void contextMenuEvent(QContextMenuEvent *event);

  /**
   * @brief Handles mouse wheel events (e.g., for zooming).
   * @param e The wheel event parameters.
   */
  void wheelEvent(QWheelEvent *e);

  /**
   * @brief General event handler.
   * @param event The event object.
   * @return True if the event was handled.
   */
  bool event(QEvent *event);
//  void showEvent(QShowEvent *event);
//  void closeEvent(QCloseEvent *event);
//  void closeEvent(QCloseEvent *event);

signals:
  /**
   * @brief Emitted when the window title should be updated (e.g., filename change).
   * @param title The new title.
   */
  void changeTitle(QString title);

  /**
   * @brief Emitted to display a message on the status bar.
   * @param message The message text.
   * @param icon The icon type.
   * @param timeout The duration to show the message.
   */
  void statusBarMessage(const QString &message, QSystemTrayIcon::MessageIcon icon = QSystemTrayIcon::Information, int timeout = 0);

  /**
   * @brief Emitted to update the cursor position indicator.
   * @param x Column number.
   * @param y Row number.
   * @param z Selection length.
   */
  void updatePrositionViewer(const int x, const int y, const int z);

  /**
   * @brief Emitted to open a URL in an external browser.
   */
  void openURL(QString url);

  /**
   * @brief Emitted when the editor window is closed.
   */
  void windowClosed();

  /**
   * @brief Emitted when zooming in.
   */
  void zoomInSignal();

  /**
   * @brief Emitted when zooming out.
   */
  void zoomOutSignal();

  /**
   * @brief Emitted when resetting zoom.
   */
  void zoomResetSignal();

public slots:
  /**
   * @brief Saves the current file to disk.
   * @return True on success.
   */
  bool saveFileActionTriggered();

  /**
   * @brief Prompts for a filename and saves the file ("Save As").
   * @return True on success.
   */
  bool saveFileAsActionTriggered();

  /**
   * @brief Opens a file dialog to load a file.
   */
  void openFileActionTriggered();

  /**
   * @brief Clears the text editor content.
   */
  void clear();

  /**
   * @brief Cuts the current line to the clipboard.
   */
  void cutLineActionTriggerd();

  /**
   * @brief Refreshes the autocompleter with MariaDB symbols.
   */
  void fillMariaDBSymbolsActionSlot();

  /**
   * @brief Sets the editor view to normal (single pane) mode.
   */
  void viewNormalActionTriggered();

  /**
   * @brief Splits the editor view vertically.
   */
  void viewVerticalActionTriggered();

  /**
   * @brief Splits the editor view horizontally.
   */
  void viewHorizoltalActionTriggered();

  /**
   * @brief Adds the current selection to the code snippets library.
   */
  void addCodeToSnippetActionSlot();

  /**
   * @brief Opens a specific file.
   * @param file The path to the file.
   */
  void openFile(QString file = QString());

private slots:
  /**
   * @brief Inserts the selected completion suggestion into the text.
   * @param completion The text to insert.
   */
  void insertCompletion(QString completion);

  /**
   * @brief Triggers the auto-completion popup manually.
   */
  void performCompletion();

  /**
   * @brief Toggles word wrapping.
   */
  void wordWrapActionToggled();

  void zoomIn();
  void zoomOut();
  void zoomReset();

  /**
   * @brief Shows the "Go to Line" dialog.
   */
  void gotoLineActionTriggered();

  /**
   * @brief Prepares the symbols menu before showing (for navigation).
   */
  void symbolsMenuAboutToShowSlot();

  /**
   * @brief Prepares the recent files menu before showing.
   */
  void openRecentFilesMenuAboutToShowSlot();

  /**
   * @brief Clears the recent files history.
   */
  void clearRecentFilesActionTriggered();

  /**
   * @brief Opens a file selected from the recent files menu.
   * @param action The action representing the file.
   */
  void recentFilesActionGroupTriggered(QAction *action);

  /**
   * @brief Handles jump-to-symbol actions.
   */
  void gotoLineActionGroupTriggered(QAction *action);

  /**
   * @brief Prints the document content.
   */
  void printActionTriggerd();

  /**
   * @brief Reloads the file from disk, discarding changes.
   */
  void reloadFileActionTriggerd();

  /**
   * @brief Updates UI state when the document modification state changes.
   * @param changed True if modified, false if saved.
   */
  void modificationChangedSlot(bool changed);

  /**
   * @brief Exports the document to PDF or PostScript.
   */
  void exportoToPdfOrPsActionTriggerd();

  /**
   * @brief Exports the document to OpenDocument Text (ODT).
   */
  void exportoToOdtActionTriggerd();

  /**
   * @brief Exports the document to HTML.
   */
  void exportoToHtmlActionTriggerd();

  /**
   * @brief Exports the document as an SVG image.
   */
  void exportoToSvgActionTriggerd();

  /**
   * @brief Exports the document as a raster image (PNG/JPG).
   */
  void exportoToImgActionTriggerd();

  /**
   * @brief Toggles comments on the selected lines.
   */
  void toggleCommentActionTriggerd();

  /**
   * @brief Displays context-sensitive help for the word under cursor.
   */
  void showMariaDBHlperActionTriggered();

  /**
   * @brief Toggles highlighting of the current line background.
   */
  void highlightCurrentLineActionTriggered();

  /**
   * @brief Updates the status bar with the current cursor position.
   */
  void updateCursorPositionSlot();

  /**
   * @brief Toggles the visibility of the symbols side pane.
   * @param checked True to show, false to hide.
   */
  void showSymbolsActionTriggered(bool checked = false);

  /**
   * @brief Jumps to the symbol selected in the side pane.
   * @param item The list item clicked.
   */
  void itemClickedSlot(QListWidgetItem *item);

  /**
   * @brief Sorts the symbols list alphabetically.
   */
  void sortSymbolsSlot();

  /**
   * @brief Normalizes text (e.g., cleans whitespace).
   */
  void normalizeTextActionTriggered();
//  void readyReadStandardOutputSlot();

  /**
   * @brief Updates the window title.
   * @param title The new title string.
   */
  void setTitle(QString title);

  /**
   * @brief Converts selected text (or keywords) to uppercase.
   */
  void uppercaseKeywordsActionSlot();

  /**
   * @brief Converts selected text (or keywords) to lowercase.
   */
  void lowercaseKeywordsActionSlot();

  /**
   * @brief Handles smart completion after typing a period (e.g., table.column).
   * @param completionPrefix The text preceding the completion.
   */
  void performPeriodCompletion(const QString &completionPrefix);

  /**
   * @brief Inserts the result of a period-triggered completion.
   */
  void insertCompletionForPeriodCompeltion(const QString &completion);
//  void readyReadStandardErrorSlot();

  /**
   * @brief Triggers the code snippet completion popup.
   */
  void performCodeSnippetCompletion();

  /**
   * @brief Inserts the selected code snippet.
   */
  void insertCompletionCodeSnippet(const QString &completion);

  /**
   * @brief Slot for actions that are placeholders or not yet implemented.
   */
  void dummyExecutionActionSlot();

  /**
   * @brief Changes the font weight of the editor text.
   * @param action The action representing the selected weight (Bold, Light, etc.).
   */
  void changeFontWeightActionGroupTriggered(QAction *action);

private:
  DBMS *serverConnection;
  QCompleter *completer;
  bool completedAndSelected;

  /**
   * @brief Internal helper to show the completer.
   * @param completionPrefix The text prefix to match.
   */
  void performCompletion(const QString &completionPrefix);

  /**
   * @brief Populates the completer model with relevant keywords.
   */
  void populateModel(const QString &completionPrefix);

  QStandardItemModel *model;

  /**
   * @brief Retrieves keywords for completion.
   */
  QStringList keywords();

  /**
   * @brief Retrieves functions for completion.
   */
  QStringList functions();

  /**
   * @brief Retrieves data types for completion.
   */
  QStringList datatypes();

  /**
   * @brief Retrieves charsets for completion.
   */
  QStringList charsets();

  /**
   * @brief Initializes all QActions.
   */
  void createActions();

  Projects *project;
  EditorTypes::EditorType editorType;
  QHash<QString, int> symbolsLineNumber;

  /**
   * @brief Updates the recent files list in settings.
   */
  void updateRecentFiles(QString fileToSave = QString());

  DSettings settings;
  QFont newFont;

  /**
   * @brief Applies font settings to editor widgets.
   */
  void setFontToWidgets(const QFont &);

  bool saveAs;

  /**
   * @brief Determines the file extension.
   */
  QString fileExtention();

  QStringList databaseSymbols;
  QPrinter printer;
  QAction *copyAction;
  QAction *cutAction;
  QAction *cutLineAction;
  QAction *pasteAction;
  QAction *undoAction;
  QAction *redoAction;
  QAction *showHideLineNumbersAction;
  QAction *wordWrapAction;
  QAction *gotoLineAction;
  QAction *toggleCommentAction;

  /**
   * @brief Helper to get the comment syntax for the current language.
   */
  inline QString commentString();

  QAction *clearAction;
  QAction *openAction;
  QAction *saveAction;
  QAction *saveAsAction;
  QAction *zoomInAction;
  QAction *zoomOutAction;
  QAction *restoreZoomAction;
  QAction *exportoToPdfOrPsAction;
  QAction *exportoToOdtAction;
  QAction *exportoToHtmlAction;
  QAction *exportoToSvgAction;
  QAction *exportoToImgAction;
  QAction *selectAllAction;
  QAction *clearRecentFilesAction;
  QActionGroup *recentFilesActionGroup;
  QActionGroup *gotoLineActionGroup;
  QAction *highlightCurrentLineAction;
  QAction *normalizeTextAction;
  SubversionedFile *subversionedFile;

  /**
   * @brief Initializes the main menu bar.
   */
  void createMenu();

  QToolBar *mainToolBar;

  /**
   * @brief Initializes the toolbar.
   */
  void createToolBars();

  QMenuBar *menuBar;
  QMenu *fileMenu;
  QMenu *editMenu;
  QMenu *optionsMenu;
  QMenu *subversionMenu;
  QMenu *viewMenu;
  FindReplace *findReplace;
  QSplitter *secondSplitter;
  QListWidget *listSymbols;
  QAction *viewNormalAction;
  QAction *viewVerticalAction;
  QAction *viewHorizoltalAction;
  BaseTextEditor *readOnlyTextEditor;

  /**
   * @brief Syncs content to the secondary read-only view (for splits).
   */
  void updateReadOnlyEditor();

  QAction *showSymbolsAction;
  QLineEdit *lineEditFile;
  QLabel *labelPosition;
  QAction *sortSymbols;

  /**
   * @brief Scans the document to populate the symbols list.
   */
  void fillMariaDBSymbols();

  /**
   * @brief Reopens the file that was open when the editor last closed.
   */
  void openLastOpenedFile();

  QMenu *openRecentFilesMenu;
  QMenu *contextualMenu;
  QMenu *symbolsMenu;
  QMenu *exportMenu;
  QAction *printAction;
  QAction *fillMariaDBSymbolsAction;

  // Action grouping helpers
  QList<QAction *> editActionList();
  QList<QAction *> viewActionList();
  QList<QAction *> fileActionList();

  QAction *showMariaDBHelperAction;
  QAction *disableCompletionAction;
//  QProcess *proc;
  BaseTextEditor *procOutput;
  QAction *contextualMenuSeparatorAction;
  QAction *viewMenuSeparatorActionView;
  QAction *viewMenuSeparatorActionSymbols;
  unsigned int windowCount;
  DTitleLabel *dTitleLabel;
  QAction *uppercaseKeywordsAction;
  QAction *lowercaseKeywordsAction;

  /**
   * @brief Helper to change case of selected text.
   */
  void changeTextCase(bool uppercse = true);

  QStandardItemModel *modelForPeriodCompeltion;
  QCompleter *completerForPeriodCompeltion;
  QMenu *versionControlMenu;
  QCompleter *codeSnippetCompleter;
  QStandardItemModel *modelCodeSnippet;
  QAction *addCodeToSnippetAction;
  QAction *mainToolBarAction;
  QAction *showCodeSnippets;

  /**
   * @brief Loads a file line by line (used for large files or processing).
   */
  void readFileLineByLine(QString fileName);

  QAction *dummyExecutionAction;
  QAction *changeFontWeightMenuSeparatorAction;
  QActionGroup *changeFontWeightActionGroup;
  QAction *changeFontWeightActionThin;
  QAction *changeFontWeightActionExtraLight;
  QAction *changeFontWeightActionLight;
  QAction *changeFontWeightActionNormal;
  QAction *changeFontWeightActionMedium;
  QAction *changeFontWeightActionDemiBold;
  QAction *changeFontWeightActionBold;
  QAction *changeFontWeightActionExtraBold;
  QAction *changeFontWeightActionBlack;
};

#endif // TEXTEDITOR_H
