/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QCompleter>
#include <QShortcut>
#include <QScrollBar>
#include <QAbstractItemView>
#include <QStandardItemModel>
#include <QFileDialog>
#include <QMenu>
#include <QApplication>
#include <QTextStream>
#include <QMessageBox>
#include <QInputDialog>
#include <QPrinter>
#include <QToolTip>
#include <QTextDocumentWriter>
#include <QtSvg/QSvgGenerator>
#include <QPainter>
#include <QImageWriter>
#include <QListWidget>
#include <QMenuBar>
#include <QVBoxLayout>
#include <QSplitter>
#include <QLabel>
#include <QWhatsThis>
#include <QPrintDialog>
#include <QActionGroup>
#include <QTextBlock>
#include <QTextDocumentFragment>

#include "texteditor.h"
#include "editortypes.h"
#include "staticfunctions.h"
#include "findreplace.h"
#include "subversionedfile.h"
#include "basetexteditor.h"
#include "projects.h"
#include "dtitlelabel.h"
#include "dbms.h"
#include "codesnippets.h"
#include "dicon.h"

#include "QDebug"

TextEditor::TextEditor(Projects *project, DBMS *serverConnection, EditorTypes::EditorType type, unsigned int windowCount, bool hideTitle)
{
  this->project = project;
  editorType = type;
  this->serverConnection = serverConnection;
  setWindowIcon(StaticFunctions::iconFromFileName(type));
//  homeKeyHitedTwice = true;
  this->windowCount = windowCount;
  readOnlyTextEditor = new BaseTextEditor(type);
  readOnlyTextEditor->setReadOnly(true);
  textEditor = new BaseTextEditor(type);
  connect(textEditor, SIGNAL(openFile(QString)), this, SLOT(openFile(QString)));
  connect(this, SIGNAL(changeTitle(QString)), this, SLOT(setTitle(QString)));
  connect(textEditor, SIGNAL(performPeriodCompletion(QString)), this, SLOT(performPeriodCompletion(QString)));
  model = new QStandardItemModel(this);
  completer = new QCompleter(this);
  completer->setWidget(this);
  completer->setCompletionMode(QCompleter::PopupCompletion);
  completer->setModel(model);
  completer->setModelSorting(QCompleter::CaseInsensitivelySortedModel);
  completer->setCaseSensitivity(Qt::CaseInsensitive);
  completer->setFilterMode(Qt::MatchContains);
  completer->setWrapAround(true);
  connect(completer, SIGNAL(activated(QString)), this, SLOT(insertCompletion(QString)));

  modelForPeriodCompeltion = new QStandardItemModel(this);
  completerForPeriodCompeltion = new QCompleter(this);
  completerForPeriodCompeltion->setWidget(this);
  completerForPeriodCompeltion->setCompletionMode(QCompleter::PopupCompletion);
  completerForPeriodCompeltion->setModel(modelForPeriodCompeltion);
  completerForPeriodCompeltion->setModelSorting(QCompleter::CaseInsensitivelySortedModel);
  completerForPeriodCompeltion->setCaseSensitivity(Qt::CaseInsensitive);
  completerForPeriodCompeltion->setFilterMode(Qt::MatchContains);
  completerForPeriodCompeltion->setWrapAround(true);
  connect(completerForPeriodCompeltion, SIGNAL(activated(QString)), this, SLOT(insertCompletionForPeriodCompeltion(QString)));

#ifdef Q_OS_MAC
    (void) new QShortcut(QKeySequence(Qt::CTRL | Qt::Key_N), this, SLOT(performCompletion()));
#else
    (void) new QShortcut(QKeySequence(Qt::CTRL | Qt::Key_Space), this, SLOT(performCompletion()));
#endif

  modelCodeSnippet = new QStandardItemModel(this);
  codeSnippetCompleter = new QCompleter(this);
  codeSnippetCompleter->setWidget(this);
  codeSnippetCompleter->setCompletionMode(QCompleter::PopupCompletion);
  codeSnippetCompleter->setModel(modelCodeSnippet);
  codeSnippetCompleter->setModelSorting(QCompleter::CaseInsensitivelySortedModel);
  codeSnippetCompleter->setCaseSensitivity(Qt::CaseInsensitive);
  codeSnippetCompleter->setFilterMode(Qt::MatchContains);
  codeSnippetCompleter->setWrapAround(true);
  connect(codeSnippetCompleter, SIGNAL(activated(QString)), this, SLOT(insertCompletionCodeSnippet(QString)));
  //(void) new QShortcut(QKeySequence(Qt::CTRL | Qt::Key_M), this, SLOT(performCodeSnippetCompletion()));

  showCodeSnippets = new QAction();
  showCodeSnippets->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_M));
  connect(showCodeSnippets, SIGNAL(triggered()), this, SLOT(performCodeSnippetCompletion()));

  findReplace = new FindReplace(this);
  createActions();
  openRecentFilesMenu = new QMenu(this);
  openRecentFilesMenu->setIcon(DIcon::DocumentOpenRecent());
  connect(openRecentFilesMenu, SIGNAL(aboutToShow()), this, SLOT(openRecentFilesMenuAboutToShowSlot()));
  symbolsMenu = new QMenu(this);
  symbolsMenu->setIcon(DIcon::Database());
  connect(symbolsMenu, SIGNAL(aboutToShow()), this, SLOT(symbolsMenuAboutToShowSlot()));
  contextualMenu = new QMenu(this);
  contextualMenu->addMenu(symbolsMenu);
  contextualMenuSeparatorAction = contextualMenu->addSeparator();
  contextualMenu->addActions(editActionList());
  exportMenu = new QMenu;
  exportMenu->setIcon(DIcon::Export());
  exportMenu->addAction(exportoToPdfOrPsAction);
  exportMenu->addAction(exportoToOdtAction);
  exportMenu->addAction(exportoToHtmlAction);
  exportMenu->addAction(exportoToSvgAction);
  exportMenu->addAction(exportoToImgAction);
  newFont = font();
  newFont.setPointSize(settings.value("TextEditor/PointSize", 12).toInt());
  setFontToWidgets(newFont);
  fillMariaDBSymbolsActionSlot();
//  connect(textEditor, SIGNAL(cursorPositionChanged()), this, SLOT(matchBracket()));

  subversionedFile = new SubversionedFile(this->fileName);
  connect(subversionedFile, SIGNAL(reverted()), this, SLOT(reloadFileActionTriggerd()));

  connect(subversionedFile, SIGNAL(statusBarMessage(QString)), this->project, SLOT(statusBarMessageSlot(QString)));
  connect(subversionedFile, SIGNAL(statusBarMessagePopup(QString)), this->project, SLOT(statusBarMessagePopupSlot(QString)));
  connect(subversionedFile, SIGNAL(addSubWindow(DMdiSubWindow*)), this->project, SLOT(addSubWindowSlot(DMdiSubWindow*)));

  listSymbols = new QListWidget;
  createToolBars();
  createMenu();

  secondSplitter = new QSplitter(Qt::Vertical, this);
  QSplitter *mainSplitter = new QSplitter(Qt::Horizontal, this);
  mainSplitter->addWidget(secondSplitter);
  mainSplitter->addWidget(listSymbols);

  secondSplitter->addWidget(textEditor);
  secondSplitter->addWidget(readOnlyTextEditor);
  secondSplitter->widget(1)->hide();
  connect(textEditor, SIGNAL(cursorPositionChanged()), this, SLOT(updateCursorPositionSlot()));
  connect(textEditor->document(), SIGNAL(contentsChanged()), this, SLOT(showSymbolsActionTriggered()));
  connect(listSymbols, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(itemClickedSlot(QListWidgetItem*)));

  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  dTitleLabel = new DTitleLabel;
  mainVLayout->addWidget(dTitleLabel);
  if (hideTitle)
    dTitleLabel->hide();
  mainVLayout->addWidget(menuBar);
  mainVLayout->addWidget(mainToolBar);
  mainVLayout->addWidget(mainSplitter);
  mainVLayout->addWidget(findReplace);

  sortSymbols = new QAction(this);
  sortSymbols->setIcon(DIcon::ViewSortAscending());
  connect(sortSymbols, SIGNAL(triggered()), this, SLOT(sortSymbolsSlot()));
  contextualMenu = new QMenu(this);
  contextualMenu->addAction(sortSymbols);
  retranslateUI();

  //QWidget *widMain = new QWidget;
  //widMain->setLayout(mainVLayout);
  //setWidget(widMain);
  setLayout(mainVLayout);
  openLastOpenedFile();
}

void TextEditor::setTitle(QString title)
{
  setWindowTitle(title);
  dTitleLabel->setText(title);
  dTitleLabel->setToolTip(dTitleLabel->text());
}

void TextEditor::uppercaseKeywordsActionSlot()
{
  changeTextCase(true);
}

void TextEditor::lowercaseKeywordsActionSlot()
{
  changeTextCase(false);
}

void TextEditor::performPeriodCompletion(const QString &completionPrefix)
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  modelForPeriodCompeltion->clear();
  foreach (QString item, serverConnection->tables()->list(completionPrefix))
    modelForPeriodCompeltion->appendRow(new QStandardItem(DIcon::DatabaseTable(), item.split(".").at(1)));
  foreach (QString item, serverConnection->views()->list(completionPrefix))
    modelForPeriodCompeltion->appendRow(new QStandardItem(DIcon::DatabaseView(), item.split(".").at(1)));
  foreach (QString item, serverConnection->events()->list(completionPrefix))
    modelForPeriodCompeltion->appendRow(new QStandardItem(DIcon::DatabaseEvent(), item.split(".").at(1)));
  foreach (QString item, serverConnection->functions()->list(completionPrefix))
    modelForPeriodCompeltion->appendRow(new QStandardItem(DIcon::DatabaseRoutine(), item.split(".").at(1)));
  foreach (QString item, serverConnection->procedures()->list(completionPrefix))
    modelForPeriodCompeltion->appendRow(new QStandardItem(DIcon::DatabaseRoutine(), item.split(".").at(1)));
  foreach (QString item, serverConnection->triggers()->list(completionPrefix))
    modelForPeriodCompeltion->appendRow(new QStandardItem(DIcon::DatabaseTrigger(), item.split(".").at(1)));
  modelForPeriodCompeltion->sort(0);

//  if (completionPrefix != completer->completionPrefix()) {
//    completer->setCompletionPrefix(completionPrefix);
//    completer->popup()->setCurrentIndex(completer->completionModel()->index(0, 0));
//  }
//  if (completer->completionCount() == 1) {
//    insertCompletion(completer->currentCompletion());
//  } else {
    completerForPeriodCompeltion->complete(QRect(textEditor->cursorRect().x() + 5
                              , textEditor->cursorRect().y() + (dTitleLabel->isVisible() ? (dTitleLabel->height() + 5) : 0)
                              , completerForPeriodCompeltion->popup()->sizeHintForColumn(0) + completerForPeriodCompeltion->popup()->verticalScrollBar()->sizeHint().width()
                              , 40));
//  }
  QApplication::restoreOverrideCursor();
}

void TextEditor::insertCompletionForPeriodCompeltion(const QString &completion)
{
  QTextCursor cursor = textEditor->textCursor();
  cursor.setPosition(cursor.position() - completerForPeriodCompeltion->completionPrefix().length());
  cursor.movePosition(QTextCursor::EndOfWord, QTextCursor::KeepAnchor);
  cursor.insertText(completion);
  textEditor->setTextCursor(cursor);
}

//void TextEditor::readyReadStandardErrorSlot()
//{
//  procOutput->insertPlainText(proc->readAllStandardError());
//}

void TextEditor::performCodeSnippetCompletion()
{
  CodeSnippets *codeSnippets = new CodeSnippets;
  modelCodeSnippet->clear();
  foreach (QString item, codeSnippets->getCodeSnippets())
    modelCodeSnippet->appendRow(new QStandardItem(DIcon::TextEditor(), item));
  modelCodeSnippet->sort(0);
  codeSnippetCompleter->complete(QRect(textEditor->cursorRect().x() + 5
                            , textEditor->cursorRect().y() + (dTitleLabel->isVisible() ? (dTitleLabel->height() + 5) : 0)
                            , codeSnippetCompleter->popup()->sizeHintForColumn(0) + codeSnippetCompleter->popup()->verticalScrollBar()->sizeHint().width()
                                       , 40));
}

void TextEditor::insertCompletionCodeSnippet(const QString &completion)
{
  CodeSnippets *codeSnippets = new CodeSnippets;
  QTextCursor cursor = textEditor->textCursor();
  cursor.setPosition(cursor.position() - completerForPeriodCompeltion->completionPrefix().length());
  cursor.movePosition(QTextCursor::EndOfWord, QTextCursor::KeepAnchor);
  cursor.insertText(codeSnippets->getCodeSnippet(completion) + "\n");
  textEditor->setTextCursor(cursor);
}

void TextEditor::dummyExecutionActionSlot()
{
  if (serverConnection->isOpened() && editorType == EditorTypes::SQLQuery && textEditor->textCursor().hasSelection()) {
    if (serverConnection->dummyExecution(textEditor->textCursor().selection().toPlainText().trimmed())) {
      QMessageBox::information(this, tr("Dummy execution"), tr("Statement executed correctly"));
    }
  }
}

void TextEditor::changeFontWeightActionGroupTriggered(QAction *action)
{
  if (action->text() == tr("Thin"))
    textEditor->setFontWeight(QFont::Thin);
  if (action->text() == tr("Extra light"))
    textEditor->setFontWeight(QFont::ExtraLight);
  if (action->text() == tr("Light"))
    textEditor->setFontWeight(QFont::Light);
  if (action->text() == tr("Normal"))
    textEditor->setFontWeight(QFont::Normal);
  if (action->text() == tr("Medium"))
    textEditor->setFontWeight(QFont::Medium);
  if (action->text() == tr("Demi bold"))
    textEditor->setFontWeight(QFont::DemiBold);
  if (action->text() == tr("Bold"))
    textEditor->setFontWeight(QFont::Bold);
  if (action->text() == tr("Extra bold"))
    textEditor->setFontWeight(QFont::ExtraBold);
  if (action->text() == tr("Black"))
    textEditor->setFontWeight(QFont::Black);
}

QList<QAction *> TextEditor::getMenus()
{
  QList<QAction *> actionsList;
  QAction *fileMenuAction = new QAction();
  fileMenuAction->setMenu(fileMenu);
  actionsList.append(fileMenuAction);

  QAction *editMenuAction = new QAction();
  editMenuAction->setMenu(editMenu);
  actionsList.append(editMenuAction);

  QAction *optionsMenuAction = new QAction();
  optionsMenuAction->setMenu(optionsMenu);
  actionsList.append(optionsMenuAction);

  QAction *versionControlMenuAction = new QAction();
  versionControlMenuAction->setMenu(versionControlMenu);
  actionsList.append(versionControlMenuAction);

  QAction *viewMenuAction = new QAction();
  viewMenuAction->setMenu(viewMenu);
  actionsList.append(viewMenuAction);

  return actionsList;
}

void TextEditor::retranslateUI()
{
  openRecentFilesMenu->setTitle(tr("Recent files"));
  openRecentFilesMenu->setToolTip(openRecentFilesMenu->title());
  symbolsMenu->setTitle(tr("Symbols"));
  symbolsMenu->setToolTip(symbolsMenu->title());
  sortSymbols->setText(tr("Sort by name"));
  sortSymbols->setToolTip(sortSymbols->text());
  contextualMenuSeparatorAction->setText(tr("Edit Menu"));
  contextualMenuSeparatorAction->setToolTip(contextualMenuSeparatorAction->text());
  exportMenu->setTitle(tr("&Export"));
  exportMenu->setToolTip(exportMenu->title());
  fileMenu->setTitle(tr("&File"));
  fileMenu->setToolTip(fileMenu->title());
  editMenu->setTitle(tr("&Edit"));
  editMenu->setToolTip(editMenu->title());
  optionsMenu->setTitle(tr("&Options"));
  optionsMenu->setToolTip(optionsMenu->title());
  subversionMenu->setTitle(tr("&Subversion"));
  subversionMenu->setToolTip(subversionMenu->title());
  viewMenu->setTitle(tr("&View"));
  viewMenu->setToolTip(viewMenu->title());
  viewMenuSeparatorActionView->setText(tr("View"));
  viewMenuSeparatorActionView->setToolTip(viewMenuSeparatorActionView->text());
  viewMenuSeparatorActionSymbols->setText(tr("Symbols"));
  viewMenuSeparatorActionSymbols->setToolTip(viewMenuSeparatorActionSymbols->text());
  changeFontWeightMenuSeparatorAction->setText(tr("Font weight"));
  changeFontWeightMenuSeparatorAction->setToolTip(changeFontWeightMenuSeparatorAction->text());
  clearAction->setText(tr("Clear"));
  clearAction->setToolTip(clearAction->text());
  openAction->setText(tr("Open"));
  openAction->setToolTip(openAction->text());
  saveAction->setText(tr("Save"));
  saveAction->setToolTip(saveAction->text());
  saveAsAction->setText(tr("Save as..."));
  saveAsAction->setToolTip(saveAsAction->text());
  copyAction->setText(tr("Copy"));
  copyAction->setToolTip(copyAction->text());
  cutAction->setText(tr("Cut"));
  cutAction->setToolTip(cutAction->text());
  pasteAction->setText(tr("Paste"));
  pasteAction->setToolTip(pasteAction->text());
  undoAction->setText(tr("Undo"));
  undoAction->setToolTip(undoAction->text());
  redoAction->setText(tr("Redo"));
  redoAction->setToolTip(redoAction->text());
  wordWrapAction->setText(tr("Word wrap"));
  wordWrapAction->setToolTip(wordWrapAction->text());
  zoomInAction->setText(tr("In"));
  zoomInAction->setToolTip(zoomInAction->text());
  zoomOutAction->setText(tr("Out"));
  zoomOutAction->setToolTip(zoomOutAction->text());
  restoreZoomAction->setText(tr("Restore"));
  restoreZoomAction->setToolTip(restoreZoomAction->text());
  gotoLineAction->setText(tr("Line..."));
  gotoLineAction->setToolTip(gotoLineAction->text());
  selectAllAction->setText(tr("Select all"));
  clearRecentFilesAction->setText(tr("Clear recent files"));
  clearRecentFilesAction->setToolTip(clearRecentFilesAction->text());
  printAction->setText(tr("Print..."));
  printAction->setToolTip(printAction->text());
  reloadFile->setText(tr("Reload file"));
  reloadFile->setToolTip(reloadFile->text());
  disableCompletionAction->setText(tr("Disable completion"));
  disableCompletionAction->setToolTip(disableCompletionAction->text());
  showHideLineNumbersAction->setText(tr("Show/Hide line numbers"));
  showHideLineNumbersAction->setToolTip(showHideLineNumbersAction->text());
  cutLineAction->setText(tr("Cut line"));
  cutLineAction->setToolTip(cutLineAction->text());
  versionControlMenu->setTitle(tr("Version &control"));
  versionControlMenu->setToolTip(versionControlMenu->title());
  mainToolBarAction->setText(tr("Show/Hide Main Toolbar"));
  mainToolBarAction->setToolTip(mainToolBarAction->text());

  switch(editorType) {
  case EditorTypes::SQLQuery:
    setWindowTitle(tr("SQL Query %1").arg(windowCount));
    fillMariaDBSymbolsAction->setText(tr("Fill MariaDB Symbols"));
    fillMariaDBSymbolsAction->setToolTip(fillMariaDBSymbolsAction->text());
    addCodeToSnippetAction->setText(tr("Add selected code to Code Snippets"));
    addCodeToSnippetAction->setToolTip(addCodeToSnippetAction->text());
    showCodeSnippets->setText(tr("Show code snippets"));
    showCodeSnippets->setToolTip(showCodeSnippets->text());
    dummyExecutionAction->setText(tr("Dummy execution"));
    dummyExecutionAction->setToolTip(tr("Execute the selected code between BEGIN and ROLLBACK just to check if it works"));
    break;
  case EditorTypes::NoEditor:
    setWindowTitle(tr("Text File %1").arg(windowCount));
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
    break;
  // case EditorTypes::Diff:
  // default: Q_ASSERT(false);
  }

  showMariaDBHelperAction->setText(tr("Show SQL Helpers in the completion"));
  showMariaDBHelperAction->setToolTip(showMariaDBHelperAction->text());
  exportoToPdfOrPsAction->setText(tr("Export to Pdf"));
  exportoToPdfOrPsAction->setToolTip(exportoToPdfOrPsAction->text());
  exportoToOdtAction->setText(tr("Export to Odt"));
  exportoToOdtAction->setToolTip(exportoToOdtAction->text());
  exportoToHtmlAction->setText(tr("Export to Html"));
  exportoToHtmlAction->setToolTip(exportoToHtmlAction->text());
  exportoToSvgAction->setText(tr("Export to Svg"));
  exportoToSvgAction->setToolTip(exportoToSvgAction->text());
  exportoToImgAction->setText(tr("Export to Image"));
  exportoToImgAction->setText(exportoToImgAction->text());
  toggleCommentAction->setText(tr("Toggle Comment"));
  toggleCommentAction->setText(toggleCommentAction->text());
  highlightCurrentLineAction->setText(tr("Highlight current line"));
  highlightCurrentLineAction->setToolTip(highlightCurrentLineAction->text());
  viewNormalAction->setText(tr("Normal"));
  viewNormalAction->setToolTip(viewNormalAction->text());
  viewVerticalAction->setText(tr("Vertical"));
  viewVerticalAction->setToolTip(viewVerticalAction->text());
  viewHorizoltalAction->setText(tr("Horizontal"));
  viewHorizoltalAction->setToolTip(viewHorizoltalAction->text());
  showSymbolsAction->setText(tr("&Show Symols"));
  showSymbolsAction->setToolTip(showSymbolsAction->text());
  normalizeTextAction->setText(tr("&Normalize text"));
  normalizeTextAction->setToolTip(normalizeTextAction->text());
  readOnlyTextEditor->retranslateUI();
  textEditor->retranslateUI();
  uppercaseKeywordsAction->setText(tr("Uppercase keywords"));
  uppercaseKeywordsAction->setToolTip(uppercaseKeywordsAction->text());
  lowercaseKeywordsAction->setText(tr("Lowercase keywords"));
  lowercaseKeywordsAction->setToolTip(lowercaseKeywordsAction->text());

  setObjectName(windowTitle());
  dTitleLabel->setText(windowTitle());

  changeFontWeightActionThin->setText(tr("Thin"));
  changeFontWeightActionThin->setToolTip(changeFontWeightActionThin->text());
  changeFontWeightActionExtraLight->setText(tr("Extra light"));
  changeFontWeightActionExtraLight->setToolTip(changeFontWeightActionExtraLight->text());
  changeFontWeightActionLight->setText(tr("Light"));
  changeFontWeightActionLight->setToolTip(changeFontWeightActionLight->text());
  changeFontWeightActionNormal->setText(tr("Normal"));
  changeFontWeightActionNormal->setToolTip(changeFontWeightActionNormal->text());
  changeFontWeightActionMedium->setText(tr("Medium"));
  changeFontWeightActionMedium->setToolTip(changeFontWeightActionMedium->text());
  changeFontWeightActionDemiBold->setText(tr("Demi bold"));
  changeFontWeightActionDemiBold->setToolTip(changeFontWeightActionDemiBold->text());
  changeFontWeightActionBold->setText(tr("Bold"));
  changeFontWeightActionBold->setToolTip(changeFontWeightActionBold->text());
  changeFontWeightActionExtraBold->setText(tr("Extra bold"));
  changeFontWeightActionExtraBold->setToolTip(changeFontWeightActionExtraBold->text());
  changeFontWeightActionBlack->setText(tr("Black"));
  changeFontWeightActionBlack->setToolTip(changeFontWeightActionBlack->text());
}

void TextEditor::setPlainText(const QString &text)
{
  textEditor->setPlainText(text);
}

QTextDocument *TextEditor::document()
{
  return textEditor->document();
}

void TextEditor::gotoLine(unsigned int line)
{
  textEditor->gotoLine(line);
}

void TextEditor::fillMariaDBSymbolsActionSlot()
{
  if (serverConnection->isOpened() && editorType == EditorTypes::SQLQuery) {
    databaseSymbols.clear();
    databaseSymbols = serverConnection->runQuerySingleColumn("SELECT `TABLE_NAME` FROM `information_schema`.`TABLES` WHERE `TABLE_SCHEMA` = DATABASE() UNION SELECT DISTINCT(`COLUMN_NAME`) FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = DATABASE() UNION SELECT DISTINCT(`name`) FROM `mysql`.`proc` WHERE `db` = DATABASE() UNION SELECT `EVENT_NAME` FROM `information_schema`.`EVENTS` WHERE `EVENT_SCHEMA` = DATABASE() UNION SELECT `TRIGGER_NAME` FROM `information_schema`.`TRIGGERS` WHERE `TRIGGER_SCHEMA` = DATABASE()");
    databaseSymbols += serverConnection->runQuerySingleColumn("SHOW DATABASES");
  }
}

void TextEditor::openLastOpenedFile()
{
  if (settings.value("GeneralSettings/OpenLastFile", false).toBool()) {
    switch(editorType) {
    case EditorTypes::SQLQuery:
      openFile(settings.value("GeneralSettings/LastMariaDBFile", "").toString());
      break;
    case EditorTypes::Diff:
    case EditorTypes::Commit:
    case EditorTypes::SVNLog:
    case EditorTypes::NoEditor:
      break;
    // default: Q_ASSERT(false);
    }
  }
}

void TextEditor::reloadFileActionTriggerd()
{
  openFile(fileName);
}

void TextEditor::printActionTriggerd()
{
  QPrinter printer;
  printer.setResolution(QPrinter::HighResolution);
  QFileInfo file(fileName);
  printer.setOutputFileName(file.completeBaseName() + ".pdf");
  printer.setPageSize(QPageSize::Letter);
  printer.setPageMargins(QMarginsF(10, 10, 10, 10), QPageLayout::Millimeter);
  QPrintDialog *printDialog = new QPrintDialog(&printer, this);
  printDialog->setWindowTitle(tr("Print Script"));
  if (printDialog->exec() == QDialog::Accepted)
    textEditor->print(&printer);
}

void TextEditor::openRecentFilesMenuAboutToShowSlot()
{
  QString setting;
  openRecentFilesMenu->clear();
  switch(editorType) {
  case EditorTypes::SQLQuery:
    setting = "RecentMariaDBFiles/Files";
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);
  }
  QStringList fileList = settings.value(setting, QStringList()).toStringList();
  fileList.sort();
  foreach (QString file, fileList)
    if (!file.isEmpty())
      recentFilesActionGroup->addAction(openRecentFilesMenu->addAction(StaticFunctions::iconFromFileName(file), file));
  openRecentFilesMenu->addSeparator()->setText(tr("Clean"));
  openRecentFilesMenu->addAction(clearRecentFilesAction);
}

void TextEditor::insertCompletion(QString completion)
{
  QTextCursor cursor = textEditor->textCursor();
  QTextCursor cursor2 = cursor;
  cursor.setPosition(cursor.position() - completer->completionPrefix().length());
  cursor.movePosition(QTextCursor::EndOfWord, QTextCursor::KeepAnchor);


  QString sufix;
  switch(editorType) {
  case EditorTypes::SQLQuery:
    cursor2.movePosition(QTextCursor::WordLeft, QTextCursor::KeepAnchor);
    cursor2.movePosition(QTextCursor::Left, QTextCursor::KeepAnchor);
    if (databaseSymbols.contains(completion)) {
      cursor.insertText((cursor2.selectedText().startsWith("`") ? "" : "`") + completion + "`");
    } else {
      cursor.insertText(completion + sufix);
    }
    break;
  default:
    cursor.insertText(completion + sufix);
    break;
  }
  textEditor->setTextCursor(cursor);
}

//void TextEditor::keyPressEvent(QKeyEvent *event)
//{
//  QPlainTextEdit::keyPressEvent(event);
//  qDebug() << event->key();
//  if (completedAndSelected && handledCompletedAndSelected(event))
//    return;
//  completedAndSelected = false;
//  if (completer->popup()->isVisible()) {
//    switch (event->key()) {
//    case Qt::Key_Up:
//    case Qt::Key_Down:
//    case Qt::Key_Enter:
//    case Qt::Key_Return:
//    case Qt::Key_Escape:
//      event->ignore();
//      return;
//    default:
//      completer->popup()->hide();
//      break;
//    }
//  }

//  switch (event->key()) {
//  case Qt::Key_QuoteDbl:
//    smartTextInsertion("\"");
//    return;
//  case Qt::Key_Apostrophe:
//    smartTextInsertion("'");
//    return;
//  case Qt::Key_Agrave:
//    smartTextInsertion("`");
//    return;
//  case Qt::Key_ParenLeft:
//    smartTextInsertion("(", ")");
//    return;
//  case Qt::Key_BracketLeft:
//    smartTextInsertion("[", "]");
//    return;
//  case Qt::Key_BraceLeft:
//    smartTextInsertion("{", "}");
//    return;
//  case Qt::Key_Tab:
//    indentAction->trigger();
//    return;
//  case Qt::Key_Escape:
//    setFocusToEditorActionTriggered();
//    break;
//  case Qt::Key_Home:
//    homeKeyHitedTwice = !homeKeyHitedTwice;
//    if (homeKeyHitedTwice) {
//      QTextCursor cursor = textEditor->textCursor();
//      cursor.movePosition(QTextCursor::NextWord, QTextCursor::MoveAnchor);
//      textEditor->setTextCursor(cursor);
//      return;
//    }
//    break;
////  case Qt::Key_Delete:
////    if (event->modifiers() == Qt::ControlModifier)
////      cutLineActionTriggerd();
////    break;
//  default:
//    homeKeyHitedTwice = true;
//  }
////  **********************QPlainTextEdit::keyPressEvent(event);

//  if (settings.value("TextEditor/AutomaticIndentation", Qt::Unchecked) == Qt::Checked)
//    switch (event->key()) {
//    case Qt::Key_Return:
//    case Qt::Key_Enter:
//      smartTextIndentation();
//      break;
//    // default: Q_ASSERT(false);
//    }
//}

//void TextEditor::smartTextIndentation()
//{
//  QString prevText(textEditor->textCursor().block().previous().text());
//  int count = 0;
//  for (int pos = 0; pos <= prevText.length(); pos++)
//    if (prevText[pos] == QChar(' '))
//      count++;
//    else
//      break;
//  QTextCursor cursor = textEditor->textCursor();
//  cursor.insertText(QString(" ").repeated(count));
//  textEditor->setTextCursor(cursor);
//}

//void TextEditor::smartTextInsertion(QString leftKey, QString rightKey)
//{
//  QTextCursor cursor = textEditor->textCursor();
//  bool hasSelection = cursor.hasSelection();
//  cursor.insertText(leftKey + cursor.selectedText() + (rightKey.isEmpty() ? leftKey : rightKey));
//  if (!hasSelection)
//    cursor.movePosition(QTextCursor::PreviousCharacter, QTextCursor::MoveAnchor);
//  textEditor->setTextCursor(cursor);
//}

//void TextEditor::mousePressEvent(QMouseEvent *event)
//{
//  if (completedAndSelected) {
//    completedAndSelected = false;
//    QTextCursor cursor = textEditor->textCursor();
//    cursor.removeSelectedText();
//    textEditor->setTextCursor(cursor);
//  }
//  QWidget::mousePressEvent(event);
//}

//bool TextEditor::handledCompletedAndSelected(QKeyEvent *event)
//{
//  completedAndSelected = false;
//  QTextCursor cursor = textEditor->textCursor();
//  switch (event->key()) {
//  case Qt::Key_Enter:
//  case Qt::Key_Return:
//    cursor.clearSelection();
//    break;
//  case Qt::Key_Escape:
//    cursor.removeSelectedText();
//    break;
//  default: return false;
//  }
//  textEditor->setTextCursor(cursor);
//  event->accept();
//  return true;
//}

void TextEditor::performCompletion()
{
  if (disableCompletionAction->isChecked())
    return;
  QTextCursor cursor = textEditor->textCursor();
  cursor.movePosition(QTextCursor::StartOfWord, QTextCursor::KeepAnchor);
  const QString completionPrefix(cursor.selectedText());

  if (!completionPrefix.isEmpty() && completionPrefix.at(completionPrefix.length() - 1).isLetter())
    performCompletion(completionPrefix);
  if (cursor.selectedText().endsWith("(") || cursor.selectedText().endsWith("`(") || cursor.selectedText().endsWith(":")) {
    cursor.movePosition(QTextCursor::PreviousWord, QTextCursor::KeepAnchor);
    QString message = "<table><tr><td><img src=\"" + DIcon::DatabaseIconPath() + "\" width=\"48\"></td><td valign=middle><p style='white-space:pre'>%1</p></td></tr></table>";
    QString messageText;
    QString text(cursor.selectedText().left(cursor.selectedText().length() - 1));
    QStringList data;
    switch(editorType) {
    case EditorTypes::SQLQuery:
      messageText = StaticFunctions::mariadbFunctionsComplete().value(text);
      if (messageText.isEmpty() && serverConnection->isOpened()) {
        data = serverConnection->runQuerySingleColumn("SELECT `param_list` FROM `mysql`.`proc` WHERE `name` = '" + StaticFunctions::unquoteSymbol(text) + "' AND `db` = DATABASE()");
        if (!data.empty())
          messageText = data.at(0);
      }
      message = message.arg(messageText);
      break;
    case EditorTypes::Diff:
    case EditorTypes::Commit:
    case EditorTypes::SVNLog:
    case EditorTypes::NoEditor:
      break;
    // default: Q_ASSERT(false);
    }
    QToolTip::showText(mapToGlobal(textEditor->cursorRect().topLeft()), message, this);
  }
}

void TextEditor::performCompletion(const QString &completionPrefix)
{
  populateModel(completionPrefix);
  if (completionPrefix != completer->completionPrefix()) {
    completer->setCompletionPrefix(completionPrefix);
    completer->popup()->setCurrentIndex(completer->completionModel()->index(0, 0));
  }
  if (completer->completionCount() == 1) {
    insertCompletion(completer->currentCompletion());
  } else {
    completer->complete(QRect(textEditor->cursorRect().x() + 5
                              , textEditor->cursorRect().y() + (dTitleLabel->isVisible() ? (dTitleLabel->height() + 5) : 0)
                              , completer->popup()->sizeHintForColumn(0) + completer->popup()->verticalScrollBar()->sizeHint().width()
                              , 40));
  }
}

void TextEditor::populateModel(const QString &completionPrefix)
{
  model->clear();
  switch(editorType) {
  case EditorTypes::SQLQuery:
    foreach (QString item, StaticFunctions::mariadbKeywords().filter(QRegularExpression(QString("^%1").arg(completionPrefix), QRegularExpression::CaseInsensitiveOption)))
      model->appendRow(new QStandardItem(DIcon::Key(), item));
    foreach (QString item, StaticFunctions::mariadbFunctions().filter(QRegularExpression(QString("^%1").arg(completionPrefix), QRegularExpression::CaseInsensitiveOption)))
      model->appendRow(new QStandardItem(DIcon::Database(), item));
    foreach (QString item, StaticFunctions::mariadbDatatypes().filter(QRegularExpression(QString("^%1").arg(completionPrefix), QRegularExpression::CaseInsensitiveOption)))
      model->appendRow(new QStandardItem(DIcon::Database(), item));
    foreach (QString item, serverConnection->getCharsets().filter(QRegularExpression(QString("^%1").arg(completionPrefix), QRegularExpression::CaseInsensitiveOption)))
      model->appendRow(new QStandardItem(DIcon::CharacterSet(), item));
    foreach (QString item, databaseSymbols.filter(QRegularExpression(QString("^%1").arg(completionPrefix), QRegularExpression::CaseInsensitiveOption)))
      model->appendRow(new QStandardItem(DIcon::Database(), item));
    if (showMariaDBHelperAction->isChecked()) {
      foreach (QString item, StaticFunctions::mariadbHelper().filter(QRegularExpression(QString("^%1").arg(completionPrefix), QRegularExpression::CaseInsensitiveOption))) {
        model->appendRow(new QStandardItem(DIcon::Key(), item));
      }
    }
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);
  }
  foreach (QString item, textEditor->wordList().filter(QRegularExpression(QString("^%1").arg(completionPrefix), QRegularExpression::CaseInsensitiveOption)))
    model->appendRow(new QStandardItem(DIcon::TextEditor(), item));
  model->sort(0);
}

void TextEditor::updateRecentFiles(QString fileToSave)
{
  if (fileToSave.isEmpty())
    fileToSave = fileName;

  QString setting;
  QStringList fileList;
  switch(editorType) {
  case EditorTypes::SQLQuery:
    setting = "RecentMariaDBFiles/Files";
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);
  }
  fileList = settings.value(setting, QStringList()).toStringList();
  if (fileList.indexOf(fileToSave) == -1)
    fileList.append(fileToSave);
  settings.setValue(setting, fileList);
  if (settings.value("GeneralSettings/OpenLastFile", false).toBool()) {
    switch(editorType) {
    case EditorTypes::SQLQuery:
      settings.setValue("GeneralSettings/LastMariaDBFile", fileToSave);
      break;
    case EditorTypes::Diff:
    case EditorTypes::Commit:
    case EditorTypes::SVNLog:
    case EditorTypes::NoEditor:
      break;
    // default: Q_ASSERT(false);
    }
  }
}

bool TextEditor::saveFileAsActionTriggered()
{
  saveAs = true;
  return saveFileActionTriggered();
}

bool TextEditor::saveFileActionTriggered()
{
  if (fileName.isEmpty() || saveAs) {
    if (textEditor->toPlainText().isEmpty()) {
      QMessageBox::warning(this, tr("Save file"), tr("Cannot save empty files."));
      return false;
    }
    QFileInfo fileInfo(fileName);
    fileName = QFileDialog::getSaveFileName(this, tr("Save file"), fileInfo.filePath(), fileExtention());
    saveAs = false;
  }
  QFile file(fileName);
  if (settings.value("TextEditor/CreateBackupFile", Qt::Checked).toBool()) {
    QFile backupFile(fileName + "~");
    backupFile.remove();
    file.copy(fileName + "~");
  }
  if (!file.open(QFile::WriteOnly | QFile::Text)) {
    emit statusBarMessage(tr("Cannot write file %1:\n%2.").arg(fileName, file.errorString()), QSystemTrayIcon::Critical);
    return false;
  }
  QTextStream out(&file);
//  if (settings.value("TextEditor/CleanWhiteSpaces", Qt::Unchecked) == Qt::Checked) {
//    QString newText;
//    foreach(QString line, toPlainText().split("\n"))
//      newText.append(line.replace(QRegularExpression("\\s*$"), "") + "\n");
//    setPlainText(newText);
//  }
  out << textEditor->toPlainText();
  file.close();
  readFileLineByLine(fileName);
  updateRecentFiles(fileName);
  lineEditFile->setText(fileName);
  textEditor->document()->setModified(false);
  emit statusBarMessage(tr("File saved at: %1").arg(fileName), QSystemTrayIcon::Information);
//  parentWidget2->updateReadOnlyEditor();
  QApplication::restoreOverrideCursor();
  return true;
}

//int TextEditor::columnAt(const QString &text, int position) const
//{
//  int column = 0;
//  for (int i = 0; i < position; ++i) {
//    if (text.at(i) == QLatin1Char('\t'))
//      column = column - (column % settings.value("TextEditor/TabSize", 2).toInt()) + settings.value("TextEditor/TabSize", 2).toInt();
//    else
//      ++column;
//  }
//  return column;
//}

void TextEditor::createActions()
{
  clearAction = new QAction(this);
  clearAction->setIcon(DIcon::Clear());
  connect(clearAction, SIGNAL(triggered()), textEditor, SLOT(clear()));
  openAction = new QAction(this);
  openAction->setIcon(DIcon::DocumentOpen());
  openAction->setShortcut(QKeySequence::Open);
  connect(openAction, SIGNAL(triggered()), this, SLOT(openFileActionTriggered()));
  saveAction = new QAction(this);
  saveAction->setIcon(DIcon::Save());
  saveAction->setShortcut(QKeySequence::Save);
  connect(saveAction, SIGNAL(triggered()), this, SLOT(saveFileActionTriggered()));
  saveAsAction = new QAction(this);
  saveAsAction->setIcon(DIcon::SaveAs());
  saveAsAction->setShortcut(QKeySequence::SaveAs);
  connect(saveAsAction, SIGNAL(triggered()), this, SLOT(saveFileAsActionTriggered()));
  copyAction = new QAction(this);
  copyAction->setIcon(DIcon::Copy());
  copyAction->setShortcut(QKeySequence::Copy);
  connect(copyAction, SIGNAL(triggered()), textEditor, SLOT(copy()));
  cutAction = new QAction(this);
  cutAction->setIcon(DIcon::Cut());
  cutAction->setShortcut(QKeySequence::Cut);
  connect(cutAction, SIGNAL(triggered()), textEditor, SLOT(cut()));
  pasteAction = new QAction(this);
  pasteAction->setIcon(DIcon::Paste());
  pasteAction->setShortcut(QKeySequence::Paste);
  connect(pasteAction, SIGNAL(triggered()), textEditor, SLOT(paste()));
  undoAction = new QAction(this);
  undoAction->setIcon(DIcon::Undo());
  undoAction->setShortcut(QKeySequence::Undo);
  connect(undoAction, SIGNAL(triggered()), textEditor, SLOT(undo()));
  redoAction = new QAction(this);
  redoAction->setIcon(DIcon::Redo());
  redoAction->setShortcut(QKeySequence::Redo);
  connect(redoAction, SIGNAL(triggered()), textEditor, SLOT(redo()));
  wordWrapAction = new QAction(this);
  wordWrapAction->setShortcut(QKeySequence(Qt::Key_F12));
  wordWrapAction->setCheckable(true);
  wordWrapAction->setChecked(textEditor->wordWrapMode());
  connect(wordWrapAction, SIGNAL(toggled(bool)), this, SLOT(wordWrapActionToggled()));
  zoomInAction = new QAction(this);
  zoomInAction->setIcon(DIcon::ZoomIn());
  zoomInAction->setShortcut(QKeySequence::ZoomIn);
  connect(zoomInAction, SIGNAL(triggered()), this, SLOT(zoomIn()));
  zoomOutAction = new QAction(this);
  zoomOutAction->setIcon(DIcon::ZoomOut());
  zoomOutAction->setShortcut(QKeySequence::ZoomOut);
  connect(zoomOutAction, SIGNAL(triggered()), this, SLOT(zoomOut()));
  restoreZoomAction = new QAction(this);
  restoreZoomAction->setIcon(DIcon::ZoomOriginal());
  restoreZoomAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_0));
  connect(restoreZoomAction, SIGNAL(triggered()), this, SLOT(zoomReset()));
  gotoLineAction = new QAction(this);
  gotoLineAction->setIcon(DIcon::GoJump());
  gotoLineAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_L));
  connect(gotoLineAction, SIGNAL(triggered()), this, SLOT(gotoLineActionTriggered()));
  selectAllAction = new QAction(this);
  selectAllAction->setIcon(DIcon::SelectAll());
  selectAllAction->setShortcut(QKeySequence::SelectAll);
  connect(selectAllAction, SIGNAL(triggered()), textEditor, SLOT(selectAll()));
  clearRecentFilesAction = new QAction(this);
  clearRecentFilesAction->setIcon(DIcon::Clear());
  connect(clearRecentFilesAction, SIGNAL(triggered()), this, SLOT(clearRecentFilesActionTriggered()));
  recentFilesActionGroup = new QActionGroup(this);
  connect(recentFilesActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(recentFilesActionGroupTriggered(QAction*)));
  gotoLineActionGroup = new QActionGroup(this);
  connect(gotoLineActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(gotoLineActionGroupTriggered(QAction*)));
  printAction = new QAction(this);
  printAction->setIcon(DIcon::Print());
  printAction->setShortcut(QKeySequence::Print);
  connect(printAction, SIGNAL(triggered()), this, SLOT(printActionTriggerd()));
  reloadFile = new QAction(this);
  reloadFile->setIcon(DIcon::Refresh());
  reloadFile->setShortcut(QKeySequence(Qt::Key_F5));
  connect(reloadFile, SIGNAL(triggered()), this, SLOT(reloadFileActionTriggerd()));
  disableCompletionAction = new QAction(this);
  disableCompletionAction->setShortcut(QKeySequence(Qt::Key_F11));
  disableCompletionAction->setCheckable(true);
  showHideLineNumbersAction = new QAction(this);
  showHideLineNumbersAction->setShortcut(QKeySequence(Qt::Key_F10));
  showHideLineNumbersAction->setCheckable(true);
  showHideLineNumbersAction->setChecked(true);
  connect(showHideLineNumbersAction, SIGNAL(toggled(bool)), textEditor, SLOT(showHideLineNumbers(bool)));
  connect(textEditor, SIGNAL(modificationChanged(bool)), this, SLOT(modificationChangedSlot(bool)));
  cutLineAction = new QAction(this);
  cutLineAction->setIcon(DIcon::Cut());
  cutLineAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_Delete));
  connect(cutLineAction, SIGNAL(triggered()), this, SLOT(cutLineActionTriggerd()));

  switch (editorType) {
  case EditorTypes::SQLQuery:
    fillMariaDBSymbolsAction = new QAction(this);
    fillMariaDBSymbolsAction->setIcon(DIcon::Key());
    fillMariaDBSymbolsAction->setShortcut(QKeySequence(Qt::Key_F6));
    connect(fillMariaDBSymbolsAction, SIGNAL(triggered()), this, SLOT(fillMariaDBSymbolsActionSlot()));
    addCodeToSnippetAction = new QAction(this);
    addCodeToSnippetAction->setIcon(DIcon::ListAdd());
    connect(addCodeToSnippetAction, SIGNAL(triggered()), this, SLOT(addCodeToSnippetActionSlot()));
    dummyExecutionAction = new QAction(this);
    dummyExecutionAction->setIcon(DIcon::Executable());
    connect(dummyExecutionAction, SIGNAL(triggered()), this, SLOT(dummyExecutionActionSlot()));
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);
  }

  showMariaDBHelperAction = new QAction(this);
  showMariaDBHelperAction->setIcon(DIcon::Executable());
  showMariaDBHelperAction->setCheckable(true);
  showMariaDBHelperAction->setChecked(settings.value("TextEditor/ShowMariaDBHelper", true).toBool());
  connect(showMariaDBHelperAction, SIGNAL(triggered()), this, SLOT(showMariaDBHlperActionTriggered()));
  exportoToPdfOrPsAction = new QAction(this);
  exportoToPdfOrPsAction->setIcon(DIcon::PDF());
  connect(exportoToPdfOrPsAction, SIGNAL(triggered()), this, SLOT(exportoToPdfOrPsActionTriggerd()));
  exportoToOdtAction = new QAction(this);
  exportoToOdtAction->setIcon(DIcon::FileTXT());
  connect(exportoToOdtAction, SIGNAL(triggered()), this, SLOT(exportoToOdtActionTriggerd()));
  exportoToHtmlAction = new QAction(this);
  exportoToHtmlAction->setIcon(DIcon::FileHTML());
  connect(exportoToHtmlAction, SIGNAL(triggered()), this, SLOT(exportoToHtmlActionTriggerd()));
  exportoToSvgAction = new QAction(this);
  exportoToSvgAction->setIcon(DIcon::ImageSVG());
  connect(exportoToSvgAction, SIGNAL(triggered()), this, SLOT(exportoToSvgActionTriggerd()));
  exportoToImgAction = new QAction(this);
  exportoToImgAction->setIcon(DIcon::ImagePNG());
  connect(exportoToImgAction, SIGNAL(triggered()), this, SLOT(exportoToImgActionTriggerd()));
  toggleCommentAction = new QAction(this);
  toggleCommentAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_D));
  connect(toggleCommentAction, SIGNAL(triggered()), this, SLOT(toggleCommentActionTriggerd()));
  highlightCurrentLineAction = new QAction(this);
  highlightCurrentLineAction->setCheckable(true);
  highlightCurrentLineAction->setChecked(settings.value("TextEditor/HighlightCurrentLine", true).toBool());
  connect(highlightCurrentLineAction, SIGNAL(triggered()), this, SLOT(highlightCurrentLineActionTriggered()));
  viewNormalAction = new QAction(this);
  viewNormalAction->setIcon(DIcon::ViewClose());
  connect(viewNormalAction, SIGNAL(triggered()), this, SLOT(viewNormalActionTriggered()));
  viewVerticalAction = new QAction(this);
  viewVerticalAction->setIcon(DIcon::ViewSplitTopBottom());
  connect(viewVerticalAction, SIGNAL(triggered()), this, SLOT(viewVerticalActionTriggered()));
  viewHorizoltalAction = new QAction(this);
  viewHorizoltalAction->setIcon(DIcon::ViewSplitLeftRight());
  connect(viewHorizoltalAction, SIGNAL(triggered()), this, SLOT(viewHorizoltalActionTriggered()));
  showSymbolsAction = new QAction(this);
  showSymbolsAction->setIcon(DIcon::Key());
  showSymbolsAction->setCheckable(true);
  showSymbolsAction->setShortcut(QKeySequence(Qt::Key_F2));
  connect(showSymbolsAction, SIGNAL(triggered(bool)), this, SLOT(showSymbolsActionTriggered(bool)));
  normalizeTextAction = new QAction(this);
  normalizeTextAction->setIcon(DIcon::Key());
  connect(normalizeTextAction, SIGNAL(triggered()), this, SLOT(normalizeTextActionTriggered()));

  uppercaseKeywordsAction = new QAction(this);
  connect(uppercaseKeywordsAction, SIGNAL(triggered()), this, SLOT(uppercaseKeywordsActionSlot()));
  lowercaseKeywordsAction = new QAction(this);
  connect(lowercaseKeywordsAction, SIGNAL(triggered()), this, SLOT(lowercaseKeywordsActionSlot()));
}

void TextEditor::showMariaDBHlperActionTriggered()
{
  settings.setValue("TextEditor/ShowMariaDBHelper", showMariaDBHelperAction->isChecked());
}

void TextEditor::highlightCurrentLineActionTriggered()
{
  settings.setValue("TextEditor/HighlightCurrentLine", highlightCurrentLineAction->isChecked());
}

void TextEditor::toggleCommentActionTriggerd()
{
  QTextCursor cursor = textEditor->textCursor();
  cursor.beginEditBlock();

  if (cursor.hasSelection()) {
    QTextDocument *doc = textEditor->document();
    QTextBlock startBlock = doc->findBlock(qMin(cursor.anchor(), cursor.position()));
    QTextBlock endBlock = doc->findBlock(qMax(cursor.anchor(), cursor.position()) - 1).next();
    for (QTextBlock block = startBlock; block != endBlock; block = block.next()) {
      cursor.setPosition(block.position());
      cursor.movePosition(QTextCursor::StartOfLine);
      cursor.select(QTextCursor::LineUnderCursor);
      if (cursor.selectedText().startsWith(commentString()))
        cursor.insertText(cursor.selectedText().right(cursor.selectedText().length() - commentString().length()));
      else
        cursor.insertText(commentString() + cursor.selectedText());
      cursor.movePosition(QTextCursor::StartOfLine);
    }
  } else {
    cursor.movePosition(QTextCursor::StartOfLine);
    cursor.select(QTextCursor::LineUnderCursor);
    if (cursor.selectedText().startsWith(commentString()))
      cursor.insertText(cursor.selectedText().right(cursor.selectedText().length() - commentString().length()));
    else
      cursor.insertText(commentString() + cursor.selectedText());
    textEditor->setTextCursor(cursor);
  }
  cursor.endEditBlock();
}

void TextEditor::cutLineActionTriggerd()
{
  QTextCursor cursor = textEditor->textCursor();
  if (!cursor.hasSelection()) {
    const QTextBlock &block = cursor.block();
    if (block.next().isValid()) {
      cursor.setPosition(block.position());
      cursor.setPosition(block.next().position(), QTextCursor::KeepAnchor);
    } else {
      cursor.movePosition(QTextCursor::EndOfBlock);
      cursor.movePosition(QTextCursor::StartOfBlock, QTextCursor::KeepAnchor);
      cursor.movePosition(QTextCursor::PreviousCharacter, QTextCursor::KeepAnchor);
    }
    textEditor->setTextCursor(cursor);
  }
  textEditor->cut();
}

QString TextEditor::commentString()
{
  QString commentCharacters;
  switch(editorType) {
  case EditorTypes::SQLQuery:
    commentCharacters = "-- ";
    break;
  case EditorTypes::Diff:
  case EditorTypes::NoEditor:
  default:
    commentCharacters = "";
    Q_ASSERT(false);
  }
  return commentCharacters;
}

bool TextEditor::okToClose()
{
  if (textEditor->document()->isModified()) {
    settings.setValue("TextEditor/PointSize", font().pointSize());
    QMessageBox msgBox;
    switch(editorType) {
    case EditorTypes::SQLQuery:
      settings.setValue("SQLQuery/LastQuery-" + qApp->property("ConnectionName").toString() + "-" + windowTitle(), textEditor->toPlainText().trimmed());
      msgBox.setWindowTitle(tr("SQL Query: Save file..."));
      break;
    case EditorTypes::Diff:
    case EditorTypes::Commit:
    case EditorTypes::SVNLog:
    case EditorTypes::NoEditor:
      break;
    // default: Q_ASSERT(false);
    }
    int option = QMessageBox::Discard;
    msgBox.setText(tr("The document has been modified."));
    msgBox.setInformativeText(tr("Do you want to save the changes?"));
    msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Discard);
    msgBox.setDefaultButton(QMessageBox::Save);
    option = msgBox.exec();
    switch (option) {
    case QMessageBox::Discard:
      return true;
    case QMessageBox::Save:
      if (saveFileActionTriggered())
        return true;
      else
        return false;
    case QMessageBox::Cancel:
      return false;
    default:
      return false;
    }
  }
  if (project->openedProject())
    project->saveFileSettings(fileName, textEditor->pointList);
  return true;
}

//void TextEditor::gotoLine(unsigned int lineNumber)
//{
//  QTextCursor cursor(textEditor->document()->findBlockByNumber(lineNumber));
//  cursor.movePosition(QTextCursor::Right, QTextCursor::MoveAnchor, 0);
//  textEditor->setTextCursor(cursor);
//  textEditor->centerCursor();
//  textEditor->setFocus();
//}

void TextEditor::gotoLineActionGroupTriggered(QAction *action)
{
  textEditor->gotoLine(symbolsLineNumber.value(action->text()));
}

void TextEditor::recentFilesActionGroupTriggered(QAction *action)
{
  openFile(action->text());
}

void TextEditor::clearRecentFilesActionTriggered()
{
  switch(editorType) {
  case EditorTypes::SQLQuery:
    settings.remove("GeneralSettings/RecentMariaDBFiles");
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);
  }
}

void TextEditor::wordWrapActionToggled()
{
  textEditor->setWordWrapMode(wordWrapAction->isChecked() ? QTextOption::WordWrap : QTextOption::NoWrap);
  readOnlyTextEditor->setWordWrapMode(textEditor->wordWrapMode());
}

void TextEditor::gotoLineActionTriggered()
{
  textEditor->gotoLine(QInputDialog::getInt(this, tr("Go To Line..."), tr("Line number")) - 1);
}

void TextEditor::clear()
{
  lineEditFile->clear();
}

void TextEditor::symbolsMenuAboutToShowSlot()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  symbolsMenu->clear();
  int counter = 0;
  symbolsLineNumber.clear();
  if (editorType == EditorTypes::SQLQuery) {
    QString icon;
    QString symbolName;
    foreach (QString line, textEditor->document()->toPlainText().split("\n", Qt::KeepEmptyParts)) {
      icon = "";
      static QRegularExpression createTableRegularExpression("^CREATE.*TABLE");
      if (line.contains(createTableRegularExpression))
        icon = DIcon::DatabaseTableIconPath();

      static QRegularExpression createViewRegularExpression("^CREATE.*VIEW");
      if (line.contains(createViewRegularExpression))
        icon = DIcon::DatabaseViewIconPath();

      static QRegularExpression createDatabaseRegularExpression("^CREATE.*DATABASE");
      if (line.contains(createDatabaseRegularExpression))
        icon = DIcon::DatabaseIconPath();

      static QRegularExpression createProcedureRegularExpression("^CREATE.*PROCEDURE");
      if (line.contains(createProcedureRegularExpression))
        icon = DIcon::DatabaseRoutineIconPath();

      static QRegularExpression createFunctionRegularExpression("^CREATE.*FUNCTION");
      if (line.contains(createFunctionRegularExpression))
        icon = DIcon::DatabaseRoutineIconPath();

      static QRegularExpression createTriggerRegularExpression("^CREATE.*TRIGGER");
      if (line.contains(createTriggerRegularExpression))
        icon = DIcon::DatabaseTriggerIconPath();

      static QRegularExpression createEventRegularExpression("^CREATE.*EVENT");
      if (line.contains(createEventRegularExpression))
        icon = DIcon::DatabaseEventIconPath();

      if (!icon.isEmpty()) {
        static QRegularExpression expression2(StaticFunctions::fullidentifierPattern());
        symbolName = expression2.match(line).capturedTexts().at(0);
        if (symbolName.isEmpty()) {
          static QRegularExpression expression3(StaticFunctions::identifierPattern());
          symbolName = expression3.match(line).capturedTexts().at(0);
        }
        symbolName = symbolName.replace("`", "");
        symbolsLineNumber.insert(symbolName, counter);
        gotoLineActionGroup->addAction(symbolsMenu->addAction(QIcon(icon), symbolName));

//        symbolsLineNumber.insert(symbolName, counter);
//        listSymbols->addItem(new QListWidgetItem(QIcon(icon), symbolName));
      }
      counter++;
    }
  }
  QApplication::restoreOverrideCursor();
}

//void TextEditor::contextMenuEvent(QContextMenuEvent *event)
//{
//  contextualMenu->exec(event->globalPos());
//}

void TextEditor::openFileActionTriggered()
{
  openFile();
}

QString TextEditor::fileExtention()
{
  QString type;
  switch(editorType) {
  case EditorTypes::SQLQuery:
    type = tr("SQL files (*.sql)");
    break;
  default:
    type = "";
    break;
  }
  return type;
}

void TextEditor::openFile(QString file)
{
//  qDebug() << file;
  if (file.isEmpty()) {
    QFileDialog fileDialog;
    fileDialog.setDirectory(QDir::home());
    fileName = fileDialog.getOpenFileName(this, tr("Open a file"), settings.value("GeneralSettings/LastPath", QDir::home().absolutePath()).toString(), fileExtention());
    settings.setValue("GeneralSettings/LastPath", fileDialog.directory().filePath(fileName));
  } else {
    fileName = file;
    settings.setValue("GeneralSettings/LastPath", fileName);
  }
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFile fileToOpen(fileName);
  if (!fileToOpen.open(QFile::ReadWrite | QFile::Text))
    emit statusBarMessage(tr("Cannot read file %1:\n%2.").arg(fileName, fileToOpen.errorString()), QSystemTrayIcon::Critical);
  textEditor->setPlainText(QString::fromUtf8(fileToOpen.readAll()));
  fileToOpen.close();
  readFileLineByLine(fileName);
  updateRecentFiles(fileName);
  lineEditFile->setText(fileName);
  updateReadOnlyEditor();
  QFileInfo fileInfo(fileName);
  emit changeTitle(fileInfo.completeBaseName() + "." + fileInfo.completeSuffix());
  if (project->openedProject())
    textEditor->pointList = project->getFileSettings(fileName);
  subversionedFile->fileName = fileName;
  QApplication::restoreOverrideCursor();
}

QList<QAction *> TextEditor::viewActionList()
{
  QList<QAction *> actionsList;
  QMenu *menu = new QMenu;
  QAction *separator = menu->addSeparator();
  separator->setText(tr("Zoom"));
  actionsList.append(separator);
  actionsList.append(zoomInAction);
  actionsList.append(zoomOutAction);
  actionsList.append(restoreZoomAction);
  return actionsList;
}

QList<QAction *> TextEditor::fileActionList()
{
  QList<QAction *> actionsList;
  actionsList.append(openAction);
  actionsList.append(saveAction);
  actionsList.append(saveAsAction);
  actionsList.append(clearAction);
  return actionsList;
}

void TextEditor::changeTextCase(bool uppercse)
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  gotoLine(0);
  QTextCursor cursor = textEditor->document()->find(QRegularExpression("\\b\\w+\\b", QRegularExpression::CaseInsensitiveOption), textEditor->textCursor(), QTextDocument::FindWholeWords);
  while (cursor.hasSelection()) {
    if (StaticFunctions::currentEditorTypeKeywords(editorType).contains(cursor.selectedText(), Qt::CaseInsensitive)) {
      if (uppercse)
        cursor.insertText(cursor.selectedText().toUpper());
      else
        cursor.insertText(cursor.selectedText().toLower());
    }
    textEditor->setTextCursor(cursor);
    cursor = textEditor->document()->find(QRegularExpression("\\b\\w+\\b", QRegularExpression::CaseInsensitiveOption), textEditor->textCursor(), QTextDocument::FindWholeWords);
  }
  gotoLine(0);
  QApplication::restoreOverrideCursor();
}

void TextEditor::readFileLineByLine(QString fileName)
{
  QFile fileToOpen2(fileName);
  bool opened = fileToOpen2.open(QIODevice::ReadOnly | QIODevice::Text);
  if (!opened) {
    return;
  }
  textEditor->fileLineByLine.clear();
  textEditor->modifiedRowList->clear();
  textEditor->lineNumberArea->update();
  while (!fileToOpen2.atEnd()) {
    textEditor->fileLineByLine.append(fileToOpen2.readLine());
//    qDebug() << fileToOpen2.readLine();
  }
}

QList<QAction *> TextEditor::editActionList()
{
  QList<QAction *> actionsList;
  QMenu *menu = new QMenu;
  actionsList.append(copyAction);
  actionsList.append(cutAction);
  actionsList.append(pasteAction);
  actionsList.append(undoAction);
  actionsList.append(redoAction);
  actionsList.append(menu->addSeparator());
  actionsList.append(textEditor->indentAction);
  actionsList.append(textEditor->unindentAction);
  actionsList.append(textEditor->foldAction);
  actionsList.append(textEditor->unfoldAction);
  actionsList.append(menu->addSeparator());
//  actionsList.append(disableCompletionAction);
  actionsList.append(showHideLineNumbersAction);
  actionsList.append(menu->addSeparator());
  actionsList.append(cutLineAction);
  actionsList.append(toggleCommentAction);
  actionsList.append(menu->addSeparator());
  actionsList.append(wordWrapAction);
  actionsList.append(uppercaseKeywordsAction);
  actionsList.append(lowercaseKeywordsAction);
  actionsList.append(textEditor->visualizeSpacesAction);
  //->setText(tr("Line"))
  actionsList.append(menu->addSeparator());
  actionsList.append(gotoLineAction);
  actionsList.append(highlightCurrentLineAction);
  actionsList.append(normalizeTextAction);
  actionsList.append(textEditor->insertLicenceTemplateAction);
  return actionsList;
}

void TextEditor::modificationChangedSlot(bool changed)
{
  Q_UNUSED(changed);
//  if (changed)
//    parentWidget2->dTitleLabel->setTitleText(parentWidget2->dTitleLabel->text() + "*");
//  else
//    parentWidget2->dTitleLabel->setTitleText(parentWidget2->dTitleLabel->text().left(parentWidget2->dTitleLabel->text().length() -1));
}

//void TextEditor::indentActionTriggered()
//{
//  indentOrUnindent(true);
//}

//void TextEditor::indentOrUnindent(bool doIndent)
//{
//  QTextCursor cursor = textEditor->textCursor();
//  cursor.beginEditBlock();

//  if (cursor.hasSelection()) {
//    int pos = cursor.position();
//    int anchor = cursor.anchor();
//    int start = qMin(anchor, pos);
//    int end = qMax(anchor, pos);

//    QTextDocument *doc = textEditor->document();
//    QTextBlock startBlock = doc->findBlock(start);
//    QTextBlock endBlock = doc->findBlock(end - 1).next();

//    for (QTextBlock block = startBlock; block != endBlock; block = block.next()) {
//      QString text(block.text());
//      int indentPosition = lineIndentPosition(text);
//      if (!doIndent && !indentPosition)
//        indentPosition = firstNonSpace(text);
//      int targetColumn = indentedColumn(columnAt(text, indentPosition), doIndent);
//      cursor.setPosition(block.position() + indentPosition);
//      cursor.insertText(indentationString(0, targetColumn, block));
//      cursor.setPosition(block.position());
//      cursor.setPosition(block.position() + indentPosition, QTextCursor::KeepAnchor);
//      cursor.removeSelectedText();
//    }
//  } else {
//    QTextBlock block = cursor.block();
//    QString text(block.text());
//    int indentPosition = cursor.position() - block.position();
//    int spaces = spacesLeftFromPosition(text, indentPosition);
//    int startColumn = columnAt(text, indentPosition - spaces);
//    int targetColumn = indentedColumn(columnAt(text, indentPosition), doIndent);
//    cursor.setPosition(block.position() + indentPosition);
//    cursor.setPosition(block.position() + indentPosition - spaces, QTextCursor::KeepAnchor);
//    cursor.removeSelectedText();
//    cursor.insertText(indentationString(startColumn, targetColumn, block));
//    textEditor->setTextCursor(cursor);
//  }
//  cursor.endEditBlock();
//}

//int TextEditor::spacesLeftFromPosition(const QString &text, int position) const
//{
//  int pos = position;
//  while (pos > 0) {
//    if (!text.at(pos-1).isSpace())
//      break;
//    --pos;
//  }
//  return position - pos;
//}

//QString TextEditor::indentationString(int startColumn, int targetColumn, const QTextBlock &block) const
//{
//  targetColumn = qMax(startColumn, targetColumn);
//  if (guessSpacesForTabs(block))
//    return QString(targetColumn - startColumn, QLatin1Char(' '));

//  QString s;
//  int alignedStart = startColumn - (startColumn % settings.value("TextEditor/TabSize", 2).toInt())
//      + settings.value("TextEditor/TabSize", 2).toInt();
//  if (alignedStart > startColumn && alignedStart <= targetColumn) {
//    s += QLatin1Char('\t');
//    startColumn = alignedStart;
//  }
//  if (int columns = targetColumn - startColumn) {
//    int tabs = columns / settings.value("TextEditor/TabSize", 2).toInt();
//    s += QString(tabs, QLatin1Char('\t'));
//    s += QString(columns - tabs * settings.value("TextEditor/TabSize", 2).toInt(), QLatin1Char(' '));
//  }
//  return s;
//}

//bool TextEditor::guessSpacesForTabs(const QTextBlock &block) const
//{
//  if (block.isValid()) {
//    const QTextDocument *doc = block.document();
//    QVector<QTextBlock> currentBlocks(2, block);
//    int maxLookAround = 100;
//    while (maxLookAround-- > 0) {
//      if (currentBlocks.at(0).isValid())
//        currentBlocks[0] = currentBlocks.at(0).previous();
//      if (currentBlocks.at(1).isValid())
//        currentBlocks[1] = currentBlocks.at(1).next();
//      bool done = true;
//      foreach (const QTextBlock &block, currentBlocks) {
//        if (block.isValid())
//          done = false;
//        if (!block.isValid() || block.length() == 0)
//          continue;
//        const QChar firstChar = doc->characterAt(block.position());
//        if (firstChar == QLatin1Char(' ')) {
//          return true;
//        } else if (firstChar == QLatin1Char('\t')) {
//          return false;
//        }
//      }
//      if (done)
//        break;
//    }
//  }
//  return true;
//}

//QString TextEditor::indentationString(const QString &text) const
//{
//  return text.left(firstNonSpace(text));
//}

//int TextEditor::indentationColumn(const QString &text) const
//{
//  return columnAt(text, firstNonSpace(text));
//}

//int TextEditor::indentedColumn(int column, bool doIndent) const
//{
//  int aligned = (column / settings.value("TextEditor/TabSize", 2).toInt()) * settings.value("TextEditor/TabSize", 2).toInt();
//  if (doIndent)
//    return aligned + settings.value("TextEditor/TabSize", 2).toInt();
//  if (aligned < column)
//    return aligned;
//  return qMax(0, aligned - settings.value("TextEditor/TabSize", 2).toInt());
//}

//int TextEditor::lineIndentPosition(const QString &text) const
//{
//  int i = 0;
//  while (i < text.size()) {
//    if (!text.at(i).isSpace())
//      break;
//    ++i;
//  }
//  int column = columnAt(text, i);
//  return i - (column % settings.value("TextEditor/TabSize", 2).toInt());
//}

//int TextEditor::firstNonSpace(const QString &text) const
//{
//  int i = 0;
//  while (i < text.size()) {
//    if (!text.at(i).isSpace())
//      return i;
//    ++i;
//  }
//  return i;
//}

//void TextEditor::unidentActionTriggered()
//{
//  indentOrUnindent(false);
//}

void TextEditor::wheelEvent(QWheelEvent *e)
{
  if (e->modifiers() == Qt::ControlModifier) {
    if (e->angleDelta().y() > 0)
      zoomIn();
    else
      zoomOut();
  }
  QWidget::wheelEvent(e);
}

void TextEditor::zoomIn()
{
  newFont = textEditor->font();
  newFont.setPointSize(textEditor->font().pointSize() + 1);
  settings.setValue("TextEditor/PointSize", newFont.pointSize());
  setFontToWidgets(newFont);
  emit zoomInSignal();
}

void TextEditor::zoomOut()
{
  newFont = textEditor->font();
  newFont.setPointSize(textEditor->font().pointSize() - 1);
  settings.setValue("TextEditor/PointSize", newFont.pointSize());
  setFontToWidgets(newFont);
  emit zoomOutSignal();
}

void TextEditor::zoomReset()
{
  newFont = textEditor->font();
  newFont.setPointSize(settings.value("TextEditor/DefaultPointSize", 11).toInt());
  settings.setValue("TextEditor/PointSize", settings.value("TextEditor/DefaultPointSize", 11).toInt());
  setFontToWidgets(newFont);
  emit zoomResetSignal();
}

void TextEditor::setFontToWidgets(const QFont &)
{
  textEditor->setFont(newFont);
  textEditor->lineNumberArea->setFont(newFont);
  completer->popup()->setFont(newFont);
  readOnlyTextEditor->setFont(newFont);
  readOnlyTextEditor->lineNumberArea->setFont(newFont);
}

void TextEditor::exportoToPdfOrPsActionTriggerd()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFileDialog fileDialog;
  fileDialog.setDirectory(QDir::home());
  QString file = fileDialog.getSaveFileName(this, tr("Save to Pdf"), settings.value("GeneralSettings/LastFilePdf", QDir::home().absolutePath()).toString(), tr("Pdf & Ps files (*.pdf *.ps)"));
  settings.setValue("GeneralSettings/LastFilePdf", fileDialog.directory().filePath(file));
  QPrinter printer(QPrinter::HighResolution);
  printer.setOutputFileName(file);
  printer.setOutputFormat(file.endsWith(".pdf") ? QPrinter::PdfFormat : QPrinter::NativeFormat);
  textEditor->document()->print(&printer);
  QApplication::restoreOverrideCursor();
  emit statusBarMessage(tr("File saved at: %1").arg(file));
}

void TextEditor::exportoToOdtActionTriggerd()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFileDialog fileDialog;
  fileDialog.setDirectory(QDir::home());
  QString file = fileDialog.getSaveFileName(this, tr("Save to Odt"), settings.value("GeneralSettings/LastFileOdt", QDir::home().absolutePath()).toString(), tr("Odt files (*.odt)"));
  settings.setValue("GeneralSettings/LastFileOdt", fileDialog.directory().filePath(file));
  QTextDocumentWriter writer(file);
  writer.write(textEditor->document());
  QApplication::restoreOverrideCursor();
  emit statusBarMessage(tr("File saved at: %1").arg(file));
}

void TextEditor::exportoToHtmlActionTriggerd()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFileDialog fileDialog;
  fileDialog.setDirectory(QDir::home());
  QString file = fileDialog.getSaveFileName(this, tr("Save to Html"), settings.value("GeneralSettings/LastFileHtml", QDir::home().absolutePath()).toString(), tr("Html files (*.html *.htm)"));
  settings.setValue("GeneralSettings/LastFileHtml", fileDialog.directory().filePath(file));
  QFile fileOut(file);
  if (!fileOut.open(QIODevice::WriteOnly|QIODevice::Text))
    emit statusBarMessage(tr("Cannot write file %1:\n%2.").arg(file, fileOut.errorString()), QSystemTrayIcon::Critical);
  QTextStream out(&fileOut);
  out.setEncoding(QStringConverter::Utf8);
  out << textEditor->document()->toHtml();
  fileOut.close();
  QApplication::restoreOverrideCursor();
  emit statusBarMessage(tr("File saved at: %1").arg(file));
}

void TextEditor::exportoToSvgActionTriggerd()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFileDialog fileDialog;
  fileDialog.setDirectory(QDir::home());
  QString file = fileDialog.getSaveFileName(this, tr("Save to Svg"), settings.value("GeneralSettings/LastFileSvg", QDir::home().absolutePath()).toString(), tr("Svg files (*.svg)"));
  settings.setValue("GeneralSettings/LastFileSvg", fileDialog.directory().filePath(file));
  QSvgGenerator svg;
  svg.setFileName(file);
  QTextDocument *doc = textEditor->document()->clone();
  QRect rectSize(0, 0, (int) doc->size().width(), (int) doc->size().height());
  QPainter painter(&svg);
  painter.setViewport(rectSize);
  doc->documentLayout()->setPaintDevice(&svg);
  doc->drawContents(&painter);
  QApplication::restoreOverrideCursor();
  emit statusBarMessage(tr("File saved at: %1").arg(file));
}

void TextEditor::exportoToImgActionTriggerd()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFileDialog fileDialog;
  fileDialog.setDirectory(QDir::home());
  QString imageFormats;
  foreach(QString format, QImageWriter::supportedImageFormats())
    imageFormats += "*." + format + " ";
  QString file = fileDialog.getSaveFileName(this, tr("Save to Image"), settings.value("GeneralSettings/LastFileImg", QDir::home().absolutePath()).toString(), tr("Image files (%1)").arg(imageFormats));
  settings.setValue("GeneralSettings/LastFileImg", fileDialog.directory().filePath(file));
  QTextDocument *doc = textEditor->document()->clone();
  QImage image(QRect(0, 0, (int) doc->size().width(), (int) doc->size().height()).size(), QImage::Format_ARGB32);
  QPainter painter(&image);
  painter.setRenderHints(QPainter::Antialiasing| QPainter::TextAntialiasing);
  painter.fillRect(painter.viewport(), Qt::white);
  painter.setViewport(printer.paperRect(QPrinter::Millimeter).toRect());
  doc->documentLayout()->setPaintDevice(&image);
  doc->drawContents(&painter);
  image.save(file);
  QApplication::restoreOverrideCursor();
  emit statusBarMessage(tr("File saved at: %1").arg(file));
}

bool TextEditor::event(QEvent *event)
{
  switch (event->type()) {
  case QEvent::WhatsThis: {
    QHelpEvent* helpEvent = static_cast<QHelpEvent*>(event);
    QTextCursor cursor = textEditor->cursorForPosition(QPoint(helpEvent->x() - (textEditor->lineNumberAreaWidth() + 5), helpEvent->y() - (menuBar->height() + 5 + (dTitleLabel->isVisible() ? (dTitleLabel->height() + 5) : 0))));
    cursor.select(QTextCursor::WordUnderCursor);
    QString message = "<table><tr><td><img src=\"" + DIcon::DatabaseIconPath() + "\" width=\"48\"></td><td valign=middle><p style='white-space:pre'><a href='%1'>%2</a></p></td></tr></table>";
    QString matchedValue = QString();
    QString url = QString();
    switch(editorType) {
    case EditorTypes::SQLQuery:
      matchedValue = StaticFunctions::mariadbFunctionsComplete().value(cursor.selectedText().toUpper());
      url = "https://mariadb.com/kb/en/" + cursor.selectedText().toLower() + "/";
      break;
    case EditorTypes::Diff:
    case EditorTypes::Commit:
    case EditorTypes::SVNLog:
    case EditorTypes::NoEditor:
      break;
      // default: Q_ASSERT(false);
    }
    if (!matchedValue.isEmpty())
      QWhatsThis::showText(helpEvent->globalPos(), message.arg(url, matchedValue), this);
    event->accept();
  }
    break;
  case QEvent::WhatsThisClicked: {
    QWhatsThisClickedEvent *whatsThisEvent = static_cast<QWhatsThisClickedEvent*>(event);
    QWhatsThis::hideText();
    //emit linkClicked( wtcEvent->href() );
    emit openURL(whatsThisEvent->href());
    event->accept();
  }
    break;
  case QEvent::ToolTip: {
    QHelpEvent* helpEvent = static_cast<QHelpEvent*>(event);
    QTextCursor cursor = textEditor->cursorForPosition(QPoint(helpEvent->x() - (textEditor->lineNumberAreaWidth() + 5), helpEvent->y() - (menuBar->height() + 5 + (dTitleLabel->isVisible() ? (dTitleLabel->height() + 5) : 0))));
    cursor.select(QTextCursor::WordUnderCursor);
    QString message = "<table><tr><td><img src=\"" + DIcon::DatabaseIconPath() + "\" width=\"48\"></td><td valign=middle><p style='white-space:pre'>%1</p></td></tr></table>";
    QString matchedValue = QString();
    switch(editorType) {
    case EditorTypes::SQLQuery:
      matchedValue = StaticFunctions::mariadbFunctionsComplete().value(cursor.selectedText().toUpper());
      break;
    case EditorTypes::Diff:
    case EditorTypes::Commit:
    case EditorTypes::SVNLog:
    case EditorTypes::NoEditor:
      break;
      // default: Q_ASSERT(false);
    }
    if (!matchedValue.isEmpty())
      QToolTip::showText(helpEvent->globalPos(), message.arg(matchedValue), this);
    event->accept();
  }
    break;
  default:
    break;
  }
  //return QMdiSubWindow::event(event);
  return QWidget::event(event);
}

//void TextEditor::showEvent(QShowEvent *event)
//{
//  textEditor->setFocus();
//  //QWidget::showEvent(event);
//  QMdiSubWindow::event(event);
//}

//void TextEditor::closeEvent(QCloseEvent *event)
//{
//  if (okToClose())
//    event->accept();
//  else
//    event->ignore();
//  QWidget::closeEvent(event);
////  emit windowClosed();
//}

///*! \brief Try to close the widgets and set the focus
// * to the Editor.
// *
// */
//void TextEditor::setFocusToEditorActionTriggered()
//{
//  qDebug() << "asdf";
//  if (findReplace->isVisible())
//    findReplace->hide();
//  setFocus(Qt::OtherFocusReason);
//}

void TextEditor::createMenu()
{
  menuBar = new QMenuBar;
  fileMenu = new QMenu(" ");
  menuBar->addMenu(fileMenu);
  fileMenu->addActions(fileActionList());
  fileMenu->addSeparator();
  fileMenu->addAction(printAction);
  fileMenu->addSeparator();
  fileMenu->addMenu(exportMenu);
  fileMenu->addSeparator();
  fileMenu->addMenu(openRecentFilesMenu);

  editMenu = new QMenu;
  menuBar->addMenu(editMenu);
  editMenu->addActions(editActionList());
  editMenu->addSeparator();
  editMenu->addAction(findReplace->toggleViewAction());

  optionsMenu = new QMenu;
  menuBar->addMenu(optionsMenu);
  optionsMenu->addAction(disableCompletionAction);

  switch (editorType) {
  case EditorTypes::SQLQuery:
    optionsMenu->addSeparator();
    optionsMenu->addAction(fillMariaDBSymbolsAction);
    optionsMenu->addAction(showMariaDBHelperAction);
    optionsMenu->addAction(addCodeToSnippetAction);
    optionsMenu->addAction(showCodeSnippets);
    optionsMenu->addAction(dummyExecutionAction);
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);
  }

  versionControlMenu = new QMenu;
  subversionMenu = new QMenu;
  subversionMenu->setIcon(DIcon::SVNLogo());
  menuBar->addMenu(versionControlMenu);
  subversionMenu->addActions(subversionedFile->getActions());
  versionControlMenu->addMenu(subversionMenu);

  viewMenu = new QMenu;
  menuBar->addMenu(viewMenu);
  viewMenu->addAction(mainToolBarAction);
  viewMenu->addActions(viewActionList());
  viewMenuSeparatorActionView = viewMenu->addSeparator();
  viewMenu->addAction(viewNormalAction);
  viewMenu->addAction(viewHorizoltalAction);
  viewMenu->addAction(viewVerticalAction);

  changeFontWeightMenuSeparatorAction = viewMenu->addSeparator();
  changeFontWeightActionGroup = new QActionGroup(this);
  connect(changeFontWeightActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(changeFontWeightActionGroupTriggered(QAction*)));

  QFont newFont;
  changeFontWeightActionThin = new QAction;
  newFont = changeFontWeightActionThin->font();
  newFont.setWeight(QFont::Thin);
  changeFontWeightActionThin->setFont(newFont);
  viewMenu->addAction(changeFontWeightActionThin);
  changeFontWeightActionGroup->addAction(changeFontWeightActionThin);
  changeFontWeightActionExtraLight = new QAction;
  newFont = changeFontWeightActionExtraLight->font();
  newFont.setWeight(QFont::ExtraLight);
  changeFontWeightActionExtraLight->setFont(newFont);
  viewMenu->addAction(changeFontWeightActionExtraLight);
  changeFontWeightActionGroup->addAction(changeFontWeightActionExtraLight);
  changeFontWeightActionLight = new QAction;
  newFont = changeFontWeightActionLight->font();
  newFont.setWeight(QFont::Light);
  changeFontWeightActionLight->setFont(newFont);
  viewMenu->addAction(changeFontWeightActionLight);
  changeFontWeightActionGroup->addAction(changeFontWeightActionLight);
  changeFontWeightActionNormal = new QAction;
  newFont = changeFontWeightActionNormal->font();
  newFont.setWeight(QFont::Normal);
  changeFontWeightActionNormal->setFont(newFont);
  viewMenu->addAction(changeFontWeightActionNormal);
  changeFontWeightActionGroup->addAction(changeFontWeightActionNormal);
  changeFontWeightActionMedium = new QAction;
  newFont = changeFontWeightActionMedium->font();
  newFont.setWeight(QFont::Medium);
  changeFontWeightActionMedium->setFont(newFont);
  viewMenu->addAction(changeFontWeightActionMedium);
  changeFontWeightActionGroup->addAction(changeFontWeightActionMedium);
  changeFontWeightActionDemiBold = new QAction;
  newFont = changeFontWeightActionDemiBold->font();
  newFont.setWeight(QFont::DemiBold);
  changeFontWeightActionDemiBold->setFont(newFont);
  viewMenu->addAction(changeFontWeightActionDemiBold);
  changeFontWeightActionGroup->addAction(changeFontWeightActionDemiBold);
  changeFontWeightActionBold = new QAction;
  newFont = changeFontWeightActionBold->font();
  newFont.setWeight(QFont::Bold);
  changeFontWeightActionBold->setFont(newFont);
  viewMenu->addAction(changeFontWeightActionBold);
  changeFontWeightActionGroup->addAction(changeFontWeightActionBold);
  changeFontWeightActionExtraBold = new QAction;
  newFont = changeFontWeightActionExtraBold->font();
  newFont.setWeight(QFont::Bold);
  changeFontWeightActionExtraBold->setFont(newFont);
  viewMenu->addAction(changeFontWeightActionExtraBold);
  changeFontWeightActionGroup->addAction(changeFontWeightActionExtraBold);
  changeFontWeightActionBlack = new QAction;
  newFont = changeFontWeightActionBlack->font();
  newFont.setWeight(QFont::Bold);
  changeFontWeightActionBlack->setFont(newFont);
  viewMenu->addAction(changeFontWeightActionBlack);
  changeFontWeightActionGroup->addAction(changeFontWeightActionBlack);

  viewMenuSeparatorActionSymbols = viewMenu->addSeparator();
  viewMenu->addAction(showSymbolsAction);

  menuBar->setMaximumHeight(menuBar->sizeHint().rheight());
}

/*! \brief Toolbar creation.
 *
 */
void TextEditor::createToolBars()
{
  mainToolBar = new QToolBar;
  mainToolBarAction = mainToolBar->toggleViewAction();
  lineEditFile = new QLineEdit(fileName);
  lineEditFile->setReadOnly(true);
  mainToolBar->addWidget(lineEditFile);
  mainToolBar->addSeparator();
  mainToolBar->addAction(showSymbolsAction);
  mainToolBar->addSeparator();
  labelPosition = new QLabel(" 0 : 0 ");
  labelPosition->setAlignment(Qt::AlignCenter);
  labelPosition->setFrameShape(QFrame::Box);
  labelPosition->setFrameShadow(QFrame::Sunken);
  mainToolBar->addWidget(labelPosition);
  mainToolBar->setIconSize(QSize(18, 18));
  mainToolBar->hide();
}

void TextEditor::viewNormalActionTriggered()
{
  secondSplitter->widget(1)->hide();
}

void TextEditor::viewVerticalActionTriggered()
{
  secondSplitter->setOrientation(Qt::Vertical);
  secondSplitter->widget(1)->show();
  updateReadOnlyEditor();
}

void TextEditor::viewHorizoltalActionTriggered()
{
  secondSplitter->setOrientation(Qt::Horizontal);
  secondSplitter->widget(1)->show();
  updateReadOnlyEditor();
}

void TextEditor::addCodeToSnippetActionSlot()
{
  QTextCursor cursor = textEditor->textCursor();
  if (cursor.hasSelection()) {
    CodeSnippets codeSnippets;
    codeSnippets.addSnippet(cursor.selectedText(), QInputDialog::getText(this, tr("Snippet name"), tr("Snippet name"), QLineEdit::Normal));
  }
}

void TextEditor::itemClickedSlot(QListWidgetItem *item)
{
  textEditor->gotoLine(symbolsLineNumber.value(item->text()));
}

void TextEditor::sortSymbolsSlot()
{
  listSymbols->sortItems(Qt::AscendingOrder);
}

void TextEditor::normalizeTextActionTriggered()
{
  bool found = true;
  while (found) {
    QTextCursor cursor = textEditor->document()->find(QRegularExpression(":(\\w|#|\")", QRegularExpression::CaseInsensitiveOption));
    if (!cursor.hasSelection())
      found = false;
    cursor.setPosition(cursor.position() - 2);
    cursor.movePosition(QTextCursor::EndOfWord, QTextCursor::KeepAnchor);
    cursor.insertText(": ");
  }
}

//void TextEditor::readyReadStandardOutputSlot()
//{
//  procOutput->insertPlainText(proc->readAllStandardOutput());
//}

void TextEditor::updateCursorPositionSlot()
{
  labelPosition->setText(QString("%1: %2, %3: %4, %5: %6").arg(tr("Line")).arg(textEditor->textCursor().block().blockNumber() + 1).arg("Column").arg(textEditor->textCursor().columnNumber() + 1).arg(tr("Selecction")).arg(textEditor->textCursor().selectedText().length()));
  emit updatePrositionViewer(textEditor->textCursor().block().blockNumber(), textEditor->textCursor().columnNumber(), textEditor->textCursor().selectedText().length());
  if (highlightCurrentLineAction->isChecked())
    textEditor->highlightCurrentLine();
  textEditor->matchBracket();
}

void TextEditor::showSymbolsActionTriggered(bool checked)
{
  showSymbolsAction->setChecked(checked);
  listSymbols->setVisible(checked);
  if (!showSymbolsAction->isChecked())
    return;
  QApplication::setOverrideCursor(Qt::WaitCursor);
  int counter = 0;
  listSymbols->clear();
  symbolsLineNumber.clear();
  QString icon;
  QString symbolName;
  switch(editorType) {
  case EditorTypes::SQLQuery:
    foreach (QString line, textEditor->document()->toPlainText().split("\n", Qt::KeepEmptyParts)) {
      icon = "";
      static QRegularExpression createTableRegularExpression("^CREATE.*TABLE");
      if (line.contains(createTableRegularExpression))
        icon = DIcon::DatabaseTableIconPath();

      static QRegularExpression createViewRegularExpression("^CREATE.*VIEW");
      if (line.contains(createViewRegularExpression))
        icon = DIcon::DatabaseViewIconPath();

      static QRegularExpression createDatabaseRegularExpression("^CREATE.*DATABASE");
      if (line.contains(createDatabaseRegularExpression))
        icon = DIcon::DatabaseIconPath();

      static QRegularExpression createProcedureRegularExpression("^CREATE.*PROCEDURE");
      if (line.contains(createProcedureRegularExpression))
        icon = DIcon::DatabaseRoutineIconPath();

      static QRegularExpression createFunctionRegularExpression("^CREATE.*FUNCTION");
      if (line.contains(createFunctionRegularExpression))
        icon = DIcon::DatabaseRoutineIconPath();

      static QRegularExpression createTriggerRegularExpression("^CREATE.*TRIGGER");
      if (line.contains(createTriggerRegularExpression))
        icon = DIcon::DatabaseTriggerIconPath();

      static QRegularExpression createEventRegularExpression("^CREATE.*EVENT");
      if (line.contains(createEventRegularExpression))
        icon = DIcon::DatabaseEventIconPath();

      if (!icon.isEmpty()) {
        static QRegularExpression expression2(StaticFunctions::fullidentifierPattern());
        QStringList data = expression2.match(line).capturedTexts();
        if (!data.empty())
          symbolName = data.at(0);
        if (symbolName.isEmpty()) {
          static QRegularExpression expression3(StaticFunctions::identifierPattern());
          symbolName = expression3.match(line).capturedTexts().at(0);
        }
        symbolName = symbolName.replace("`", "");
        symbolsLineNumber.insert(symbolName, counter);
        listSymbols->addItem(new QListWidgetItem(QIcon(icon), symbolName));
      }
      counter++;
    }
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);
  }
  QApplication::restoreOverrideCursor();
}

void TextEditor::updateReadOnlyEditor()
{
  if (readOnlyTextEditor->isVisible())
    readOnlyTextEditor->setPlainText(textEditor->toPlainText());
}

//void TextEditor::openFile(QString file)
//{
//  textEditor->openFile(file);
//}

void TextEditor::fillMariaDBSymbols()
{
  fillMariaDBSymbolsActionSlot();
}
