/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef TABLEMAINTENANCE_H
#define TABLEMAINTENANCE_H

#include <QWizard>
#include <QWizardPage>

class QTreeWidget;
class BaseTextEditor;
class QTreeWidgetItem;
class QCheckBox;
class QRadioButton;
class DTitleLabel;
class QGroupBox;

#include "dbms.h"
#include "dmdisubwindow.h"

/**
 * @brief The TableMaintenance class provides a GUI for administrative table operations.
 *
 * This class inherits from DMdiSubWindow and allows the user to perform standard
 * database maintenance tasks such as OPTIMIZE, REPAIR, CHECK, CHECKSUM, and ANALYZE
 * on multiple selected tables. It supports various MySQL/MariaDB specific flags
 * (e.g., QUICK, EXTENDED, USE_FRM).
 */
class TableMaintenance : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the TableMaintenance window.
   * @param serverConnection Pointer to the active DBMS connection.
   */
  TableMaintenance(DBMS *serverConnection);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels, buttons, and option text to reflect the current application language.
   */
  void retranslateUI();

private:
  DBMS *serverConnection;
  QTreeWidget *tablesListWidget;
  BaseTextEditor *resultEditor;
  QPushButton *analyzePushButton;
  QPushButton *checkPushButton;
  QPushButton *checksumPushButton;
  QPushButton *optimizePushButton;
  QPushButton *repairPushButton;
  QList<QTreeWidgetItem *> tables;

  // Options corresponding to SQL keywords
  QCheckBox *optionLOCAL;           // NO_WRITE_TO_BINLOG
  QRadioButton *optionFORUPGRADE;   // FOR UPGRADE
  QRadioButton *optionQUICK;        // QUICK
  QRadioButton *optionFAST;         // FAST
  QRadioButton *optionMEDIUM;       // MEDIUM
  QRadioButton *optionEXTENDED;     // EXTENDED
  QRadioButton *optionCHANGED;      // CHANGED
  QRadioButton *optionUSE_FRM;      // USE_FRM

  DTitleLabel *dTitleLabel;
  QGroupBox *groupBoxAction;
  QPushButton *clearSelectionPushButton;
  QPushButton *selectAllTablesPushButton;
  QPushButton *selectAllLocalTablesPushButton;
  bool selectAllLocalTables;

private:
  /**
   * @brief Generates a comma-separated string of selected tables.
   *
   * Iterates through the tree widget to find checked items and formats them
   * for inclusion in the SQL statement (e.g., `db.table1, db.table2`).
   *
   * @return A QString containing the list of tables.
   */
  QString tableList();

  /**
   * @brief Executes the generated maintenance SQL statement.
   * @param statement The full SQL command (e.g., "OPTIMIZE TABLE ...").
   */
  void executeStatement(const QString statement);

signals:
  /**
   * @brief Emitted to update the progress of data loading (e.g., fetching table lists).
   * @param progress The completion percentage (0-100).
   */
  void loadProgress(const int progress);

private slots:
  /**
   * @brief Populates the tree widget with available databases and tables.
   */
  void fillDatabasesSlot();

  /**
   * @brief Generates and executes the ANALYZE TABLE statement.
   */
  void analyzePushButtonSlot();

  /**
   * @brief Generates and executes the CHECK TABLE statement.
   */
  void checkPushButtonSlot();

  /**
   * @brief Generates and executes the CHECKSUM TABLE statement.
   */
  void checksumPushButtonSlot();

  /**
   * @brief Generates and executes the OPTIMIZE TABLE statement.
   */
  void optimizePushButtonSlot();

  /**
   * @brief Generates and executes the REPAIR TABLE statement.
   */
  void repairPushButtonSlot();

  /**
   * @brief Handles item activation (clicks) in the table tree.
   * @param item The item clicked.
   * @param column The column clicked.
   */
  void itemActivatedSlot(QTreeWidgetItem *item, int column);

  /**
   * @brief Deselects all tables in the tree.
   */
  void clearSelectionPushButtonSlot();

  /**
   * @brief Selects all tables across all databases.
   */
  void selectAllTablesPushButtonSlot();

  /**
   * @brief Selects all tables in the currently active/local database.
   */
  void selectAllLocalTablesPushButtonSlot();
};

//class IntroPageTableMaintenance : public QWizardPage
//{
//public:
//  IntroPageTableMaintenance();
//};

//class SelectDatabase : public QWizardPage
//{
//  Q_OBJECT

//public:
//  SelectDatabase();

//private:
//  QListWidget *listWidget;

////private slots:
////  void itemSelectionChangedSlot();
//};

//class SelectTables : public QWizardPage
//{
//  Q_OBJECT

//public:
//  SelectTables();

//private:
//  QPushButton *button;
//  QString databases;

//protected:
//  void initializePage();
//};

//class TableMaintenanceWizard : public QWizard
//{
//public:
//  TableMaintenanceWizard();
//  enum {Intro_Page, SelectDatabase_Page, SelectTables_Page};

//private:
//  SelectDatabase *selectDatabases;
//  SelectTables *selectTables;
////  QString database;
//};

#endif // TABLEMAINTENANCE_H
