/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef SUBVERSIONEDFILE_H
#define SUBVERSIONEDFILE_H

#include <QWidget>

//#include "texteditor.h"

class QAction;
class QProcess;
class DMdiSubWindow;

/**
 * @brief The SubversionedFile class acts as a wrapper for SVN operations on a specific file.
 *
 * This class manages the interaction between the application and the Subversion (SVN)
 * version control system. It provides QActions for common SVN commands (Diff, Commit,
 * Revert, Log, etc.) and executes them using the command-line `svn` tool via QProcess.
 */
class SubversionedFile : public QWidget
{
  Q_OBJECT

public:
  /**
   * @brief Constructs a SubversionedFile handler.
   * @param file The absolute path to the local file to be managed.
   */
  SubversionedFile(QString file);

  /**
   * @brief Retrieves the list of available SVN actions for this file.
   *
   * These actions are typically added to the context menu or toolbar of the text editor
   * or file browser.
   *
   * @return A list of QAction pointers.
   */
  QList<QAction *> getActions();

  /**
   * @brief The path to the file being managed.
   */
  QString fileName;

  /**
   * @brief Static helper to schedule a file for addition to version control.
   *
   * Executes `svn add` on the specified path.
   *
   * @param filePath The path to the file.
   * @return The standard output/error from the SVN command.
   */
  static QString svnAdd(QString filePath);

private:
  /**
   * @brief Checks if the file is currently under version control.
   *
   * Runs `svn info` or `svn status` to determine if the file is tracked.
   * @return True if versioned, false otherwise.
   */
  bool validateVersionedFile();

  /**
   * @brief Executes a raw SVN command.
   * @param commands A list of arguments to pass to the `svn` executable.
   * @return The text output from the command.
   */
  QString svnCommand(QStringList commands);

  QAction *svnDiffAction;
  QAction *svnCommitAction;
  QAction *svnRevertAction;
  QAction *svnInfoAction;
  QAction *svnStatusAction;
  QAction *svnUpdateAction;
  QAction *svnAddAction;
  QAction *svnDeleteAction;
  QAction *svnLogAction;

private slots:
  /**
   * @brief Triggers `svn diff` and displays the differences.
   */
  void svnDiffActionTriggered();

  /**
   * @brief Triggers `svn commit` dialog.
   */
  void svnCommitActionTriggered();

  /**
   * @brief Triggers `svn revert` to discard local changes.
   */
  void svnRevertActionTriggered();

  /**
   * @brief Triggers `svn info` to show repository details.
   */
  void svnInfoActionTriggered();

  /**
   * @brief Triggers `svn status` to show the current file state.
   */
  void svnStatusActionTriggered();

  /**
   * @brief Triggers `svn update` to fetch the latest changes from the repository.
   */
  void svnUpdateActionTriggered();

  /**
   * @brief Triggers `svn add` to start tracking the file.
   */
  void svnAddActionTriggered();

  /**
   * @brief Triggers `svn delete` to remove the file from version control.
   */
  void svnDeleteActionTriggered();

  /**
   * @brief Triggers `svn log` to view the commit history.
   */
  void svnLogActionTriggered();

signals:
  /**
   * @brief Emitted when the file has been reverted.
   *
   * Used to notify the editor that it should reload the file from disk.
   */
  void reverted();

  /**
   * @brief Emitted to display a message on the main status bar.
   * @param message The message text.
   */
  void statusBarMessage(QString message);

  /**
   * @brief Emitted to show a temporary popup message.
   * @param message The message text.
   */
  void statusBarMessagePopup(QString message);

  /**
   * @brief Emitted to open a new sub-window (e.g., for Diff or Log views) in the MDI area.
   * @param dmdiSubWindow The new window instance to display.
   */
  void addSubWindow(DMdiSubWindow *dmdiSubWindow);
};

#endif // SUBVERSIONEDFILE_H
