/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef STATICSYMBOLS_H
#define STATICSYMBOLS_H

#include <QStringList>
#include <QHash>
#include <QLocale>

#include "editortypes.h"

class QIcon;
class EditorTypes;
class QFont;

/**
 * @brief The StaticFunctions class provides a collection of global utility methods.
 *
 * This class acts as a static library for the application, offering helper functions
 * for text formatting, database metadata retrieval (keywords, data types),
 * encryption, font management, and random data generation.
 */
class StaticFunctions : public QObject
{
  Q_OBJECT

public:
  /**
   * @brief Retrieves the current working directory of the application.
   * @return The absolute path as a QString.
   */
  static QString workingDirectory();

  /**
   * @brief Retrieves the standard fixed-width font used in code editors.
   * @return A QFont object configured for monospace display.
   */
  static QFont fixedWidthFont();

  /**
   * @brief Retrieves a list of reserved MariaDB keywords.
   * @return A QStringList of keywords.
   */
  static QStringList mariadbKeywords();

  /**
   * @brief Retrieves a list of helper commands or context-sensitive keywords.
   * @return A QStringList of helper terms.
   */
  static QStringList mariadbHelper();

  /**
   * @brief Retrieves a list of standard MariaDB functions.
   * @return A QStringList of function names (e.g., COUNT, SUM).
   */
  static QStringList mariadbFunctions();

  /**
   * @brief Retrieves a list of MariaDB data types.
   * @return A QStringList of data types (e.g., VARCHAR, INT).
   */
  static QStringList mariadbDatatypes();

  /**
   * @brief Retrieves a list of available character sets.
   * @return A QStringList of charset names.
   */
  static QStringList mariadbCharsets();

  /**
   * @brief Retrieves a list of available collations.
   * @return A QStringList of collation names.
   */
  static QStringList mariadbCharsetsCollations();

  /**
   * @brief Retrieves error patterns related to missing commas on both sides.
   * @return A list of syntax rule strings.
   */
  static QStringList mariadbCodeErrorsNotCommaBothSides();

  /**
   * @brief Retrieves error patterns related to missing commas on the left side.
   * @return A list of syntax rule strings.
   */
  static QStringList mariadbCodeErrorsNotCommaOnLeft();

  /**
   * @brief Converts a string representation of bytes into a human-readable format.
   * @param bytes The byte count as a string.
   * @return A formatted string (e.g., "1.5 MB").
   */
  static QString bytesConvertor(QString bytes);

  /**
   * @brief Converts a numeric byte count into a human-readable format.
   * @param bytes The byte count.
   * @return A formatted string (e.g., "2.0 GB").
   */
  static QString bytesConvertor(qulonglong bytes);

  /**
   * @brief Encloses the text in backticks or quotes for SQL safety.
   * @param text The identifier to quote.
   * @return The quoted string.
   */
  static QString quoteSymbol(QString text);

  /**
   * @brief Removes backticks or quotes from the text.
   * @param text The quoted identifier.
   * @return The raw identifier string.
   */
  static QString unquoteSymbol(QString text);

  /**
   * @brief Encrypts or decrypts a password string.
   * @param pass The password string.
   * @param encrypt True to encrypt, False to decrypt.
   * @return The processed password string.
   */
  static QString password(QString pass = QString(), bool encrypt = false);

  /**
   * @brief Retrieves a complete map of MariaDB functions with descriptions/signatures.
   * @return A QHash mapping function names to their details.
   */
  static QHash<QString, QString> mariadbFunctionsComplete();

  /**
   * @brief Finds the matching right bracket for a given left bracket.
   * @param leftBracket The opening character (e.g., '(').
   * @return The closing character (e.g., ')'), or null if not found.
   */
  static char rightBracketMatch(const char leftBracket);

  /**
   * @brief Finds the matching left bracket for a given right bracket.
   * @param rightBracket The closing character (e.g., '}').
   * @return The opening character (e.g., '{'), or null if not found.
   */
  static char leftBracketMatch(const char rightBracket);

  /**
   * @brief Retrieves the list of supported opening bracket characters.
   * @return A list of characters (e.g., '(', '{', '[').
   */
  static QList<char> bracketLeftList();

  /**
   * @brief Returns the identifier for a numeric item delegate.
   */
  static QString DelegateTypeNumber();

  /**
   * @brief Returns the identifier for a null/no-op delegate.
   */
  static QString DelegateTypeNoDelegate();

  /**
   * @brief Returns the identifier for a Date-Time item delegate.
   */
  static QString DelegateTypeDateTime();

  /**
   * @brief Returns the identifier for a Date item delegate.
   */
  static QString DelegateTypeDate();

  /**
   * @brief Returns the identifier for a Password item delegate.
   */
  static QString DelegateTypePassword();

  /**
   * @brief Returns the identifier for an Enum item delegate.
   */
  static QString DelegateTypeEnum();

  /**
   * @brief Retrieves an icon appropriate for a specific file name/extension.
   * @param fileName The path or name of the file.
   * @return The corresponding QIcon.
   */
  static QIcon iconFromFileName(const QString fileName);

  /**
   * @brief Determines the editor type based on a file extension.
   * @param fileName The name of the file.
   * @return The matching EditorType.
   */
  static EditorTypes::EditorType findEditorType(const QString fileName);

  /**
   * @brief Retrieves the standard icon associated with an editor type.
   * @param editorType The type of editor.
   * @return The corresponding QIcon.
   */
  static QIcon iconFromFileName(const EditorTypes::EditorType editorType);

  /**
   * @brief Returns a list of enabled/supported DBMS names.
   */
  static QStringList dbmsEnabled();

  /**
   * @brief Enumeration of supported Database Management Systems.
   */
  enum dbmsTypes {Undefined, MySQL, MariaDB, PostgreSQL};

  /**
   * @brief Returns the standard format string for SQL DateTime.
   */
  static QString DBMSDateTimeFormat();

  /**
   * @brief Returns the standard format string for SQL Date.
   */
  static QString DBMSDateFormat();

  /**
   * @brief Returns the name of the default database system (e.g., "mysql").
   */
  static QString DBMSDefaultDatabase();

//  static QMap<QString, QVariant> explodeConnectionString(QString connectionName);

  /**
   * @brief Generates a random alphanumeric string.
   * @param length The desired length of the string.
   * @return The generated string.
   */
  static QString randomString(unsigned int length = 1);

  /**
   * @brief Generates a random name (e.g., for temporary tables).
   * @param length The desired length.
   * @return The generated name.
   */
  static QString randomName(unsigned int length = 1);

  /**
   * @brief Retrieves syntax keywords for the currently active editor type.
   * @param editorType The current editor context.
   * @return A list of applicable keywords.
   */
  static QStringList currentEditorTypeKeywords(const EditorTypes::EditorType editorType);

  /**
   * @brief Retrieves the application's current locale settings.
   */
  static QLocale currentLocale();

  /**
   * @brief Returns the SQL query used to fetch server information.
   */
  static QString serverInformationQuery();

  /**
   * @brief Returns the SQL query used to fetch slow queries.
   */
  static QString slowQueriesQuery();

  /**
   * @brief Enumeration for search flags (Direction, Case, Whole Word).
   */
  enum findFlags {FindForward, FindBackward, FindCaseSensitively, FindWholeWords};

  /**
   * @brief Returns the Regex pattern used to identify valid SQL identifiers.
   */
  static QString identifierPattern();

  /**
   * @brief Returns the full Regex pattern for identifiers including qualification.
   */
  static QString fullidentifierPattern();
};

#endif // STATICSYMBOLS_H
