/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QFontDatabase>
#include <QTime>
#include <QFileInfo>
//#include <QApplication>
#include <QRandomGenerator>
#include <QDir>

#include "staticfunctions.h"
#include "dicon.h"
#include "dsettings.h"

#include <QDebug>

QString StaticFunctions::DelegateTypeNumber() { return "DelegateTypeNumber"; }
QString StaticFunctions::DelegateTypeNoDelegate() { return "DelegateTypeNoDelegate"; }
QString StaticFunctions::DelegateTypeDateTime() { return "DelegateTypeDateTime"; }
QString StaticFunctions::DelegateTypeDate() { return "DelegateTypeDate"; }
QString StaticFunctions::DelegateTypePassword() { return "DelegateTypePassword"; }
QString StaticFunctions::DelegateTypeEnum() { return "DelegateTypeEnum"; }

QString StaticFunctions::DBMSDateTimeFormat() {

  return "yyyy-MM-dd hh:mm:ss";
}

QString StaticFunctions::DBMSDateFormat() {

  return "yyyy-MM-dd";
}

QString StaticFunctions::DBMSDefaultDatabase()
{
  return "mysql";
}

EditorTypes::EditorType StaticFunctions::findEditorType(const QString fileName)
{
  QFileInfo fileInfo(fileName);
  QString completeSuffix(fileInfo.completeSuffix().toLower());
  QString suffix(fileInfo.suffix().toLower());
  DSettings settings;
  if (settings.value("FileAssociations/SQLFiles", "").toString().split(",").contains(completeSuffix)
      || settings.value("FileAssociations/SQLFiles", "").toString().split(",").contains(suffix))
    return EditorTypes::SQLQuery;
  return EditorTypes::NoEditor;
}

QIcon StaticFunctions::iconFromFileName(const EditorTypes::EditorType editorType)
{
  switch(editorType) {
  case EditorTypes::SQLQuery:
    return iconFromFileName("file.sql");
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
    //  case EditorTypes::NoEditor:
    //    return iconFromFileName("file.");
    //    break;
  // case EditorTypes::Diff:
  // default: Q_ASSERT(false);
  }
  return iconFromFileName("");
}

QStringList StaticFunctions::dbmsEnabled()
{
  return QStringList() << "--" << "MySQL" << "MariaDB";// << "PostgreSQL";
}

QIcon StaticFunctions::iconFromFileName(const QString fileName)
{
  switch(findEditorType(fileName)) {
  case EditorTypes::SQLQuery:
    return DIcon::FileSQL();
    break;
  case EditorTypes::NoEditor:
    return DIcon::DocumentNew();
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
    break;
  // case EditorTypes::Diff:
  // default: Q_ASSERT(false);
  }

  return QIcon();
}

char StaticFunctions::rightBracketMatch(const char leftBracket)
{
  if (leftBracket == '(')
    return ')';
  if (leftBracket == '{')
    return '}';
  if (leftBracket == '[')
    return ']';
  return '(';
}

char StaticFunctions::leftBracketMatch(const char rightBracket)
{
  if (rightBracket == ')')
    return '(';
  if (rightBracket == '}')
    return '{';
  if (rightBracket == ']')
    return '[';
  return ')';
}

QList<char> StaticFunctions::bracketLeftList()
{
  QList<char> list;
  list << '(' << '[' << '{';
  return list;

}

QHash<QString, QString> StaticFunctions::mariadbFunctionsComplete()
{
  QHash<QString, QString> data;
  // --- A ---
  data.insert("ABS", QString("<u>ABS</u>(<b>X</b>) <br />%1").arg(tr("Returns the absolute value of <b>X</b>")));
  data.insert("ACOS", QString("<u>ACOS</u>(<b>X</b>) <br />%1").arg(tr("Returns the arc cosine of X. Returns NULL if X is not in the range -1 to 1.")));
  data.insert("ADDDATE", QString("<u>ADDDATE</u>(<b>date, INTERVAL expr unit</b>) <br />%1").arg(tr("Adds a time value (interval) to a date value.")));
  data.insert("ADDTIME", QString("<u>ADDTIME</u>(<b>expr1, expr2</b>) <br />%1").arg(tr("Adds expr2 to expr1 and returns the result.<br />expr1 is a time or datetime expression, and expr2 is a time expression.")));
  data.insert("AES_DECRYPT", QString("binary <u>AES_DECRYPT</u>(<b>crypt_str, key_str</b>) <br />%1").arg(tr("This function decrypts data using the official AES (Advanced Encryption Standard) algorithm.")));
  data.insert("AES_ENCRYPT", QString("binary <u>AES_ENCRYPT</u>(<b>crypt_str, key_str</b>) <br />%1").arg(tr("This function encrypts data using the official AES (Advanced Encryption Standard) algorithm.")));
  data.insert("ASCII", QString("<u>ASCII</u>(<b>str</b>) <br />%1").arg(tr("Returns the numeric value of the leftmost character of the string str.")));
  data.insert("ASIN", QString("<u>ASIN</u>(<b>X</b>) <br />%1").arg(tr("Returns the arc sine of X. Returns NULL if X is not in the range -1 to 1.")));
  data.insert("ATAN", QString("<u>ATAN</u>(<b>X</b>) <br />%1").arg(tr("Returns the arc tangent of X.")));
  data.insert("ATAN2", QString("<u>ATAN2</u>(<b>Y, X</b>) <br />%1").arg(tr("Returns the arc tangent of the two variables X and Y.")));
  data.insert("AVG", QString("<u>AVG</u>(<b>[DISTINCT] expr</b>) <br />%1").arg(tr("Returns the average value of expr.<br />The DISTINCT option returns the average of the distinct values of expr.")));

  // --- B ---
  data.insert("BENCHMARK", QString("<u>BENCHMARK</u>(<b>count, expr</b>) <br />%1").arg(tr("Executes the expression expr repeatedly count times.")));
  data.insert("BIN", QString("<u>BIN</u>(<b>N</b>) <br />%1").arg(tr("Returns a string representation of the binary value of N.")));
  data.insert("BIT_AND", QString("<u>BIT_AND</u>(<b>expr</b>) <br />%1").arg(tr("Returns the bitwise AND of all bits in expr (64-bit precision).")));
  data.insert("BIT_COUNT", QString("<u>BIT_COUNT</u>(<b>N</b>) <br />%1").arg(tr("Returns the number of bits that are set in the argument N.")));
  data.insert("BIT_LENGTH", QString("<u>BIT_LENGTH</u>(<b>str</b>) <br />%1").arg(tr("Returns the length of the string str in bits.")));
  data.insert("BIT_OR", QString("<u>BIT_OR</u>(<b>expr</b>) <br />%1").arg(tr("Returns the bitwise OR of all bits in expr (64-bit precision).")));
  data.insert("BIT_XOR", QString("<u>BIT_XOR</u>(<b>expr</b>) <br />%1").arg(tr("Returns the bitwise XOR of all bits in expr (64-bit precision).")));

  // --- C ---
  data.insert("CAST", QString("<u>CAST</u>(<b>expr AS type</b>) <br />%1").arg(tr("Takes an expression of any type and produces a result value of a specified type.")));
  data.insert("CEIL", QString("<u>CEIL</u>(<b>x</b>) <br />%1").arg(tr("Returns the smallest integer value not less than X.")));
  data.insert("CEILING", QString("<u>CEILING</u>(<b>x</b>) <br />%1").arg(tr("Returns the smallest integer value not less than X.")));
  data.insert("CHAR_LENGTH", QString("<u>CHAR_LENGTH</u>(<b>str</b>) <br />%1").arg(tr("Returns the length of the string str, measured in characters.")));
  data.insert("CHARACTER_LENGTH", QString("<u>CHARACTER_LENGTH</u>(<b>str</b>) <br />%1").arg(tr("Returns the length of the string str, measured in characters.")));
  data.insert("COALESCE", QString("<u>COALESCE</u>(<b>value, ...</b>) <br />%1").arg(tr("Returns the first non-NULL value in the list.")));
  data.insert("COERCIBILITY", QString("<u>COERCIBILITY</u>(<b>str</b>) <br />%1").arg(tr("Returns the collation coercibility value of the string argument.")));
  data.insert("COLLATION", QString("<u>COLLATION</u>(<b>str</b>) <br />%1").arg(tr("Returns the collation of the string argument.")));
  data.insert("COMPRESS", QString("<u>COMPRESS</u>(<b>string_to_compress</b>) <br />%1").arg(tr("Compresses a string and returns the result as a binary string.")));
  data.insert("CONCAT", QString("<u>CONCAT</u>(<b>str1, str2, ...</b>) <br />%1").arg(tr("Returns the string that results from concatenating the arguments.")));
  data.insert("CONCAT_WS", QString("string <u>CONCAT_WS</u>(<b>separator, str1[,str2,...]</b>) <br />%1").arg(tr("Concatenate the strings with the given separator.")));
  data.insert("CONNECTION_ID", QString("<u>CONNECTION_ID</u>() <br />%1").arg(tr("Returns the connection ID (thread ID) for the connection.")));
  data.insert("CONV", QString("<u>CONV</u>(<b>N, from_base, to_base</b>) <br />%1").arg(tr("Converts numbers between different number bases.")));
  data.insert("CONVERT_TZ", QString("<u>CONVERT_TZ</u>(<b>dt, from_tz, to_tz</b>) <br />%1").arg(tr("Converts a datetime value dt from one time zone to another.")));
  data.insert("COS", QString("<u>COS</u>(<b>X</b>) <br />%1").arg(tr("Returns the cosine of X, where X is given in radians.")));
  data.insert("COT", QString("<u>COT</u>(<b>X</b>) <br />%1").arg(tr("Returns the cotangent of X.")));
  data.insert("COUNT", QString("<u>COUNT</u>(<b>expr</b>) <br />%1").arg(tr("Returns a count of the number of non-NULL values of expr in the rows retrieved by a SELECT statement.")));
  data.insert("CRC32", QString("<u>CRC32</u>(<b>expr</b>) <br />%1").arg(tr("Computes a cyclic redundancy check value and returns a 32-bit unsigned value.")));
  data.insert("CUME_DIST", QString("<u>CUME_DIST</u>() <br />%1").arg(tr("Window function: returns the cumulative distribution of a given row.")));
  data.insert("CURDATE", QString("<u>CURDATE</u>() <br />%1").arg(tr("Returns the current date.")));
  data.insert("CURRENT_DATE", QString("<u>CURRENT_DATE</u>() <br />%1").arg(tr("Returns the current date.")));
  data.insert("CURRENT_ROLE", QString("<u>CURRENT_ROLE</u>() <br />%1").arg(tr("Returns the current role name.")));
  data.insert("CURRENT_TIME", QString("<u>CURRENT_TIME</u>() <br />%1").arg(tr("Returns the current time.")));
  data.insert("CURRENT_TIMESTAMP", QString("<u>CURRENT_TIMESTAMP</u>() <br />%1").arg(tr("Returns the current date and time.")));
  data.insert("CURRENT_USER", QString("<u>CURRENT_USER</u>() <br />%1").arg(tr("Returns the user name and host name combination for the authenticated account.")));
  data.insert("CURTIME", QString("<u>CURTIME</u>() <br />%1").arg(tr("Returns the current time.")));

  // --- D ---
  data.insert("DATABASE", QString("<u>DATABASE</u>() <br />%1").arg(tr("Returns the default (current) database name.")));
  data.insert("DATE", QString("<u>DATE</u>(<b>expr</b>) <br />%1").arg(tr("Extracts the date part of a date or datetime expression.")));
  data.insert("DATE_ADD", QString("<u>DATE_ADD</u>(<b>date, INTERVAL expr unit</b>) <br />%1").arg(tr("Performs date arithmetic (adds interval).")));
  data.insert("DATE_FORMAT", QString("<u>DATE_FORMAT</u>(<b>date, format</b>) <br />%1").arg(tr("Returns a string containing the date in the specified format.")));
  data.insert("DATE_SUB", QString("<u>DATE_SUB</u>(<b>date, INTERVAL expr unit</b>) <br />%1").arg(tr("Performs date arithmetic (subtracts interval).")));
  data.insert("DATEDIFF", QString("<u>DATEDIFF</u>(<b>expr1, expr2</b>) <br />%1").arg(tr("Returns expr1 - expr2 expressed as a value in days.")));
  data.insert("DAY", QString("<u>DAY</u>(<b>date</b>) <br />%1").arg(tr("Returns the day of the month (1-31).")));
  data.insert("DAY_HOUR", QString("<u>DAY_HOUR</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("DAY_MICROSECOND", QString("<u>DAY_MICROSECOND</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("DAY_MINUTE", QString("<u>DAY_MINUTE</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("DAY_SECOND", QString("<u>DAY_SECOND</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("DAYNAME", QString("<u>DAYNAME</u>(<b>date</b>) <br />%1").arg(tr("Returns the name of the weekday for date.")));
  data.insert("DAYOFMONTH", QString("<u>DAYOFMONTH</u>(<b>date</b>) <br />%1").arg(tr("Returns the day of the month for date (1-31).")));
  data.insert("DAYOFWEEK", QString("<u>DAYOFWEEK</u>(<b>date</b>) <br />%1").arg(tr("Returns the weekday index for date (1 = Sunday...).")));
  data.insert("DAYOFYEAR", QString("<u>DAYOFYEAR</u>(<b>date</b>) <br />%1").arg(tr("Returns the day of the year for date (1-366).")));
  data.insert("DECODE", QString("<u>DECODE</u>(<b>crypt_str, pass_str</b>) <br />%1").arg(tr("Decrypts the encrypted string crypt_str using pass_str.")));
  data.insert("DEGREES", QString("<u>DEGREES</u>(<b>X</b>) <br />%1").arg(tr("Returns the argument X, converted from radians to degrees.")));
  data.insert("DENSE_RANK", QString("<u>DENSE_RANK</u>() <br />%1").arg(tr("Window function: Returns the rank of the current row within its partition (no gaps).")));
  data.insert("DES_DECRYPT", QString("<u>DES_DECRYPT</u>(<b>crypt_str[, key_str]</b>) <br />%1").arg(tr("Decrypts a string encrypted with Triple-DES.")));
  data.insert("DES_ENCRYPT", QString("<u>DES_ENCRYPT</u>(<b>str[, key]</b>) <br />%1").arg(tr("Encrypts the string with Triple-DES.")));
  data.insert("DIV", QString("<u>DIV</u> <br />%1").arg(tr("Integer division.")));

  // --- E ---
  data.insert("ELT", QString("<u>ELT</u>(<b>N, str1, str2, ...</b>) <br />%1").arg(tr("Returns string1 if N = 1, string2 if N = 2, and so on.")));
  data.insert("ENCODE", QString("<u>ENCODE</u>(<b>str, pass_str</b>) <br />%1").arg(tr("Encrypts str using pass_str as the password.")));
  data.insert("ENCRYPT", QString("<u>ENCRYPT</u>(<b>str[, salt]</b>) <br />%1").arg(tr("Encrypts str using the Unix crypt() system call.")));
  data.insert("EXP", QString("<u>EXP</u>(<b>X</b>) <br />%1").arg(tr("Returns the value of e raised to the power of X.")));
  data.insert("EXPORT_SET", QString("<u>EXPORT_SET</u>(<b>bits, on, off...</b>) <br />%1").arg(tr("Returns a string where bits set return 'on' and unset return 'off'.")));
  data.insert("EXTRACT", QString("<u>EXTRACT</u>(<b>unit FROM date</b>) <br />%1").arg(tr("Extracts parts from the date.")));
  data.insert("EXTRACTVALUE", QString("<u>EXTRACTVALUE</u>(<b>xml_frag, xpath_expr</b>) <br />%1").arg(tr("Extracts a value from an XML string using XPath notation.")));

  // --- F ---
  data.insert("FIELD", QString("<u>FIELD</u>(<b>str, str1, str2, ...</b>) <br />%1").arg(tr("Returns the index (position) of str in the list.")));
  data.insert("FIND_IN_SET", QString("<u>FIND_IN_SET</u>(<b>str, strlist</b>) <br />%1").arg(tr("Returns the position of str in the string list strlist.")));
  data.insert("FLOOR", QString("<u>FLOOR</u>(<b>x</b>) <br />%1").arg(tr("Returns the largest integer value not greater than x.")));
  data.insert("FORMAT", QString("<u>FORMAT</u>(<b>X, D</b>) <br />%1").arg(tr("Formats the number X to a format like '#,###,###.##'.")));
  data.insert("FOUND_ROWS", QString("<u>FOUND_ROWS</u>() <br />%1").arg(tr("Returns the number of rows from the last SELECT (ignoring LIMIT if SQL_CALC_FOUND_ROWS used).")));
  data.insert("FROM_DAYS", QString("<u>FROM_DAYS</u>(<b>N</b>) <br />%1").arg(tr("Given a day number N, returns a DATE value.")));
  data.insert("FROM_UNIXTIME", QString("date <u>FROM_UNIXTIME</u>(<b>unix_timestamp [, format])</b>) <br />%1").arg(tr("Returns a representation of the unix_timestamp argument as a date/time value.")));

  // --- G ---
  data.insert("GET_FORMAT", QString("<u>GET_FORMAT</u>(<b>type, standard</b>) <br />%1").arg(tr("Returns a format string for date/time functions.")));
  data.insert("GET_LOCK", QString("<u>GET_LOCK</u>(<b>str, timeout</b>) <br />%1").arg(tr("Tries to obtain a named lock.")));
  data.insert("GREATEST", QString("<u>GREATEST</u>(<b>value1, value2, ...</b>) <br />%1").arg(tr("Returns the largest (maximum-valued) argument.")));
  data.insert("GROUP_CONCAT", QString("<u>GROUP_CONCAT</u>(<b>expr</b>) <br />%1").arg(tr("Returns a string result with the concatenated non-NULL values from a group.")));

  // --- H ---
  data.insert("HEX", QString("string <u>HEX</u>(<b>str_or_num</b>) <br />%1").arg(tr("Returns a hexadecimal string representation of the value.")));
  data.insert("HOUR_MICROSECOND", QString("<u>HOUR_MICROSECOND</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("HOUR_MINUTE", QString("<u>HOUR_MINUTE</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("HOUR_SECOND", QString("<u>HOUR_SECOND</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));

  // --- I ---
  data.insert("IF", QString("<u>IF</u>(<b>expr1, expr2, expr3</b>) <br />%1").arg(tr("If expr1 is TRUE, returns expr2; otherwise, returns expr3.")));
  data.insert("IFNULL", QString("<u>IFNULL</u>(<b>expr1, expr2</b>) <br />%1").arg(tr("If expr1 is not NULL, returns expr1; otherwise it returns expr2.")));
  data.insert("INET_ATON", QString("<u>INET_ATON</u>(<b>expr</b>) <br />%1").arg(tr("Given an IPv4 address string, returns an integer representation.")));
  data.insert("INET_NTOA", QString("<u>INET_NTOA</u>(<b>expr</b>) <br />%1").arg(tr("Given a numeric IPv4 address, returns the dotted-quad string representation.")));
  data.insert("INSTR", QString("<u>INSTR</u>(<b>str, substr</b>) <br />%1").arg(tr("Returns the position of the first occurrence of substring substr in string str.")));
  data.insert("INTERVAL", QString("<u>INTERVAL</u>(<b>N, N1, N2, ...</b>) <br />%1").arg(tr("Returns the index of the argument that is less than the first argument.")));
  data.insert("IS_FREE_LOCK", QString("<u>IS_FREE_LOCK</u>(<b>str</b>) <br />%1").arg(tr("Checks whether the lock named str is free to use.")));
  data.insert("IS_USED_LOCK", QString("<u>IS_USED_LOCK</u>(<b>str</b>) <br />%1").arg(tr("Checks whether the lock named str is in use.")));
  data.insert("ISNULL", QString("<u>ISNULL</u>(<b>expr</b>) <br />%1").arg(tr("If expr is NULL, returns 1, otherwise it returns 0.")));

  // --- J (JSON Functions) ---
  data.insert("JSON_ARRAY", QString("<u>JSON_ARRAY</u>(<b>[value[, value ...]]</b>) <br />%1").arg(tr("Returns a JSON array containing the listed values.")));
  data.insert("JSON_ARRAYAGG", QString("<u>JSON_ARRAYAGG</u>(<b>expr</b>) <br />%1").arg(tr("Aggregates a result set into a single JSON array.")));
  data.insert("JSON_ARRAY_APPEND", QString("<u>JSON_ARRAY_APPEND</u>(<b>json_doc, path, value...</b>) <br />%1").arg(tr("Appends values to the end of the indicated arrays within a JSON document.")));
  data.insert("JSON_ARRAY_INSERT", QString("<u>JSON_ARRAY_INSERT</u>(<b>json_doc, path, value...</b>) <br />%1").arg(tr("Inserts values into arrays within a JSON document.")));
  data.insert("JSON_ARRAY_INTERSECT", QString("<u>JSON_ARRAY_INTERSECT</u>(<b>arr1, arr2</b>) <br />%1").arg(tr("Returns the intersection of two JSON arrays.")));
  data.insert("JSON_COMPACT", QString("<u>JSON_COMPACT</u>(<b>json_doc</b>) <br />%1").arg(tr("Removes all unnecessary whitespace from a JSON document.")));
  data.insert("JSON_CONTAINS", QString("<u>JSON_CONTAINS</u>(<b>json_doc, value[, path]</b>) <br />%1").arg(tr("Checks if a specific value exists within the JSON document.")));
  data.insert("JSON_CONTAINS_PATH", QString("<u>JSON_CONTAINS_PATH</u>(<b>json_doc, one_or_all, path...</b>) <br />%1").arg(tr("Checks if paths exist within the JSON document.")));
  data.insert("JSON_DEPTH", QString("<u>JSON_DEPTH</u>(<b>json_doc</b>) <br />%1").arg(tr("Returns the maximum depth of a JSON document.")));
  data.insert("JSON_DETAILED", QString("<u>JSON_DETAILED</u>(<b>json_doc</b>) <br />%1").arg(tr("Formats JSON to be human-readable.")));
  data.insert("JSON_EQUALS", QString("<u>JSON_EQUALS</u>(<b>doc1, doc2</b>) <br />%1").arg(tr("Checks if two JSON documents are equal.")));
  data.insert("JSON_EXISTS", QString("<u>JSON_EXISTS</u>(<b>json_doc, path</b>) <br />%1").arg(tr("Checks if the specified path exists in the JSON document.")));
  data.insert("JSON_EXTRACT", QString("<u>JSON_EXTRACT</u>(<b>json_doc, path...</b>) <br />%1").arg(tr("Returns extracted data from specific paths in the JSON document.")));
  data.insert("JSON_INSERT", QString("<u>JSON_INSERT</u>(<b>json_doc, path, value...</b>) <br />%1").arg(tr("Inserts data into a JSON document without replacing existing data.")));
  data.insert("JSON_KEY_VALUE", QString("<u>JSON_KEY_VALUE</u>(<b>json_doc</b>) <br />%1").arg(tr("Converts object elements into key-value structure.")));
  data.insert("JSON_KEYS", QString("<u>JSON_KEYS</u>(<b>json_doc[, path]</b>) <br />%1").arg(tr("Returns an array of keys from the top-level value.")));
  data.insert("JSON_LENGTH", QString("<u>JSON_LENGTH</u>(<b>json_doc[, path]</b>) <br />%1").arg(tr("Returns the length of a JSON document.")));
  data.insert("JSON_LOOSE", QString("<u>JSON_LOOSE</u>(<b>json_doc</b>) <br />%1").arg(tr("Adds whitespace to a JSON document to make it more readable.")));
  data.insert("JSON_MERGE", QString("<u>JSON_MERGE</u>(<b>doc1, doc2...</b>) <br />%1").arg(tr("Merges multiple JSON documents.")));
  data.insert("JSON_MERGE_PATCH", QString("<u>JSON_MERGE_PATCH</u>(<b>doc1, doc2...</b>) <br />%1").arg(tr("Merges JSON documents following RFC 7396.")));
  data.insert("JSON_MERGE_PRESERVE", QString("<u>JSON_MERGE_PRESERVE</u>(<b>doc1, doc2...</b>) <br />%1").arg(tr("Merges JSON documents preserving all values.")));
  data.insert("JSON_NORMALIZE", QString("<u>JSON_NORMALIZE</u>(<b>json_doc</b>) <br />%1").arg(tr("Normalizes the JSON document.")));
  data.insert("JSON_OBJECT", QString("<u>JSON_OBJECT</u>(<b>[key, value...]</b>) <br />%1").arg(tr("Returns a JSON object containing the listed key-value pairs.")));
  data.insert("JSON_OBJECT_FILTER_KEYS", QString("<u>JSON_OBJECT_FILTER_KEYS</u>(<b>doc, keys</b>) <br />%1").arg(tr("Returns a JSON object containing only the specified keys.")));
  data.insert("JSON_OBJECT_TO_ARRAY", QString("<u>JSON_OBJECT_TO_ARRAY</u>(<b>json_doc</b>) <br />%1").arg(tr("Converts a JSON object to a JSON array of key-value pairs.")));
  data.insert("JSON_OBJECTAGG", QString("<u>JSON_OBJECTAGG</u>(<b>key, value</b>) <br />%1").arg(tr("Aggregates a result set into a single JSON object.")));
  data.insert("JSON_OVERLAPS", QString("<u>JSON_OVERLAPS</u>(<b>doc1, doc2</b>) <br />%1").arg(tr("Checks if two JSON documents have any common parts.")));
  data.insert("JSON_PRETTY", QString("<u>JSON_PRETTY</u>(<b>json_doc</b>) <br />%1").arg(tr("Formats JSON with indentation and newlines.")));
  data.insert("JSON_QUERY", QString("<u>JSON_QUERY</u>(<b>json_doc, path</b>) <br />%1").arg(tr("Returns an object or array from a JSON document.")));
  data.insert("JSON_QUOTE", QString("<u>JSON_QUOTE</u>(<b>string</b>) <br />%1").arg(tr("Quotes a string as a JSON value.")));
  data.insert("JSON_REMOVE", QString("<u>JSON_REMOVE</u>(<b>json_doc, path...</b>) <br />%1").arg(tr("Removes data from a JSON document at the specified paths.")));
  data.insert("JSON_REPLACE", QString("<u>JSON_REPLACE</u>(<b>json_doc, path, value...</b>) <br />%1").arg(tr("Replaces existing values in a JSON document.")));
  data.insert("JSON_SCHEMA_VALID", QString("<u>JSON_SCHEMA_VALID</u>(<b>schema, doc</b>) <br />%1").arg(tr("Validates a JSON document against a JSON schema.")));
  data.insert("JSON_SEARCH", QString("<u>JSON_SEARCH</u>(<b>json_doc, one_or_all, str...</b>) <br />%1").arg(tr("Returns the path to the given string within a JSON document.")));
  data.insert("JSON_SET", QString("<u>JSON_SET</u>(<b>json_doc, path, value...</b>) <br />%1").arg(tr("Inserts or updates data in a JSON document.")));
  data.insert("JSON_TABLE", QString("<u>JSON_TABLE</u>(<b>json_doc, path COLUMNS...</b>) <br />%1").arg(tr("Extracts data from a JSON document and returns it as a relational table.")));
  data.insert("JSON_TYPE", QString("<u>JSON_TYPE</u>(<b>json_val</b>) <br />%1").arg(tr("Returns a string indicating the type of the JSON value.")));
  data.insert("JSON_UNQUOTE", QString("<u>JSON_UNQUOTE</u>(<b>json_val</b>) <br />%1").arg(tr("Unquotes a JSON value, returning the result as a string.")));
  data.insert("JSON_VALID", QString("<u>JSON_VALID</u>(<b>val</b>) <br />%1").arg(tr("Returns 1 if the value is valid JSON, otherwise 0.")));
  data.insert("JSON_VALUE", QString("<u>JSON_VALUE</u>(<b>json_doc, path</b>) <br />%1").arg(tr("Extracts a scalar value from a JSON document.")));

  // --- L ---
  data.insert("LAG", QString("<u>LAG</u>(<b>expr[, N]</b>) <br />%1").arg(tr("Window Function: Accesses data from a previous row.")));
  data.insert("LAST_DAY", QString("<u>LAST_DAY</u>(<b>date</b>) <br />%1").arg(tr("Returns the last day of the month for the date argument.")));
  data.insert("LAST_INSERT_ID", QString("<u>LAST_INSERT_ID</u>(<b>[expr]</b>) <br />%1").arg(tr("Returns the first automatically generated value successfully inserted for an AUTO_INCREMENT column.")));
  data.insert("LCASE", QString("<u>LCASE</u>(<b>str</b>) <br />%1").arg(tr("Synonym for LOWER().")));
  data.insert("LEAD", QString("<u>LEAD</u>(<b>expr[, N]</b>) <br />%1").arg(tr("Window Function: Accesses data from a following row.")));
  data.insert("LEAST", QString("<u>LEAST</u>(<b>value1, value2, ...</b>) <br />%1").arg(tr("Returns the smallest (minimum-valued) argument.")));
  data.insert("LEFT", QString("<u>LEFT</u>(<b>str, len</b>) <br />%1").arg(tr("Returns the leftmost len characters from the string str.")));
  data.insert("LENGTH", QString("<u>LENGTH</u>(<b>str</b>) <br />%1").arg(tr("Returns the length of the string str, measured in bytes.")));
  data.insert("LN", QString("<u>LN</u>(<b>X</b>) <br />%1").arg(tr("Returns the natural logarithm of X.")));
  data.insert("LOAD_FILE", QString("<u>LOAD_FILE</u>(<b>path</b>) <br />%1").arg(tr("Loads the given file.")));
  data.insert("LOCALTIMESTAMP", QString("<u>LOCALTIMESTAMP</u>() <br />%1").arg(tr("Returns the current date and time.")));
  data.insert("LOCATE", QString("<u>LOCATE</u>(<b>substr, str[, pos])</b>) <br />%1").arg(tr("Returns the position of the first occurrence of substring substr in string str.")));
  data.insert("LOG", QString("<u>LOG</u>(<b>X</b>) <br />%1").arg(tr("Returns the natural logarithm of X.")));
  data.insert("LOG10", QString("<u>LOG10</u>(<b>X</b>) <br />%1").arg(tr("Returns the base-10 logarithm of X.")));
  data.insert("LOG2", QString("<u>LOG2</u>(<b>X</b>) <br />%1").arg(tr("Returns the base-2 logarithm of X.")));
  data.insert("LOWER", QString("<u>LOWER</u>(<b>str</b>) <br />%1").arg(tr("Returns the string str with all characters changed to lowercase.")));
  data.insert("LPAD", QString("<u>LPAD</u>(<b>str, len, padstr</b>) <br />%1").arg(tr("Returns the string str, left-padded with the string padstr to a length of len characters.")));
  data.insert("LTRIM", QString("<u>LTRIM</u>(<b>str</b>) <br />%1").arg(tr("Returns the string str with leading space characters removed.")));

  // --- M ---
  data.insert("MAKE_SET", QString("<u>MAKE_SET</u>(<b>bits, str1, str2...</b>) <br />%1").arg(tr("Returns a set value based on bits.")));
  data.insert("MAKEDATE", QString("<u>MAKEDATE</u>(<b>year, dayofyear</b>) <br />%1").arg(tr("Returns a date, given year and day-of-year values.")));
  data.insert("MAKETIME", QString("<u>MAKETIME</u>(<b>hour, minute, second</b>) <br />%1").arg(tr("Returns a time value calculated from the arguments.")));
  data.insert("MASTER_POS_WAIT", QString("<u>MASTER_POS_WAIT</u>(<b>log, pos</b>) <br />%1").arg(tr("Blocks until the replica has read and applied updates to the specified position.")));
  data.insert("MAX", QString("<u>MAX</u>(<b>[DISTINCT] expr)</b>) <br />%1").arg(tr("Returns the maximum value of expr.")));
  data.insert("MD5", QString("<u>MD5</u>(<b>str</b>) <br />%1").arg(tr("Calculates an MD5 128-bit checksum for the string.")));
  data.insert("MICROSECOND", QString("<u>MICROSECOND</u>(<b>expr</b>) <br />%1").arg(tr("Returns the microseconds from the time expression.")));
  data.insert("MID", QString("<u>MID</u>(<b>str, pos, len</b>) <br />%1").arg(tr("Returns a substring starting at position pos.")));
  data.insert("MIN", QString("<u>MIN</u>(<b>[DISTINCT] expr</b>) <br />%1").arg(tr("Returns the minimum value of expr.")));
  data.insert("MINUTE_MICROSECOND", QString("<u>MINUTE_MICROSECOND</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("MINUTE_SECOND", QString("<u>MINUTE_SECOND</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("MOD", QString("<u>MOD</u>(<b>N, M</b>) <br />%1").arg(tr("Modulo operation. Returns the remainder of N divided by M.")));
  data.insert("MONTH", QString("<u>MONTH</u>(<b>date</b>) <br />%1").arg(tr("Returns the month for date (1-12).")));
  data.insert("MONTHNAME", QString("<u>MONTHNAME</u>(<b>date</b>) <br />%1").arg(tr("Returns the full name of the month for date.")));

  // --- N ---
  data.insert("NAME_CONST", QString("<u>NAME_CONST</u>(<b>name, value</b>) <br />%1").arg(tr("Returns the given value and causes the column to have the given name.")));
  data.insert("NOW", QString("<u>NOW</u>() <br />%1").arg(tr("Returns the current date and time.")));
  data.insert("NTILE", QString("<u>NTILE</u>(<b>N</b>) <br />%1").arg(tr("Window function: Divides the partition into N groups.")));
  data.insert("NULLIF", QString("<u>NULLIF</u>(<b>expr1, expr2</b>) <br />%1").arg(tr("Returns NULL if expr1 = expr2 is true, otherwise returns expr1.")));

  // --- O ---
  data.insert("OCT", QString("<u>OCT</u>(<b>N</b>) <br />%1").arg(tr("Returns a string representation of the octal value of N.")));
  data.insert("OCTET_LENGTH", QString("<u>OCTET_LENGTH</u>(<b>str</b>) <br />%1").arg(tr("Returns the length of the string str, measured in bytes.")));
  data.insert("OLD_PASSWORD", QString("<u>OLD_PASSWORD</u>(<b>str</b>) <br />%1").arg(tr("Returns the pre-4.1 password hash.")));
  data.insert("ORD", QString("<u>ORD</u>(<b>str</b>) <br />%1").arg(tr("Returns the code for the leftmost character.")));

  // --- P ---
  data.insert("PASSWORD", QString("<u>PASSWORD</u>(<b>str</b>) <br />%1").arg(tr("Calculates and returns a password string (Deprecated).")));
  data.insert("PERCENT_RANK", QString("<u>PERCENT_RANK</u>() <br />%1").arg(tr("Window function: Returns the relative percent rank of a given row.")));
  data.insert("PERIOD_ADD", QString("<u>PERIOD_ADD</u>(<b>P, N</b>) <br />%1").arg(tr("Adds N months to period P.")));
  data.insert("PERIOD_DIFF", QString("<u>PERIOD_DIFF</u>(<b>P1, P2</b>) <br />%1").arg(tr("Returns the number of months between periods P1 and P2.")));
  data.insert("PI", QString("<u>PI</u>() <br />%1").arg(tr("Returns the value of PI.")));
  data.insert("PLUGINS", QString("<u>PLUGINS</u> <br />%1").arg(tr("Use 'SHOW PLUGINS' to display information about server plugins.")));
  data.insert("POSITION", QString("<u>POSITION</u>(<b>substr IN str</b>) <br />%1").arg(tr("Returns the position of the first occurrence of substring.")));
  data.insert("POW", QString("<u>POW</u>(<b>X, Y</b>) <br />%1").arg(tr("Returns the value of X raised to the power of Y.")));
  data.insert("POWER", QString("<u>POWER</u>(<b>X, Y</b>) <br />%1").arg(tr("Returns the value of X raised to the power of Y.")));

  // --- Q ---
  data.insert("QUOTE", QString("<u>QUOTE</u>(<b>str</b>) <br />%1").arg(tr("Quotes a string to produce a result usable in SQL.")));

  // --- R ---
  data.insert("RADIANS", QString("<u>RADIANS</u>(<b>X</b>) <br />%1").arg(tr("Returns the argument X, converted from degrees to radians.")));
  data.insert("RAND", QString("<u>RAND</u>(<b>[N]</b>) <br />%1").arg(tr("Returns a random floating-point value.")));
  data.insert("RANK", QString("<u>RANK</u>() <br />%1").arg(tr("Window function: Returns the rank of the current row (with gaps).")));
  data.insert("REGEXP_REPLACE", QString("<u>REGEXP_REPLACE</u>(<b>str, pattern, replace</b>) <br />%1").arg(tr("Replaces occurrences of the pattern with the replacement string.")));
  data.insert("REGEXP_SUBSTR", QString("<u>REGEXP_SUBSTR</u>(<b>str, pattern</b>) <br />%1").arg(tr("Returns the substring that matches the regular expression.")));
  data.insert("RELEASE_LOCK", QString("<u>RELEASE_LOCK</u>(<b>str</b>) <br />%1").arg(tr("Releases the named lock.")));
  data.insert("REPLACE", QString("<u>REPLACE</u>(<b>str, from, to</b>) <br />%1").arg(tr("Replaces all occurrences of a specified string.")));
  data.insert("REVERSE", QString("<u>REVERSE</u>(<b>str</b>) <br />%1").arg(tr("Returns the string str with the order of the characters reversed.")));
  data.insert("RIGHT", QString("<u>RIGHT</u>(<b>str, len</b>) <br />%1").arg(tr("Returns the rightmost len characters from the string str.")));
  data.insert("ROUND", QString("<u>ROUND</u>(<b>X [, D]</b>) <br />%1").arg(tr("Rounds the argument X to D decimal places.")));
  data.insert("ROW_COUNT", QString("<u>ROW_COUNT</u>() <br />%1").arg(tr("Returns the number of rows changed, deleted, or inserted by the last statement.")));
  data.insert("ROW_NUMBER", QString("<u>ROW_NUMBER</u>() <br />%1").arg(tr("Window function: Returns the number of the current row.")));
  data.insert("RPAD", QString("<u>RPAD</u>(<b>str, len, padstr</b>) <br />%1").arg(tr("Returns the string str, right-padded with the string padstr.")));
  data.insert("RTRIM", QString("<u>RTRIM</u>(<b>str</b>) <br />%1").arg(tr("Returns the string str with trailing space characters removed.")));

  // --- S ---
  data.insert("SEC_TO_TIME", QString("<u>SEC_TO_TIME</u>(<b>seconds</b>) <br />%1").arg(tr("Returns the seconds argument, converted to a TIME value.")));
  data.insert("SECOND_MICROSECOND", QString("<u>SECOND_MICROSECOND</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("SESSION_USER", QString("<u>SESSION_USER</u>() <br />%1").arg(tr("Returns the current user name and host name.")));
  data.insert("SHA", QString("<u>SHA</u>(<b>str</b>) <br />%1").arg(tr("Calculates an SHA-1 160-bit checksum for the string.")));
  data.insert("SHA1", QString("<u>SHA1</u>(<b>str</b>) <br />%1").arg(tr("Calculates an SHA-1 160-bit checksum for the string.")));
  data.insert("SHA2", QString("<u>SHA2</u>(<b>str, len</b>) <br />%1").arg(tr("Calculates the SHA-2 family of hash functions.")));
  data.insert("SIGN", QString("<u>SIGN</u>(<b>X</b>) <br />%1").arg(tr("Returns the sign of the argument (-1, 0, 1).")));
  data.insert("SIN", QString("<u>SIN</u>(<b>X</b>) <br />%1").arg(tr("Returns the sine of X.")));
  data.insert("SLEEP", QString("0 <u>SLEEP</u>(<b>duration</b>) <br />%1").arg(tr("Sleeps (pauses) for the number of seconds given by the duration.")));
  data.insert("SOUNDEX", QString("<u>SOUNDEX</u>(<b>str</b>) <br />%1").arg(tr("Returns a soundex string from str.")));
  data.insert("SPACE", QString("<u>SPACE</u>(<b>N</b>) <br />%1").arg(tr("Returns a string consisting of N space characters.")));
  data.insert("SQL_CALC_FOUND_ROWS", QString("<u>SQL_CALC_FOUND_ROWS</u> <br />%1").arg(tr("Keyword used in SELECT to calculate total rows ignoring LIMIT.")));
  data.insert("SQR", QString("<u>SQR</u>(<b>X</b>) <br />%1").arg(tr("Typically refers to square.")));
  data.insert("SQRT", QString("<u>SQRT</u>(<b>X</b>) <br />%1").arg(tr("Returns the square root of a non-negative number X.")));
  data.insert("STD", QString("<u>STD</u>(<b>expr</b>) <br />%1").arg(tr("Returns the population standard deviation of expr.")));
  data.insert("STDDEV", QString("<u>STDDEV</u>(<b>expr</b>) <br />%1").arg(tr("Returns the population standard deviation of expr.")));
  data.insert("STDDEV_POP", QString("<u>STDDEV_POP</u>(<b>expr</b>) <br />%1").arg(tr("Returns the population standard deviation of expr.")));
  data.insert("STDDEV_SAMP", QString("<u>STDDEV_SAMP</u>(<b>expr</b>) <br />%1").arg(tr("Returns the sample standard deviation of expr.")));
  data.insert("STR_TO_DATE", QString("<u>STR_TO_DATE</u>(<b>str, format</b>) <br />%1").arg(tr("Converts a string to a date value.")));
  data.insert("STRCMP", QString("<u>STRCMP</u>(<b>expr1, expr2</b>) <br />%1").arg(tr("Compares two strings. Returns 0 if equal, -1 if smaller, 1 if larger.")));
  data.insert("SUBDATE", QString("<u>SUBDATE</u>(<b>date, INTERVAL expr unit</b>) <br />%1").arg(tr("Subtracts a time value (interval) from a date.")));
  data.insert("SUBSTR", QString("<u>SUBSTR</u>(<b>str, pos, len</b>) <br />%1").arg(tr("Returns a substring.")));
  data.insert("SUBSTRING", QString("<u>SUBSTRING</u>(<b>str, pos, len</b>) <br />%1").arg(tr("Returns a substring.")));
  data.insert("SUBSTRING_INDEX", QString("<u>SUBSTRING_INDEX</u>(<b>str, delim, count</b>) <br />%1").arg(tr("Returns the substring from string str before count occurrences of the delimiter.")));
  data.insert("SUBTIME", QString("<u>SUBTIME</u>(<b>expr1, expr2</b>) <br />%1").arg(tr("Returns expr1 – expr2 expressed as a value in the same format as expr1.")));
  data.insert("SUM", QString("<u>SUM</u>(<b>[DISTINCT] expr</b>) <br />%1").arg(tr("Returns the sum of expr.")));
  data.insert("SYSDATE", QString("<u>SYSDATE</u>() <br />%1").arg(tr("Returns the current date and time (at execution time).")));
  data.insert("SYSTEM_USER", QString("<u>SYSTEM_USER</u>() <br />%1").arg(tr("Synonym for USER().")));

  // --- T ---
  data.insert("TAN", QString("<u>TAN</u>(<b>X</b>) <br />%1").arg(tr("Returns the tangent of X.")));
  data.insert("TIME_FORMAT", QString("<u>TIME_FORMAT</u>(<b>time, format</b>) <br />%1").arg(tr("Returns a string containing the time in the specified format.")));
  data.insert("TIME_TO_SEC", QString("<u>TIME_TO_SEC</u>(<b>time</b>) <br />%1").arg(tr("Returns the time argument, converted to seconds.")));
  data.insert("TIMEDIFF", QString("<u>TIMEDIFF</u>(<b>expr1, expr2</b>) <br />%1").arg(tr("Returns expr1 - expr2 expressed as a time value.")));
  data.insert("TIMESTAMP", QString("<u>TIMESTAMP</u>(<b>expr</b>) <br />%1").arg(tr("Returns the argument as a datetime value.")));
  data.insert("TIMESTAMPADD", QString("<u>TIMESTAMPADD</u>(<b>unit, interval, expr</b>) <br />%1").arg(tr("Adds the integer expression interval to the date or datetime expression.")));
  data.insert("TIMESTAMPDIFF", QString("<u>TIMESTAMPDIFF</u>(<b>unit, expr1, expr2</b>) <br />%1").arg(tr("Returns expr2 - expr1.")));
  data.insert("TO_DAYS", QString("<u>TO_DAYS</u>(<b>date</b>) <br />%1").arg(tr("Returns a day number (the number of days since year 0).")));
  data.insert("TRIM", QString("<u>TRIM</u>(<b>[remstr FROM] str</b>) <br />%1").arg(tr("Returns the string str with all remstr prefixes or suffixes removed.")));
  data.insert("TRUNCATE", QString("<u>TRUNCATE</u>(<b>X, D</b>) <br />%1").arg(tr("Returns the number X, truncated to D decimal places.")));

  // --- U ---
  data.insert("UCASE", QString("<u>UCASE</u>(<b>str</b>) <br />%1").arg(tr("Synonym for UPPER().")));
  data.insert("UNCOMPRESS", QString("<u>UNCOMPRESS</u>(<b>str</b>) <br />%1").arg(tr("Uncompresses a string compressed by the COMPRESS() function.")));
  data.insert("UNCOMPRESSED_LENGTH", QString("<u>UNCOMPRESSED_LENGTH</u>(<b>str</b>) <br />%1").arg(tr("Returns the length of the compressed string before compression.")));
  data.insert("UNHEX", QString("<u>UNHEX</u>(<b>str</b>) <br />%1").arg(tr("Performs the inverse operation of HEX(str).")));
  data.insert("UNIX_TIMESTAMP", QString("<u>UNIX_TIMESTAMP</u>(<b>[date]</b>) <br />%1").arg(tr("Returns a Unix timestamp.")));
  data.insert("UpdateXML", QString("<u>UpdateXML</u>(<b>target, xpath, new_xml</b>) <br />%1").arg(tr("Replaces a portion of XML markup.")));
  data.insert("UPPER", QString("<u>UPPER</u>(<b>str</b>) <br />%1").arg(tr("Returns the string str with all characters changed to uppercase.")));
  data.insert("USER", QString("<u>USER</u>() <br />%1").arg(tr("Returns the current MySQL user name and host name.")));
  data.insert("UTC_DATE", QString("<u>UTC_DATE</u>() <br />%1").arg(tr("Returns the current UTC date.")));
  data.insert("UTC_TIME", QString("<u>UTC_TIME</u>() <br />%1").arg(tr("Returns the current UTC time.")));
  data.insert("UTC_TIMESTAMP", QString("<u>UTC_TIMESTAMP</u>() <br />%1").arg(tr("Returns the current UTC date and time.")));
  data.insert("UUID", QString("string <u>UUID</u>() <br />%1").arg(tr("Returns a Universal Unique Identifier (UUID).")));

  // --- V ---
  data.insert("VAR_POP", QString("<u>VAR_POP</u>(<b>expr</b>) <br />%1").arg(tr("Returns the population standard variance of expr.")));
  data.insert("VAR_SAMP", QString("<u>VAR_SAMP</u>(<b>expr</b>) <br />%1").arg(tr("Returns the sample standard variance of expr.")));
  data.insert("VARIANCE", QString("<u>VARIANCE</u>(<b>expr</b>) <br />%1").arg(tr("Returns the population standard variance of expr.")));
  data.insert("VERSION", QString("<u>VERSION</u>() <br />%1").arg(tr("Returns a string that indicates the MySQL server version.")));

  // --- W ---
  data.insert("WEEK", QString("<u>WEEK</u>(<b>date [,mode]</b>) <br />%1").arg(tr("Returns the week number for date.")));
  data.insert("WEEKDAY", QString("<u>WEEKDAY</u>(<b>date</b>) <br />%1").arg(tr("Returns the weekday index for date (0-6).")));
  data.insert("WEEKOFYEAR", QString("<u>WEEKOFYEAR</u>(<b>date</b>) <br />%1").arg(tr("Returns the calendar week of the date (1-53).")));

  // --- Y ---
  data.insert("YEAR", QString("<u>YEAR</u>(<b>date</b>) <br />%1").arg(tr("Returns the year for the given date.")));
  data.insert("YEAR_MONTH", QString("<u>YEAR_MONTH</u> <br />%1").arg(tr("Keyword used in INTERVAL expressions.")));
  data.insert("YEARWEEK", QString("<u>YEARWEEK</u>(<b>date [, mode]</b>) <br />%1").arg(tr("Returns the year and week for a date.")));
  //Descripction for Keywords
  data.insert("BETWEEN", QString("expr BETWEEN min AND max <br />%1").arg(tr("If expr is greater than or equal to min and expr is less than or equal to max, BETWEEN returns 1, otherwise it returns 0.")));
  return data;
}

QString StaticFunctions::password(QString pass, bool encrypt)
{
  QString pas;

  if (encrypt) {
    QChar *data;
    QRandomGenerator random(QTime::currentTime().second());
    pas = "*";
    data = pass.data();
    while (!data->isNull()) {
      int seed = (int) random.bounded(32767) % 9;
      pas += QString("%1%2").arg(seed).arg(QChar(data->unicode() + seed));
      ++data;
    }
  } else {
    for (int i = 1; i < pass.length(); i++) {
      pas += QString("%1").arg(QChar(pass.at(i + 1).unicode() - QString(pass.at(i)).toInt()));
      i = i + 1;
    }
  }
  return pas;
}

QString StaticFunctions::quoteSymbol(QString text)
{
  return (text.startsWith("`") ? "" : "`") + text + (text.endsWith("`") ? "" : "`");
}

QString StaticFunctions::unquoteSymbol(QString text)
{
  if (text.startsWith("`"))
    text = text.right(text.length() - 1);
  if (text.endsWith("`"))
    text = text.left(text.length() - 1);
  return text;
}

QString StaticFunctions::workingDirectory()
{
  return QDir::tempPath() + QDir::separator() + "Caliope-TempFiles" + QDir::separator();
}

QFont StaticFunctions::fixedWidthFont()
{
  foreach (QString family, QFontDatabase::families())
    if (QFontDatabase::isFixedPitch(family))
      return QFont(family);
  return QFont();
}

QStringList StaticFunctions::mariadbKeywords()
{
  // https://mariadb.com/kb/en/information-schema-keywords-table/
  return QStringList() << "&&"
<< "<="
<< "<>"
<< "!="
<< ">="
<< "<<"
<< ">>"
<< "<=>"
  << "ACCESSIBLE" << "ACCOUNT" << "ACTION" << "ADD" << "ADMIN" << "AFTER" << "AGAINST" << "AGGREGATE"
  << "ALGORITHM" << "ALL" << "ALTER" << "ALWAYS" << "ANALYZE" << "AND" << "ANY" << "AS"
  << "ASC" << "ASCII" << "ASENSITIVE" << "AT" << "ATOMIC" << "AUTHORS" << "AUTO" << "AUTO_INCREMENT"
  << "AUTOEXTEND_SIZE" << "AVG" << "AVG_ROW_LENGTH" << "BACKUP" << "BEFORE" << "BEGIN" << "BETWEEN" << "BIGINT"
  << "BINARY" << "BINLOG" << "BIT" << "BLOB" << "BLOCK" << "BODY" << "BOOL" << "BOOLEAN"
  << "BOTH" << "BTREE" << "BY" << "BYTE" << "CACHE" << "CALL" << "CASCADE" << "CASCADED"
  << "CASE" << "CATALOG_NAME" << "CHAIN" << "CHANGE" << "CHANGED" << "CHAR" << "CHARACTER" << "CHARSET"
  << "CHECK" << "CHECKPOINT" << "CHECKSUM" << "CIPHER" << "CLASS_ORIGIN" << "CLIENT" << "CLOB" << "CLOSE"
  << "COALESCE" << "CODE" << "COLLATE" << "COLLATION" << "COLUMN" << "COLUMN_ADD" << "COLUMN_CHECK" << "COLUMN_CREATE"
  << "COLUMN_DELETE" << "COLUMN_GET" << "COLUMN_NAME" << "COLUMNS" << "COMMENT" << "COMMIT" << "COMMITTED" << "COMPACT"
  << "COMPLETION" << "COMPRESSED" << "CONCURRENT" << "CONDITION" << "CONNECTION" << "CONSISTENT" << "CONSTRAINT" << "CONSTRAINT_CATALOG"
  << "CONSTRAINT_NAME" << "CONSTRAINT_SCHEMA" << "CONTAINS" << "CONTEXT" << "CONTINUE" << "CONTRIBUTORS" << "CONVERT" << "CPU"
  << "CREATE" << "CROSS" << "CUBE" << "CURRENT" << "CURRENT_DATE" << "CURRENT_POS" << "CURRENT_ROLE" << "CURRENT_TIME"
  << "CURRENT_TIMESTAMP" << "CURRENT_USER" << "CURSOR" << "CURSOR_NAME" << "CYCLE" << "DATA" << "DATABASE" << "DATABASES"
  << "DATAFILE" << "DATE" << "DATETIME" << "DAY" << "DAY_HOUR" << "DAY_MICROSECOND" << "DAY_MINUTE" << "DAY_SECOND"
  << "DEALLOCATE" << "DEC" << "DECIMAL" << "DECLARE" << "DEFAULT" << "DEFINER" << "DELAY_KEY_WRITE" << "DELAYED"
  << "DELETE" << "DELETE_DOMAIN_ID" << "DES_KEY_FILE" << "DESC" << "DESCRIBE" << "DETERMINISTIC" << "DIAGNOSTICS" << "DIRECTORY"
  << "DISABLE" << "DISCARD" << "DISK" << "DISTINCT" << "DISTINCTROW" << "DIV" << "DO" << "DO_DOMAIN_IDS"
  << "DOUBLE" << "DROP" << "DUAL" << "DUMPFILE" << "DUPLICATE" << "DYNAMIC" << "EACH" << "ELSE"
  << "ELSEIF" << "ELSIF" << "ENABLE" << "ENCLOSED" << "END" << "ENDS" << "ENGINE" << "ENGINES"
  << "ENUM" << "ERROR" << "ERRORS" << "ESCAPE" << "ESCAPED" << "EVENT" << "EVENTS" << "EVERY"
  << "EXAMINED" << "EXCEPT" << "EXCEPTION" << "EXCHANGE" << "EXCLUDE" << "EXECUTE" << "EXISTS" << "EXIT"
  << "EXPANSION" << "EXPLAIN" << "EXPIRE" << "EXPORT" << "EXTENDED" << "EXTENT_SIZE" << "FALSE" << "FAST"
  << "FAULTS" << "FEDERATED" << "FETCH" << "FIELDS" << "FILE" << "FIRST" << "FIXED" << "FLOAT"
  << "FLOAT4" << "FLOAT8" << "FLUSH" << "FOLLOWING" << "FOLLOWS" << "FOR" << "FORCE" << "FOREIGN"
  << "FORMAT" << "FOUND" << "FROM" << "FULL" << "FULLTEXT" << "FUNCTION" << "GENERAL" << "GENERATED"
  << "GET" << "GET_FORMAT" << "GLOBAL" << "GOTO" << "GRANT" << "GRANTS" << "GROUP" << "HANDLER"
  << "HARD" << "HASH" << "HAVING" << "HELP" << "HIGH_PRIORITY" << "HISTORY" << "HOST" << "HOSTS"
  << "HOUR" << "HOUR_MICROSECOND" << "HOUR_MINUTE" << "HOUR_SECOND" << "ID" << "IDENTIFIED" << "IF" << "IGNORE"
  << "IGNORE_DOMAIN_IDS" << "IGNORE_SERVER_IDS" << "IMMEDIATE" << "IMPORT" << "IN" << "INCREMENT" << "INDEX" << "INDEXES"
  << "INFILE" << "INITIAL_SIZE" << "INNER" << "INOUT" << "INSENSITIVE" << "INSERT" << "INSERT_METHOD" << "INSTALL"
  << "INT" << "INT1" << "INT2" << "INT3" << "INT4" << "INT8" << "INTEGER"
  << "INTERSECT" << "INTERVAL" << "INTO" << "INVISIBLE" << "INVOKER" << "IO" << "IO_THREAD" << "IPC"
  << "IS" << "ISOLATION" << "ISOPEN" << "ISSUER" << "ITERATE" << "JOIN" << "JSON" << "KEY"
  << "KEY_BLOCK_SIZE" << "KEYS" << "KILL" << "LANGUAGE" << "LAST" << "LAST_VALUE" << "LASTVAL" << "LEADING"
  << "LEAVE" << "LEAVES" << "LEFT" << "LESS" << "LEVEL" << "LIKE" << "LIMIT" << "LINEAR"
  << "LINES" << "LIST" << "LOAD" << "LOCAL" << "LOCALTIME" << "LOCALTIMESTAMP" << "LOCK" << "LOCKS"
  << "LOGFILE" << "LOGS" << "LONG" << "LONGBLOB" << "LONGTEXT" << "LOOP" << "LOW_PRIORITY" << "MASTER"
  << "MASTER_CONNECT_RETRY" << "MASTER_DELAY" << "MASTER_GTID_POS" << "MASTER_HEARTBEAT_PERIOD" << "MASTER_HOST" << "MASTER_LOG_FILE" << "MASTER_LOG_POS" << "MASTER_PASSWORD"
  << "MASTER_PORT" << "MASTER_SERVER_ID" << "MASTER_SSL" << "MASTER_SSL_CA" << "MASTER_SSL_CAPATH" << "MASTER_SSL_CERT" << "MASTER_SSL_CIPHER" << "MASTER_SSL_CRL"
  << "MASTER_SSL_CRLPATH" << "MASTER_SSL_KEY" << "MASTER_SSL_VERIFY_SERVER_CERT" << "MASTER_USE_GTID" << "MASTER_USER" << "MATCH" << "MAX_CONNECTIONS_PER_HOUR" << "MAX_QUERIES_PER_HOUR"
  << "MAX_ROWS" << "MAX_SIZE" << "MAX_STATEMENT_TIME" << "MAX_UPDATES_PER_HOUR" << "MAX_USER_CONNECTIONS" << "MAXVALUE" << "MEDIUM" << "MEDIUMBLOB"
  << "MEDIUMINT" << "MEDIUMTEXT" << "MEMORY" << "MERGE" << "MESSAGE_TEXT" << "MICROSECOND" << "MIDDLEINT" << "MIGRATE"
  << "MIN_ROWS" << "MINUTE" << "MINUTE_MICROSECOND" << "MINUTE_SECOND" << "MINVALUE" << "MOD" << "MODE" << "MODIFIES"
  << "MODIFY" << "MONITOR" << "MONTH" << "MUTEX" << "MYSQL" << "MYSQL_ERRNO" << "NAME" << "NAMES"
  << "NATIONAL" << "NATURAL" << "NCHAR" << "NEVER" << "NEW" << "NEXT" << "NEXTVAL" << "NO"
  << "NO_WAIT" << "NO_WRITE_TO_BINLOG" << "NOCACHE" << "NOCYCLE" << "NODEGROUP" << "NOMAXVALUE" << "NOMINVALUE" << "NONE"
  << "NOT" << "NOTFOUND" << "NOWAIT" << "NULL" << "NUMBER" << "NUMERIC" << "NVARCHAR" << "OF"
  << "OFFSET" << "OLD_PASSWORD" << "ON" << "ONE" << "ONLINE" << "ONLY" << "OPEN" << "OPTIMIZE"
  << "OPTION" << "OPTIONALLY" << "OPTIONS" << "OR" << "ORDER" << "OTHERS" << "OUT" << "OUTER"
  << "OUTFILE" << "OVER" << "OVERLAPS" << "OWNER" << "PACK_KEYS" << "PACKAGE" << "PAGE" << "PAGE_CHECKSUM"
  << "PARSE_VCOL_EXPR" << "PARSER" << "PARTIAL" << "PARTITION" << "PARTITIONING" << "PARTITIONS" << "PASSWORD" << "PERIOD"
  << "PERSISTENT" << "PHASE" << "PLUGIN" << "PLUGINS" << "PORT" << "PORTION" << "PRECEDES" << "PRECEDING"
  << "PRECISION" << "PREPARE" << "PRESERVE" << "PREV" << "PREVIOUS" << "PRIMARY" << "PRIVILEGES" << "PROCEDURE"
  << "PROCESS" << "PROCESSLIST" << "PROFILE" << "PROFILES" << "PROXY" << "PURGE" << "QUARTER" << "QUERY"
  << "QUICK" << "RAISE" << "RANGE" << "RAW" << "READ" << "READ_ONLY" << "READ_WRITE" << "READS"
  << "REAL" << "REBUILD" << "RECOVER" << "RECURSIVE" << "REDO_BUFFER_SIZE" << "REDOFILE" << "REDUNDANT" << "REF_SYSTEM_ID"
  << "REFERENCES" << "REGEXP" << "RELAY" << "RELAY_LOG_FILE" << "RELAY_LOG_POS" << "RELAY_THREAD" << "RELAYLOG" << "RELEASE"
  << "RELOAD" << "REMOVE" << "RENAME" << "REORGANIZE" << "REPAIR" << "REPEAT" << "REPEATABLE" << "REPLACE"
  << "REPLAY" << "REPLICA" << "REPLICA_POS" << "REPLICAS" << "REPLICATION" << "REQUIRE" << "RESET" << "RESIGNAL"
  << "RESTART" << "RESTORE" << "RESTRICT" << "RESUME" << "RETURN" << "RETURNED_SQLSTATE" << "RETURNING" << "RETURNS"
  << "REUSE" << "REVERSE" << "REVOKE" << "RIGHT" << "RLIKE" << "ROLE" << "ROLLBACK" << "ROLLUP"
  << "ROUTINE" << "ROW" << "ROW_COUNT" << "ROW_FORMAT" << "ROWCOUNT" << "ROWS" << "ROWTYPE" << "RTREE"
  << "SAVEPOINT" << "SCHEDULE" << "SCHEMA" << "SCHEMA_NAME" << "SCHEMAS" << "SECOND" << "SECOND_MICROSECOND" << "SECURITY"
  << "SELECT" << "SENSITIVE" << "SEPARATOR" << "SEQUENCE" << "SERIAL" << "SERIALIZABLE" << "SERVER" << "SESSION"
  << "SET" << "SETVAL" << "SHARE" << "SHOW" << "SHUTDOWN" << "SIGNAL" << "SIGNED" << "SIMPLE"
  << "SLAVE" << "SLAVE_POS" << "SLAVES" << "SLOW" << "SMALLINT" << "SNAPSHOT" << "SOCKET" << "SOFT"
  << "SOME" << "SONAME" << "SOUNDS" << "SOURCE" << "SPATIAL" << "SPECIFIC" << "SQL" << "SQL_BIG_RESULT"
  << "SQL_BUFFER_RESULT" << "SQL_CACHE" << "SQL_CALC_FOUND_ROWS" << "SQL_NO_CACHE" << "SQL_SMALL_RESULT" << "SQL_THREAD" << "SQL_TSI_DAY" << "SQL_TSI_HOUR"
  << "SQL_TSI_MINUTE" << "SQL_TSI_MONTH" << "SQL_TSI_QUARTER" << "SQL_TSI_SECOND" << "SQL_TSI_WEEK" << "SQL_TSI_YEAR" << "SQLEXCEPTION" << "SQLSTATE"
  << "SQLWARNING" << "SSL" << "STAGE" << "START" << "STARTING" << "STARTS" << "STATEMENT" << "STATS_AUTO_RECALC"
  << "STATS_PERSISTENT" << "STATS_SAMPLE_PAGES" << "STATUS" << "STOP" << "STORAGE" << "STORED" << "STRAIGHT_JOIN" << "STRING"
  << "SUBCLASS_ORIGIN" << "SUBJECT" << "SUBPARTITION" << "SUBPARTITIONS" << "SUPER" << "SUSPEND" << "SWAPS" << "SWITCHES"
  << "SYSTEM" << "SYSTEM_TIME" << "TABLE" << "TABLE_CHECKSUM" << "TABLE_NAME" << "TABLES" << "TABLESPACE" << "TEMPORARY"
  << "TEMPTABLE" << "TERMINATED" << "TEXT" << "THAN" << "THEN" << "THREADS" << "TIES" << "TIME"
  << "TIMESTAMP" << "TIMESTAMPADD" << "TIMESTAMPDIFF" << "TINYBLOB" << "TINYINT" << "TINYTEXT" << "TO" << "TRAILING"
  << "TRANSACTION" << "TRANSACTIONAL" << "TRIGGER" << "TRIGGERS" << "TRUE" << "TRUNCATE" << "TYPE" << "TYPES"
  << "UNBOUNDED" << "UNCOMMITTED" << "UNDEFINED" << "UNDO" << "UNDO_BUFFER_SIZE" << "UNDOFILE" << "UNICODE" << "UNINSTALL"
  << "UNION" << "UNIQUE" << "UNKNOWN" << "UNLOCK" << "UNSIGNED" << "UNTIL" << "UPDATE" << "UPGRADE"
  << "USAGE" << "USE" << "USE_FRM" << "USER" << "USER_RESOURCES" << "USING" << "UTC_DATE" << "UTC_TIME"
  << "UTC_TIMESTAMP" << "VALUE" << "VALUES" << "VARBINARY" << "VARCHAR" << "VARCHARACTER" << "VARCHAR2" << "VARIABLES"
  << "VARYING" << "VERSIONING" << "VIA" << "VIEW" << "VIRTUAL" << "VISIBLE" << "WAIT" << "WARNINGS"
  << "WEEK" << "WEIGHT_STRING" << "WHEN" << "WHERE" << "WHILE" << "WINDOW" << "WITH" << "WITHIN"
  << "WITHOUT" << "WORK" << "WRAPPER" << "WRITE" << "X509" << "XA" << "XML" << "XOR"
  << "YEAR" << "YEAR_MONTH" << "ZEROFILL";
}

QStringList StaticFunctions::mariadbHelper()
{

  return QStringList()
      << "ALTER TABLE"
      << "ANALYZE [NO_WRITE_TO_BINLOG | LOCAL] TABLE tbl_name [, tbl_name]"
      << "CHANGE MASTER TO MASTER_BIND = 'interface_name', MASTER_HOST = 'host_name', MASTER_USER = 'user_name', MASTER_PASSWORD = 'password', MASTER_PORT = port_num, MASTER_CONNECT_RETRY = interval, MASTER_HEARTBEAT_PERIOD = interval, MASTER_LOG_FILE = 'master_log_name', MASTER_LOG_POS = master_log_pos, RELAY_LOG_FILE = 'relay_log_name', RELAY_LOG_POS = relay_log_pos, MASTER_SSL = {0|1}, MASTER_SSL_CA = 'ca_file_name', MASTER_SSL_CAPATH = 'ca_directory_name', MASTER_SSL_CERT = 'cert_file_name', MASTER_SSL_KEY = 'key_file_name', MASTER_SSL_CIPHER = 'cipher_list', MASTER_SSL_VERIFY_SERVER_CERT = {0|1}, IGNORE_SERVER_IDS = [server_id [, server_id] ... ]"
      << "CHECK TABLE tbl_name [, tbl_name] {FOR UPGRADE | QUICK | FAST | MEDIUM | EXTENDED | CHANGED}"
      << "CHECKSUM TABLE tbl_name [, tbl_name] [QUICK | EXTENDED]"
      << "CREATE DATABASE `DatabaseName` DEFAULT CHARACTER SET utf8 COLLATE utf8_spanish_ci;"
      << "COMMIT; SET AUTOCOMMIT:= 0; SET FOREIGN_KEY_CHECKS := 0; SET UNIQUE_CHECKS := 0;"
      << "DELETE FROM"
      << "DROP DATABASE IF EXISTS"
      << "DROP DATABASE"
      << "DROP FUNCTION IF EXISTS"
      << "DROP FUNCTION"
      << "DROP TABLE IF EXISTS"
      << "DROP TABLE"
      << "DROP USER"
      << "DROP VIEW IF EXISTS"
      << "DROP VIEW"
      << "EXPORT DATA FOR INSERT"
      << "HELP 'Topic'"
      << "FLUSH DES_KEY_FILE"
      << "FLUSH HOSTS"
      << "FLUSH LOGS"
      << "FLUSH MASTER"
      << "FLUSH PRIVILEGES"
      << "FLUSH QUERY CACHE"
      << "FLUSH SLAVE"
      << "FLUSH STATUS"
      << "FLUSH TABLES tbl_name [, tbl_name]"
      << "FLUSH TABLES WITH READ LOCK"
      << "FLUSH TABLES"
      << "FLUSH USER_RESOURCES"
      << "INSERT INTO"
      << "INSTALL PLUGIN Plugin_Name SONAME 'Plugin_Name.SO'"
      << "IS NULL"
      << "LIMIT 10"
      << "LIMIT 100"
      << "LIMIT 1000"
      << "LOAD DATA LOCAL INFILE 'localfile' INTO TABLE `` FIELDS TERMINATED BY '\t' ENCLOSED BY '' ESCAPED BY '\\';"
      << "LOAD DATA LOCAL INFILE 'localfile' INTO TABLE `` LINES TERMINATED BY '\n' STARTING BY '';"
      << "NOT NULL DEFAULT"
      << "NOT NULL"
      << "ON DUPLICATE KEY UPDATE"
      << "OPTIMIZE [NO_WRITE_TO_BINLOG | LOCAL] TABLE tbl_name [, tbl_name] REPAIR [NO_WRITE_TO_BINLOG | LOCAL] TABLE tbl_name [, tbl_name] [QUICK] [EXTENDED] [USE_FRM]"
      << "ORDER BY"
      << "PROCEDURE ANALYSE()"
      << "PURGE BINARY LOGS BEFORE NOW();"
      << "RESET SLAVE ['connection_name'] [ALL]"
      << "SELECT * FOR UPDATE"
      << "SELECT * FROM"
      << "SELECT * LOCK IN SHARE MODE"
      << "SELECT COUNT(*) FROM"
      << "SELECT SQL_CACHE"
      << "SELECT SQL_NO_CACHE"
      << "SET AUTOCOMMIT:= 0; SET FOREIGN_KEY_CHECKS := 0; SET UNIQUE_CHECKS := 0;"
      << "SET FOREIGN_KEY_CHECKS := 0;"
      << "SET GLOBAL `show_compatibility_56` := ON;"
      << "SET PASSWORD FOR 'user'@'host' = PASSWORD('cleartext password');"
      << "SHOW ALL SLAVES STATUS"
      << "SHOW AUTHORS"
      << "SHOW BINARY LOGS"
      << "SHOW BINLOG EVENTS"
      << "SHOW CHARACTER SET"
      << "SHOW COLLATION"
      << "SHOW COLUMNS"
      << "SHOW CONTRIBUTORS"
      << "SHOW CREATE DATABASE"
      << "SHOW CREATE EVENT"
      << "SHOW CREATE FUNCTION"
      << "SHOW CREATE PROCEDURE"
      << "SHOW CREATE TABLE"
      << "SHOW CREATE TRIGGER"
      << "SHOW CREATE VIEW"
      << "SHOW DATABASES"
      << "SHOW ENGINE engineName STATUS"
      << "SHOW ENGINE INNODB STATUS"
      << "SHOW ENGINE INNODB MUTEX"
      << "SHOW ENGINE {NDB | NDBCLUSTER} STATUS"
      << "SHOW ENGINE PERFORMANCE_SCHEMA STATUS"
      << "SHOW ENGINE"
      << "SHOW ENGINES"
      << "SHOW ERRORS"
      << "SHOW EVENTS"
      << "SHOW FULL TABLES"
      << "SHOW FUNCTION CODE"
      << "SHOW FUNCTION STATUS"
      << "SHOW GRANTS"
      << "SHOW INDEX"
      << "SHOW INDEXES"
      << "SHOW MASTER STATUS"
      << "SHOW OPEN TABLES"
      << "SHOW PLUGINS"
      << "SHOW PRIVILEGES"
      << "SHOW PROCEDURE CODE"
      << "SHOW PROCEDURE STATUS"
      << "SHOW PROCESSLIST"
      << "SHOW PROFILE"
      << "SHOW PROFILES"
      << "SHOW RELAYLOG EVENTS"
      << "SHOW SLAVE HOSTS"
      << "SHOW SLAVE STATUS"
      << "SHOW STATUS"
      << "SHOW TABLE STATUS"
      << "SHOW TABLES"
      << "SHOW TRIGGERS"
      << "SHOW VARIABLES"
      << "SHOW WARNINGS"
      << "START ALL SLAVES;"
      << "START SLAVE IO_THREAD;"
      << "START SLAVE SQL_THREAD;"
      << "START SLAVE;"
      << "STOP ALL SLAVES;"
      << "STOP SLAVE IO_THREAD;"
      << "STOP SLAVE SQL_THREAD;"
      << "STOP SLAVE; SET GLOBAL SQL_SLAVE_SKIP_COUNTER = 1; START SLAVE;"
      << "STOP SLAVE;"
      << "TRIM COLUMNS IN"
      << "TRUNCATE TABLE"
      << "UNIQUE KEY";
}

QStringList StaticFunctions::mariadbFunctions()
{
  QStringList list = QStringList();
  QHashIterator<QString, QString> hashIterator(mariadbFunctionsComplete());
  while (hashIterator.hasNext()) {
    hashIterator.next();
    list.append(hashIterator.key());
  }
  return list;
}

QStringList StaticFunctions::mariadbDatatypes()
{
  return QStringList() // Numeric Types
                         << "BIT" << "TINYINT" << "BOOL" << "BOOLEAN" << "SMALLINT" << "MEDIUMINT"
                         << "INT" << "INTEGER" << "BIGINT" << "SERIAL" << "DECIMAL" << "DEC"
                         << "NUMERIC" << "FIXED" << "FLOAT" << "DOUBLE" << "DOUBLE PRECISION" << "REAL"

                         // Date and Time Types
                         << "DATE" << "DATETIME" << "TIMESTAMP" << "TIME" << "YEAR"

                         // String Types
                         << "CHAR" << "VARCHAR" << "BINARY" << "VARBINARY" << "TINYBLOB" << "BLOB"
                         << "MEDIUMBLOB" << "LONGBLOB" << "TINYTEXT" << "TEXT" << "MEDIUMTEXT" << "LONGTEXT"
                         << "ENUM" << "SET" << "JSON"

                         // Spatial Types (GIS)
                         << "GEOMETRY" << "POINT" << "LINESTRING" << "POLYGON" << "MULTIPOINT"
                         << "MULTILINESTRING" << "MULTIPOLYGON" << "GEOMETRYCOLLECTION"

                         // MariaDB Specific / Special Types
                         << "UUID" << "INET4" << "INET6" << "ROW";
}

QStringList StaticFunctions::mariadbCharsets()
{
  // Must be updated once in a while
  // SELECT DISTINCT `CHARACTER_SET_NAME` FROM `information_schema`.`CHARACTER_SETS` ORDER BY `CHARACTER_SET_NAME`;
  return QStringList() << "armscii8" << "ascii" << "big5" << "binary"
                       << "cp1250" << "cp1251" << "cp1256" << "cp1257"
                       << "cp850" << "cp852" << "cp866" << "cp932"
                       << "dec8" << "eucjpms" << "euckr" << "gb2312"
                       << "gbk" << "geostd8" << "greek" << "hebrew"
                       << "hp8" << "keybcs2" << "koi8r" << "koi8u"
                       << "latin1" << "latin2" << "latin5" << "latin7"
                       << "macce" << "macroman" << "sjis" << "swe7"
                       << "tis620" << "ucs2" << "ujis" << "utf16"
                       << "utf16le" << "utf32" << "utf8mb3" << "utf8mb4";
}

QStringList StaticFunctions::mariadbCharsetsCollations()
{
  // Must be updated once in a while
  // SELECT DISTINCT `COLLATION_NAME` FROM `information_schema`.`COLLATION_CHARACTER_SET_APPLICABILITY` ORDER BY `COLLATION_NAME`;
  return QStringList() << "armscii8_bin" << "armscii8_general_ci" << "armscii8_general_nopad_ci" << "armscii8_nopad_bin"
                       << "ascii_bin" << "ascii_general_ci" << "ascii_general_nopad_ci" << "ascii_nopad_bin"
                       << "big5_bin" << "big5_chinese_ci" << "big5_chinese_nopad_ci" << "big5_nopad_bin" << "binary"
                       << "cp1250_bin" << "cp1250_croatian_ci" << "cp1250_czech_cs" << "cp1250_general_ci"
                       << "cp1250_general_nopad_ci" << "cp1250_nopad_bin" << "cp1250_polish_ci"
                       << "cp1251_bin" << "cp1251_bulgarian_ci" << "cp1251_general_ci" << "cp1251_general_cs"
                       << "cp1251_general_nopad_ci" << "cp1251_nopad_bin" << "cp1251_ukrainian_ci"
                       << "cp1256_bin" << "cp1256_general_ci" << "cp1256_general_nopad_ci" << "cp1256_nopad_bin"
                       << "cp1257_bin" << "cp1257_general_ci" << "cp1257_general_nopad_ci" << "cp1257_lithuanian_ci" << "cp1257_nopad_bin"
                       << "cp850_bin" << "cp850_general_ci" << "cp850_general_nopad_ci" << "cp850_nopad_bin"
                       << "cp852_bin" << "cp852_general_ci" << "cp852_general_nopad_ci" << "cp852_nopad_bin"
                       << "cp866_bin" << "cp866_general_ci" << "cp866_general_nopad_ci" << "cp866_nopad_bin"
                       << "cp932_bin" << "cp932_japanese_ci" << "cp932_japanese_nopad_ci" << "cp932_nopad_bin"
                       << "dec8_bin" << "dec8_nopad_bin" << "dec8_swedish_ci" << "dec8_swedish_nopad_ci"
                       << "eucjpms_bin" << "eucjpms_japanese_ci" << "eucjpms_japanese_nopad_ci" << "eucjpms_nopad_bin"
                       << "euckr_bin" << "euckr_korean_ci" << "euckr_korean_nopad_ci" << "euckr_nopad_bin"
                       << "gb2312_bin" << "gb2312_chinese_ci" << "gb2312_chinese_nopad_ci" << "gb2312_nopad_bin"
                       << "gbk_bin" << "gbk_chinese_ci" << "gbk_chinese_nopad_ci" << "gbk_nopad_bin"
                       << "geostd8_bin" << "geostd8_general_ci" << "geostd8_general_nopad_ci" << "geostd8_nopad_bin"
                       << "greek_bin" << "greek_general_ci" << "greek_general_nopad_ci" << "greek_nopad_bin"
                       << "hebrew_bin" << "hebrew_general_ci" << "hebrew_general_nopad_ci" << "hebrew_nopad_bin"
                       << "hp8_bin" << "hp8_english_ci" << "hp8_english_nopad_ci" << "hp8_nopad_bin"
                       << "keybcs2_bin" << "keybcs2_general_ci" << "keybcs2_general_nopad_ci" << "keybcs2_nopad_bin"
                       << "koi8r_bin" << "koi8r_general_ci" << "koi8r_general_nopad_ci" << "koi8r_nopad_bin"
                       << "koi8u_bin" << "koi8u_general_ci" << "koi8u_general_nopad_ci" << "koi8u_nopad_bin"
                       << "latin1_bin" << "latin1_danish_ci" << "latin1_general_ci" << "latin1_general_cs"
                       << "latin1_german1_ci" << "latin1_german2_ci" << "latin1_nopad_bin" << "latin1_spanish_ci"
                       << "latin1_swedish_ci" << "latin1_swedish_nopad_ci"
                       << "latin2_bin" << "latin2_croatian_ci" << "latin2_czech_cs" << "latin2_general_ci"
                       << "latin2_general_nopad_ci" << "latin2_hungarian_ci" << "latin2_nopad_bin"
                       << "latin5_bin" << "latin5_nopad_bin" << "latin5_turkish_ci" << "latin5_turkish_nopad_ci"
                       << "latin7_bin" << "latin7_estonian_cs" << "latin7_general_ci" << "latin7_general_cs"
                       << "latin7_general_nopad_ci" << "latin7_nopad_bin"
                       << "macce_bin" << "macce_general_ci" << "macce_general_nopad_ci" << "macce_nopad_bin"
                       << "macroman_bin" << "macroman_general_ci" << "macroman_general_nopad_ci" << "macroman_nopad_bin"
                       << "sjis_bin" << "sjis_japanese_ci" << "sjis_japanese_nopad_ci" << "sjis_nopad_bin"
                       << "swe7_bin" << "swe7_nopad_bin" << "swe7_swedish_ci" << "swe7_swedish_nopad_ci"
                       << "tis620_bin" << "tis620_nopad_bin" << "tis620_thai_ci" << "tis620_thai_nopad_ci"
                       << "uca1400_ai_ci" << "uca1400_ai_cs" << "uca1400_as_ci" << "uca1400_as_cs"
                       << "uca1400_croatian_ai_ci" << "uca1400_croatian_ai_cs" << "uca1400_croatian_as_ci" << "uca1400_croatian_as_cs"
                       << "uca1400_croatian_nopad_ai_ci" << "uca1400_croatian_nopad_ai_cs" << "uca1400_croatian_nopad_as_ci" << "uca1400_croatian_nopad_as_cs"
                       << "uca1400_czech_ai_ci" << "uca1400_czech_ai_cs" << "uca1400_czech_as_ci" << "uca1400_czech_as_cs"
                       << "uca1400_czech_nopad_ai_ci" << "uca1400_czech_nopad_ai_cs" << "uca1400_czech_nopad_as_ci" << "uca1400_czech_nopad_as_cs"
                       << "uca1400_danish_ai_ci" << "uca1400_danish_ai_cs" << "uca1400_danish_as_ci" << "uca1400_danish_as_cs"
                       << "uca1400_danish_nopad_ai_ci" << "uca1400_danish_nopad_ai_cs" << "uca1400_danish_nopad_as_ci" << "uca1400_danish_nopad_as_cs"
                       << "uca1400_esperanto_ai_ci" << "uca1400_esperanto_ai_cs" << "uca1400_esperanto_as_ci" << "uca1400_esperanto_as_cs"
                       << "uca1400_esperanto_nopad_ai_ci" << "uca1400_esperanto_nopad_ai_cs" << "uca1400_esperanto_nopad_as_ci" << "uca1400_esperanto_nopad_as_cs"
                       << "uca1400_estonian_ai_ci" << "uca1400_estonian_ai_cs" << "uca1400_estonian_as_ci" << "uca1400_estonian_as_cs"
                       << "uca1400_estonian_nopad_ai_ci" << "uca1400_estonian_nopad_ai_cs" << "uca1400_estonian_nopad_as_ci" << "uca1400_estonian_nopad_as_cs"
                       << "uca1400_german2_ai_ci" << "uca1400_german2_ai_cs" << "uca1400_german2_as_ci" << "uca1400_german2_as_cs"
                       << "uca1400_german2_nopad_ai_ci" << "uca1400_german2_nopad_ai_cs" << "uca1400_german2_nopad_as_ci" << "uca1400_german2_nopad_as_cs"
                       << "uca1400_hungarian_ai_ci" << "uca1400_hungarian_ai_cs" << "uca1400_hungarian_as_ci" << "uca1400_hungarian_as_cs"
                       << "uca1400_hungarian_nopad_ai_ci" << "uca1400_hungarian_nopad_ai_cs" << "uca1400_hungarian_nopad_as_ci" << "uca1400_hungarian_nopad_as_cs"
                       << "uca1400_icelandic_ai_ci" << "uca1400_icelandic_ai_cs" << "uca1400_icelandic_as_ci" << "uca1400_icelandic_as_cs"
                       << "uca1400_icelandic_nopad_ai_ci" << "uca1400_icelandic_nopad_ai_cs" << "uca1400_icelandic_nopad_as_ci" << "uca1400_icelandic_nopad_as_cs"
                       << "uca1400_latvian_ai_ci" << "uca1400_latvian_ai_cs" << "uca1400_latvian_as_ci" << "uca1400_latvian_as_cs"
                       << "uca1400_latvian_nopad_ai_ci" << "uca1400_latvian_nopad_ai_cs" << "uca1400_latvian_nopad_as_ci" << "uca1400_latvian_nopad_as_cs"
                       << "uca1400_lithuanian_ai_ci" << "uca1400_lithuanian_ai_cs" << "uca1400_lithuanian_as_ci" << "uca1400_lithuanian_as_cs"
                       << "uca1400_lithuanian_nopad_ai_ci" << "uca1400_lithuanian_nopad_ai_cs" << "uca1400_lithuanian_nopad_as_ci" << "uca1400_lithuanian_nopad_as_cs"
                       << "uca1400_nopad_ai_ci" << "uca1400_nopad_ai_cs" << "uca1400_nopad_as_ci" << "uca1400_nopad_as_cs"
                       << "uca1400_persian_ai_ci" << "uca1400_persian_ai_cs" << "uca1400_persian_as_ci" << "uca1400_persian_as_cs"
                       << "uca1400_persian_nopad_ai_ci" << "uca1400_persian_nopad_ai_cs" << "uca1400_persian_nopad_as_ci" << "uca1400_persian_nopad_as_cs"
                       << "uca1400_polish_ai_ci" << "uca1400_polish_ai_cs" << "uca1400_polish_as_ci" << "uca1400_polish_as_cs"
                       << "uca1400_polish_nopad_ai_ci" << "uca1400_polish_nopad_ai_cs" << "uca1400_polish_nopad_as_ci" << "uca1400_polish_nopad_as_cs"
                       << "uca1400_romanian_ai_ci" << "uca1400_romanian_ai_cs" << "uca1400_romanian_as_ci" << "uca1400_romanian_as_cs"
                       << "uca1400_romanian_nopad_ai_ci" << "uca1400_romanian_nopad_ai_cs" << "uca1400_romanian_nopad_as_ci" << "uca1400_romanian_nopad_as_cs"
                       << "uca1400_roman_ai_ci" << "uca1400_roman_ai_cs" << "uca1400_roman_as_ci" << "uca1400_roman_as_cs"
                       << "uca1400_roman_nopad_ai_ci" << "uca1400_roman_nopad_ai_cs" << "uca1400_roman_nopad_as_ci" << "uca1400_roman_nopad_as_cs"
                       << "uca1400_sinhala_ai_ci" << "uca1400_sinhala_ai_cs" << "uca1400_sinhala_as_ci" << "uca1400_sinhala_as_cs"
                       << "uca1400_sinhala_nopad_ai_ci" << "uca1400_sinhala_nopad_ai_cs" << "uca1400_sinhala_nopad_as_ci" << "uca1400_sinhala_nopad_as_cs"
                       << "uca1400_slovak_ai_ci" << "uca1400_slovak_ai_cs" << "uca1400_slovak_as_ci" << "uca1400_slovak_as_cs"
                       << "uca1400_slovak_nopad_ai_ci" << "uca1400_slovak_nopad_ai_cs" << "uca1400_slovak_nopad_as_ci" << "uca1400_slovak_nopad_as_cs"
                       << "uca1400_slovenian_ai_ci" << "uca1400_slovenian_ai_cs" << "uca1400_slovenian_as_ci" << "uca1400_slovenian_as_cs"
                       << "uca1400_slovenian_nopad_ai_ci" << "uca1400_slovenian_nopad_ai_cs" << "uca1400_slovenian_nopad_as_ci" << "uca1400_slovenian_nopad_as_cs"
                       << "uca1400_spanish2_ai_ci" << "uca1400_spanish2_ai_cs" << "uca1400_spanish2_as_ci" << "uca1400_spanish2_as_cs"
                       << "uca1400_spanish2_nopad_ai_ci" << "uca1400_spanish2_nopad_ai_cs" << "uca1400_spanish2_nopad_as_ci" << "uca1400_spanish2_nopad_as_cs"
                       << "uca1400_spanish_ai_ci" << "uca1400_spanish_ai_cs" << "uca1400_spanish_as_ci" << "uca1400_spanish_as_cs"
                       << "uca1400_spanish_nopad_ai_ci" << "uca1400_spanish_nopad_ai_cs" << "uca1400_spanish_nopad_as_ci" << "uca1400_spanish_nopad_as_cs"
                       << "uca1400_swedish_ai_ci" << "uca1400_swedish_ai_cs" << "uca1400_swedish_as_ci" << "uca1400_swedish_as_cs"
                       << "uca1400_swedish_nopad_ai_ci" << "uca1400_swedish_nopad_ai_cs" << "uca1400_swedish_nopad_as_ci" << "uca1400_swedish_nopad_as_cs"
                       << "uca1400_turkish_ai_ci" << "uca1400_turkish_ai_cs" << "uca1400_turkish_as_ci" << "uca1400_turkish_as_cs"
                       << "uca1400_turkish_nopad_ai_ci" << "uca1400_turkish_nopad_ai_cs" << "uca1400_turkish_nopad_as_ci" << "uca1400_turkish_nopad_as_cs"
                       << "uca1400_vietnamese_ai_ci" << "uca1400_vietnamese_ai_cs" << "uca1400_vietnamese_as_ci" << "uca1400_vietnamese_as_cs"
                       << "uca1400_vietnamese_nopad_ai_ci" << "uca1400_vietnamese_nopad_ai_cs" << "uca1400_vietnamese_nopad_as_ci" << "uca1400_vietnamese_nopad_as_cs"
                       << "ucs2_bin" << "ucs2_croatian_ci" << "ucs2_croatian_mysql561_ci" << "ucs2_czech_ci"
                       << "ucs2_danish_ci" << "ucs2_esperanto_ci" << "ucs2_estonian_ci" << "ucs2_general_ci"
                       << "ucs2_general_mysql500_ci" << "ucs2_general_nopad_ci" << "ucs2_german2_ci" << "ucs2_hungarian_ci"
                       << "ucs2_icelandic_ci" << "ucs2_latvian_ci" << "ucs2_lithuanian_ci" << "ucs2_myanmar_ci"
                       << "ucs2_nopad_bin" << "ucs2_persian_ci" << "ucs2_polish_ci" << "ucs2_romanian_ci"
                       << "ucs2_roman_ci" << "ucs2_sinhala_ci" << "ucs2_slovak_ci" << "ucs2_slovenian_ci"
                       << "ucs2_spanish2_ci" << "ucs2_spanish_ci" << "ucs2_swedish_ci" << "ucs2_thai_520_w2"
                       << "ucs2_turkish_ci" << "ucs2_unicode_520_ci" << "ucs2_unicode_520_nopad_ci" << "ucs2_unicode_ci"
                       << "ucs2_unicode_nopad_ci" << "ucs2_vietnamese_ci"
                       << "ujis_bin" << "ujis_japanese_ci" << "ujis_japanese_nopad_ci" << "ujis_nopad_bin"
                       << "utf16le_bin" << "utf16le_general_ci" << "utf16le_general_nopad_ci" << "utf16le_nopad_bin"
                       << "utf16_bin" << "utf16_croatian_ci" << "utf16_croatian_mysql561_ci" << "utf16_czech_ci"
                       << "utf16_danish_ci" << "utf16_esperanto_ci" << "utf16_estonian_ci" << "utf16_general_ci"
                       << "utf16_general_nopad_ci" << "utf16_german2_ci" << "utf16_hungarian_ci" << "utf16_icelandic_ci"
                       << "utf16_latvian_ci" << "utf16_lithuanian_ci" << "utf16_myanmar_ci" << "utf16_nopad_bin"
                       << "utf16_persian_ci" << "utf16_polish_ci" << "utf16_romanian_ci" << "utf16_roman_ci"
                       << "utf16_sinhala_ci" << "utf16_slovak_ci" << "utf16_slovenian_ci" << "utf16_spanish2_ci"
                       << "utf16_spanish_ci" << "utf16_swedish_ci" << "utf16_thai_520_w2" << "utf16_turkish_ci"
                       << "utf16_unicode_520_ci" << "utf16_unicode_520_nopad_ci" << "utf16_unicode_ci"
                       << "utf16_unicode_nopad_ci" << "utf16_vietnamese_ci"
                       << "utf32_bin" << "utf32_croatian_ci" << "utf32_croatian_mysql561_ci" << "utf32_czech_ci"
                       << "utf32_danish_ci" << "utf32_esperanto_ci" << "utf32_estonian_ci" << "utf32_general_ci"
                       << "utf32_general_nopad_ci" << "utf32_german2_ci" << "utf32_hungarian_ci" << "utf32_icelandic_ci"
                       << "utf32_latvian_ci" << "utf32_lithuanian_ci" << "utf32_myanmar_ci" << "utf32_nopad_bin"
                       << "utf32_persian_ci" << "utf32_polish_ci" << "utf32_romanian_ci" << "utf32_roman_ci"
                       << "utf32_sinhala_ci" << "utf32_slovak_ci" << "utf32_slovenian_ci" << "utf32_spanish2_ci"
                       << "utf32_spanish_ci" << "utf32_swedish_ci" << "utf32_thai_520_w2" << "utf32_turkish_ci"
                       << "utf32_unicode_520_ci" << "utf32_unicode_520_nopad_ci" << "utf32_unicode_ci"
                       << "utf32_unicode_nopad_ci" << "utf32_vietnamese_ci"
                       << "utf8mb3_bin" << "utf8mb3_croatian_ci" << "utf8mb3_croatian_mysql561_ci" << "utf8mb3_czech_ci"
                       << "utf8mb3_danish_ci" << "utf8mb3_esperanto_ci" << "utf8mb3_estonian_ci" << "utf8mb3_general_ci"
                       << "utf8mb3_general_mysql500_ci" << "utf8mb3_general_nopad_ci" << "utf8mb3_german2_ci"
                       << "utf8mb3_hungarian_ci" << "utf8mb3_icelandic_ci" << "utf8mb3_latvian_ci" << "utf8mb3_lithuanian_ci"
                       << "utf8mb3_myanmar_ci" << "utf8mb3_nopad_bin" << "utf8mb3_persian_ci" << "utf8mb3_polish_ci"
                       << "utf8mb3_romanian_ci" << "utf8mb3_roman_ci" << "utf8mb3_sinhala_ci" << "utf8mb3_slovak_ci"
                       << "utf8mb3_slovenian_ci" << "utf8mb3_spanish2_ci" << "utf8mb3_spanish_ci" << "utf8mb3_swedish_ci"
                       << "utf8mb3_thai_520_w2" << "utf8mb3_turkish_ci" << "utf8mb3_unicode_520_ci"
                       << "utf8mb3_unicode_520_nopad_ci" << "utf8mb3_unicode_ci" << "utf8mb3_unicode_nopad_ci" << "utf8mb3_vietnamese_ci"
                       << "utf8mb4_bin" << "utf8mb4_croatian_ci" << "utf8mb4_croatian_mysql561_ci" << "utf8mb4_czech_ci"
                       << "utf8mb4_danish_ci" << "utf8mb4_esperanto_ci" << "utf8mb4_estonian_ci" << "utf8mb4_general_ci"
                       << "utf8mb4_general_nopad_ci" << "utf8mb4_german2_ci" << "utf8mb4_hungarian_ci"
                       << "utf8mb4_icelandic_ci" << "utf8mb4_latvian_ci" << "utf8mb4_lithuanian_ci" << "utf8mb4_myanmar_ci"
                       << "utf8mb4_nopad_bin" << "utf8mb4_persian_ci" << "utf8mb4_polish_ci" << "utf8mb4_romanian_ci"
                       << "utf8mb4_roman_ci" << "utf8mb4_sinhala_ci" << "utf8mb4_slovak_ci" << "utf8mb4_slovenian_ci"
                       << "utf8mb4_spanish2_ci" << "utf8mb4_spanish_ci" << "utf8mb4_swedish_ci" << "utf8mb4_thai_520_w2"
                       << "utf8mb4_turkish_ci" << "utf8mb4_unicode_520_ci" << "utf8mb4_unicode_520_nopad_ci"
                       << "utf8mb4_unicode_ci" << "utf8mb4_unicode_nopad_ci" << "utf8mb4_vietnamese_ci";
}

QStringList StaticFunctions::mariadbCodeErrorsNotCommaBothSides()
{
  return QStringList() << "AND"
                       << "FROM"
                          ;
}

QStringList StaticFunctions::mariadbCodeErrorsNotCommaOnLeft()
{
  return QStringList() << "SELECT"
                          ;
}

QString StaticFunctions::bytesConvertor(QString bytes)
{
  unsigned long long num = bytes.toULongLong();
  if (num <= 1024) {
    return QString("%1 B").arg(num);
  } else {
    num /= 1024;
    if (num <= 1024) {
      return QString("%1 KB").arg(num);
    } else {
      num /= 1024;
      if (num <= 1024) {
        return QString("%1 MB").arg(num);
      } else {
        num /= 1024;
        if (num <= 1024) {
          return QString("%1 GB").arg(num);
        } else {
          num /= 1024;
          if (num <= 1024) {
            return QString("%1 TB").arg(num);
          } else {
            num /= 1024;
            if (num <= 1024) {
              return QString("%1 PB").arg(num);
            } else {
              num /= 1024;
              if (num <= 1024)
                return QString("%1 EB").arg(num);
            }
          }
        }
      }
    }
  }
  return "Not known";
}

QString StaticFunctions::bytesConvertor(qulonglong bytes)
{
  return StaticFunctions::bytesConvertor(QString("%1").arg(bytes));
}

//QMap<QString, QVariant> StaticFunctions::explodeConnectionString(QString connectionName)
//{
//  QMap<QString, QVariant> connectionParameters;
////  QSettings settings;
//////  QStringList params;
////  int startIndex = 0;
////  int endIndex = 0;
////  QString defaultValue = "MariaDB:root@localhost:3306/mysql Count:0 Collation:utf8mb4|utf8mb4_general_ci Password:*99";
////  QString connection(settings.value("ServerConnections/" + connectionName, defaultValue).toString());

////  endIndex = connection.indexOf(":", startIndex);
////  //0 - Connection type
////  connectionParameters.insert("ConnectionType", connection.mid(startIndex, endIndex));
//////  params.append(connection.mid(startIndex, endIndex));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf("@", startIndex) - startIndex;
////  //1 - User
//////  params.append(connection.mid(startIndex, endIndex));
////  connectionParameters.insert("User", connection.mid(startIndex, endIndex));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf(":", startIndex) - startIndex;
////  //2 - Host
//////  params.append(connection.mid(startIndex, endIndex));
////  connectionParameters.insert("Host", connection.mid(startIndex, endIndex));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf("/", startIndex) - startIndex;
////  //3 - Port
////  connectionParameters.insert("Port", connection.mid(startIndex, endIndex));
//////  params.append(connection.mid(startIndex, endIndex));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf(" ", startIndex) - startIndex;
////  //4 - Database
//////  params.append(connection.mid(startIndex, endIndex));
////  connectionParameters.insert("Database", connection.mid(startIndex, endIndex));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf(" ", startIndex) - startIndex;
////  //5 - Conexion count
////  connectionParameters.insert("ConnectionsCount", connection.mid(startIndex, endIndex).split(":").at(1));
//////  params.append(connection.mid(startIndex, endIndex).split(":").at(1));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf(" ", startIndex) - startIndex;
////  //6 - Collation
////  connectionParameters.insert("Collation", connection.mid(startIndex, endIndex).split(":").at(1));
//////  params.append(connection.mid(startIndex, endIndex).split(":").at(1));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf(" ", startIndex) - startIndex;
////  //7 - UseSSL
////  connectionParameters.insert("UseSSL", connection.mid(startIndex, endIndex).split(":").at(1));
//////  params.append(connection.mid(startIndex, endIndex).split(":").at(1));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf(" ", startIndex) - startIndex;
////  //8 - KeyFile
////  connectionParameters.insert("KeyFile", connection.mid(startIndex, endIndex).split(":").at(1));
//////  params.append(connection.mid(startIndex, endIndex).split(":").at(1));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf(" ", startIndex) - startIndex;
////  //9 - CertFile
////  connectionParameters.insert("CertFile", connection.mid(startIndex, endIndex).split(":").at(1));
//////  params.append(connection.mid(startIndex, endIndex).split(":").at(1));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf(" ", startIndex) - startIndex;
////  //10 - Password
////  connectionParameters.insert("Password", connection.mid(startIndex, endIndex).mid(9));
//////  params.append(connection.mid(startIndex, endIndex).mid(9));
////  startIndex += ++endIndex;
////  endIndex = connection.indexOf(" ", startIndex) - startIndex;
////  Q_UNUSED(endIndex);
////  //11 - Connection name
////  if (connection != defaultValue) {
////    connectionParameters.insert("Name", connectionName);
//////    params.append(connectionName);
////  } else {
////    connectionParameters.insert("Name", "DefaultConnection");
//////    params.append("DefaultConnection");
////  }

//  qApp->setProperty("ConnectionName", connectionName);
//  return connectionParameters;
//}

QString StaticFunctions::randomString(unsigned int length)
{
  QString string;
  QString possibleCharacters("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789 ;.,+¿*-/");
  QRandomGenerator random(QTime::currentTime().msec());
  for(unsigned int counter = 0; counter < length; ++counter)
    string.append(possibleCharacters.at(random.bounded(32767) % possibleCharacters.length()));
  return string;
}

QString StaticFunctions::randomName(unsigned int length)
{
  QString string;
  QString possibleCharacters("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789");
  QRandomGenerator random(QTime::currentTime().msec());
  for(unsigned int counter = 0; counter < length; ++counter)
    string.append(possibleCharacters.at(random.bounded(32767) % possibleCharacters.length()));
  return string;
}

QStringList StaticFunctions::currentEditorTypeKeywords(const EditorTypes::EditorType editorType)
{
  switch(editorType) {
  case EditorTypes::SQLQuery:
    return mariadbKeywords();
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);
  }
  return QStringList();
}

QLocale StaticFunctions::currentLocale()
{
  QStringList languageData = qApp->property("ApplicationLanguage").toString().split("/");
  QLocale::Language language(QLocale::English);
  if (languageData.at(0) == "es")
    language = QLocale::Spanish;
  QLocale::Country country(QLocale::UnitedStates);
  if (languageData.at(1) == "CR")
    country = QLocale::CostaRica;
  return QLocale(language, country);
}

QString StaticFunctions::serverInformationQuery()
{
  return "SELECT '" + tr("Uptime in days") + "' AS `" + tr("Variable") + "`,  LPAD(FORMAT(`VARIABLE_VALUE` / 60 / 60 / 24, 2), 15, ' ') AS `" + tr("Value") + "`, '" + tr("Number of days the server has been running.") + "' AS `" + tr("Description") + "`, 'UPTIME' AS `" + tr("Variable mame") + "` FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'UPTIME'"
      " UNION"
      " SELECT '" + tr("Aborted clients") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of aborted client connections.") + "', 'ABORTED_CLIENTS' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'ABORTED_CLIENTS'"
      " UNION"
      " SELECT '" + tr("Aborted clients per day") + "', LPAD(FORMAT("
      "  (SELECT `VARIABLE_VALUE` FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'ABORTED_CLIENTS')"
      "   / (SELECT `VARIABLE_VALUE` / 60 /60 / 24 AS `Value` FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'UPTIME'), 2), 15, ' ')"
      " , '" + tr("Number of aborted client connections per day.") + "', '-'"
      " UNION"
      " SELECT '" + tr("Aborted connections") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of failed server connection attempts.") + "', 'ABORTED_CONNECTS' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'ABORTED_CONNECTS'"
      " UNION"
      " SELECT '" + tr("Aborted connections per day") + "', LPAD(FORMAT("
      "  (SELECT `VARIABLE_VALUE` FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'ABORTED_CONNECTS')"
      "   / (SELECT `VARIABLE_VALUE` / 60 /60 / 24 AS `Value` FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'UPTIME'), 2), 15, ' ')"
      " , '" + tr("Number of failed server connection attempts per day.") + "', '-'"
      " UNION"
      " SELECT '" + tr("Executed rollbacks") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of ROLLBACK commands executed.") + "', 'COM_ROLLBACK' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'COM_ROLLBACK'"
      " UNION"
      " SELECT '" + tr("Rollbacks per second") + "', LPAD(FORMAT("
      "   ((SELECT `VARIABLE_VALUE` FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'COM_ROLLBACK')"
      "   / (SELECT `VARIABLE_VALUE` FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'UPTIME')), 2), 15, ' ')"
      " , '" + tr("Number of ROLLBACK commands executed every second.") + "', '-'"
      " UNION"
      " SELECT '" + tr("Executed queries") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of statements executed by the server.") + "', 'QUERIES' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'QUERIES'"
      " UNION"
      " SELECT '" + tr("Temporary disk tables created") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of on-disk temporary tables created.") + "', 'CREATED_TMP_DISK_TABLES' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'CREATED_TMP_DISK_TABLES'"
      " UNION"
      " SELECT '" + tr("Free cache memory") + "', LPAD(FORMAT(`VARIABLE_VALUE` / 1024, 0), 15, ' '), '" + tr("Amount of free query cache memory.") + "', 'QCACHE_FREE_MEMORY' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'QCACHE_FREE_MEMORY'"
      " UNION"
      " SELECT '" + tr("Joins with full table scan") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of joins which did not use an index. If not zero, you may need to check table indexes.") + "', 'SELECT_FULL_JOIN' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'SELECT_FULL_JOIN'"
      " UNION"
      " SELECT '" + tr("Log slow queries") + "', LPAD(`VARIABLE_VALUE`, 15, ' '), '" + tr("Is the slow queries logging enabled?") + "', 'LOG_SLOW_QUERIES' FROM `information_schema`.`GLOBAL_VARIABLES` WHERE `VARIABLE_NAME` = 'SLOW_QUERY_LOG'"
      " UNION"
      " SELECT '" + tr("Slow queries time in seconds") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("The number of seconds that determinate a slow query.") + "', 'LONG_QUERY_TIME' FROM `information_schema`.`GLOBAL_VARIABLES` WHERE `VARIABLE_NAME` = 'LONG_QUERY_TIME'"
      " UNION"
      " SELECT '" + tr("Count of slow queries") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of queries which took longer than long_query_time to run.") + "', 'SLOW_QUERIES' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'SLOW_QUERIES'"
      " UNION"
      " SELECT '" + tr("Idle connections") + "', LPAD(COUNT(*), 15, ' '), '" + tr("The number of connections that has been idle for more than 30 seconds.") + "', '-' FROM `information_schema`.`PROCESSLIST` WHERE `TIME` >  30 AND `COMMAND` NOT IN ('Daemon', 'Binlog Dump')"
      " UNION"
      " SELECT '" + tr("Active connections") + "', LPAD(COUNT(*), 15, ' '), '" + tr("The number of connections that has been idle for less than 30 seconds.") + "', '-' FROM `information_schema`.`PROCESSLIST` WHERE `TIME` <  30 AND `COMMAND` NOT IN ('Daemon', 'Binlog Dump')"
      " UNION"
      " SELECT '" + tr("Total connections") + "', LPAD(COUNT(*), 15, ' '), '" + tr("The number of active connections on the server.") + "', '-' FROM `information_schema`.`PROCESSLIST` WHERE `COMMAND` NOT IN ('Daemon', 'Binlog Dump')"
      " UNION"
      " SELECT '" + tr("Max connections aviable") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("The maximum number of simultaneous client connections.") + "', 'MAX_CONNECTIONS' FROM `information_schema`.`GLOBAL_VARIABLES` WHERE `VARIABLE_NAME` = 'MAX_CONNECTIONS'"
      " UNION"
      " SELECT '" + tr("Used connections") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Max number of connections ever open at the same time.") + "', 'MAX_USED_CONNECTIONS' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'MAX_USED_CONNECTIONS'"
      " UNION"
      " SELECT '" + tr("DELETE commands executed") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of DELETE commands executed.") + "', 'COM_DELETE' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'COM_DELETE'"
      " UNION"
      " SELECT '" + tr("INSERT commands executed") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of INSERT commands executed.") + "', 'COM_INSERT' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'COM_INSERT'"
      " UNION"
      " SELECT '" + tr("UPDATE commands executed") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of UPDATE commands executed.") + "', 'COM_UPDATE' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'COM_UPDATE'"
      " UNION"
      " SELECT '" + tr("SELECT commands executed") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of SELECT commands executed.") + "', 'COM_SELECT' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'COM_SELECT'"
      " UNION"
      " SELECT '" + tr("Requests of the first index row") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of requests to read the first row from an index. A high value indicates many full index scans.") + "', 'HANDLER_READ_FIRST' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'HANDLER_READ_FIRST'"
      " UNION"
      " SELECT '" + tr("Rate of SELECTs per full index scans") + "', LPAD(FORMAT((SELECT `VARIABLE_VALUE` * 100 FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'HANDLER_READ_FIRST') / (SELECT `VARIABLE_VALUE` FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'COM_SELECT'), 0), 15, ' '), '" + tr("Rate of SELECTs per full index scans. A value higher than 100 means you do more table scan than SELECTs.") + "', '-'"
      " UNION"
      " SELECT '" + tr("Read requests based on an index value") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Number of row read requests based on an index value. A high value indicates indexes are regularly being used.") + "', 'HANDLER_READ_KEY' FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'HANDLER_READ_KEY'"
      " UNION"
      " SELECT '" + tr("Wait timeout") + "', LPAD(FORMAT(`VARIABLE_VALUE`, 0), 15, ' '), '" + tr("Time in seconds that the server waits for a connection to become active before closing it.") + "', 'WAIT_TIMEOUT' FROM `information_schema`.`GLOBAL_VARIABLES` WHERE `VARIABLE_NAME` = 'WAIT_TIMEOUT'"
      ;
}

QString StaticFunctions::slowQueriesQuery()
{
  return "SELECT DATE(`start_time`) AS `" + tr("Date") + "`, DAYNAME(`start_time`) AS `" + tr("Day") + "`, LPAD(FORMAT(COUNT(*), 0), 13, ' ') AS `" + tr("Total queries") + "`"
      ", LPAD(FORMAT(AVG(`query_time`), 2), 20,  ' ') AS `" + tr("Average (in seconds)") + "` FROM `mysql`.`slow_log` WHERE `start_time` >= (SELECT FROM_UNIXTIME("
      " (SELECT UNIX_TIMESTAMP(CURRENT_TIMESTAMP()))"
      " - (SELECT `VARIABLE_VALUE` FROM `information_schema`.`GLOBAL_STATUS` WHERE `VARIABLE_NAME` = 'UPTIME')))"
      " GROUP BY DATE(`start_time`)";
}

QString StaticFunctions::identifierPattern()
{
  return "`[A-Za-z_\\d%\\- \\(\\)\\./]*`";
}

QString StaticFunctions::fullidentifierPattern()
{
  return identifierPattern() + "\\." + identifierPattern();
}
