/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef STATEMENTS_H
#define STATEMENTS_H

#include <QDialog>

class QMenu;
class DTableWidget;

/**
 * @brief The Statements class provides a dialog to review and manage a list of SQL statements.
 *
 * This dialog is typically used to show the user a batch of generated SQL queries
 * (e.g., from a table editor or migration tool) before they are committed to the database.
 * Users can view, copy, or remove specific statements from the execution list.
 */
class Statements : public QDialog
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the Statements dialog.
   *
   * Initializes the table widget and the user interface layout.
   */
  Statements();

public slots:
  /**
   * @brief Displays the dialog populated with a specific list of queries.
   *
   * @param queries to the list of SQL statements to be reviewed.
   */
  void execSlot(QStringList queries);

private:
  DTableWidget *queriesDTableWidget;
  QAction *copyAction;
  QClipboard *clipboard;
  QMenu *menu;
  QStringList queryList;
  QAction *removeStatementAction;

  /**
   * @brief Initializes the actions for the context menu (Copy, Remove, etc.).
   */
  void createActions();

  /**
   * @brief Populates the table widget with the data from the query list.
   */
  void fillGrid();

private slots:
  /**
   * @brief Handles double-click events on the table cells.
   *
   * Typically serves as a shortcut to copy the cell content or view it in detail.
   */
  void queriesCellDoubleClickedSlot();

  /**
   * @brief Copies the content of the selected statement to the system clipboard.
   */
  void copyToClipboard();

  /**
   * @brief Removes the selected statement from the list.
   *
   * This allows the user to exclude specific queries from execution.
   */
  void removeStatementActionTriggered();

protected:
  /**
   * @brief Handles the context menu event.
   *
   * Shows the custom menu containing actions to copy or remove statements.
   * @param e The context menu event parameters.
   */
  void contextMenuEvent(QContextMenuEvent *e);
};


#endif // STATEMENTS_H
