/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef SQLQUERY_H
#define SQLQUERY_H

class BaseTextEditor;
class QHBoxLayout;
class QToolBar;
class Statements;
class TextEditor;
class DTitleLabel;
class QPushButton;
class QLabel;
class QElapsedTimer;
class QSqlDatabase;
class Projects;
class QComboBox;
class QMovie;

#include "dmdisubwindow.h"
#include "dbms.h"
#include "dtablewidget.h"
#include "dsettings.h"

/**
 * @brief The SQLQuery class provides an interface for writing and executing SQL scripts.
 *
 * This class inherits from DMdiSubWindow and is the primary workspace for database developers.
 * It features a syntax-highlighting text editor for writing SQL, toolbars for execution control,
 * and multiple views for displaying results (Text or Table).
 *
 * It also supports advanced features like the "SQL Player" (stepping through queries),
 * transaction management, EXPLAIN analysis, and background thread execution.
 */
class SQLQuery : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the SQLQuery window.
   * @param project Pointer to the project context (if any).
   * @param serverConnection Pointer to the active database connection.
   * @param windowCount A unique identifier/index for this window instance.
   */
  SQLQuery(Projects *project, DBMS *serverConnection, unsigned int windowCount = 0);

  /**
   * @brief Checks if the window can be safely closed.
   *
   * Verifies if there are running queries or unsaved changes.
   * @return True if safe to close, false otherwise.
   */
  bool okToClose();

  /**
   * @brief Populates the editor's autocomplete/highlighter with MariaDB/MySQL symbols.
   */
  void fillMariaDBSymbols();

  /**
   * @brief Retrieves the main toolbar associated with this window.
   * @return Pointer to the QToolBar.
   */
  QToolBar *getToolBar();

private:
  TextEditor *scriptEditor;
  BaseTextEditor *resultEditor;
  DTableWidget *dTableWidgetResult;
  DBMS *serverConnection;
  QToolBar *queryToolBar;
  Projects *project;
  QComboBox *externalConnectionName;

  // Output formatting helpers
  void outputT();
  void outputV();
  void outputVV();
  void outputVVV();

  /**
   * @brief Initializes all QActions and connects their signals.
   */
  void createActions();

  /**
   * @brief Retrieves the current text selected or the whole text in the editor.
   * @return The SQL string.
   */
  QString statement();

  QMovie *runningSnakeMovie;
  QHBoxLayout *toolbarHorizontalLayout;
  QWidget *widToolbar;

  // Actions
  QAction *executeAction;
  QAction *executeActionOnAThread;
  QAction *executeExternalQueryAction;
  QAction *exportAction;
  QAction *viewHistoryAction;
  Statements *statementsDialog;
  DSettings settings;
  QAction *showNewLinesAction;
  QAction *splitAction;
  unsigned int windowCount;
  DTitleLabel *dTitleLabel;

  // SQL Player Members (Step-by-step execution)
  QAction *startSQLPlayerAction;
  QStringList queriesToBePlayed;
  QToolBar *queryPlayerToolBar;
  QAction *queryPlayerRunQueryAction;
  QDialog *dialogQueryPlayer;

  /**
   * @brief Internal helper to execute a specific SQL string.
   * @param statement The SQL to run.
   */
  void executeStatement(QString statement = QString());

  float nextQueryToExecute;
  QAction *queryPlayerStopAction;
  QAction *queryPlayerRunPreviousQueryAction;
  QAction *queryPlayerRunNextQueryAction;
  QAction *queryPlayerRunFirstQueryAction;
  QAction *queryPlayerRunLastQueryAction;
  bool queryPlayerStopBecauseAnError;

  QAction *concatenateOutputAction;
  QAction *exportTableDataForInsertAction;
  QAction *explainSelectAction;
  QAction *explainInsertAction;
  QAction *safeStatementsAction;
  QLabel *timeLabel;
  unsigned long long elapsedSeconds;
  QTimer *elapsedSecondsTimer;
  QThread *threadElapsedSecondsTimer;
  QThread *threadQueryRunQueryAction;
  QAction *showStatementsErrorAction;
  QAction *explainUpdateAction;
  QAction *repeatQueryExecutionAction;
  unsigned int repeatQueryExecutionTime;
  QAction *wordWrapOnResultAction;
  QAction *logStatementsAction;
  bool logStatements;
  QAction *trimColumnsAction;
  QAction *checkTablesAction;
  QAction *explainSelectActionWithAliasAction;

  /**
   * @brief Runs an EXPLAIN query on the current selection.
   * @param withAlias If true, uses extended alias information.
   */
  void explainSELECT(bool withAlias = false);

  QAction *beginTransacctionAction;
  QAction *commitTransacctionAction;
  QAction *rollbackTransacctionAction;
  QAction *exportResultDataForInsertAction;
  QComboBox *comboDelimiter;
  QComboBox *comboOutput;
//  DBMS *threadedQueryExecutionServerConnection;

  /**
   * @brief Executes a CHECK TABLE command.
   * @param statement The check statement.
   */
  void checkTablesExecuteTheCheck(QString statement);

  QAction *deleteCurrentSQLQueryAction;
  QAction *runBenchmarkAction;

signals:
  /**
   * @brief Emitted to toggle the enabled state of UI actions during execution.
   */
  void enableDisableAction();

  /**
   * @brief Emitted when query execution begins.
   * @param progress Initial progress value.
   */
  void executionStarted(double progress);

  /**
   * @brief Emitted when query execution finishes.
   * @param progress Final progress value.
   */
  void executionFinished(double progress);

  /**
   * @brief Emitted to update progress during execution.
   * @param progress Current progress value.
   */
  void executionProgress(double progress);

  /**
   * @brief Emitted to request opening a result tab in the main window.
   */
  void showResultTab(QString table, QString database, QString where);

  /**
   * @brief Emitted to show a message on the status bar.
   */
  void statusBarMessage(QString message);

  /**
   * @brief Emitted to open a URL in an external browser.
   */
  void openURL(QString url);

  /**
   * @brief Emitted to update the cursor position viewer in the status bar.
   * @param x Column.
   * @param y Row.
   * @param z Selection length.
   */
  void updatePrositionViewer(const int x, const int y, const int z);

  /**
   * @brief Emitted to set the application busy state.
   */
  void busy(bool busy);

  /**
   * @brief Emitted when the window is deleted/closed to cleanup resources.
   */
  void deleteAndClose(QString queryElement);

private slots:
  /**
   * @brief Standard execution of the current script.
   */
  void executeActionTriggered();

  /**
   * @brief Executions the current script in a background thread.
   */
  void executeActionOnAThreadTriggered();

  /**
   * @brief Executes the query against a different (external) connection.
   */
  void executeExternalQueryActionTriggered();

  /**
   * @brief Sets the content of the script editor.
   * @param text The SQL text.
   */
  void setScriptText(QString text);

  /**
   * @brief Opens the query history dialog.
   */
  void viewHistoryActionTriggered();

  /**
   * @brief Toggles the display of newline characters.
   */
  void showNewLinesActionTriggered(bool triggered);

  /**
   * @brief Toggles split view.
   */
  void splitActionTriggered(bool triggered);

  /**
   * @brief Activates the SQL Player mode (step-by-step execution).
   */
  void startSQLPlayerActionTriggered();

  // SQL Player Control Slots
  void queryPlayerRunQueryActionTriggered();
  void queryPlayerStopActionTriggered();
  void queryPlayerRunPreviousQueryActionTriggered();
  QString queryToBePlayed(int position);
  void queryPlayerRunNextQueryActionTriggered();
  void enableDisableActionSlot();
  void queryPlayerRunFirstQueryActionTriggered();
  void queryPlayerRunLastQueryActionTriggered();

  void exportTableDataForInsertActionTriggered();
  void exportResultDataForInsertActionTriggered();
  void explainSelectActionTriggered();
  void explainInsertActionTriggered();

  /**
   * @brief Toggles "Safe Updates" mode (preventing UPDATE/DELETE without WHERE).
   */
  void safeStatementsActionTriggered(bool triggered);

  /**
   * @brief Updates the execution timer label.
   */
  void incrementSeconds();

  void runQueryActionTriggered();
  void showStatementsErrorActionTriggered();
  void errorAcceptedSlot();
  void errorOccurredSlot();
  void explainUpdateActionTriggered();

  /**
   * @brief Configures repeated execution of the query (polling).
   */
  void repeatQueryExecutionActionTriggered();

  void wordWrapOnResultActionToggled();

  /**
   * @brief Reloads the last executed query from history.
   */
  void loadLastQuery();

  void logStatementsActionToggled();
  void trimColumnsActionTriggered();
  void checkTablesActionTriggered();
  void explainSelectActionWithAliasActionTriggered();

  /**
   * @brief Forwards the position update signal.
   */
  void emitUpdatePrositionViewer(const int x, const int y, const int z);

  // Transaction Slots
  void beginTransacctionActionTriggered();
  void commitTransacctionActionTriggered();
  void rollbackTransacctionActionTriggered();

  /**
   * @brief Executes a batch of generated statements (e.g., from table edits).
   */
  void executeStatements(QStringList statements);

  /**
   * @brief Updates the delimiter combo box based on text input.
   */
  void comboDelimiterEditTextChanged(const QString &text);

  /**
   * @brief Changes the SQL delimiter used for parsing.
   */
  void changeDelimiter(QString delimiter = ";");

  /**
   * @brief Processes the result string from a threaded execution.
   */
  void handleResultsOfThreadedQueryExecution(QString result);

  /**
   * @brief cleans up after a threaded execution finishes.
   */
  void handleFinishedOfThreadedQueryExecution();

  /**
   * @brief Retranslates the UI.
   */
  void retranslateUI();

  void deleteCurrentSQLQueryActionTriggered();
  void runBenchmarkActionTriggered();

};

#endif // SQLQUERY_H
