/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QVBoxLayout>
#include <QToolBar>
#include <QAction>
#include <QSplitter>
#include <QTextDocumentFragment>
#include <QApplication>
#include <QPushButton>
#include <QTimer>
#include <QTextBlock>
#include <QMessageBox>
#include <QThread>
#include <QDateTime>
#include <QInputDialog>
#include <QFileDialog>
#include <QComboBox>
#include <QMovie>
#include <memory>

#include "sqlquery.h"
#include "dtitlelabel.h"
#include "staticfunctions.h"
#include "statements.h"
#include "texteditor.h"
#include "basetexteditor.h"
#include "dicon.h"

#include "QDebug"

SQLQuery::SQLQuery(Projects *project, DBMS *serverConnection, unsigned int windowCount)
{
  setObjectName("SQLQuery");
  this->project = project;
  this->serverConnection = serverConnection;
  this->queryPlayerStopBecauseAnError = false;
  connect(this->serverConnection, SIGNAL(errorOccurred()), this, SLOT(errorOccurredSlot()));
  connect(this->serverConnection, SIGNAL(errorMessageAccepted()), this, SLOT(errorAcceptedSlot()));
  connect(this->serverConnection, SIGNAL(changeDelimiter(QString)), this, SLOT(changeDelimiter(QString)));
  ///connect(this->serverConnection, SIGNAL(errorOccurred()), this, SLOT(queryPlayerStopActionTriggered()));
  this->windowCount = windowCount;
  setWindowIcon(DIcon::DatabaseQuery());
  statementsDialog = new Statements;
  QVBoxLayout *mainVLayout = new QVBoxLayout;
  dTitleLabel = new DTitleLabel;
  mainVLayout->addWidget(dTitleLabel);
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  QSplitter *mainSplitter = new QSplitter(Qt::Vertical);
  timeLabel = new QLabel("\n\n\n");
  elapsedSeconds = 0;
  repeatQueryExecutionTime = 0;
  createActions();
  queryToolBar = new QToolBar;
  comboOutput = new QComboBox;
  comboDelimiter = new QComboBox;
  comboDelimiter->addItem(";");
  comboDelimiter->addItem("|");
  comboDelimiter->setEditable(true);
  //comboDelimiter->setSizeAdjustPolicy(QComboBox::AdjustToMinimumContentsLengthWithIcon);
  connect(comboDelimiter, SIGNAL(editTextChanged(QString)), this, SLOT(comboDelimiterEditTextChanged(QString)));
  queryToolBar->addAction(beginTransacctionAction);
  queryToolBar->addAction(rollbackTransacctionAction);
  queryToolBar->addAction(commitTransacctionAction);
  queryToolBar->addSeparator();
  queryToolBar->addAction(safeStatementsAction);
  queryToolBar->addAction(executeAction);
  queryToolBar->addAction(exportAction);
  queryToolBar->addAction(concatenateOutputAction);
  queryToolBar->addAction(showStatementsErrorAction);
  queryToolBar->addSeparator();
  queryToolBar->addWidget(comboOutput);
  queryToolBar->addSeparator();
  queryToolBar->addAction(showNewLinesAction);
  queryToolBar->addWidget(comboDelimiter);
  queryToolBar->addAction(splitAction);
  queryToolBar->addSeparator();
  queryToolBar->addAction(viewHistoryAction);
  queryToolBar->addSeparator();
  queryToolBar->addAction(startSQLPlayerAction);
  queryToolBar->setIconSize(QSize(24, 24));
  queryToolBar->addSeparator();
  queryToolBar->addAction(explainSelectAction);
  queryToolBar->addAction(explainSelectActionWithAliasAction);
  queryToolBar->addAction(explainInsertAction);
  queryToolBar->addAction(explainUpdateAction);
  queryToolBar->addAction(exportTableDataForInsertAction);
  queryToolBar->addAction(exportResultDataForInsertAction);
  queryToolBar->addAction(repeatQueryExecutionAction);
  queryToolBar->addAction(wordWrapOnResultAction);
  queryToolBar->addAction(logStatementsAction);
  queryToolBar->addAction(trimColumnsAction);
  queryToolBar->addAction(checkTablesAction);
  queryToolBar->addSeparator();
  queryToolBar->addAction(executeActionOnAThread);
  runningSnakeMovie = new QMovie(DIcon::Loading(), this);
  runningSnakeMovie->setScaledSize(QSize(24, 24));
  QLabel *processLabel = new QLabel(this);
  processLabel->setMovie(runningSnakeMovie);
  queryToolBar->addWidget(processLabel);
  runningSnakeMovie->start();
  runningSnakeMovie->stop();
  mainVLayout->addWidget(queryToolBar);

  queryToolBar->addSeparator();
  externalConnectionName = new QComboBox();
  externalConnectionName->setMaximumWidth(100);
  externalConnectionName->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  externalConnectionName->setToolTip(tr("List of aviable database connections"));
  externalConnectionName->setStatusTip(externalConnectionName->toolTip());
  externalConnectionName->setWhatsThis(externalConnectionName->toolTip());
  QStringList connections = settings.allKeys("ServerConnections");
  int valueToRemove = connections.indexOf("StorePassword");
  if (valueToRemove >= 0)
      connections.removeAt(valueToRemove);
  valueToRemove = connections.indexOf("SortConnectionList");
  if (valueToRemove >= 0)
      connections.removeAt(valueToRemove);
  externalConnectionName->insertItems(0, connections);
  queryToolBar->addWidget(externalConnectionName);
  queryToolBar->addAction(executeExternalQueryAction);
  queryToolBar->addSeparator();
  queryToolBar->addAction(deleteCurrentSQLQueryAction);

  queryToolBar->addSeparator();
  queryToolBar->addAction(runBenchmarkAction);

  QFrame* separatorFrame = new QFrame();
  separatorFrame->setFrameShape(QFrame::HLine);
  mainVLayout->addWidget(separatorFrame);
  scriptEditor = new TextEditor(project, serverConnection, EditorTypes::SQLQuery, 0, true);
#ifdef Q_OS_MAC
    queryToolBar->addSeparator();
    queryToolBar->addActions(scriptEditor->getMenus());
#endif
  connect(scriptEditor, SIGNAL(updatePrositionViewer(int,int,int)), this, SLOT(emitUpdatePrositionViewer(int,int,int)));
  ///connect(scriptEditor, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), project, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  mainSplitter->addWidget(scriptEditor);
  resultEditor = new BaseTextEditor(EditorTypes::NoEditor);
  connect(scriptEditor, SIGNAL(zoomInSignal()), resultEditor, SLOT(zoomIn()));
  connect(scriptEditor, SIGNAL(zoomOutSignal()), resultEditor, SLOT(zoomOut()));
  connect(scriptEditor, SIGNAL(zoomResetSignal()), resultEditor, SLOT(zoomReset()));
  resultEditor->setWordWrapMode(settings.value("SQLQuery/WordWrapOnResul", false).toBool() ? QTextOption::WordWrap : QTextOption::NoWrap);
  mainSplitter->addWidget(resultEditor);
  dTableWidgetResult = new DTableWidget(QStringList(), QAbstractItemView::MultiSelection);
  connect(dTableWidgetResult, SIGNAL(executeStatements(QStringList)), this, SLOT(executeStatements(QStringList)));
  dTableWidgetResult->setVisible(false);
  mainSplitter->addWidget(dTableWidgetResult);
  queriesToBePlayed = QStringList();
  dialogQueryPlayer = new QDialog;
//   dialogQueryPlayer->setWindowFlags(Qt::FramelessWindowHint);
  QVBoxLayout *queryPlayerVLayout = new QVBoxLayout;
  queryPlayerVLayout->setContentsMargins(0, 0, 0, 0);
  queryPlayerVLayout->addWidget(queryPlayerToolBar);
  queryPlayerVLayout->addWidget(timeLabel);
  dialogQueryPlayer->setLayout(queryPlayerVLayout);
  connect(this, SIGNAL(enableDisableAction()), this, SLOT(enableDisableActionSlot()));

  QWidget *widMain = new QWidget;
  mainVLayout->addWidget(mainSplitter);
  widMain->setLayout(mainVLayout);
  retranslateUI();
  setWidget(widMain);
  scriptEditor->setFocus();
  scriptEditor->setWindowFlags(Qt::FramelessWindowHint);

  threadElapsedSecondsTimer = new QThread;
  elapsedSecondsTimer = new QTimer;
  elapsedSecondsTimer->moveToThread(threadElapsedSecondsTimer);
  connect(threadElapsedSecondsTimer, SIGNAL(started()), elapsedSecondsTimer, SLOT(start()));
  connect(threadElapsedSecondsTimer, SIGNAL(finished()), elapsedSecondsTimer, SLOT(stop()));

  threadQueryRunQueryAction = new QThread;
  connect(elapsedSecondsTimer, SIGNAL(timeout()), this, SLOT(incrementSeconds()));
  elapsedSecondsTimer->setInterval(1000);
  connect(threadQueryRunQueryAction, SIGNAL(started()), this, SLOT(runQueryActionTriggered()));
  QTimer::singleShot(0, this, SLOT(loadLastQuery()));
}

void SQLQuery::retranslateUI()
{
  setWindowTitle(tr("Query %1").arg(windowCount));
//  setObjectName(windowTitle());
  dTitleLabel->setText(windowTitle());
  dTitleLabel->setToolTip(dTitleLabel->text());
  executeAction->setText(tr("Run"));
  executeAction->setToolTip(tr("Runs a query"));
  executeActionOnAThread->setText(tr("Run on thread"));
  executeActionOnAThread->setToolTip(tr("Runs a query on a thread"));
  exportAction->setText(tr("Export"));
  exportAction->setToolTip(tr("Export query output"));
  viewHistoryAction->setText(tr("History"));
  viewHistoryAction->setToolTip(viewHistoryAction->text());
  showNewLinesAction->setText(tr("Show New Lines"));
  showNewLinesAction->setToolTip(showNewLinesAction->text());
  startSQLPlayerAction->setText(tr("Start SQL Player"));
  startSQLPlayerAction->setToolTip(startSQLPlayerAction->text());
  splitAction->setText(tr("Split query"));
  splitAction->setToolTip(splitAction->text());
  queryPlayerToolBar->setWindowTitle(tr("Actions"));
  queryPlayerToolBar->setToolTip(queryPlayerToolBar->windowTitle());
  queryPlayerRunQueryAction->setText(tr("Start/Pause execution"));
  queryPlayerRunQueryAction->setToolTip(queryPlayerRunQueryAction->text());
  queryPlayerStopAction->setText(tr("Stop execution"));
  queryPlayerStopAction->setToolTip(queryPlayerStopAction->text());
  queryPlayerRunPreviousQueryAction->setText(tr("Run previous query"));
  queryPlayerRunPreviousQueryAction->setToolTip(queryPlayerRunPreviousQueryAction->text());
  queryPlayerRunNextQueryAction->setText(tr("Run next query"));
  queryPlayerRunNextQueryAction->setToolTip(queryPlayerRunNextQueryAction->text());
  queryPlayerRunFirstQueryAction->setText(tr("Run first query"));
  queryPlayerRunFirstQueryAction->setToolTip(queryPlayerRunFirstQueryAction->text());
  queryPlayerRunLastQueryAction->setText(tr("Run last query"));
  queryPlayerRunLastQueryAction->setToolTip(queryPlayerRunLastQueryAction->text());
  concatenateOutputAction->setText(tr("Concatenate query output"));
  concatenateOutputAction->setToolTip(concatenateOutputAction->text());
  exportTableDataForInsertAction->setText(tr("Export table data for INSERT"));
  exportTableDataForInsertAction->setToolTip(exportTableDataForInsertAction->text());
  exportResultDataForInsertAction->setText(tr("Export result data for INSERT"));
  exportResultDataForInsertAction->setToolTip(exportResultDataForInsertAction->text());
  explainSelectAction->setText(tr("Explain SELECT"));
  explainSelectAction->setToolTip(explainSelectAction->text());
  explainInsertAction->setText(tr("Explain INSERT"));
  explainInsertAction->setToolTip(explainInsertAction->text());
  safeStatementsAction->setText(tr("Safe statements"));
  safeStatementsAction->setToolTip(safeStatementsAction->text());
  showStatementsErrorAction->setText(tr("Show statements with error"));
  showStatementsErrorAction->setToolTip(showStatementsErrorAction->text());
  explainUpdateAction->setText(tr("Explain UPDATE"));
  explainUpdateAction->setToolTip(explainUpdateAction->text());
  repeatQueryExecutionAction->setText(tr("Repeate execution"));
  repeatQueryExecutionAction->setToolTip(repeatQueryExecutionAction->text());
  wordWrapOnResultAction->setText(tr("Word wrap on result"));
  wordWrapOnResultAction->setToolTip(wordWrapOnResultAction->text());
  logStatementsAction->setText(tr("Log statements"));
  logStatementsAction->setToolTip(logStatementsAction->text());
  trimColumnsAction->setText(tr("TRIM columns in table"));
  trimColumnsAction->setToolTip(trimColumnsAction->text());
  checkTablesAction->setText(tr("Check tables status"));
  checkTablesAction->setToolTip(checkTablesAction->text());
  explainSelectActionWithAliasAction->setText(tr("Explain SELECT with Alias"));
  explainSelectActionWithAliasAction->setToolTip(explainSelectActionWithAliasAction->text());
  deleteCurrentSQLQueryAction->setText(tr("Delete current query"));
  deleteCurrentSQLQueryAction->setToolTip(deleteCurrentSQLQueryAction->text());

//  radioT->setToolTip(tr("Output as table."));
//  radioT2->setToolTip(tr("Alternate output as table."));
//  radioX->setToolTip(tr("Same output as -v but with no headers."));
//  radioV->setToolTip(tr("Same output as -vv but with no query."));
//  radioVV->setToolTip(tr("Output TAB separated with the query."));
//  radioVVV->setToolTip(tr("Same output as -t but with the query."));
//  radioHTML->setToolTip(tr("Output as HTML."));
//  radioTXT->setToolTip(tr("Output as text."));
//  radioXML->setToolTip(tr("Output as XML."));
//  radioPDF->setToolTip(tr("Output in PDF."));
//  radioG->setToolTip(tr("Outputs columns as rows."));

  beginTransacctionAction->setText(tr("Begin transaction"));
  beginTransacctionAction->setToolTip(beginTransacctionAction->text());
  rollbackTransacctionAction->setText(tr("Rollback transaction"));
  rollbackTransacctionAction->setToolTip(rollbackTransacctionAction->text());
  commitTransacctionAction->setText(tr("Commit transaction"));
  commitTransacctionAction->setToolTip(commitTransacctionAction->text());

  comboDelimiter->setToolTip(tr("Statement delimiter"));

  comboOutput->setToolTip(tr("Output type"));
  comboOutput->clear();
  comboOutput->addItem(tr("Table"), "-t");
  comboOutput->addItem(tr("Table with the query"), "-vvv");
  comboOutput->addItem(tr("Alternative"), "-A");
  comboOutput->addItem(tr("Headerless"), "-x");
  comboOutput->addItem(tr("Tab separeted"), "-v");
  comboOutput->addItem(tr("Tab separeted with the query"), "-vv");
  comboOutput->addItem("HTML", "HTML");
  comboOutput->addItem("TXT (Text)", "-TXT");
  comboOutput->addItem("XML", "-XML");
  comboOutput->addItem("PDF", "-PDF");
  comboOutput->addItem(tr("Columnar"), "-G");
  comboOutput->addItem("JSON", "-J");
  comboOutput->addItem("DTableWidget", "DTableWidget");
  comboOutput->setMaximumWidth(100);
  runBenchmarkAction->setText(tr("Run a benchmark"));
  runBenchmarkAction->setToolTip(runBenchmarkAction->text());
  executeExternalQueryAction->setText(tr("Run query on an external connection"));
  executeExternalQueryAction->setToolTip(executeExternalQueryAction->text());

  scriptEditor->retranslateUI();
  resultEditor->retranslateUI();
}

void SQLQuery::deleteCurrentSQLQueryActionTriggered()
{
  emit deleteAndClose("SQLQuery/LastQuery-" + qApp->property("ConnectionName").toString() + "-" + windowTitle());
  close();
}

void SQLQuery::runBenchmarkActionTriggered()
{
  bool ok;
  uint tests = QInputDialog::getInt(this, tr("What number of tests you want to run?")
                                    , tr("What number of tests you want to run?"), 0, 1, 1000, 1, &ok);
  if (ok) {
    QString statementToExecute = statement();
    QMessageBox msgBox;
    msgBox.setText(tr("Script exectution"));
    msgBox.setInformativeText(tr("Do you really want to execute this script?"));
    msgBox.setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
    msgBox.setDefaultButton(QMessageBox::Cancel);
    msgBox.setDetailedText(statementToExecute);
    if (msgBox.exec() == QMessageBox::Ok) {
      for(uint counter = 1; counter <= tests; counter++) {
        ThreadedQueryExecution *threadedQueryExecution = new ThreadedQueryExecution(this->serverConnection->getConnectionParameters(), statementToExecute);
        threadedQueryExecution->setOnlyExecuteSafeStatements(safeStatementsAction->isChecked());
        threadedQueryExecution->setOutputType(comboOutput->currentData().toString());
        threadedQueryExecution->setSaveToFile(exportAction->isChecked());
        threadedQueryExecution->setShowNewLines(showNewLinesAction->isChecked());
        threadedQueryExecution->start();
      }
    }
  }
}

void SQLQuery::createActions()
{
  executeAction = new QAction(this);
  executeAction->setIcon(DIcon::QueryExecution());
  executeAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_Return));
  connect(executeAction, SIGNAL(triggered()), this, SLOT(executeActionTriggered()));
  executeActionOnAThread = new QAction(this);
  executeActionOnAThread->setIcon(DIcon::QueryExecution());
  connect(executeActionOnAThread, SIGNAL(triggered()), this, SLOT(executeActionOnAThreadTriggered()));
  executeExternalQueryAction = new QAction(this);
  executeExternalQueryAction->setIcon(DIcon::QueryExecution());
  connect(executeExternalQueryAction, SIGNAL(triggered()), this, SLOT(executeExternalQueryActionTriggered()));
  exportAction = new QAction(this);
  exportAction->setIcon(DIcon::Export());
  exportAction->setCheckable(true);
  exportAction->setChecked(false);
  viewHistoryAction = new QAction(this);
  viewHistoryAction->setIcon(DIcon::DatabaseQueryHistory());
  connect(viewHistoryAction, SIGNAL(triggered()), this, SLOT(viewHistoryActionTriggered()));
  showNewLinesAction = new QAction(this);
  showNewLinesAction->setIcon(DIcon::Export());
  showNewLinesAction->setShortcut(QKeySequence(Qt::Key_F8));
  showNewLinesAction->setCheckable(true);
  showNewLinesAction->setChecked(settings.value("SQLQuery/ReplaceNewLines", true).toBool());
  connect(showNewLinesAction, SIGNAL(triggered(bool)), this, SLOT(showNewLinesActionTriggered(bool)));
  splitAction = new QAction(this);
  splitAction->setIcon(DIcon::Cut());
  splitAction->setCheckable(true);
  splitAction->setChecked(settings.value("SQLQuery/SplitQueries", true).toBool());
  connect(splitAction, SIGNAL(triggered(bool)), this, SLOT(splitActionTriggered(bool)));
  startSQLPlayerAction = new QAction(this);
  startSQLPlayerAction->setIcon(DIcon::MediaPlaybackStart());
  connect(startSQLPlayerAction, SIGNAL(triggered()), this, SLOT(startSQLPlayerActionTriggered()));
  queryPlayerRunQueryAction = new QAction(this);
  queryPlayerRunQueryAction->setIcon(DIcon::MediaPlaybackStart());
  connect(queryPlayerRunQueryAction, SIGNAL(triggered()), this, SLOT(queryPlayerRunQueryActionTriggered()));
  queryPlayerRunQueryAction->setCheckable(true);
  queryPlayerStopAction = new QAction(this);
  queryPlayerStopAction->setIcon(DIcon::MediaPlaybackStop());
  connect(queryPlayerStopAction, SIGNAL(triggered()), this, SLOT(queryPlayerStopActionTriggered()));
  showStatementsErrorAction = new QAction(this);
  showStatementsErrorAction->setIcon(DIcon::Error());
  connect(showStatementsErrorAction, SIGNAL(triggered()), this, SLOT(showStatementsErrorActionTriggered()));
  queryPlayerRunPreviousQueryAction = new QAction(this);
  queryPlayerRunPreviousQueryAction->setIcon(DIcon::MediaSeekBackward());
  connect(queryPlayerRunPreviousQueryAction, SIGNAL(triggered()), this, SLOT(queryPlayerRunPreviousQueryActionTriggered()));
  queryPlayerRunNextQueryAction = new QAction(this);
  queryPlayerRunNextQueryAction->setIcon(DIcon::MediaSeekForward());
  connect(queryPlayerRunNextQueryAction, SIGNAL(triggered()), this, SLOT(queryPlayerRunNextQueryActionTriggered()));
  queryPlayerRunFirstQueryAction = new QAction(this);
  queryPlayerRunFirstQueryAction->setIcon(DIcon::MediaSkipBackward());
  connect(queryPlayerRunFirstQueryAction, SIGNAL(triggered()), this, SLOT(queryPlayerRunFirstQueryActionTriggered()));
  queryPlayerRunLastQueryAction = new QAction(this);
  queryPlayerRunLastQueryAction->setIcon(DIcon::MediaSkipForward());
  connect(queryPlayerRunLastQueryAction, SIGNAL(triggered()), this, SLOT(queryPlayerRunLastQueryActionTriggered()));
  concatenateOutputAction = new QAction(this);
  concatenateOutputAction->setIcon(DIcon::AttachResult());
  concatenateOutputAction->setCheckable(true);
  concatenateOutputAction->setChecked(false);
  exportTableDataForInsertAction = new QAction(this);
  exportTableDataForInsertAction->setIcon(DIcon::Export());
  connect(exportTableDataForInsertAction, SIGNAL(triggered()), this, SLOT(exportTableDataForInsertActionTriggered()));
  exportResultDataForInsertAction = new QAction(this);
  exportResultDataForInsertAction->setIcon(DIcon::Export());
  connect(exportResultDataForInsertAction, SIGNAL(triggered()), this, SLOT(exportResultDataForInsertActionTriggered()));
  explainSelectAction = new QAction(this);
  explainSelectAction->setIcon(DIcon::DatabaseSELECT());
  explainSelectAction->setShortcut(QKeySequence(Qt::Key_F7));
  connect(explainSelectAction, SIGNAL(triggered()), this, SLOT(explainSelectActionTriggered()));
  explainInsertAction = new QAction(this);
  explainInsertAction->setIcon(DIcon::DatabaseINSERT());
  explainInsertAction->setShortcut(QKeySequence(Qt::Key_F9));
  connect(explainInsertAction, SIGNAL(triggered()), this, SLOT(explainInsertActionTriggered()));
  safeStatementsAction = new QAction(this);
  safeStatementsAction->setIcon(DIcon::LockQueryExecution());
  safeStatementsAction->setCheckable(true);
  safeStatementsAction->setChecked(settings.value("SQLQuery/ExecuteSafeStatements-" + qApp->property("ConnectionName").toString(), true).toBool());
  connect(safeStatementsAction, SIGNAL(triggered(bool)), this, SLOT(safeStatementsActionTriggered(bool)));
  explainUpdateAction = new QAction(this);
  explainUpdateAction->setIcon(DIcon::DatabaseUPDATE());
  connect(explainUpdateAction, SIGNAL(triggered()), this, SLOT(explainUpdateActionTriggered()));
  repeatQueryExecutionAction = new QAction(this);
  repeatQueryExecutionAction->setIcon(DIcon::RepeateQueryExecution());
  connect(repeatQueryExecutionAction, SIGNAL(triggered()), this, SLOT(repeatQueryExecutionActionTriggered()));
  repeatQueryExecutionAction->setCheckable(true);
  logStatementsAction = new QAction(this);
  logStatementsAction->setIcon(DIcon::DatabaseQueryLog());
  logStatementsAction->setCheckable(true);
  logStatementsAction->setChecked(settings.value("SQLQuery/LogStatements", true).toBool());
  connect(logStatementsAction, SIGNAL(toggled(bool)), this, SLOT(logStatementsActionToggled()));
  logStatements = logStatementsAction->isChecked();
  trimColumnsAction = new QAction(this);
  trimColumnsAction->setIcon(DIcon::Executable());
  connect(trimColumnsAction, SIGNAL(triggered()), this, SLOT(trimColumnsActionTriggered()));
  deleteCurrentSQLQueryAction = new QAction(this);
  deleteCurrentSQLQueryAction->setIcon(DIcon::Delete());
  connect(deleteCurrentSQLQueryAction, SIGNAL(triggered()), this, SLOT(deleteCurrentSQLQueryActionTriggered()));

  wordWrapOnResultAction = new QAction(this);
  wordWrapOnResultAction->setIcon(DIcon::WordWrap());
  wordWrapOnResultAction->setCheckable(true);
  wordWrapOnResultAction->setChecked(settings.value("SQLQuery/WordWrapOnResul", false).toBool());
  connect(wordWrapOnResultAction, SIGNAL(toggled(bool)), this, SLOT(wordWrapOnResultActionToggled()));

  checkTablesAction = new QAction(this);
  checkTablesAction->setIcon(DIcon::DatabaseTableStatus());
  connect(checkTablesAction, SIGNAL(triggered()), this, SLOT(checkTablesActionTriggered()));

  explainSelectActionWithAliasAction = new QAction(this);
  explainSelectActionWithAliasAction->setIcon(DIcon::DatabaseSELECT());
  explainSelectActionWithAliasAction->setShortcut(QKeySequence(Qt::SHIFT | Qt::Key_F7));
  connect(explainSelectActionWithAliasAction, SIGNAL(triggered()), this, SLOT(explainSelectActionWithAliasActionTriggered()));

  queryPlayerToolBar = new QToolBar();
  queryPlayerToolBar->addAction(queryPlayerRunFirstQueryAction);
  queryPlayerToolBar->addAction(queryPlayerRunPreviousQueryAction);
  queryPlayerToolBar->addAction(queryPlayerRunQueryAction);
  queryPlayerToolBar->addAction(queryPlayerRunNextQueryAction);
  queryPlayerToolBar->addAction(queryPlayerRunLastQueryAction);
  queryPlayerToolBar->addSeparator();
  queryPlayerToolBar->addAction(queryPlayerStopAction);
  queryPlayerToolBar->addSeparator();
  queryPlayerToolBar->addAction(explainSelectAction);
  queryPlayerToolBar->addAction(explainSelectActionWithAliasAction);
  queryPlayerToolBar->addAction(explainInsertAction);
  queryPlayerToolBar->addAction(explainUpdateAction);
  queryPlayerToolBar->addAction(concatenateOutputAction);

  beginTransacctionAction = new QAction(this);
  beginTransacctionAction->setIcon(DIcon::Database());
  beginTransacctionAction->setCheckable(true);
  connect(beginTransacctionAction, SIGNAL(triggered()), this, SLOT(beginTransacctionActionTriggered()));
  commitTransacctionAction = new QAction(this);
  commitTransacctionAction->setIcon(DIcon::GoUp());
  commitTransacctionAction->setEnabled(false);
  connect(commitTransacctionAction, SIGNAL(triggered()), this, SLOT(commitTransacctionActionTriggered()));
  rollbackTransacctionAction = new QAction(this);
  rollbackTransacctionAction->setIcon(DIcon::DatabaseRollback());
  rollbackTransacctionAction->setEnabled(false);
  connect(rollbackTransacctionAction, SIGNAL(triggered()), this, SLOT(rollbackTransacctionActionTriggered()));
  runBenchmarkAction = new QAction(this);
  runBenchmarkAction->setIcon(DIcon::Dashboard());
  connect(runBenchmarkAction, SIGNAL(triggered()), this, SLOT(runBenchmarkActionTriggered()));
}

void SQLQuery::viewHistoryActionTriggered()
{
  statementsDialog->execSlot(serverConnection->executedQueries);
}

void SQLQuery::showNewLinesActionTriggered(bool triggered)
{
  settings.setValue("SQLQuery/ReplaceNewLines", triggered);
}

void SQLQuery::splitActionTriggered(bool triggered)
{
  settings.setValue("SQLQuery/SplitQueries", triggered);
}

void SQLQuery::startSQLPlayerActionTriggered()
{
  queriesToBePlayed = QStringList(statement().split(QRegularExpression("\\" + comboDelimiter->currentText() + "\\s+"), Qt::SkipEmptyParts));
  nextQueryToExecute = 0;
  emit enableDisableAction();
  timeLabel->setText("\n\n\n");
  dialogQueryPlayer->show();
  emit executionStarted(0);
}

void SQLQuery::queryPlayerRunQueryActionTriggered()
{
  threadElapsedSecondsTimer->start();
  threadQueryRunQueryAction->start();
}

void SQLQuery::queryPlayerStopActionTriggered()
{
  queryPlayerRunQueryAction->setChecked(false);
  queryPlayerRunQueryAction->setIcon(DIcon::MediaPlaybackStart());
  threadQueryRunQueryAction->quit();
  threadElapsedSecondsTimer->quit();
  elapsedSeconds = 0;
  dialogQueryPlayer->close();
}

void SQLQuery::queryPlayerRunPreviousQueryActionTriggered()
{
  checkTablesExecuteTheCheck(queryToBePlayed(--nextQueryToExecute));
  emit enableDisableAction();
}

QString SQLQuery::queryToBePlayed(int position)
{
  if ((position < queriesToBePlayed.count()) && (position >= 0)) {
    return queriesToBePlayed.at(position);
  } else {
    queryPlayerRunQueryAction->setChecked(false);
    queryPlayerRunQueryAction->setIcon(DIcon::MediaPlaybackStart());
  }
  if (position == queriesToBePlayed.count())
    return queriesToBePlayed.last();
  return QString();
}

void SQLQuery::queryPlayerRunNextQueryActionTriggered()
{
  checkTablesExecuteTheCheck(queryToBePlayed(++nextQueryToExecute));
  emit enableDisableAction();
}

void SQLQuery::enableDisableActionSlot()
{
  if (nextQueryToExecute <= 0) {
    queryPlayerRunPreviousQueryAction->setEnabled(false);
    queryPlayerStopAction->setEnabled(false);
    queryPlayerRunFirstQueryAction->setEnabled(false);
    emit executionStarted(0);
  } else {
    queryPlayerRunPreviousQueryAction->setEnabled(true);
    queryPlayerStopAction->setEnabled(true);
    queryPlayerRunFirstQueryAction->setEnabled(true);
  }
  if (nextQueryToExecute >= queriesToBePlayed.count() - 1) {
    queryPlayerRunNextQueryAction->setEnabled(false);
    queryPlayerRunQueryAction->setEnabled(false);
    queryPlayerRunLastQueryAction->setEnabled(false);
    emit executionFinished(100);
  } else {
    queryPlayerRunNextQueryAction->setEnabled(true);
    queryPlayerRunQueryAction->setEnabled(true);
    queryPlayerRunLastQueryAction->setEnabled(true);
  }
}

void SQLQuery::queryPlayerRunFirstQueryActionTriggered()
{
  nextQueryToExecute = 0;
  checkTablesExecuteTheCheck(queryToBePlayed(nextQueryToExecute));
  emit enableDisableAction();
}

void SQLQuery::queryPlayerRunLastQueryActionTriggered()
{
  nextQueryToExecute =  queriesToBePlayed.count() - 1;
  checkTablesExecuteTheCheck(queryToBePlayed(nextQueryToExecute));
  emit enableDisableAction();
}

QString SQLQuery::statement()
{
  if (scriptEditor->textEditor->textCursor().hasSelection()) {
    return scriptEditor->textEditor->textCursor().selection().toPlainText().trimmed();
  } else {
    return scriptEditor->textEditor->toPlainText().trimmed();
  }
}

void SQLQuery::executeStatement(QString statement)
{
  if (!statement.isEmpty()) {
    if (!scriptEditor->textEditor->textCursor().hasSelection()) {
      if (QMessageBox::question(this, tr("Script exectution"),
                                tr("Do you really want to execute the hole script?"),
                                QMessageBox::Ok | QMessageBox::Cancel, QMessageBox::Cancel) == QMessageBox::Cancel) {
        QApplication::restoreOverrideCursor();
        return;
      }
    }
    QApplication::setOverrideCursor(Qt::WaitCursor);
    if (statement.trimmed().startsWith("DELIMITER", Qt::CaseInsensitive)) {
      changeDelimiter(statement);
//      QString delimiter = statement.split(QRegularExpression("( |\\t|\\r|\\n)"), Qt::SkipEmptyParts).at(1);
//      delimiter = delimiter.trimmed();
//      if (comboDelimiter->findText(delimiter) < 0)
//        comboDelimiter->addItem(delimiter);
//      comboDelimiter->setCurrentText(delimiter);
//      statement.remove(statement.indexOf("DELIMITER", 0, Qt::CaseInsensitive), 9);
//      statement = statement.trimmed();
//      statement.remove(statement.indexOf(delimiter, Qt::CaseInsensitive), delimiter.length());
//      statement = statement.trimmed();
//      if (statement.isEmpty()) {
//        QApplication::restoreOverrideCursor();
//        return;
//      }
    }
    static QRegularExpression re("(ALTER|CHANGE|CREATE|DELETE|DROP|GRANT|LOAD|RENAME|SET|START|STOP|TRUNCATE|UPDATE)", QRegularExpression::CaseInsensitiveOption);
    if (statement.contains(re)
        && safeStatementsAction->isChecked()) {
      QString message(tr("Could not execute statement on safe mode."));
      QMessageBox::warning(this, tr("Safe mode"), message);
      resultEditor->setPlainText(message + "\n" + statement);
      QApplication::restoreOverrideCursor();
      return;
    }
    settings.sync();
    serverConnection->executedQueries.append(statement);
    if (settings.value("GeneralSettings/SaveQueriesBeforeExecution", true).toBool())
      settings.setValue("SQLQuery/LastQuery-" + qApp->property("ConnectionName").toString() + "-" + windowTitle(), scriptEditor->textEditor->toPlainText().trimmed());
    if (!concatenateOutputAction->isChecked())
      resultEditor->clear();
    if (comboOutput->currentData() == "-t")
      resultEditor->setPlainText(serverConnection->outputAsTable(statement, false, exportAction->isChecked()
                                                                 , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                                 , true, comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-A")
      resultEditor->setPlainText(serverConnection->outputAsTable(statement, false, exportAction->isChecked()
                                                                 , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                                 , true, comboDelimiter->currentText(), true)
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-x")
      resultEditor->setPlainText(serverConnection->outputAsV(statement, false, exportAction->isChecked()
                                                             , showNewLinesAction->isChecked(), splitAction->isChecked(), true
                                                             , comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-v")
      resultEditor->setPlainText(serverConnection->outputAsV(statement, false, exportAction->isChecked()
                                                             , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                             , false, comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-vv")
      resultEditor->setPlainText(serverConnection->outputAsVV(statement, exportAction->isChecked()
                                                              , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                              , comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-vvv")
      resultEditor->setPlainText(serverConnection->outputAsTable(statement, true, exportAction->isChecked()
                                                                 , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                                 , true, comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-HTML")
      resultEditor->setPlainText(serverConnection->outputAsHTML(statement, exportAction->isChecked()
                                                                , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                                , comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-TXT")
      resultEditor->setPlainText(serverConnection->outputAsTable(statement, false, exportAction->isChecked()
                                                                 , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                                 , true, comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-XML")
      resultEditor->setPlainText(serverConnection->outputAsXML(statement, exportAction->isChecked()
                                                               , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                               , comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-G")
      resultEditor->setPlainText(serverConnection->outputAsG(statement, exportAction->isChecked()
                                                             , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                             , true, comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-J")
      resultEditor->setPlainText(serverConnection->outputAsJ(statement, exportAction->isChecked()
                                                             , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                             , true, comboDelimiter->currentText())
                                 , concatenateOutputAction->isChecked());
    if (comboOutput->currentData() == "-PDF") {
      resultEditor->setPlainText(serverConnection->outputAsTable(statement, false, false, showNewLinesAction->isChecked(), splitAction->isChecked(), true, comboDelimiter->currentText()));
      QFileDialog fileDialog;
      fileDialog.setDirectory(QDir::home());
      QString file = fileDialog.getSaveFileName(this, tr("Save to Pdf"), settings.value("GeneralSettings/LastFilePdf", QDir::home().absolutePath()).toString(), tr("Pdf & Ps files (*.pdf *.ps)"));
      settings.setValue("GeneralSettings/LastFilePdf", fileDialog.directory().filePath(file));
      QPrinter printer(QPrinter::HighResolution);
      printer.setOutputFileName(file);
      printer.setOutputFormat(file.endsWith(".pdf") ? QPrinter::PdfFormat : QPrinter::NativeFormat);
      resultEditor->document()->print(&printer);
      emit statusBarMessage(tr("File saved at: %1").arg(file));
    }
    if (comboOutput->currentData() == "DTableWidget") {
      dTableWidgetResult->clear();
      dTableWidgetResult->setTableName(QString());
      std::unique_ptr<QList<QStringList>> rows;
      static QRegularExpression re("SELECT \\* FROM " + StaticFunctions::identifierPattern() + "." + StaticFunctions::identifierPattern(), QRegularExpression::CaseInsensitiveOption);
      if (statement.contains(re)) {
        dTableWidgetResult->setPasteActionEnabled(true);
        rows.reset(serverConnection->runQuery(statement, true, false));
        if (rows) {
          dTableWidgetResult->setHeaders(rows->takeFirst());
          rows->takeLast();
        }
        static QRegularExpression expression(StaticFunctions::identifierPattern() + "." + StaticFunctions::identifierPattern());
        QRegularExpressionMatch regularExpressionMatch = expression.match(statement);
        if (regularExpressionMatch.hasMatch()) {
          dTableWidgetResult->setTableName(regularExpressionMatch.captured(0));
        } else {
          QMessageBox::information(this, "DTableWidget", tr("Error determining the table name."));
        }
      } else {
        QMessageBox::information(this, "DTableWidget", tr("This option is only available fully qualified table names on SELECT * FROM `Database`.`Table` queries. Paste action will be disabled."));
        dTableWidgetResult->setPasteActionEnabled(false);
      }
      resultEditor->setVisible(false);
      dTableWidgetResult->setVisible(true);
      if (rows) {
        dTableWidgetResult->fillTable(rows.get());
      }
    } else {
      resultEditor->setVisible(true);
      dTableWidgetResult->setVisible(false);
    }
    if (logStatements) //Use a variable here because is faster
      serverConnection->logStatement(statement, resultEditor->toPlainText());
    //resultEditor->setPlainText(statement);
    QApplication::restoreOverrideCursor();
  }
}

void SQLQuery::explainSELECT(bool withAlias)
{
  QTextBlock block = scriptEditor->textEditor->textCursor().block();
  QTextCursor cursor = scriptEditor->textEditor->textCursor();
  bool showErrorHint = false;
  QRegularExpression expression;
  QRegularExpressionMatch regularExpressionMatch;
  QString database;
  QString table;
  if (block.text().startsWith("SELECT * FROM")) {
    expression.setPattern(StaticFunctions::fullidentifierPattern());
    regularExpressionMatch = expression.match(block.text());
    if (regularExpressionMatch.hasMatch()) {
      QStringList list(regularExpressionMatch.captured(0).split("."));
      database = list.at(0).mid(1, list.at(0).length() - 2);
      table = list.at(1).mid(1, list.at(1).length() - 2);
    } else {
      expression.setPattern(StaticFunctions::identifierPattern());
      regularExpressionMatch = expression.match(block.text());
      if (regularExpressionMatch.hasMatch()) {
        database = serverConnection->getDatabase();
        table = regularExpressionMatch.captured(0).mid(1, regularExpressionMatch.captured(0).length() - 2);
      } else {
        showErrorHint = true;
      }
    }
    QString queryString("SELECT `COLUMN_NAME` FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + database + "' AND `TABLE_NAME` = '" + table + "'");
    QString outPut;
    foreach (QString column, serverConnection->runQuerySingleColumn(queryString))
      outPut += (withAlias ? "`a`.`" : "`") + column + "`, ";
    cursor.movePosition(QTextCursor::StartOfBlock);
    cursor.movePosition(QTextCursor::NextCharacter, QTextCursor::KeepAnchor, 8);
    cursor.insertText("SELECT " + outPut.mid(0, outPut.length() - 2));
  } else {
    showErrorHint = true;
  }
  if (showErrorHint)
    resultEditor->setPlainText(tr("Incorrect use of the EXPLAIN SELECT option. Example: SELECT * FROM `mysql`.`columns_priv`. The line must starts with 'SELECT * FROM' and use the ` symbol for table name and database name."));
}

void SQLQuery::checkTablesExecuteTheCheck(QString statement)
{
  resultEditor->setPlainText(serverConnection->outputAsTable(statement, true, false, false, false, true, ""), true);
}

void SQLQuery::executeActionTriggered()
{
  if (statement().startsWith("EDIT ", Qt::CaseInsensitive)) {
    QString stmnt(statement());
    QString where;
    QString database;
    QString table;
    int indexOfWhere = stmnt.indexOf(" WHERE ", 0, Qt::CaseInsensitive);
    table = stmnt.mid(5, indexOfWhere - 5);
    if (table.contains(".")) {
      QStringList list = table.split(".");
      database = list.at(0);
      table = list.at(1);
    }
    if (indexOfWhere > 1) {
      indexOfWhere += 7;
      where = stmnt.mid(indexOfWhere, stmnt.length() - indexOfWhere);
      if (where.endsWith(";"))
        where = where.mid(0, where.length() - 1);
    }
    emit showResultTab(table, database, where);
  } else {
    emit busy(true);
    queryPlayerStopBecauseAnError = false;
    executeStatement(statement());
    emit busy(false);
  }
}

void SQLQuery::executeActionOnAThreadTriggered()
{
  ThreadedQueryExecution *threadedQueryExecution = new ThreadedQueryExecution(this->serverConnection->getConnectionParameters(), statement());
  threadedQueryExecution->setOnlyExecuteSafeStatements(safeStatementsAction->isChecked());
  threadedQueryExecution->setOutputType(comboOutput->currentData().toString());
  threadedQueryExecution->setSaveToFile(exportAction->isChecked());
  threadedQueryExecution->setShowNewLines(showNewLinesAction->isChecked());
  connect(threadedQueryExecution, SIGNAL(stringResultReady(QString)), this, SLOT(handleResultsOfThreadedQueryExecution(QString)));
  connect(threadedQueryExecution, SIGNAL(finished()), this, SLOT(handleFinishedOfThreadedQueryExecution()));
  threadedQueryExecution->start();
  runningSnakeMovie->start();
}

void SQLQuery::executeExternalQueryActionTriggered()
{
  DBMS *externalServerConnection = new DBMS(true);
  QMap<QString, QVariant> params = settings.value("ServerConnections/" + externalConnectionName->currentText()).toMap();
  externalServerConnection->setConnectionParameters(params);
  if (params.value("ConnectionType") == "--")
    externalServerConnection->setDBMSType(StaticFunctions::Undefined);
  if (params.value("ConnectionType")  == "MySQL")
    externalServerConnection->setDBMSType(StaticFunctions::MySQL);
  if (params.value("ConnectionType")  == "MariaDB")
    externalServerConnection->setDBMSType(StaticFunctions::MariaDB);
  if (!externalServerConnection->open())
    QMessageBox::critical(this, tr("Cannot connect to the server"), externalServerConnection->lastError());

  QString statement;
  if (scriptEditor->textEditor->textCursor().hasSelection()) {
    statement = scriptEditor->textEditor->textCursor().selection().toPlainText().trimmed();
  } else {
    if (QMessageBox::question(this, tr("Script exectution"),
                              tr("Do you really want to execute the hole script?"),
                              QMessageBox::Ok | QMessageBox::Cancel, QMessageBox::Cancel) == QMessageBox::Cancel) {
      return;
    } else {
      statement = scriptEditor->textEditor->toPlainText().trimmed();
    }
  }
  resultEditor->setPlainText(tr("Execution on: ") + params.value("Host").toString() + "\n"
                             + externalServerConnection->outputAsTable(statement, true, false
                                                                     , showNewLinesAction->isChecked(), splitAction->isChecked()
                                                                     , true, comboDelimiter->currentText())
                             , concatenateOutputAction->isChecked());
  externalServerConnection->close();
}

void SQLQuery::setScriptText(QString text)
{
  scriptEditor->textEditor->setPlainText(text);
  scriptEditor->textEditor->setFocus(Qt::OtherFocusReason);
}

void SQLQuery::fillMariaDBSymbols()
{
  scriptEditor->fillMariaDBSymbolsActionSlot();
}

QToolBar *SQLQuery::getToolBar()
{
  return queryToolBar;
}

void SQLQuery::exportTableDataForInsertActionTriggered()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QTextBlock block = scriptEditor->textEditor->textCursor().block();
//  QTextCursor cursor = scriptEditor->textEditor->textCursor();
  bool showErrorHint = false;
  QRegularExpression expression;
  QRegularExpressionMatch regularExpressionMatch;
  QString database;
  QString table;
  if (block.text().startsWith("EXPORT DATA FOR INSERT")) {
    expression.setPattern(StaticFunctions::fullidentifierPattern());
    regularExpressionMatch = expression.match(block.text());
    if (regularExpressionMatch.hasMatch()) {
      QStringList list(regularExpressionMatch.captured(0).split("."));
      database = list.at(0).mid(1, list.at(0).length() - 2);
      table = list.at(1).mid(1, list.at(1).length() - 2);
    } else {
      expression.setPattern(StaticFunctions::identifierPattern());
      regularExpressionMatch = expression.match(block.text());
      if (regularExpressionMatch.hasMatch()) {
        database = serverConnection->getDatabase();
        table = regularExpressionMatch.captured(0).mid(1, regularExpressionMatch.captured(0).length() - 2);
      } else {
        showErrorHint = true;
      }
    }
    QString fields;
    QString queryString("SELECT `COLUMN_NAME` FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + database + "' AND `TABLE_NAME` = '" + table + "'");
    foreach (QString column, serverConnection->runQuerySingleColumn(queryString))
      fields += "`" + column + "`, ";
    fields = fields.mid(0, fields.length() - 2);
    QList<QStringList> *rows = serverConnection->runQuery("SELECT * FROM `" + database + "`.`" + table + "`");
    QString rowTMP;
    QString outPut;
    QString rowData("INSERT INTO `" + database + "`.`" + table + "` (" + fields + ") VALUES (");
    for (int row = 0; row < rows->count() - 1; row++) {
      for (int row2 = 0; row2 < rows->at(row).count(); row2++) {
        rowTMP += "'" + rows->at(row).at(row2) + "', ";
      }
      outPut += rowData + rowTMP.mid(0, rowTMP.length() - 2) + ");\n";
      rowTMP = "";
    }
    if (comboOutput->currentData() == "-TXT") {
      QString fileName = QFileDialog::getSaveFileName(this, tr("Select a file"), settings.value("GeneralSettings/LastSQLFile", "").toString(), "SQL Files (*.sql)");
      QFile file(fileName);
      if (!file.open(QFile::WriteOnly | QFile::Text))
        emit statusBarMessage(tr("Cannot write file %1:\n%2.").arg(fileName, file.errorString()));
      QTextStream out(&file);
      out << outPut.toUtf8();
      file.close();
    } else {
      resultEditor->setPlainText(resultEditor->toPlainText() + "\n" + outPut);
    }
    if (exportAction->isChecked()) {
      serverConnection->saveOutputToFile(resultEditor->toPlainText(), "Text Files (*.txt)", settings.value("GeneralSettings/LastTextFile", "").toString());
    }
  } else {
    showErrorHint = true;
  }
  if (showErrorHint)
    resultEditor->setPlainText(tr("Incorrect use of the EXPORT DATA FOR INSERT option. Example: EXPORT DATA FOR INSERT `mysql`.`columns_priv`. The line must starts with 'EXPORT DATA FOR INSERT' and use the ` symbol for table name and database name."));
  QApplication::restoreOverrideCursor();
}

void SQLQuery::exportResultDataForInsertActionTriggered()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QTextBlock block = scriptEditor->textEditor->textCursor().block();
//  QTextCursor cursor = scriptEditor->textEditor->textCursor();
  bool showErrorHint = false;
  QRegularExpression expression;
  QRegularExpressionMatch regularExpressionMatch;
  QString database;
  QString table;
  if (block.text().startsWith("EXPORT RESULT FOR INSERT SELECT * FROM")) {
    expression.setPattern(StaticFunctions::fullidentifierPattern());
    regularExpressionMatch = expression.match(block.text());
    if (regularExpressionMatch.hasMatch()) {
      QStringList list(regularExpressionMatch.captured(0).split("."));
      database = list.at(0).mid(1, list.at(0).length() - 2);
      table = list.at(1).mid(1, list.at(1).length() - 2);
    } else {
      expression.setPattern(StaticFunctions::identifierPattern());
      regularExpressionMatch = expression.match(block.text());
      if (regularExpressionMatch.hasMatch()) {
        database = serverConnection->getDatabase();
        table = regularExpressionMatch.captured(0).mid(1, regularExpressionMatch.captured(0).length() - 2);
      } else {
        showErrorHint = true;
      }
    }
    QString fields;
    QString queryString("SELECT `COLUMN_NAME` FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + database + "' AND `TABLE_NAME` = '" + table + "'");
    foreach (QString column, serverConnection->runQuerySingleColumn(queryString))
      fields += "`" + column + "`, ";
    fields = fields.mid(0, fields.length() - 2);
    QList<QStringList> *rows = serverConnection->runQuery(block.text().mid(25));
    QString rowTMP;
    QString outPut;
    QString rowData("INSERT INTO `" + database + "`.`" + table + "` (" + fields + ") VALUES (");
    for (int row = 0; row < rows->count() - 1; row++) {
      for (int row2 = 0; row2 < rows->at(row).count(); row2++) {
        rowTMP += "'" + rows->at(row).at(row2) + "', ";
      }
      outPut += rowData + rowTMP.mid(0, rowTMP.length() - 2) + ");\n";
      rowTMP = "";
    }
    if (comboOutput->currentData() == "-TXT") {
      QString fileName = QFileDialog::getSaveFileName(this, tr("Select a file"), settings.value("GeneralSettings/LastSQLFile", "").toString(), "SQL Files (*.sql)");
      QFile file(fileName);
      if (!file.open(QFile::WriteOnly | QFile::Text))
        emit statusBarMessage(tr("Cannot write file %1:\n%2.").arg(fileName, file.errorString()));
      QTextStream out(&file);
      out << outPut.toUtf8();
      file.close();
    } else {
      resultEditor->setPlainText(resultEditor->toPlainText() + "\n" + outPut);
    }
    if (exportAction->isChecked()) {
      serverConnection->saveOutputToFile(resultEditor->toPlainText(), "Text Files (*.txt)", settings.value("GeneralSettings/LastTextFile", "").toString());
    }
  } else {
    showErrorHint = true;
  }
  if (showErrorHint)
    resultEditor->setPlainText(tr("Incorrect use of the EXPORT RESULT FOR INSERT option. Example: EXPORT RESULT FOR INSERT SELECT * FROM `mysql`.`columns_priv`. The line must starts with 'EXPORT RESULT FOR INSERT SELECT * FROM' and use the ` symbol for table name and database name."));
  QApplication::restoreOverrideCursor();
}

void SQLQuery::explainSelectActionTriggered()
{
  explainSELECT(false);
}

void SQLQuery::explainInsertActionTriggered()
{
  QTextBlock block = scriptEditor->textEditor->textCursor().block();
  QTextCursor cursor = scriptEditor->textEditor->textCursor();
  bool showErrorHint = false;
  QRegularExpression expression;
  QRegularExpressionMatch regularExpressionMatch;
  QString database;
  QString table;
  if (block.text().startsWith("INSERT INTO")) {
    expression.setPattern(StaticFunctions::fullidentifierPattern());
    regularExpressionMatch = expression.match(block.text());
    if (regularExpressionMatch.hasMatch()) {
      QStringList list(regularExpressionMatch.captured(0).split("."));
      database = list.at(0).mid(1, list.at(0).length() - 2);
      table = list.at(1).mid(1, list.at(1).length() - 2);
    } else {
      expression.setPattern(StaticFunctions::identifierPattern());
      regularExpressionMatch = expression.match(block.text());
      if (regularExpressionMatch.hasMatch()) {
        database = serverConnection->getDatabase();
        table = regularExpressionMatch.captured(0).mid(1, regularExpressionMatch.captured(0).length() - 2);
      } else {
        showErrorHint = true;
      }
    }
    QString queryString("SELECT `COLUMN_NAME` FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + database + "' AND `TABLE_NAME` = '" + table + "'");
    QString outPut;
    foreach (QString column, serverConnection->runQuerySingleColumn(queryString))
      outPut += "TRIM('" + column + "'), ";
    cursor.movePosition(QTextCursor::EndOfBlock);
    cursor.insertText(" VALUES (" + outPut.mid(0, outPut.length() - 2) + ");");
  } else {
    showErrorHint = true;
  }
  if (showErrorHint)
    resultEditor->setPlainText(tr("Incorrect use of the EXPLAIN INSERT option. Example: INSERT INTO `mydatabase`.`mytable`. The line must starts with 'INSERT INTO' and use the ` symbol for table name and database name."));

}

void SQLQuery::safeStatementsActionTriggered(bool triggered)
{
  settings.setValue("SQLQuery/ExecuteSafeStatements-" + qApp->property("ConnectionName").toString(), triggered);
}

void SQLQuery::incrementSeconds()
{
  elapsedSeconds++;
  float elapsedTime = elapsedSeconds;
  QString timeUnit(tr("seconds"));
  QString timeUnit2 = timeUnit;
  if (elapsedSeconds > 60) {
    elapsedTime /= 60 ;
    timeUnit = tr("minutes");
  }
  float remaniningTime = (queriesToBePlayed.count() - nextQueryToExecute) * (elapsedSeconds / nextQueryToExecute);
  if (remaniningTime > 60) {
    remaniningTime /= 60;
    timeUnit2 = tr("minutes");
  }
  timeLabel->setText(tr("Elapsed: %1 %2. Remaining: %3 %4.\nTotal of queries: %5, remaining: %6.\nQueries per second: %7.")
                     .arg(elapsedTime)
                     .arg(timeUnit)
                     .arg(remaniningTime)
                     .arg(timeUnit2)
                     .arg(queriesToBePlayed.count())
                     .arg(queriesToBePlayed.count() - nextQueryToExecute)
                     .arg(nextQueryToExecute / elapsedSeconds));
  if (nextQueryToExecute >= queriesToBePlayed.count())
    elapsedSecondsTimer->stop();
}

void SQLQuery::runQueryActionTriggered()
{
  if (queryPlayerRunQueryAction->isChecked()) {
    queryPlayerRunQueryAction->setIcon(DIcon::MediaPause());
  } else {
    queryPlayerRunQueryAction->setIcon(DIcon::MediaPlaybackStart());
  }
  if (!queryPlayerStopBecauseAnError)
    checkTablesExecuteTheCheck(queryToBePlayed(nextQueryToExecute++));
  emit enableDisableAction();
  if (queryPlayerRunQueryAction->isChecked() && nextQueryToExecute < queriesToBePlayed.count())
    QTimer::singleShot(0, this, SLOT(runQueryActionTriggered()));
  emit executionProgress(nextQueryToExecute * 100 / queriesToBePlayed.count());
}

void SQLQuery::showStatementsErrorActionTriggered()
{
  resultEditor->setPlainText(serverConnection->getfailedQueries());
}

void SQLQuery::errorAcceptedSlot()
{
  queryPlayerStopBecauseAnError = false;
}

void SQLQuery::errorOccurredSlot()
{
  queryPlayerStopBecauseAnError = true;
}

void SQLQuery::explainUpdateActionTriggered()
{
  QTextBlock block = scriptEditor->textEditor->textCursor().block();
  QTextCursor cursor = scriptEditor->textEditor->textCursor();
  bool showErrorHint = false;
  QRegularExpression expression;
  QRegularExpressionMatch regularExpressionMatch;
  QString database;
  QString table;
  if (block.text().startsWith("UPDATE")) {
    expression.setPattern(StaticFunctions::fullidentifierPattern());
    regularExpressionMatch = expression.match(block.text());
    if (regularExpressionMatch.hasMatch()) {
      QStringList list(regularExpressionMatch.captured(0).split("."));
      database = list.at(0).mid(1, list.at(0).length() - 2);
      table = list.at(1).mid(1, list.at(1).length() - 2);
    } else {
      expression.setPattern(StaticFunctions::identifierPattern());
      regularExpressionMatch = expression.match(block.text());
      if (regularExpressionMatch.hasMatch()) {
        database = serverConnection->getDatabase();
        table = regularExpressionMatch.captured(0).mid(1, regularExpressionMatch.captured(0).length() - 2);
      } else {
        showErrorHint = true;
      }
    }
    QString queryString("SELECT `COLUMN_NAME` FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + database + "' AND `TABLE_NAME` = '" + table + "' AND `COLUMN_KEY` = 'PRI'");
    QString columns("SELECT REPLACE(GROUP_CONCAT(CONCAT('`', `COLUMN_NAME`, \"` = TRIM('\", `COLUMN_NAME`, \"')\")), ',', ', ') FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + database + "' AND `TABLE_NAME` = '" + table + "' AND `COLUMN_KEY` <> 'PRI'");
    QString outPut(" SET " + serverConnection->runQuerySingleColumn(columns).at(0) + " WHERE ");
    foreach (QString column, serverConnection->runQuerySingleColumn(queryString))
      outPut += "`" + column + "` = TRIM('" + column + "') AND ";
    cursor.movePosition(QTextCursor::EndOfBlock);
    cursor.insertText(outPut.mid(0, outPut.length() - 5) + ";");
  } else {
    showErrorHint = true;
  }
  if (showErrorHint)
    resultEditor->setPlainText(tr("Incorrect use of the EXPLAIN UPDATE option. Example: UPDATE `mysql`.`columns_priv`. The line must starts with 'UPDATE' and use the ` symbol for table name and database name."));
}

void SQLQuery::repeatQueryExecutionActionTriggered()
{
  scriptEditor->setEnabled(!repeatQueryExecutionAction->isChecked());
  if (repeatQueryExecutionAction->isChecked()) {
    if (repeatQueryExecutionTime == 0)
      repeatQueryExecutionTime = QInputDialog::getInt(this, tr("Repeat time in seconds"), tr("Repeat time in seconds"), 1, 1);
    executeActionTriggered();
    QTimer::singleShot(repeatQueryExecutionTime * 1000, this, SLOT(repeatQueryExecutionActionTriggered()));
  } else {
    repeatQueryExecutionTime = 0;
  }
}

void SQLQuery::wordWrapOnResultActionToggled()
{
  resultEditor->setWordWrapMode(wordWrapOnResultAction->isChecked() ? QTextOption::WordWrap : QTextOption::NoWrap);
  settings.setValue("SQLQuery/WordWrapOnResul", wordWrapOnResultAction->isChecked());
}

void SQLQuery::loadLastQuery()
{
  scriptEditor->textEditor->setPlainText(settings.value("SQLQuery/LastQuery-" + qApp->property("ConnectionName").toString() + "-" + windowTitle(), "").toString());
}

void SQLQuery::logStatementsActionToggled()
{
  settings.setValue("SQLQuery/LogStatements", logStatementsAction->isChecked());
  logStatements = logStatementsAction->isChecked();
}

void SQLQuery::trimColumnsActionTriggered()
{
//  QTextCursor cursor = scriptEditor->textEditor->textCursor();
//  QTextDocument *doc = scriptEditor->textEditor->document();
//  QTextBlock block = doc->findBlock(qMin(cursor.anchor(), cursor.position()));
//  if (block.text().startsWith("TRIM COLUMNS IN `")) {
//    QRegularExpression expression("StaticFunctions::identifierPattern()");
//    QRegularExpressionMatch regularExpressionMatch = expression.match(block.text());
//    QString queryString("SELECT `COLUMN_NAME` FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + serverConnection->getDatabase() + "' AND `TABLE_NAME` = '" + regularExpressionMatch.captured(0).mid(1, regularExpressionMatch.captured(0).length() - 2) + "' AND `DATA_TYPE` IN ('char', 'varchar', 'text', 'mediumtext', 'longtext', 'tinytext')");
//    QString outPut("UPDATE `" + regularExpressionMatch.captured(0).mid(1, regularExpressionMatch.captured(0).length() - 2) + "` SET ");
//    foreach (QString column, serverConnection->runQuerySingleColumn(queryString))
//      outPut += "`" + column + "` = TRIM(`" + column + "`), ";
//    resultEditor->setPlainText(outPut.mid(0, outPut.length() - 2) + ";");
//  } else {
//    resultEditor->setPlainText(tr("Incorrect use of the TRIM columns option. Example: TRIM COLUMNS IN `columns_priv`, it only works for the current database."));
//  }
  QTextBlock block = scriptEditor->textEditor->textCursor().block();
  QTextCursor cursor = scriptEditor->textEditor->textCursor();
  bool showErrorHint = false;
  QRegularExpression expression;
  QRegularExpressionMatch regularExpressionMatch;
  QString database;
  QString table;
  if (block.text().startsWith("TRIM COLUMNS IN")) {
    expression.setPattern(StaticFunctions::fullidentifierPattern());
    regularExpressionMatch = expression.match(block.text());
    if (regularExpressionMatch.hasMatch()) {
      QStringList list(regularExpressionMatch.captured(0).split("."));
      database = list.at(0).mid(1, list.at(0).length() - 2);
      table = list.at(1).mid(1, list.at(1).length() - 2);
    } else {
      expression.setPattern("StaticFunctions::identifierPattern()");
      regularExpressionMatch = expression.match(block.text());
      if (regularExpressionMatch.hasMatch()) {
        database = serverConnection->getDatabase();
        table = regularExpressionMatch.captured(0).mid(1, regularExpressionMatch.captured(0).length() - 2);
      } else {
        showErrorHint = true;
      }
    }
    QString queryString("SELECT `COLUMN_NAME` FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + database + "' AND `TABLE_NAME` = '" + table + "'");
    QString outPut("UPDATE `" + database + "`.`" + table + "` SET ");
    foreach (QString column, serverConnection->runQuerySingleColumn(queryString))
      outPut += "`" + column + "` = TRIM(`" + column + "`), ";
    cursor.movePosition(QTextCursor::StartOfBlock);
    cursor.movePosition(QTextCursor::NextWord, QTextCursor::KeepAnchor, 9);
    cursor.insertText(outPut.mid(0, outPut.length() - 2) + ";");
  } else {
    showErrorHint = true;
  }
  if (showErrorHint)
    resultEditor->setPlainText(tr("Incorrect use of the TRIM COLUMNS option. Example: TRIM COLUMNS IN `mysql`.`columns_priv`. The line must starts with 'TRIM COLUMNS IN' and use the ` symbol for table name and database name."));
}

void SQLQuery::checkTablesActionTriggered()
{
  nextQueryToExecute = 0;
  emit executionStarted(0);
  queriesToBePlayed.clear();
  concatenateOutputAction->setChecked(true);
  foreach (QString table, serverConnection->database()->getTables())
    queriesToBePlayed.append("SELECT * FROM `" + table + "` LIMIT 1");
  emit enableDisableAction();
  timeLabel->setText("\n\n\n");
  dialogQueryPlayer->show();
}

void SQLQuery::explainSelectActionWithAliasActionTriggered()
{
  explainSELECT(true);
}

void SQLQuery::emitUpdatePrositionViewer(const int x, const int y, const int z)
{
  emit updatePrositionViewer(x, y, z);
}

void SQLQuery::beginTransacctionActionTriggered()
{
  serverConnection->transaction()->beginTransacction();
  beginTransacctionAction->setChecked(true);
  beginTransacctionAction->setEnabled(false);
  commitTransacctionAction->setEnabled(true);
  rollbackTransacctionAction->setEnabled(true);
}

void SQLQuery::commitTransacctionActionTriggered()
{
  if (QMessageBox::question(this, tr("Commit transacction"), tr("Do you really want to commint changes?")
                            , QMessageBox::Ok | QMessageBox::Cancel, QMessageBox::Cancel) == QMessageBox::Ok)
  {
    serverConnection->transaction()->commitTransacction();
    beginTransacctionAction->setChecked(false);
    beginTransacctionAction->setEnabled(true);
    commitTransacctionAction->setEnabled(false);
    rollbackTransacctionAction->setEnabled(false);
  }
}

void SQLQuery::rollbackTransacctionActionTriggered()
{
  if (QMessageBox::question(this, tr("Rollback transacction"), tr("Do you really want to rollback changes?")
                            , QMessageBox::Ok | QMessageBox::Cancel, QMessageBox::Cancel) == QMessageBox::Ok)
  {
    serverConnection->transaction()->rollbackTransacction();
    beginTransacctionAction->setChecked(false);
    beginTransacctionAction->setEnabled(true);
    commitTransacctionAction->setEnabled(false);
    rollbackTransacctionAction->setEnabled(false);
  }
}

void SQLQuery::executeStatements(QStringList statements)
{
  for (int counter = 0; counter < statements.count(); counter++)
    serverConnection->executeQuery(statements.at(counter));
}

void SQLQuery::comboDelimiterEditTextChanged(const QString &text)
{
  if (comboDelimiter->findText(text) < 0)
    comboDelimiter->addItem(text);
}

void SQLQuery::changeDelimiter(QString delimiter)
{
  static QRegularExpression re("( |\\t|\\r|\\n)");
  QString delimiterText = delimiter.split(re, Qt::SkipEmptyParts).at(1);
  delimiterText = delimiterText.trimmed();
  if (comboDelimiter->findText(delimiterText) < 0)
    comboDelimiter->addItem(delimiterText);
  comboDelimiter->setCurrentText(delimiterText);

}

void SQLQuery::handleResultsOfThreadedQueryExecution(QString result)
{
  resultEditor->setPlainText(result, concatenateOutputAction->isChecked());
  serverConnection->insertOnExecutedQueries(qApp->property("SessionId").toString() + "-ThreadedExecution"
                                            , this->serverConnection->getConnectionString()
                                            , statement()
                                            , result
                                            );
}

void SQLQuery::handleFinishedOfThreadedQueryExecution()
{
  runningSnakeMovie->stop();
//  threadedQueryExecutionServerConnection->close();
}
