/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef SERVERINFORMATION_H
#define SERVERINFORMATION_H

#include "dsettings.h"

class QPlainTextEdit;
class QTabWidget;
class QPushButton;
class QSpinBox;
class QLabel;
class QFormLayout;
class QGroupBox;
class DTitleLabel;
class DLineEdit;
class DTableView;
class QComboBox;

#include "dmdisubwindow.h"
#include "dbms.h"

/**
 * @brief The DBarChartWidget class is a custom widget for rendering real-time server statistics.
 *
 * This widget draws simple bar charts to visualize metrics such as active connections,
 * kilobits sent, and queries executed per second.
 */
class DBarChartWidget : public QWidget
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DBarChartWidget.
   */
  DBarChartWidget();

  /**
   * @brief Adds a new set of data points to the chart history.
   * @param point Value for Connections.
   * @param point2 Value for Network Traffic (KB sent).
   * @param point3 Value for Queries per second.
   */
  void addPoints(double point, double point2, double point3);

protected:
  /**
   * @brief Handles the painting event to render the charts.
   * @param event The paint event parameters.
   */
  void paintEvent(QPaintEvent *event);

private:
  QList<double> pointsConnections;
  QList<double> pointsKbSent;
  QList<double> pointsExecutedQueries;

  /**
   * @brief Calculates the vertical pixel position for a bar based on its value.
   * @param recHeight The total height of the drawing area.
   * @param value The value of the specific data point.
   * @param maximunPoint The maximum value in the dataset (for scaling).
   * @return The Y coordinate.
   */
  inline float calculateYPosition(double recHeight, double value, double maximunPoint);

  /**
   * @brief Helper to draw text within a bounding rectangle.
   */
  inline QRect drawWrapText(QPainter &painter, const QRect &rectangle, const QString &text, int flags = Qt::AlignCenter);

  /**
   * @brief Helper to draw a specific graph area (title, borders, and bars).
   */
  inline QRect drawGraphicArea(QPainter &painter, QRect rect, const QString title, const QList<double> points);
};

/**
 * @brief The ServerInformation class provides a comprehensive dashboard of server health.
 *
 * This window includes multiple tabs for viewing:
 * - Global Variables and Status
 * - Replication Status (Slave/Master)
 * - Real-time Performance Graphs
 * - HDD/Table Usage
 * - Slow Query Logs
 */
class ServerInformation : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the ServerInformation window.
   * @param serverConnection Pointer to the active DBMS connection.
   */
  ServerInformation(DBMS *serverConnection);

  /**
   * @brief Programmatically switches the active information tab.
   * @param tabNumber The index of the tab to select.
   */
  void setCurrentTab(unsigned int tabNumber);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels, titles, and button text for language changes.
   */
  void retranslateUI();

private:
  DBMS *serverConnection;
  QPlainTextEdit *globalVariables;
  QPlainTextEdit *replicationStatus;
  QPlainTextEdit *serverStatus;
  QPlainTextEdit *serverGraphs;
  QPlainTextEdit *hddUsage;
  QString statementServerGraphs;

  // Network Traffic Tracking Variables
  qulonglong tBytesSent0;
  qulonglong tBytesSent1;
  qulonglong tBytesSent2;
  qulonglong tBytesSent3;
  qulonglong tBytesSent4;
  qulonglong tBytesSent5;
  qulonglong rateBytesSent;

  DBarChartWidget *graphicsWidget;
  QTabWidget *serverInformationTab;

  // Network Reception Tracking Variables
  qulonglong tBytesReceived0;
  qulonglong tBytesReceived1;
  qulonglong tBytesReceived2;
  qulonglong tBytesReceived3;
  qulonglong tBytesReceived4;
  qulonglong tBytesReceived5;
  qulonglong rateBytesReceived;

  // Graph Data Points
  double kbSentGraph1;
  double kbSentGraph2;
  double executedQueriesGraph1;
  double executedQueriesGraph2;

  /**
   * @brief Formats the server graph data as a text report (for MariaDB specific fields).
   * @return The formatted string.
   */
  QString serverGraphsDataTxtMariaDB();

  QTimer *timerServerGraphsDataTxt;
  QTimer *timerServerGraphsData;
  QTimer *timerReplicationStatusTxt;

  // Replication Control Buttons
  QPushButton *pushButtonSkip0Error;
  QPushButton *pushButtonSkip1Error;
  QPushButton *pushButtonSkip10Error;
  QPushButton *pushButtonSkip100Error;
  QPushButton *pushButtonSkip1000Error;
  QPushButton *pushButtonStopRefreshingReplicator;

  QPushButton *pushButtonStopRefreshingServerGraphicsTXT;
  QPushButton *pushButtonStopRefreshingServerGraphics;
  QPushButton *pushButtonRefreshServerStatus;

  /**
   * @brief Skips a specified number of replication errors (SQL_SLAVE_SKIP_COUNTER).
   * @param count The number of events to skip.
   */
  void skipReplicationErrors(int unsigned count = 0);

  QSpinBox *spinBoxTableSize;
  DSettings settings;

  // Header Labels
  QLabel *labelUser;
  QLabel *labelHost;
  QLabel *labelSocket;
  QLabel *labelPort;
  QLabel *labelVersion;
  QLabel *labelConnectionId;

  QFormLayout *formLayout;
  QGroupBox *informationGroupBox;
  QGroupBox *groupServer;
  DTitleLabel *dTitleLabelReplicationStatus;
  QGroupBox *buttonGroupReplication;
  DTitleLabel *dTitleLabelGlobalVariables;
  DTitleLabel *dTitleLabelServerGraphics;
  QGroupBox *buttonGroupServerGraphicsTXT;
  DTitleLabel *dTitleLabelServerGraphicsData;
  QGroupBox *buttonGroupServerGraphics;
  QGroupBox *buttonGroupHDDUsage;
  DTitleLabel *dTitleLabelHDDUsage;
  DTitleLabel *dTitleLabel;
  QGroupBox *buttonGroupVariables;
  QPushButton *pushButtonShowGlobalVariables;
  QPushButton *pushButtonShowGlobalStatus;
  QPushButton *pushButtonShowSessionStatus;
  QPushButton *pushButtonShowSessionVariables;
  QLabel *labelFilter;
  DLineEdit *lineEditFilter;
  QWidget *widgetServerGraphs2;
  QPushButton *pushButtonServerGraphicsFullScreen;
  QPushButton *pushButtonShowSlowQueries;
  QLabel *labelServerStatus;
  //DLineEdit *lineEditConnectioName;
  DTableView *slowQueriesDTableView;
  QList<QStringList> *result;
  QComboBox *slavesListComboBox;
  QSpinBox *spinBoxReplicationRefreshRate;
  QPushButton *pushButtonSwitchMasterSalve;

public slots:
  /**
   * @brief Refreshes the information for the selected tab.
   * @param tabIndex The index of the tab being viewed.
   */
  void showInformation(int tabIndex);

  /**
   * @brief Updates the textual server graph data.
   */
  void serverGraphsDataTxt();

  /**
   * @brief Updates the visual server graph data (bar charts).
   */
  void serverGraphsData();

  /**
   * @brief Updates the replication status text.
   */
  void replicationStatusTxt();

  /**
   * @brief Updates the HDD/Table size usage data.
   */
  void hddUsageData();

  /**
   * @brief Updates the general server status text.
   */
  void serverStatusTxt();

  /**
   * @brief Fetches and displays the slow query log.
   */
  void serverSlowQueriesTxt();

private slots:
  void pushButtonSkip0ErrorClicked();
  void pushButtonSkip1ErrorClicked();
  void pushButtonSkip10ErrorClicked();
  void pushButtonSkip100ErrorClicked();
  void pushButtonSkip1000ErrorClicked();

  /**
   * @brief Updates the HDD view when the limit of tables to show changes.
   * @param value The number of tables to display.
   */
  void spinBoxTableSizeValueChanged(int value);

  void pushButtonShowGlobalVariablesSlot();
  void pushButtonShowGlobalStatusSlot();
  void pushButtonShowSessionStatusSlot();
  void pushButtonShowSessionVariablesSlot();

  /**
   * @brief Filters the variable/status lists based on user input.
   * @param filter The text to filter by.
   */
  void lineEditFilterTextChangedSlot(QString filter);

  /**
   * @brief Toggles full-screen mode for the graphs.
   * @param checked True if full screen.
   */
  void pushButtonServerGraphicsFullScreenSlot(bool checked = false);

  //void lineEditConnectioNameClicked();

  /**
   * @brief Changes the master connection being monitored.
   * @param index The index of the selected connection in the combo box.
   */
  void changeDefaultMasterConnection(int index);

  /**
   * @brief Updates the refresh rate for replication status monitoring.
   * @param value The refresh interval in seconds.
   */
  void replicationRefreshRateSlot(const int value);

  /**
   * @brief Toggles the replication role (Master <-> Slave) if supported.
   */
  void pushButtonSwitchMasterSlaveClicked();
};

#endif // SERVERINFORMATION_H
