/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef RESULT_H
#define RESULT_H

#include <QVariant>

class DBMS;
class DTableView;
class QMenu;
class DTitleLabel;
class QToolBar;
class Statements;
class QStandardItem;

#include "dmdisubwindow.h"

/**
 * @brief The Result class displays and manages the result set of a database query or table view.
 *
 * This class inherits from DMdiSubWindow and provides an editable data grid (DTableView)
 * representing the contents of a specific database table. It tracks changes made by the user
 * (edits, deletions, insertions) and generates the corresponding SQL statements to apply
 * those changes back to the database.
 */
class Result : public DMdiSubWindow
{
  Q_OBJECT

//  Q_PROPERTY(QString table READ getTable WRITE setTable CONSTANT)
//  Q_PROPERTY(QString database READ getDatabase WRITE setDatabase CONSTANT)
//  Q_PROPERTY(QString where READ getWhere WRITE setWhere CONSTANT)

public:
  /**
   * @brief Constructs the Result window.
   * @param serverConnection Pointer to the active DBMS connection.
   * @param table The name of the table to display.
   * @param database The name of the database containing the table.
   * @param where An optional WHERE clause to filter the initial data.
   */
  Result(DBMS *serverConnection, QString table, QString database = QString(), QString where = QString());

  /**
   * @brief Gets the name of the current table.
   * @return The table name.
   */
  QString getTable();

  /**
   * @brief Gets the name of the current database.
   * @return The database name.
   */
  QString getDatabase();

  /**
   * @brief Gets the WHERE clause used for filtering.
   * @return The WHERE clause string.
   */
  QString getWhere();

  /**
   * @brief Sets the table to display.
   * @param tableName The name of the table.
   */
  void setTable(QString tableName = QString());

  /**
   * @brief Sets the database context.
   * @param databaseName The name of the database.
   */
  void setDatabase(QString databaseName = QString());

  /**
   * @brief Sets the filter condition.
   * @param where The SQL WHERE clause.
   */
  void setWhere(QString where = QString());

private:
  DBMS *serverConnection;
  DTableView *resultDTableView;
  QList<QStringList> *result;
  QAction *refreshAction;
  QAction *applyStatementsAction;
  QAction *discardStatementsAction;
  QAction *deleteCurrentLineAction;
  QAction *cloneCurrentLineAction;
  QMenu *menu;
  QString p_table;
  QString p_database;
  QString p_where;
  DTitleLabel *dTitleLabel;
  QList<QStringList> *headers;
  QStringList statements;
  QToolBar *resultToolBar;

  /**
   * @brief Initializes the toolbar and context menu actions.
   */
  void createActions();

  Statements *statementsDialog;
  QAction *viewChangesAction;
  QAction *addRowAction;

  /**
   * @brief Retranslates the user interface.
   *
   * Updates titles, menus, and actions to match the current application language.
   */
  void retranslateUI();

public slots:
  /**
   * @brief Refreshes the data in the table view from the database.
   *
   * Re-executes the SELECT query based on the current table and filter settings.
   */
  void updateResult();

private slots:
  /**
   * @brief Slot called when a cell value is changed by the user.
   * @param item The QStandardItem that was modified.
   */
  void itemChangedSlot(QStandardItem *item);

  /**
   * @brief Applies pending changes to the database.
   *
   * Executes the generated UPDATE, INSERT, or DELETE statements.
   */
  void applyStatementsActionTriggered();

  /**
   * @brief Discards all pending changes.
   *
   * Clears the list of generated SQL statements and refreshes the view.
   */
  void discardStatementsActionTriggered();

  /**
   * @brief Marks the selected row for deletion.
   */
  void deleteCurrentLineActionTriggered();

  /**
   * @brief Duplicates the selected row.
   */
  void cloneCurrentLineActionTriggered();

  /**
   * @brief Opens a dialog to view the pending SQL statements before applying them.
   */
  void viewChangesActionTriggered();

  /**
   * @brief Adds a new empty row to the table for data insertion.
   */
  void addRow();

  /**
   * @brief Updates the status bar with loading progress.
   * @param message The status message.
   * @param timeout The duration to show the message.
   * @param progress The completion percentage (0-100).
   */
  void statusBarProgressMessageSlot(const QString &message, const unsigned int timeout, const double progress);

signals:
  /**
   * @brief Emitted to display a temporary popup message on the status bar.
   * @param message The message text.
   */
  void statusBarMessagePopup(QString message);

  /**
   * @brief Emitted to update the progress bar and status message.
   * @param message The status message.
   * @param timeout The duration to show the message.
   * @param progress The completion percentage.
   */
  void statusBarProgressMessage(const QString &message, const unsigned int timeout, const double progress);

protected:
  /**
   * @brief Handles context menu events (right-click).
   * @param event The context menu event parameters.
   */
  void contextMenuEvent(QContextMenuEvent *event);

};

#endif // RESULT_H
