/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef RESTOREBACKUPFILE_H
#define RESTOREBACKUPFILE_H

#include "dsettings.h"

class QCheckBox;
class QLabel;
class QListWidgetItem;
class QDialogButtonBox;
class QGroupBox;
class DFileSelector;
class DTitleLabel;
class QFormLayout;
class QGridLayout;
class QProgressDialog;
class BaseTextEditor;
class DMariaDBRealQueryOnQThread;

#include "dmdisubwindow.h"
#include "dbms.h"

/**
 * @brief The RestoreBackupFile class provides an interface for importing SQL dump files.
 *
 * This class inherits from DMdiSubWindow and handles the parsing and execution of
 * SQL backup files to restore a database. It utilizes a background thread
 * (DMariaDBRealQueryOnQThread) to prevent the GUI from freezing during large
 * restore operations and provides progress feedback via a dialog.
 */
class RestoreBackupFile : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the RestoreBackupFile window.
   * @param serverConnection Pointer to the active DBMS connection where data will be restored.
   */
  RestoreBackupFile(DBMS *serverConnection);

private:
  DBMS *serverConnection;
  QCheckBox *force;
  QLabel *fileLabel;
  QLabel *dataLengthLabel;
  QDialogButtonBox *buttonBox;
  DFileSelector *dFileSelector;
  DTitleLabel *dTitleLabel;
  QGroupBox *fileGroupBox;
  QFormLayout *fileGroupBoxLayout;
  QGroupBox *optionsGroupBox;
  QGridLayout *optionsGroupBoxLayout;
  QProgressDialog *restoreProgressDialog;
  BaseTextEditor *baseTextEditor;
  DSettings settings;
  bool cancelRestore;
  bool insertingData;
  MYSQL *restoreConnection;
  DMariaDBRealQueryOnQThread *dMariaDBRealQueryOnQThread;
  unsigned int threadCounter;

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels, buttons, and status text to reflect the current application language.
   */
  void retranslateUI();

private slots:
  /**
   * @brief Reads and displays information about the selected backup file.
   *
   * Updates the UI with the file size and other relevant details when a file is selected.
   * @param file The path to the selected file.
   */
  void getFileInfo(QString file = QString());

  /**
   * @brief Initiates the restore process.
   *
   * Called when the user confirms the dialog. Starts parsing the file and executing statements.
   */
  void buttonBoxAcceptSlot();

  /**
   * @brief Cancels the ongoing restore operation.
   *
   * Signals the worker loop/thread to stop processing further statements.
   */
  void cancelRestoreProgressDialogSlot();

  /**
   * @brief Executes a specific SQL statement or block.
   *
   * Typically hands the statement off to the background thread for execution.
   * @param statement The SQL string to execute.
   */
  void executeStatement(std::string statement);
};

#endif // RESTOREBACKUPFILE_H
