/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef PROJECTS_H
#define PROJECTS_H

#include <QObject>
#include <QStringList>

#include "dsettings.h"

class QAction;
class QMenu;
class QActionGroup;
class QDialog;
class DFileSelector;
class QListView;
class QPushButton;
class ProjectFindReplace;
class SubversionedFile;
class DMdiSubWindow;
class QLineEdit;
class QDomNodeList;

/**
 * @brief The Projects class manages the project system within the application.
 *
 * This class handles the creation, opening, and saving of projects, which are collections
 * of files and configurations. It manages the project menu, file lists, integration
 * with version control (Subversion), and project-wide search functionality.
 */
class Projects : public QObject
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the Projects manager.
   */
  Projects();

  /**
   * @brief Retrieves the list of actions representing files in the current project.
   * @return A list of QActions, typically used to populate a menu or toolbar.
   */
  QList<QAction *> getProjectFiles();

  /**
   * @brief Gets the name of the currently open project.
   * @return The project name.
   */
  QString name();

  /**
   * @brief Populates the provided menu with project-related actions.
   *
   * Adds actions like New, Open, Close, and recent project lists to the menu.
   * @param menu The QMenu to populate.
   */
  void buildMenu(QMenu *menu);

  /**
   * @brief Checks if a project is currently open.
   * @return True if a project is open, false otherwise.
   */
  bool openedProject();

  /**
   * @brief Retrieves stored settings for a specific file within the project.
   * @param filename The name of the file.
   * @return A list of settings (e.g., cursor position, bookmarks).
   */
  QStringList getFileSettings(const QString filename);

  /**
   * @brief Saves settings for a specific file within the project.
   * @param filename The name of the file.
   * @param options The list of settings to save.
   */
  void saveFileSettings(QString filename, QStringList options);

  QMenu *projectFilesMenu;

  /**
   * @brief Gets the root directory path of the current project.
   * @return The absolute path.
   */
  QString getProjectPath();

  /**
   * @brief Retrieves the list of all files contained in the project.
   * @return A QStringList of file paths.
   */
  QStringList getProjectFileList();

  /**
   * @brief Scans project files to build a map of function names and their locations.
   * @return A Hash where keys are function names and values are file contexts/signatures.
   */
  QHash<QString, QString> getProjectFunctionListComplete();

  /**
   * @brief Retrieves a simple list of functions found in the project.
   * @return A QStringList of function names.
   */
  QStringList getProjectFunctionList();

  /**
   * @brief Retranslates the user interface actions and menus.
   */
  void retranslateUI();

  /**
   * @brief Retrieves the button used to access project files (e.g., for a sidebar).
   * @return Pointer to the QPushButton.
   */
  QPushButton *getProjectFilesPushButton();

private:
  QAction *newProjectAction;
  QAction *openProjectAction;
  QMenu *openRecentFilesMenu;
  QActionGroup *recentFilesActionGroup;
  QAction *clearRecentFilesAction;
  QDialog *dialog;
  DFileSelector *projectFile;
  QListView *filesList;
  QPushButton *loadFilesPushButton;
  QAction *loadFilesAction;
  QStringList projectFiles;

  /**
   * @brief Populates the internal file list view or model.
   */
  void fillFilesList();

  DSettings settings;
  QActionGroup *projectFilesActionGroup;
  QString fileName;
  QString workingFolder;
  QAction *closeProjectAction;
  QAction *findInProjectAction;
  ProjectFindReplace *projectFindReplace;

  /**
   * @brief Saves the project configuration to disk.
   * @param fileList The list of files to include in the project configuration.
   */
  void saveProjectFile(const QStringList fileList);

  /**
   * @brief Loads the project configuration from disk.
   * @return The list of files loaded from the project file.
   */
  QStringList loadProjectFile();

  /**
   * @brief Toggles the enabled state of menu options based on project state.
   * @param enable True to enable project-dependent options.
   */
  void enableMenuOption(bool enable = false);

  QMenu *subversionMenu;
//  QAction *subversionDiffRepository;
  SubversionedFile *subversionedFile;
  QAction *addFileToProjectAction;
  QAction *addExistingFileToProjectAction;
  QMenu *versionControlMenu;
  QLineEdit *nameLineEdit;

  /**
   * @brief Adds a list of files to the current project.
   * @param files List of file paths.
   */
  void addFilesToProject(QStringList files);

  /**
   * @brief Parses the project XML to retrieve bookmarks/dock marks.
   * @return A list of XML nodes representing the marks.
   */
  QDomNodeList getProjectDockMarks();

  /**
   * @brief Saves bookmarks/dock marks to the project XML.
   * @param fileName The file associated with the marks.
   * @param marks The list of marks to save.
   */
  void addDockMarksToProject(QString fileName, QStringList marks);

  QPushButton *projectFilesPushButton;

private slots:
  /**
   * @brief Opens a specific project file.
   * @param file The path to the project file (optional).
   */
  void openProjectActionSlot(QString file = QString());

  /**
   * @brief Populates the recent projects menu before it is shown.
   */
  void openRecentFilesMenuAboutToShowSlot();

  /**
   * @brief Handles the selection of a recent project.
   * @param action The action representing the file to open.
   */
  void recentFilesActionGroupTriggered(QAction *action);

  /**
   * @brief Clears the history of recent projects.
   */
  void clearRecentFilesActionTriggered();

  /**
   * @brief Initiates the creation of a new project.
   */
  void newProjectActionSlot();

  /**
   * @brief Triggers the logic to load files into the project.
   */
  void loadFilesActionTriggered();

  /**
   * @brief Handles clicking on a specific file within the project menu.
   * @param action The action representing the file.
   */
  void projectFilesActionGroupTriggered(QAction *action);

  /**
   * @brief Closes the currently active project.
   */
  void closeProjectActionTriggered();

  /**
   * @brief Opens the "Find in Project" dialog.
   */
  void findInProjectActionTriggered();

  /**
   * @brief Generic slot for handling project file interactions.
   */
  void projectFilesSlot();
//  void subversionDiffRepositoryActionTriggered();

  /**
   * @brief Opens a dialog to create and add a new file to the project.
   */
  void addFileToProjectActionTriggered();

  /**
   * @brief Opens a dialog to add an existing file to the project.
   */
  void addExistingFileToProjectActionTriggered();

  /**
   * @brief Signals the main window to open a file in the editor.
   * @param fileName The path to the file.
   * @param line The line number to jump to.
   */
  void openFileSlot(QString fileName = QString(), unsigned int line = 1);

  /**
   * @brief Handles changes to the project name in the creation dialog.
   * @param text The new name.
   */
  void nameLineEditTextEditedSlot(const QString &text);

public slots:
  /**
   * @brief Displays a popup message on the status bar.
   * @param message The message text.
   */
  void statusBarMessagePopupSlot(QString message);

  /**
   * @brief Displays a standard message on the status bar.
   * @param message The message text.
   */
  void statusBarMessageSlot(QString message);

  /**
   * @brief Requests adding a sub-window (e.g., search results) to the main MDI area.
   * @param window The sub-window to add.
   */
  void addSubWindowSlot(DMdiSubWindow *window);

signals:
  /**
   * @brief Emitted when the project state changes, requiring a window title update.
   */
  void updateTitle();

  /**
   * @brief Signal to update the main status bar message.
   * @param message The message text.
   */
  void statusBarMessage(QString message);

  /**
   * @brief Signal to add a new sub-window to the main application area.
   * @param window The sub-window to add.
   */
  void addSubWindow(DMdiSubWindow *window);

  /**
   * @brief Signal to open a file in the main editor.
   * @param fileName The file path.
   * @param line The line number.
   */
  void openFile(QString fileName = QString(), unsigned int line = 1);

  /**
   * @brief Signal to show a temporary popup message.
   * @param message The message text.
   */
  void statusBarMessagePopup(QString message);
};

#endif // PROJECTS_H
