/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QAction>
#include <QApplication>
#include <QMenu>
#include <QDialog>
#include <QLineEdit>
#include <QFormLayout>
#include <QGroupBox>
#include <QDialogButtonBox>
#include <QListView>
#include <QPushButton>
#include <QStandardItemModel>
#include <QFileDialog>
#include <QTextStream>
#include <QInputDialog>
#include <QDomDocument>
#include <QXmlStreamReader>
#include <QXmlStreamWriter>
#include <QProcess>
#include <QMessageBox>
#include <QActionGroup>

#include "projects.h"
#include "dtitlelabel.h"
#include "dfileselector.h"
#include "staticfunctions.h"
#include "projectfindreplace.h"
#include "subversionedfile.h"
#include "dicon.h"

#include "QDebug"

Projects::Projects()
{
  newProjectAction = new QAction(this);
  newProjectAction->setIcon(DIcon::DocumentNew());
  connect(newProjectAction, SIGNAL(triggered()), this, SLOT(newProjectActionSlot()));
  openProjectAction = new QAction(this);
  openProjectAction->setIcon(DIcon::DocumentOpen());
  openProjectAction->setShortcut(QKeySequence::Open);
  connect(openProjectAction, SIGNAL(triggered()), this, SLOT(openProjectActionSlot()));
  openRecentFilesMenu = new QMenu();
  openRecentFilesMenu->setIcon(DIcon::DocumentOpenRecent());
  connect(openRecentFilesMenu, SIGNAL(aboutToShow()), this, SLOT(openRecentFilesMenuAboutToShowSlot()));
  recentFilesActionGroup = new QActionGroup(this);
  connect(recentFilesActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(recentFilesActionGroupTriggered(QAction*)));
  clearRecentFilesAction = new QAction(this);
  clearRecentFilesAction->setIcon(DIcon::Clear());
  connect(clearRecentFilesAction, SIGNAL(triggered()), this, SLOT(clearRecentFilesActionTriggered()));
  projectFilesActionGroup = new QActionGroup(this);
  connect(projectFilesActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(projectFilesActionGroupTriggered(QAction*)));
  closeProjectAction = new QAction(this);
  closeProjectAction->setIcon(DIcon::DocumentClose());
  connect(closeProjectAction, SIGNAL(triggered()), this, SLOT(closeProjectActionTriggered()));
  findInProjectAction = new QAction(this);
  findInProjectAction->setIcon(DIcon::Find());
  connect(findInProjectAction, SIGNAL(triggered()), this, SLOT(findInProjectActionTriggered()));
  projectFilesMenu = new QMenu();
  projectFilesMenu->setIcon(DIcon::TextEditor());
  connect(projectFilesMenu, SIGNAL(aboutToShow()), this, SLOT(projectFilesSlot()));
  versionControlMenu = new QMenu;
  subversionMenu = new QMenu();
  versionControlMenu->addMenu(subversionMenu);
  subversionMenu->setIcon(DIcon::SVNLogo());
//  subversionDiffRepository = new QAction(DIcon::Database(), tr("Diff repository"), this);
//  subversionDiffRepository->setToolTip(subversionDiffRepository->text());
//  connect(subversionDiffRepository, SIGNAL(triggered()), this, SLOT(subversionDiffRepositoryActionTriggered()));
//  subversionMenu->addAction(subversionDiffRepository);
  subversionedFile = new SubversionedFile(getProjectPath());
  connect(subversionedFile, SIGNAL(statusBarMessage(QString)), this, SLOT(statusBarMessageSlot(QString)));
  connect(subversionedFile, SIGNAL(statusBarMessagePopup(QString)), this, SLOT(statusBarMessagePopupSlot(QString)));
  connect(subversionedFile, SIGNAL(addSubWindow(DMdiSubWindow*)), this, SLOT(addSubWindowSlot(DMdiSubWindow*)));
  subversionMenu->addActions(subversionedFile->getActions());
  addFileToProjectAction = new QAction(this);
  addFileToProjectAction->setIcon(DIcon::ListAdd());
  connect(addFileToProjectAction, SIGNAL(triggered()), this, SLOT(addFileToProjectActionTriggered()));
  addExistingFileToProjectAction = new QAction(this);
  addExistingFileToProjectAction->setIcon(DIcon::ListAdd());
  connect(addExistingFileToProjectAction, SIGNAL(triggered()), this, SLOT(addExistingFileToProjectActionTriggered()));

  projectFilesPushButton = new QPushButton();
  projectFilesPushButton->setIcon(DIcon::TextEditor());
  projectFilesPushButton->setMenu(projectFilesMenu);

  retranslateUI();
  enableMenuOption(false);
}

void Projects::clearRecentFilesActionTriggered()
{
  settings.remove("Projects/RecentFiles");
}

void Projects::recentFilesActionGroupTriggered(QAction *action)
{
  openProjectActionSlot(action->text());
}

void Projects::openRecentFilesMenuAboutToShowSlot()
{
  openRecentFilesMenu->clear();
  QStringList fileList(settings.value("Projects/RecentFiles", QStringList()).toStringList());
  fileList.sort();
  foreach (QString file, fileList)
    if (!file.isEmpty())
      recentFilesActionGroup->addAction(openRecentFilesMenu->addAction(DIcon::DocumentOpenRecent(), file));
  openRecentFilesMenu->addSeparator()->setText(tr("Clean"));
  openRecentFilesMenu->addAction(clearRecentFilesAction);
}

QList<QAction *> Projects::getProjectFiles()
{
  foreach (QAction *action, projectFilesActionGroup->actions())
    projectFilesActionGroup->removeAction(action);
  if (!fileName.isEmpty()) {
    foreach (QString file, getProjectFileList()) {
      QFileInfo fileInfo(file);
      projectFilesActionGroup->addAction(StaticFunctions::iconFromFileName(file), fileInfo.completeBaseName() + "." + fileInfo.suffix());
    }
  }
  return projectFilesActionGroup->actions();
}

QString Projects::name()
{
  if (fileName.isEmpty()) {
    return tr("No projet");
  } else {
    QFile file(fileName);
    bool opened = file.open(QIODevice::ReadOnly | QIODevice::Text);
    Q_UNUSED(opened);
    QDomDocument xmlDocument;
    xmlDocument.setContent(&file);
    file.close();
    workingFolder = xmlDocument.firstChildElement("ProjectInformation").firstChildElement("Path").text();

//    QProcess *process = new QProcess(this);
//    process->start("svnversion", QStringList() << xmlDocument.firstChildElement("ProjectInformation").firstChildElement("Path").text());
//    process->waitForFinished();
//    QByteArray output = process->readAll();
//    QString result;
//    for (int counter = 0; counter < output.size(); ++counter)
//      result += output.at(counter);
    return xmlDocument.firstChildElement("ProjectInformation").firstChildElement("Name").text()/* + " - " + result*/;
  }
}

void Projects::buildMenu(QMenu *menu)
{
  menu->addAction(newProjectAction);
  menu->addAction(openProjectAction);
  menu->addAction(closeProjectAction);
  menu->addSeparator();
  menu->addMenu(openRecentFilesMenu);
  menu->addSeparator();
  menu->addAction(findInProjectAction);
  menu->addAction(addFileToProjectAction);
  menu->addAction(addExistingFileToProjectAction);
  menu->addSeparator();
  menu->addMenu(projectFilesMenu);
//  menu->addSeparator();
//  menu->addMenu(versionControlMenu);
}

bool Projects::openedProject()
{
  if (!fileName.isEmpty())
    return true;
  else
    return false;
}

QStringList Projects::getFileSettings(const QString filename)
{
  QFile file(fileName);
  bool opened = file.open(QIODevice::ReadOnly | QIODevice::Text);
  Q_UNUSED(opened);
  QDomDocument xmlDocument;
  xmlDocument.setContent(&file);
  file.close();
  QDomNodeList gitLabFileNodes = xmlDocument.firstChildElement("ProjectInformation").firstChildElement("DockMarks").elementsByTagName("File");
  for(int counter = 0; counter < gitLabFileNodes.count(); counter++) {
    if (gitLabFileNodes.at(counter).toElement().attribute("name") == filename.split(QDir::separator()).last())
      return gitLabFileNodes.at(counter).toElement().text().split(",");
  }
  return QStringList();
}

void Projects::saveFileSettings(QString filename, QStringList options)
{
  if (options.count() > 0) {
    addDockMarksToProject(filename.split(QDir::separator()).takeLast(), options);
  }
}

QString Projects::getProjectPath()
{
  QString path(".");
  if (!fileName.isEmpty()) {
    QFile file(fileName);
    bool opened = file.open(QIODevice::ReadOnly | QIODevice::Text);
    Q_UNUSED(opened);
    QDomDocument xmlDocument;
    xmlDocument.setContent(&file);
    file.close();
    path = xmlDocument.firstChildElement("ProjectInformation").firstChildElement("Path").text();
  }
  return path;
}

QStringList Projects::getProjectFileList()
{
  QStringList list;
  QFile file(fileName);
  bool opened = file.open(QIODevice::ReadOnly | QIODevice::Text);
  Q_UNUSED(opened);
  QDomDocument xmlDocument;
  xmlDocument.setContent(&file);
  file.close();
  QDomNodeList gitLabFileNodes = xmlDocument.firstChildElement("ProjectInformation").firstChildElement("Files").elementsByTagName("File");
  for(int counter = 0; counter < gitLabFileNodes.count(); counter++) {
    list.append(gitLabFileNodes.at(counter).toElement().text());
  }
  return list;
}

QHash<QString, QString> Projects::getProjectFunctionListComplete()
{
  QHash<QString, QString> data;
//  unsigned int lineNumber = 0;
  QString param1;
  QString param3;
  QString param4;
  QString previousLine1;
  QString previousLine2;
  QString previousLine3;
  QString previousLine4;
  QString previousLine5;
  foreach (QString fileName, getProjectFileList()) {
    QFile file(fileName);
    if(file.open(QFile::ReadOnly | QFile::Text)) {
      QTextStream in(&file);
      in.setEncoding(QStringConverter::Utf8);
      while(!in.atEnd()) {
//        ++lineNumber;
        QString line = in.readLine();
        previousLine5 = previousLine4;
        previousLine4 = previousLine3;
        previousLine3 = previousLine2;
        previousLine2 = previousLine1;
        previousLine1 = line;
        static QRegularExpression re("^function.*\\(.*\\) \\{$", QRegularExpression::CaseInsensitiveOption);
        if (line.contains(re)) {
          if (previousLine5.startsWith(" * @summary:"))
            param4 = previousLine5.mid(12);
          if (previousLine4.startsWith(" * @param:"))
            param3 = previousLine4.mid(10);
          if (previousLine3.startsWith(" * @return:"))
            param1 = previousLine3.mid(11);
          data.insert(line.mid(9, line.indexOf("(") - 9), QString("%1 <u>%2</u>(<b>%3</b>) <br />%4").arg(param1.trimmed(), line.mid(9, line.indexOf("(") - 9), param3.trimmed(), param4.trimmed()));
        }
        param1 = QString();
        param3 = QString();
        param4 = QString();
      }
      file.close();
//      lineNumber = 0;
    }
  }
  return data;
}

QStringList Projects::getProjectFunctionList()
{
  QStringList list = QStringList();
  QHashIterator<QString, QString> hashIterator(getProjectFunctionListComplete());
  while (hashIterator.hasNext()) {
    hashIterator.next();
    list.append(hashIterator.key());
  }
  return list;
}

void Projects::retranslateUI()
{
  newProjectAction->setText(tr("New Project..."));
  newProjectAction->setToolTip(newProjectAction->text());
  openProjectAction->setText(tr("Open Project..."));
  openProjectAction->setToolTip(openProjectAction->text());
  openRecentFilesMenu->setTitle(tr("Recent files"));
  openRecentFilesMenu->setToolTip(openRecentFilesMenu->title());
  clearRecentFilesAction->setText(tr("Clear recent files"));
  clearRecentFilesAction->setToolTip(clearRecentFilesAction->text());
  closeProjectAction->setText(tr("Close current project"));
  closeProjectAction->setToolTip(closeProjectAction->text());
  findInProjectAction->setText(tr("Find in project"));
  findInProjectAction->setToolTip(findInProjectAction->text());
  projectFilesMenu->setTitle(tr("Project files"));
  subversionMenu->setTitle(tr("Subversion"));
  addFileToProjectAction->setText(tr("Add new file to project"));
  addFileToProjectAction->setToolTip(addFileToProjectAction->text());
  addExistingFileToProjectAction->setText(tr("Add existing file to project"));
  addExistingFileToProjectAction->setToolTip(addExistingFileToProjectAction->text());
  versionControlMenu->setTitle(tr("Version &control"));
}

void Projects::openProjectActionSlot(QString file)
{
  QFileInfo fileInfo(file);
  if (file.isEmpty() || !fileInfo.exists()) {
    QFileDialog fileDialog;
    fileDialog.setDirectory(QDir::home());
    fileName = fileDialog.getOpenFileName(0, tr("Open a project"), QDir::home().absolutePath(), tr("Project files (*.xml)"));
  } else {
    fileName = file;
  }
  if (fileName.isEmpty())
    return;
  QDomDocument xmlDocument;
  QFile file2(fileName);
  xmlDocument.setContent(&file2);
  file2.close();
  workingFolder = xmlDocument.firstChildElement("ProjectInformation").firstChildElement("Path").text();

  QStringList fileList;
  fileList = settings.value("Projects/RecentFiles", QStringList()).toStringList();
  if (fileList.indexOf(fileName) == -1)
    fileList.append(fileName);
  settings.setValue("Projects/RecentFiles", fileList);

  emit updateTitle();
  enableMenuOption(true);
  subversionedFile->fileName = getProjectPath();
}

void Projects::newProjectActionSlot()
{
  dialog = new QDialog;
  dialog->setWindowTitle(tr("New project"));
  dialog->setObjectName(dialog->windowTitle());
  dialog->setWindowIcon(DIcon::Database());
  dialog->resize(600, 600);

  nameLineEdit = new QLineEdit;
  connect(nameLineEdit, SIGNAL(textEdited(const QString)), this, SLOT(nameLineEditTextEditedSlot(const QString)));

  //Project info
  QFormLayout *formLayout = new QFormLayout;
  formLayout->addRow(tr("&Name:"), nameLineEdit);
  projectFile = new DFileSelector(DFileSelectorContexts::ProjectFile, tr("Destination:"), true);
  projectFile->setPlaceholderText(tr("The project file and all its files must be in the same path."));

  projectFile->setStatusTip(projectFile->placeholderText());
  projectFile->setToolTip(projectFile->placeholderText());

  formLayout->addRow(projectFile);
  QGroupBox *projectGroupBox = new QGroupBox(tr("Poject information"));
  projectGroupBox->setLayout(formLayout);

  //Project files
  loadFilesPushButton = new QPushButton(DIcon::Database(), tr("Load files"));
  loadFilesAction = new QAction("Load files", this);
  connect(loadFilesPushButton, SIGNAL(clicked()), this, SLOT(loadFilesActionTriggered()));
  if (!projectFile->getFileName().isEmpty()) {
    projectFiles.clear();
    loadProjectFile();
    fillFilesList();
  }
  filesList = new QListView;
  QVBoxLayout *projectFilesLayout = new QVBoxLayout;
  projectFilesLayout->addWidget(loadFilesPushButton);
  projectFilesLayout->addWidget(filesList);
  QGroupBox *projectFilesGroupBox = new QGroupBox(tr("Poject files"));
  projectFilesGroupBox->setLayout(projectFilesLayout);

  QDialogButtonBox *buttonBox = new QDialogButtonBox(QDialogButtonBox::Save | QDialogButtonBox::Cancel);
  connect(buttonBox, SIGNAL(accepted()), dialog, SLOT(accept()));
  connect(buttonBox, SIGNAL(rejected()), dialog, SLOT(reject()));

  QVBoxLayout *verticalLayout = new QVBoxLayout;
  verticalLayout->addWidget(new DTitleLabel(tr("Create new Project")));
  verticalLayout->addWidget(projectGroupBox);
  verticalLayout->addWidget(projectFilesGroupBox);
  verticalLayout->addWidget(buttonBox);
  dialog->setLayout(verticalLayout);
  if (dialog->exec() == QDialog::Accepted) {
    QStringList fileList;
    for (int row = 0; row < projectFiles.count(); row++)
      fileList.append(projectFiles.at(row));
    fileName = projectFile->getFileName();
    saveProjectFile(fileList);
    emit statusBarMessage(tr("Information saved."));
    openProjectActionSlot(projectFile->getFileName());
  }
}

void Projects::fillFilesList()
{
  if (projectFiles.empty())
    return;

  QStandardItemModel *model = new QStandardItemModel(projectFiles.count(), 1);
  for (int row = 0; row < projectFiles.count(); row++)
    model->setItem(row, 0, new QStandardItem(StaticFunctions::iconFromFileName(projectFiles.at(row)), projectFiles.at(row)));
  filesList->setModel(model);
}

void Projects::saveProjectFile(const QStringList fileList)
{
  QFileInfo infoOfTheFile(fileName);
  QDir tempDirectory(QDir::tempPath());
  tempDirectory.mkpath(infoOfTheFile.absoluteDir().path());

  QDomDocument xmlDocument;
  if (!fileName.endsWith(".xml"))
    fileName.append(".xml");
  QFile file2(fileName);
  QDomElement rootElement = xmlDocument.createElement("ProjectInformation");
  xmlDocument.appendChild(rootElement);
  if (!file2.open(QIODevice::WriteOnly | QIODevice::Text))
    return;

  QDomElement projectName = xmlDocument.createElement("Name");
  projectName.appendChild(xmlDocument.createTextNode(nameLineEdit->text()));
  rootElement.appendChild(projectName);
  QDomElement projectPath = xmlDocument.createElement("Path");
  projectPath.appendChild(xmlDocument.createTextNode(projectFile->fileDir().path()));
  rootElement.appendChild(projectPath);
  QDomElement projectFiles = xmlDocument.createElement("Files");
  QDomElement projectFile;
  foreach (QString fileElement, fileList) {
    projectFile = xmlDocument.createElement("File");
    projectFile.appendChild(xmlDocument.createTextNode(fileElement));
//    projectFile.setAttribute("LocalFilename", "PossibleName");
    projectFiles.appendChild(projectFile);
  }
  rootElement.appendChild(projectFiles);
  QDomElement dockMarks = xmlDocument.createElement("DockMarks");
  rootElement.appendChild(dockMarks);

  QTextStream out2(&file2);
  out2 << xmlDocument.toString();
  file2.flush();
  file2.close();
}

QStringList Projects::loadProjectFile()
{
  return getProjectFileList();
}

void Projects::enableMenuOption(bool enable)
{
  closeProjectAction->setEnabled(enable);
  findInProjectAction->setEnabled(enable);
  projectFilesMenu->setEnabled(enable);
  subversionMenu->setEnabled(enable);
  versionControlMenu->setEnabled(enable);
  addFileToProjectAction->setEnabled(enable);
  addExistingFileToProjectAction->setEnabled(enable);
  projectFilesPushButton->setEnabled(enable);
}

void Projects::addFilesToProject(QStringList files)
{
  QFile xmlFile(fileName);
  bool opened = xmlFile.open(QIODevice::ReadWrite | QIODevice::Text);
  Q_UNUSED(opened);
  QByteArray xmlData(xmlFile.readAll());
  QDomDocument xmlDocument;
  xmlDocument.setContent(xmlData);
  QDomElement rootElement = xmlDocument.documentElement();
  QDomElement nodeTag = rootElement.firstChildElement("Files");
  QDomElement projectFiles = xmlDocument.createElement("Files");
  QDomElement projectFile;
  foreach (QString fileElement, QStringList() << getProjectFileList() << files) {
    projectFile = xmlDocument.createElement("File");
    projectFile.appendChild(xmlDocument.createTextNode(fileElement));
//    projectFile.setAttribute("LocalFilename", "PossibleName");
    projectFiles.appendChild(projectFile);
  }
  rootElement.replaceChild(projectFiles, nodeTag);
  xmlFile.resize(0);
  QTextStream stream;
  stream.setDevice(&xmlFile);
  xmlDocument.save(stream, 4);
  xmlFile.close();
}

QDomNodeList Projects::getProjectDockMarks()
{
  QFile file(fileName);
  bool opened = file.open(QIODevice::ReadOnly | QIODevice::Text);
  Q_UNUSED(opened);
  QDomDocument xmlDocument;
  xmlDocument.setContent(&file);
  file.close();
  return xmlDocument.firstChildElement("ProjectInformation").firstChildElement("DockMarks").elementsByTagName("File");
}

void Projects::addDockMarksToProject(QString fileName, QStringList marks)
{
  QFile xmlFile(this->fileName);
  bool opened = xmlFile.open(QIODevice::ReadWrite | QIODevice::Text);
  Q_UNUSED(opened);
  QByteArray xmlData(xmlFile.readAll());
  QDomDocument xmlDocument;
  xmlDocument.setContent(xmlData);
  QDomElement rootElement = xmlDocument.documentElement();
  QDomElement nodeTag = rootElement.firstChildElement("DockMarks");
  QDomElement dockerMarks = xmlDocument.createElement("DockMarks");
  QDomElement dockerMark;
  for(int counter = 0; counter < getProjectDockMarks().count(); counter++) {
    if (getProjectDockMarks().at(counter).toElement().attribute("name") != fileName) {
      dockerMark = xmlDocument.createElement("File");
      dockerMark.appendChild(xmlDocument.createTextNode(getProjectDockMarks().at(counter).toElement().text()));
      dockerMark.setAttribute("name", getProjectDockMarks().at(counter).toElement().attribute("name"));
      dockerMarks.appendChild(dockerMark);
    }
  }
  dockerMark = xmlDocument.createElement("File");
  dockerMark.appendChild(xmlDocument.createTextNode(marks.join(",")));
  dockerMark.setAttribute("name", fileName);
  dockerMarks.appendChild(dockerMark);
  rootElement.replaceChild(dockerMarks, nodeTag);
  xmlFile.resize(0);
  QTextStream stream;
  stream.setDevice(&xmlFile);
  xmlDocument.save(stream, 4);
  xmlFile.close();
}

QPushButton *Projects::getProjectFilesPushButton()
{
  return projectFilesPushButton;
}

void Projects::loadFilesActionTriggered()
{
  projectFiles = QFileDialog::getOpenFileNames(0, tr("Select files"), projectFile->fileDir().absolutePath());
  projectFiles.sort();
  fillFilesList();
}

void Projects::projectFilesActionGroupTriggered(QAction *action)
{
  emit openFile(workingFolder + QDir::separator() + action->text());
}

void Projects::closeProjectActionTriggered()
{
  fileName.clear();
  emit updateTitle();
  enableMenuOption(false);
}

void Projects::findInProjectActionTriggered()
{
  projectFindReplace = new ProjectFindReplace(this);
  connect(projectFindReplace, SIGNAL(openFile(QString,uint)), this, SLOT(openFileSlot(QString,uint)));
  emit addSubWindow(projectFindReplace);
}

void Projects::projectFilesSlot()
{
  projectFilesMenu->clear();
  projectFilesMenu->addActions(getProjectFiles());
}

void Projects::addFileToProjectActionTriggered()
{
  QString fileName = getProjectPath() + QDir::separator() + QInputDialog::getText(0, tr("File name"), tr("File name")).trimmed();
//  qDebug() << fileName;
  addFilesToProject(QStringList() << fileName);
  emit statusBarMessage(tr("Information saved."));
  QFile file(fileName);
  if (!file.open(QFile::ReadWrite | QFile::Text))
    emit statusBarMessage(tr("Cannot write file %1:\n%2.").arg(fileName, file.errorString()));
  file.close();
//  emit statusBarMessage(SubversionedFile::svnAdd(fileName));
}

void Projects::addExistingFileToProjectActionTriggered()
{
  addFilesToProject(QStringList() << QFileDialog::getOpenFileName(0, tr("Select a file"), getProjectPath()).trimmed());
  emit statusBarMessage(tr("Information saved."));
}

void Projects::openFileSlot(QString fileName, unsigned int line)
{
  emit openFile(fileName, line);
}

void Projects::nameLineEditTextEditedSlot(const QString &text)
{
  projectFile->setFileName(QDir::homePath() + "/" + text + "/" + text + ".xml");
}

void Projects::statusBarMessageSlot(QString message)
{
  emit statusBarMessage(message);
}

void Projects::statusBarMessagePopupSlot(QString message)
{
  emit statusBarMessagePopup(message);
}

void Projects::addSubWindowSlot(DMdiSubWindow *window)
{
  emit addSubWindow(window);
}

//void Projects::subversionDiffRepositoryActionTriggered()
//{
//  QString path;
//  QFile file(fileName);
//  file.open(QIODevice::ReadOnly | QIODevice::Text);
//  QTextStream in(&file);
//  while (!in.atEnd()) {
//    QString line(in.readLine());
//    if (line.startsWith("Path"))
//      path = line.split("=").at(1);
//  }
//  addSubWindow(new DiffFile(path));
//}
