/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef PROJECTFINDREPLACE_H
#define PROJECTFINDREPLACE_H

#include "dmdisubwindow.h"
#include "projects.h"
#include "dsettings.h"

class QGroupBox;
class QLineEdit;
class QFormLayout;
class QTreeWidget;
class QTreeWidgetItem;
class QCheckBox;
class QCompleter;
class DTitleLabel;

/**
 * @brief The ProjectFindReplace class provides a global search and replace tool.
 *
 * This window allows the user to search for text strings across all files
 * contained within the current active project. It supports standard search
 * options like case sensitivity, whole words, and regular expressions, displaying
 * results in a navigable tree view.
 */
class ProjectFindReplace : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the ProjectFindReplace window.
   * @param project Pointer to the current project instance to search within.
   */
  ProjectFindReplace(Projects *project);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels, buttons, and titles to reflect the current application language.
   */
  void retranslateUI();

private:
  Projects *project;
  QGroupBox *mainGroupBox;
  QLineEdit *lineEditTextToFind;
  QFormLayout *formLayout;
  QTreeWidget *findsListWidget;
  QPushButton *pushButtonFind;
  QCheckBox *checkBoxCaseSensitive;
  QCheckBox *checkBoxWholeWords;
  QCheckBox *checkBoxRegularExp;
  QLineEdit *lineEditTextToReplace;
  DSettings settings;
  QCompleter *completer;
  QPushButton *pushButtonReplace;
  DTitleLabel *dTitleLabel;

signals:
  /**
   * @brief Emitted to update the main window status bar.
   * @param message The message text.
   */
  void statusBarMessage(QString message);

  /**
   * @brief Emitted when a search result is clicked to open the file.
   * @param file The path to the file containing the match.
   * @param line The line number of the match.
   */
  void openFile(QString file, unsigned int line);

protected:
  /**
   * @brief Filters events from watched objects.
   *
   * Used to intercept key events (like Enter) on the input fields to trigger
   * the search action.
   *
   * @param obj The object receiving the event.
   * @param event The event being processed.
   * @return True if the event was handled, false otherwise.
   */
  bool eventFilter(QObject *obj, QEvent *event);

private slots:
  /**
   * @brief Executes the search logic across project files.
   *
   * Scans files, finds matches based on criteria, and populates the results tree.
   */
  void findSlot();

  /**
   * @brief Handles double-clicks on search results.
   *
   * Triggers the openFile signal to navigate to the specific match.
   *
   * @param item The tree item clicked.
   * @param column The column clicked.
   */
  void itemDoubleClickedSlot(QTreeWidgetItem *item, int column);

  /**
   * @brief Executes the replace logic across project files.
   *
   * Replaces occurrences of the search text with the replacement text
   * and saves the modified files.
   */
  void replaceSlot();

  /**
   * @brief Validates user input.
   *
   * Enables or disables the Find/Replace buttons based on whether the
   * search field is empty.
   */
  void validateInputs();
};

#endif // PROJECTFINDREPLACE_H
