/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef PROCESSLIST_H
#define PROCESSLIST_H

#include "dmdisubwindow.h"
#include "dbms.h"
#include "dsettings.h"

class QMenu;
class DTableView;
class QGroupBox;
class DTitleLabel;
class QPushButton;
class QSpinBox;
class QRadioButton;
class QComboBox;

/**
 * @brief The ProcessList class monitors database server threads and connections.
 *
 * This class inherits from DMdiSubWindow and provides a real-time view of
 * currently executing queries and connected users (similar to `SHOW PROCESSLIST`).
 * It includes functionality to auto-refresh data, filter by user, and terminate
 * specific threads or queries (KILL command).
 */
class ProcessList : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the ProcessList window.
   * @param serverConnection Pointer to the active DBMS connection used to fetch process data.
   */
  ProcessList(DBMS *serverConnection);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels, buttons, and menu actions to reflect the current language.
   */
  void retranslateUI();

private:
  DBMS *serverConnection;
  DTableView *processTable;
  QList<QStringList> *result;
  QAction *killThread;
  QMenu *menu;
  QPushButton *pushButtonStopRefreshing;
  QTimer *timerRefresh;
  QGroupBox *buttonGroup;
  DTitleLabel *dTitleLabel;
  QPushButton *pushButtonKillIdleThreads;
  QSpinBox *spinBoxTimeLimit;
  DSettings settings;
  QSpinBox *spinBoxRefreshRate;
  QAction *killQuery;
  QList<QStringList> *headers;
  bool useTable;
  QRadioButton *optionPROCESSLIST;
  QRadioButton *optionPROCESSES;

  /**
   * @brief Configures the default table headers.
   */
  void setHeaders();

  /**
   * @brief Updates headers based on the selected view mode.
   * @param checked State of the radio button triggering the change.
   */
  void reSetHeaders(bool checked);

  QPushButton *pushButtonKillBusyThreads;
  QSpinBox *KillBusyThreadsTimeLimitSpinBox;
  QComboBox *userListComboBox;

private slots:
  /**
   * @brief Fetches the latest process list data from the server.
   *
   * Triggered automatically by the refresh timer or manually.
   */
  void reloadData();

  /**
   * @brief Kills the selected connection/thread (`KILL CONNECTION id`).
   */
  void killThreadSlot();

  /**
   * @brief Kills all threads that have been in 'Sleep' state longer than the limit.
   */
  void killIdleThreadsSlot();

  /**
   * @brief Updates the auto-refresh interval.
   * @param value The new interval in seconds.
   */
  void refreshRateSlot(const int value);

  /**
   * @brief Kills the query running on the selected thread (`KILL QUERY id`).
   */
  void killQuerySlot();

  /**
   * @brief Switches view to standard `SHOW PROCESSLIST` output.
   * @param checked True if selected.
   */
  void optionPROCESSLISTClicked(bool checked);

  /**
   * @brief Switches view to `INFORMATION_SCHEMA.PROCESSLIST` output.
   * @param checked True if selected.
   */
  void optionPROCESSESClicked(bool checked);

  /**
   * @brief Kills threads running queries longer than the specified limit.
   */
  void pushButtonKillBusyThreadsSlot();

  /**
   * @brief Filters the process list by a specific user.
   * @param user The username selected in the combo box.
   */
  void userListComboBoxTextActivated(QString user);

protected:
  /**
   * @brief Handles context menu events (right-click).
   *
   * Displays the menu to Kill Thread or Kill Query.
   * @param event The context menu event parameters.
   */
  void contextMenuEvent(QContextMenuEvent *event);
};

#endif // PROCESSLIST_H
