/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QLabel>
#include <QVBoxLayout>
#include <QAction>
#include <QTimer>
#include <QMenu>
#include <QContextMenuEvent>
#include <QApplication>
#include <QGroupBox>
#include <QPushButton>
#include <QRadioButton>
#include <QSpinBox>
#include <QComboBox>

#include "processlist.h"
#include "dtitlelabel.h"
#include "dtableview.h"
#include "dicon.h"

#include "QDebug"

ProcessList::ProcessList(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
  setWindowIcon(DIcon::DatabaseProcessList());

  useTable = settings.value("Processes/UseTable", false).toBool();

//  /**
//  * Pending translation
//  */
//  switch(serverConnection->getDBMSType()) {
//  case StaticFunctions::MySQL:
//    headers->append(QStringList() << tr("Id") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("User") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Host") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Database") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Command") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Time") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("State") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Info") << "NoDelegate"  << "" << "Left");
//    break;
//  case StaticFunctions::MariaDB:
//    headers->append(QStringList() << tr("Id") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("User") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Host") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Database") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Command") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Time") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("State") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Info") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Milliseconds") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Stage") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Max Stage") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Progress") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Memory") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Examined Rows") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Query Id") << "NoDelegate"  << "" << "Left");
//    headers->append(QStringList() << tr("Binary data information") << "NoDelegate"  << "" << "Left");
//    break;
//  case StaticFunctions::Undefined:
//  default:
//    break;
//  }

  timerRefresh = new QTimer(this);
  connect(timerRefresh, SIGNAL(timeout()), this, SLOT(reloadData()));
  setHeaders();
  processTable = new DTableView(headers);
//  connect(processTable, SIGNAL(loadStarted(QString,uint,double)), parentWidget, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
//  connect(processTable, SIGNAL(loadFinished(QString,uint,double)), parentWidget, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
//  connect(processTable, SIGNAL(loadProgress(QString,uint,double)), parentWidget, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
  QVBoxLayout *mainLayout = new QVBoxLayout;
  mainLayout->setContentsMargins(0, 0, 0, 0);
  dTitleLabel = new DTitleLabel;
  mainLayout->addWidget(dTitleLabel);
  buttonGroup = new QGroupBox(this);
  QVBoxLayout *qVBoxLayout = new QVBoxLayout;
  QHBoxLayout *groupBoxHLayout = new QHBoxLayout;
  QHBoxLayout *killingBoxHLayout = new QHBoxLayout;
  qVBoxLayout->addLayout(groupBoxHLayout);
  qVBoxLayout->addLayout(killingBoxHLayout);
  pushButtonStopRefreshing = new QPushButton;
  pushButtonStopRefreshing->setIcon(DIcon::Stop());
  pushButtonStopRefreshing->setCheckable(true);
  pushButtonStopRefreshing->setChecked(false);
  connect(pushButtonStopRefreshing, SIGNAL(toggled(bool)), timerRefresh, SLOT(start()));
  groupBoxHLayout->addWidget(pushButtonStopRefreshing);

  pushButtonKillIdleThreads = new QPushButton(this);
  pushButtonKillIdleThreads->setIcon(DIcon::Trash());
  connect(pushButtonKillIdleThreads, SIGNAL(clicked()), this, SLOT(killIdleThreadsSlot()));
  killingBoxHLayout->addWidget(pushButtonKillIdleThreads);
  spinBoxTimeLimit = new QSpinBox;
  spinBoxTimeLimit->setRange(0, 2147483647);
  killingBoxHLayout->addWidget(spinBoxTimeLimit);
  spinBoxTimeLimit->setValue(settings.value("Processes/TimeToKillThreads", 30).toUInt());

  pushButtonKillBusyThreads = new QPushButton(this);
  pushButtonKillBusyThreads->setIcon(DIcon::Trash());
  connect(pushButtonKillBusyThreads, SIGNAL(clicked()), this, SLOT(pushButtonKillBusyThreadsSlot()));
  killingBoxHLayout->addWidget(pushButtonKillBusyThreads);
  KillBusyThreadsTimeLimitSpinBox = new QSpinBox;
  KillBusyThreadsTimeLimitSpinBox->setRange(300, 2147483647);
  killingBoxHLayout->addWidget(KillBusyThreadsTimeLimitSpinBox);
  KillBusyThreadsTimeLimitSpinBox->setValue(settings.value("Processes/TimeToKillBusyThreads", 300).toUInt());

  spinBoxRefreshRate = new QSpinBox;
  spinBoxRefreshRate->setRange(0, 2147483647);
  groupBoxHLayout->addWidget(spinBoxRefreshRate);
  spinBoxRefreshRate->setValue(settings.value("Processes/RefreshRate", 1).toUInt());
  connect(spinBoxRefreshRate, SIGNAL(valueChanged(int)), this, SLOT(refreshRateSlot(int)));

  optionPROCESSLIST = new QRadioButton();
  optionPROCESSLIST->setChecked(!useTable);
  connect(optionPROCESSLIST, SIGNAL(clicked(bool)), this, SLOT(optionPROCESSLISTClicked(bool)));
  optionPROCESSES = new QRadioButton();
  optionPROCESSES->setChecked(useTable);
  connect(optionPROCESSES, SIGNAL(clicked(bool)), this, SLOT(optionPROCESSESClicked(bool)));
  groupBoxHLayout->addWidget(optionPROCESSLIST);
  groupBoxHLayout->addWidget(optionPROCESSES);

//  QFrame *line = new QFrame(this);
//  line->setFrameShape(QFrame::HLine);
//  line->setLineWidth(10);
//  killingBoxHLayout->addWidget(line);

  userListComboBox = new QComboBox();
  userListComboBox->insertItems(0, QStringList() << tr("All users"));
  userListComboBox->insertItems(1, serverConnection->runQuerySingleColumn("SELECT DISTINCT `User` FROM `information_schema`.`PROCESSLIST` ORDER BY `User`", false));
  connect(userListComboBox, SIGNAL(textActivated(QString)), this, SLOT(userListComboBoxTextActivated(QString)));
  groupBoxHLayout->addWidget(userListComboBox);
  groupBoxHLayout->addStretch(1);
  killingBoxHLayout->addStretch(1);
  buttonGroup->setLayout(qVBoxLayout);
  mainLayout->addWidget(buttonGroup);
  mainLayout->addWidget(processTable);
  QWidget *widMain = new QWidget;
  widMain->setLayout(mainLayout);
  killThread = new QAction(this);
  killThread->setIcon(DIcon::Close());
  connect(killThread, SIGNAL(triggered()), this, SLOT(killThreadSlot()));
  menu = new QMenu(this);
  menu->addAction(killThread);
  killQuery = new QAction(this);
  killQuery->setIcon(DIcon::Close());
  connect(killQuery, SIGNAL(triggered()), this, SLOT(killQuerySlot()));
  menu->addAction(killQuery);
  timerRefresh->setInterval(1000 * settings.value("Processes/RefreshRate", 1).toInt());
  timerRefresh->start();
  retranslateUI();
  setWidget(widMain);
}

void ProcessList::retranslateUI()
{
  setWindowTitle(tr("Process List"));
  setObjectName(windowTitle());
  dTitleLabel->setText(windowTitle());
  buttonGroup->setTitle(tr("Actions"));
  pushButtonStopRefreshing->setText(tr("Stop refreshing"));
  killThread->setText(tr("Kill thread"));
  killThread->setToolTip(tr("Kills the given thread."));
  pushButtonKillIdleThreads->setText(tr("Kill idle threads"));
  pushButtonKillIdleThreads->setToolTip(tr("Kills thread exeding the given seconds inactive."));
  spinBoxTimeLimit->setToolTip(tr("Time to kill threads."));
  spinBoxTimeLimit->setSuffix(" " + tr("seconds"));
  spinBoxRefreshRate->setToolTip(tr("Refresh rate."));
  spinBoxRefreshRate->setPrefix(tr("Refresh rate:") + " ");
  spinBoxRefreshRate->setSuffix(" " + tr("seconds"));
  killQuery->setText(tr("Kill query"));
  killQuery->setToolTip(tr("Kills the given query."));
  optionPROCESSLIST->setText(tr("Process list"));
  optionPROCESSLIST->setToolTip(optionPROCESSLIST->text());
  optionPROCESSES->setText(tr("Process table"));
  optionPROCESSES->setToolTip(optionPROCESSES->text());
  pushButtonKillBusyThreads->setText(tr("Kill busy threads"));
  pushButtonKillBusyThreads->setToolTip(tr("Kills thread exeding the given seconds busy."));
  KillBusyThreadsTimeLimitSpinBox->setToolTip(tr("Time to kill busy threads."));
  KillBusyThreadsTimeLimitSpinBox->setSuffix(" " + tr("seconds"));
}

void ProcessList::setHeaders()
{
  QStringList headerList = this->serverConnection->processes()->getHeaderList(useTable);
  headers = new QList<QStringList>;
  for (int counter = 0; counter < headerList.count(); counter++) {
    headers->append(QStringList() << headerList.at(counter) << "NoDelegate"  << "" << "Left");
  }
}

void ProcessList::reSetHeaders(bool checked)
{
  useTable = checked;
  settings.setValue("Processes/UseTable", useTable);
  setHeaders();
  processTable->setHeaders(headers);
}

void ProcessList::killThreadSlot()
{
  serverConnection->processes()->killThread(killThread->text().split(": ").at(1).toLongLong());
}

void ProcessList::killIdleThreadsSlot()
{
  settings.setValue("Processes/TimeToKillThreads", spinBoxTimeLimit->value());
  serverConnection->processes()->killIdleThreads(spinBoxTimeLimit->value());
}

void ProcessList::refreshRateSlot(const int value)
{
  timerRefresh->setInterval(1000 * value);
  settings.setValue("Processes/RefreshRate", value);
}

void ProcessList::killQuerySlot()
{
  serverConnection->processes()->killQuery(killQuery->text().split(": ").at(1).toLongLong());
}

void ProcessList::optionPROCESSLISTClicked(bool checked)
{
  reSetHeaders(!checked);
  userListComboBox->setCurrentIndex(0);
}

void ProcessList::optionPROCESSESClicked(bool checked)
{
  reSetHeaders(checked);
}

void ProcessList::pushButtonKillBusyThreadsSlot()
{
  settings.setValue("Processes/TimeToKillBusyThreads", KillBusyThreadsTimeLimitSpinBox->value());
  serverConnection->processes()->killBusyThreads(KillBusyThreadsTimeLimitSpinBox->value());
}

void ProcessList::userListComboBoxTextActivated(QString user)
{
  if (!(user == tr("All users"))) {
    optionPROCESSES->setChecked(true);
    optionPROCESSESClicked(optionPROCESSES->isChecked());
  }
}

void ProcessList::reloadData()
{
  result = serverConnection->processes()->getProcessList(useTable, userListComboBox->currentText());
  if (result->count() > 0)
    processTable->setModelData(result, true, 0, false);
  if (pushButtonStopRefreshing->isChecked())
    timerRefresh->stop();
}

void ProcessList::contextMenuEvent(QContextMenuEvent *event)
{
  killThread->setText(tr("Kill thread: %1").arg(processTable->indexData(processTable->currentIndexItem().row(), 0, Qt::DisplayRole).toString()));
  killQuery->setText(tr("Kill query: %1").arg(processTable->indexData(processTable->currentIndexItem().row(), 0, Qt::DisplayRole).toString()));
  menu->exec(event->globalPos());
}
