/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef PREFERENCES_H
#define PREFERENCES_H

#include "dsettings.h"

class QComboBox;
class QCheckBox;
class QSpinBox;
class QLineEdit;
class DTableView;
class FileAssociations;
class QStandardItem;
class DFileSelector;
class DTitleLabel;
class QFormLayout;
class QGroupBox;
class LicenseTemplate;
class NetworkSettings;
class DStackedWidget;
class ApplicationTheme;
class CodeSnippets;
class QToolButton;
class QMenu;

#include "dmdisubwindow.h"
#include "dbms.h"

/**
 * @brief The Preferences class manages application-wide configuration.
 *
 * This class inherits from DMdiSubWindow and provides a centralized interface
 * for users to modify settings related to:
 * - General behavior (startup tips, session restoration).
 * - Appearance (themes, fonts, background images).
 * - Text Editor (indentation, whitespace, keywords).
 * - Network (proxies).
 * - Database Logging and Connections.
 * - File Associations and Templates.
 */
class Preferences : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the Preferences window.
   * @param serverConnection Pointer to the active DBMS connection (used for logging settings).
   */
  Preferences(DBMS *serverConnection);

private:
  DBMS *serverConnection;
  QComboBox *stylesCombo;
  QComboBox *iconsStylesCombo;
  QSpinBox *tabSizeSpinBox;
  QCheckBox *checkBoxRememberWindows;
  QCheckBox *checkBoxOpenLastFile;
  DSettings settings;
  QCheckBox *checkBoxAutomaticIndentation;
  QCheckBox *checkBoxCleanwhiteSpaces;
  QCheckBox *checkBoxSaveABackupFile;
  DTableView *listConnections;
  QList<QStringList> *connectionsData;
  FileAssociations *fileAssociationsTab;
  QCheckBox *checkBoxSaveQueryBeforeExecution;
  //QCheckBox *checkBoxShowTabsAndSpaces;
  DTitleLabel *dTitleLabel;
  QFormLayout *styleFLayout;
  QGroupBox *styleGroupBox;
  QFormLayout *textEditorFLayout;
  LicenseTemplate *licenseTemplate;
  NetworkSettings *networkSettings;
  FileAssociations *fileAssociations;

  /**
   * @brief Populates the connection list and other data models.
   */
  void fillModelData();

  DStackedWidget *dStackedWidget;
  QCheckBox *checkBoxEnableQueryLog;
  QCheckBox *checkBoxEnableErrorLog;
  ApplicationTheme *applicationTheme;
  DFileSelector *fileSelectorBackgroundImage;
  QCheckBox *checkBoxAutoreconnect;
  CodeSnippets *codeSnippets;
  QCheckBox *checkBoxSelectOnPaste;
  QCheckBox *checkBoxShowTrayIconMessges;
  QToolButton *restoreBackgroudImageToolButton;
  QToolButton *backgroudImageHelpToolButton;
  QComboBox *remoteConnextionForLogginComboBox;

  /**
   * @brief Retranslates the user interface.
   *
   * Updates all labels, tooltips, and lists to reflect the current language.
   */
  void retranslateUI();

  QCheckBox *checkBoxUppercaseKeywordsAndReservedWords;
  QMenu *menu;
  QAction *deleteCurrentLineAction;
  QCheckBox *showStartupTips;

private slots:
  /**
   * @brief Updates the application style (e.g., Fusion, Windows).
   * @param index The index of the selected style.
   */
  void stylesComboCurrentIndexChanged(const int index);

  /**
   * @brief Updates the application icon theme.
   * @param index The index of the selected icon set.
   */
  void iconsStylesComboCurrentIndexChanged(const int index);

  /**
   * @brief Toggles saving the window state on exit.
   */
  void checkBoxRememberWindowsStateChanged();

  /**
   * @brief Toggles opening the last used file on startup.
   */
  void checkBoxOpenLastFileStateChanged();

  /**
   * @brief Updates the tab size setting for the text editor.
   * @param value The number of spaces per tab.
   */
  void tabSizeSpinBoxValueChanged(int value);

  /**
   * @brief Toggles automatic indentation in the editor.
   * @param value The checkbox state.
   */
  void checkBoxAutomaticIndentationValueChanged(int value);

  /**
   * @brief Toggles cleaning trailing white spaces on save.
   * @param value The checkbox state.
   */
  void checkBoxCleanwhiteSpacesValueChanged(int value);

  /**
   * @brief Toggles creating a backup file (.bak) on save.
   * @param value The checkbox state.
   */
  void checkBoxSaveABackupFileValueChanged(int value);

  /**
   * @brief Handles changes to items in the connections list.
   * @param item The item that was modified.
   */
  void connectioItemChangedSlot(QStandardItem *item);

  /**
   * @brief Toggles the prompt to save queries before execution.
   */
  void checkBoxSaveQueryBeforeExecutionStateChanged();
  //void checkBoxShowTabsAndSpacesValueChanged(int value);

  /**
   * @brief Toggles the general query log.
   * @param value The checkbox state.
   */
  void checkBoxEnableQueryLogValueChanged(int value);

  /**
   * @brief Toggles the error log.
   * @param value The checkbox state.
   */
  void checkBoxEnableErrorLogValueChanged(int value);

  /**
   * @brief Toggles showing messages in the system tray.
   * @param value The checkbox state.
   */
  void checkBoxShowTrayIconMessgesValueChanged(int value);

  /**
   * @brief Handles the selection of a custom background image.
   */
  void fileSelectorBackgroundImageSlot();

  /**
   * @brief Toggles automatic reconnection features.
   * @param value The checkbox state.
   */
  void checkBoxAutoreconnectValueChanged(int value);

  /**
   * @brief Toggles selecting text after pasting.
   * @param value The checkbox state.
   */
  void checkBoxSelectOnPasteValueChanged(int value);

  /**
   * @brief Resets the background image to default.
   */
  void restoreBackgroudImageToolButtonClicked();

  /**
   * @brief Shows help regarding the background image feature.
   */
  void backgroudImageHelpToolButtonClicked();

  /**
   * @brief Updates the connection used for remote logging.
   * @param value The index of the selected connection.
   */
  void remoteConnextionForLogginComboBoxCurrentIndexChanged(int value);

  /**
   * @brief Toggles automatic uppercasing of SQL keywords.
   */
  void checkBoxUppercaseKeywordsAndReservedWordsStateChanged();

  /**
   * @brief Deletes the selected line/connection from the list.
   */
  void deleteCurrentLineActionTriggered();

  /**
   * @brief Toggles showing tips on startup.
   * @param value The checkbox state.
   */
  void showStartupTipsValueChanged(int value);

protected:
  /**
   * @brief Handles context menu events (e.g., for the connection list).
   * @param event The context menu event parameters.
   */
  void contextMenuEvent(QContextMenuEvent *event);
};

#endif // PREFERENCES_H
