/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QApplication>
#include <QStyleFactory>
#include <QVBoxLayout>
#include <QFormLayout>
#include <QGroupBox>
#include <QComboBox>
#include <QCheckBox>
#include <QSpinBox>
#include <QLineEdit>
#include <QListWidget>
#include <QStandardItemModel>
#include <QToolButton>
#include <QMessageBox>
#include <QMenu>
#include <QContextMenuEvent>

#include "preferences.h"
#include "dtitlelabel.h"
#include "licensetemplate.h"
#include "networksettings.h"
#include "dtableview.h"
#include "staticfunctions.h"
//#include "fileassociations.h"
#include "dfileselector.h"
#include "dstackedwidget.h"
#include "applicationtheme.h"
#include "codesnippets.h"
#include "dicon.h"

#include "QDebug"

Preferences::Preferences(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
  setWindowIcon(DIcon::Preferences());

  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  dTitleLabel = new DTitleLabel;
  mainVLayout->addWidget(dTitleLabel);

  dStackedWidget = new DStackedWidget;

  //General
  styleFLayout = new QFormLayout;
  stylesCombo = new QComboBox;
  stylesCombo->addItems(QStyleFactory::keys());
  stylesCombo->setCurrentIndex(stylesCombo->findText(settings.value("GeneralSettings/Style", QApplication::style()->objectName()).toString(), Qt::MatchFixedString));
  connect(stylesCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(stylesComboCurrentIndexChanged(int)));
  styleFLayout->addRow(" ", stylesCombo);
  iconsStylesCombo = new QComboBox;
  iconsStylesCombo->addItem("Calíope - 1");
  iconsStylesCombo->addItem("Calíope - 2");
  iconsStylesCombo->addItem("Calíope - 3");
  iconsStylesCombo->addItem("Calíope - 4");
  iconsStylesCombo->setCurrentIndex(iconsStylesCombo->findText(settings.value("GeneralSettings/IconsStyle", "Calíope - 1").toString(), Qt::MatchFixedString));
  connect(iconsStylesCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(iconsStylesComboCurrentIndexChanged(int)));
  styleFLayout->addRow(" ", iconsStylesCombo);
  checkBoxRememberWindows = new QCheckBox;
  checkBoxRememberWindows->setCheckState(settings.value("GeneralSettings/RememberOpenedWindows", false).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(checkBoxRememberWindows, SIGNAL(stateChanged(int)), this, SLOT(checkBoxRememberWindowsStateChanged()));
  styleFLayout->addRow(checkBoxRememberWindows);
  checkBoxOpenLastFile = new QCheckBox;
  checkBoxOpenLastFile->setCheckState(settings.value("GeneralSettings/OpenLastFile", false).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(checkBoxOpenLastFile, SIGNAL(stateChanged(int)), this, SLOT(checkBoxOpenLastFileStateChanged()));
  styleFLayout->addRow(checkBoxOpenLastFile);
  checkBoxSaveQueryBeforeExecution = new QCheckBox;
  checkBoxSaveQueryBeforeExecution->setCheckState(settings.value("GeneralSettings/SaveQueriesBeforeExecution", true).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(checkBoxSaveQueryBeforeExecution, SIGNAL(stateChanged(int)), this, SLOT(checkBoxSaveQueryBeforeExecutionStateChanged()));
  checkBoxEnableQueryLog = new QCheckBox;
  checkBoxEnableQueryLog->setCheckState(settings.value("GeneralSettings/EnableQueryLog", false).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(checkBoxEnableQueryLog, SIGNAL(stateChanged(int)), this, SLOT(checkBoxEnableQueryLogValueChanged(int)));
  remoteConnextionForLogginComboBox = new QComboBox;
  remoteConnextionForLogginComboBox->addItem(tr("None"), "None");
  QStringList connections = settings.allKeys("ServerConnections");
  int valueToRemove = connections.indexOf("StorePassword");
  if (valueToRemove >= 0)
      connections.removeAt(valueToRemove);
  valueToRemove = connections.indexOf("SortConnectionList");
  if (valueToRemove >= 0)
      connections.removeAt(valueToRemove);
  remoteConnextionForLogginComboBox->insertItems(1, connections);
  remoteConnextionForLogginComboBox->setCurrentIndex(remoteConnextionForLogginComboBox->findText(settings.value("GeneralSettings/QueryLogOnRemoteDatabase", "None").toString(), Qt::MatchFixedString));
  connect(remoteConnextionForLogginComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(remoteConnextionForLogginComboBoxCurrentIndexChanged(int)));
  checkBoxEnableErrorLog = new QCheckBox;
  checkBoxEnableErrorLog->setCheckState(settings.value("GeneralSettings/EnableErrorLog", false).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(checkBoxEnableErrorLog, SIGNAL(stateChanged(int)), this, SLOT(checkBoxEnableErrorLogValueChanged(int)));

  checkBoxUppercaseKeywordsAndReservedWords = new QCheckBox;
  checkBoxUppercaseKeywordsAndReservedWords->setCheckState(settings.value("GeneralSettings/UppercaseKeywordsAndReservedWords", false).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(checkBoxUppercaseKeywordsAndReservedWords, SIGNAL(stateChanged(int)), this, SLOT(checkBoxUppercaseKeywordsAndReservedWordsStateChanged()));

  checkBoxShowTrayIconMessges = new QCheckBox;
  checkBoxShowTrayIconMessges->setCheckState(settings.value("GeneralSettings/ShowTrayIconMessges", false).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(checkBoxShowTrayIconMessges, SIGNAL(stateChanged(int)), this, SLOT(checkBoxShowTrayIconMessgesValueChanged(int)));

  showStartupTips = new QCheckBox;
  showStartupTips->setCheckState(settings.value("GeneralSettings/ShowStartupTips", false).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(showStartupTips, SIGNAL(stateChanged(int)), this, SLOT(showStartupTipsValueChanged(int)));

  styleFLayout->addRow(checkBoxEnableQueryLog);
  styleFLayout->addRow(" ", remoteConnextionForLogginComboBox);
  styleFLayout->addRow(checkBoxEnableErrorLog);
  styleFLayout->addRow(checkBoxUppercaseKeywordsAndReservedWords);
  styleFLayout->addRow(checkBoxSaveQueryBeforeExecution);
  styleFLayout->addRow(checkBoxShowTrayIconMessges);
  styleFLayout->addRow(showStartupTips);
  checkBoxAutoreconnect = new QCheckBox;
  checkBoxAutoreconnect->setCheckState(settings.value("DBMS/Reconnect", 1).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(checkBoxAutoreconnect, SIGNAL(stateChanged(int)), this, SLOT(checkBoxAutoreconnectValueChanged(int)));
  styleFLayout->addRow(checkBoxAutoreconnect);
  styleGroupBox= new QGroupBox;
  styleGroupBox->setLayout(styleFLayout);
  fileSelectorBackgroundImage = new DFileSelector(DFileSelectorContexts::Image);
  fileSelectorBackgroundImage->setFileName(settings.value("GeneralSettings/BackgroundImage", ":/images/png/512/Caliope-1.png").toString());
  connect(fileSelectorBackgroundImage, SIGNAL(changed()), this, SLOT(fileSelectorBackgroundImageSlot()));
  fileSelectorBackgroundImage->setText(tr("Change default background image"));
  fileSelectorBackgroundImage->retranslateUI();
  backgroudImageHelpToolButton = new QToolButton(this);
  backgroudImageHelpToolButton->setIcon(DIcon::Help());
  connect(backgroudImageHelpToolButton, SIGNAL(clicked()), this, SLOT(backgroudImageHelpToolButtonClicked()));
  restoreBackgroudImageToolButton = new QToolButton(this);
  restoreBackgroudImageToolButton->setIcon(DIcon::Clear());
  connect(restoreBackgroudImageToolButton, SIGNAL(clicked()), this, SLOT(restoreBackgroudImageToolButtonClicked()));

  QHBoxLayout *backgroundImageLayout = new QHBoxLayout();
  backgroundImageLayout->addWidget(fileSelectorBackgroundImage);
  backgroundImageLayout->addWidget(backgroudImageHelpToolButton);
  backgroundImageLayout->addWidget(restoreBackgroudImageToolButton);

  styleFLayout->addRow(backgroundImageLayout);
  QVBoxLayout *generalLayout = new QVBoxLayout;
  generalLayout->addWidget(styleGroupBox);
  generalLayout->addStretch();
  QWidget *widgetGeneral = new QWidget;
  widgetGeneral->setLayout(generalLayout);
  dStackedWidget->addWidget(widgetGeneral, DIcon::Database(), tr("General"));

  //Texteditor
  textEditorFLayout = new QFormLayout;
  tabSizeSpinBox = new QSpinBox;
  tabSizeSpinBox->setRange(2, 30);
  tabSizeSpinBox->setValue(settings.value("TextEditor/TabSize", tabSizeSpinBox->minimum()).toInt());
  connect(tabSizeSpinBox, SIGNAL(valueChanged(int)), this, SLOT(tabSizeSpinBoxValueChanged(int)));
  textEditorFLayout->addRow(" ", tabSizeSpinBox);
  checkBoxAutomaticIndentation = new QCheckBox;
  checkBoxAutomaticIndentation->setChecked(settings.value("TextEditor/AutomaticIndentation", false).toBool());
  connect(checkBoxAutomaticIndentation, SIGNAL(stateChanged(int)), this, SLOT(checkBoxAutomaticIndentationValueChanged(int)));
  textEditorFLayout->addRow(checkBoxAutomaticIndentation);
  checkBoxCleanwhiteSpaces = new QCheckBox;
  checkBoxCleanwhiteSpaces->setChecked(settings.value("TextEditor/CleanWhiteSpaces", false).toBool());
  connect(checkBoxCleanwhiteSpaces, SIGNAL(stateChanged(int)), this, SLOT(checkBoxCleanwhiteSpacesValueChanged(int)));
  textEditorFLayout->addRow(checkBoxCleanwhiteSpaces);
  checkBoxSaveABackupFile = new QCheckBox;
  checkBoxSaveABackupFile->setChecked(settings.value("TextEditor/CreateBackupFile", true).toBool());
  connect(checkBoxSaveABackupFile, SIGNAL(stateChanged(int)), this, SLOT(checkBoxSaveABackupFileValueChanged(int)));
  textEditorFLayout->addRow(checkBoxSaveABackupFile);

  checkBoxSelectOnPaste = new QCheckBox;
  checkBoxSelectOnPaste->setChecked(settings.value("TextEditor/SelectOnPaste", true).toBool());
  connect(checkBoxSelectOnPaste, SIGNAL(stateChanged(int)), this, SLOT(checkBoxSelectOnPasteValueChanged(int)));
  textEditorFLayout->addRow(checkBoxSelectOnPaste);

//  checkBoxShowTabsAndSpaces = new QCheckBox;
//  checkBoxShowTabsAndSpaces->setChecked(settings.value("TextEditor/ShowTabsAndSpaces", false).toBool());
//  connect(checkBoxShowTabsAndSpaces, SIGNAL(stateChanged(int)), this, SLOT(checkBoxShowTabsAndSpacesValueChanged(int)));
//  textEditorFLayout->addRow(checkBoxShowTabsAndSpaces);
  QWidget *widgetTextEditor = new QWidget;
  widgetTextEditor->setLayout(textEditorFLayout);
  dStackedWidget->addWidget(widgetTextEditor, DIcon::TextEditor(), tr("Text Editor"));

  //License template
  licenseTemplate = new LicenseTemplate;
  dStackedWidget->addWidget(licenseTemplate, DIcon::Database(), tr("License Template"));

  //Network Settings
  networkSettings = new NetworkSettings;
  dStackedWidget->addWidget(networkSettings, DIcon::Preferences(), tr("Network Settings"));

  //Connections settings
  /**
  * Pending translation
  */
  QString NoDelegate = StaticFunctions::DelegateTypeNoDelegate();
  QList<QStringList> *connectionsDTableViewHeaders = new QList<QStringList>;
  //0 - Connection type
  //1 - User
  //2 - Host
  //3 - Port
  //4 - Database
  //5 - Conexion count -- No parsed but keeped the space.
  //6 - Collation
  //7 - UseSSL
  //8 - KeyFile
  //9 - CertFile
  //10 - Password
  //11 - Connection name

  QString collationsJoined;
  QList<QStringList> *collations = serverConnection->getCollationsApplicability();
  for (int row = 0; row < collations->count() - 1; row++)
    collationsJoined += collations->at(row).at(1) + "|" + collations->at(row).at(0) + ",";
  collationsJoined = collationsJoined.left(-1);

  connectionsDTableViewHeaders->append(QStringList() << tr("Name") << NoDelegate << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("Type") << (StaticFunctions::DelegateTypeEnum() + "(" + StaticFunctions::dbmsEnabled().join(",") + ")") << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("User") << NoDelegate << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("Server") << NoDelegate << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("Port") << StaticFunctions::DelegateTypeNumber() << "" << "Right");
  connectionsDTableViewHeaders->append(QStringList() << tr("Database") << NoDelegate << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("Count") << StaticFunctions::DelegateTypeNumber() << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("Collation") << (StaticFunctions::DelegateTypeEnum() + "(" + collationsJoined + ")") << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("Use SSL") << NoDelegate << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("Client Key File") << NoDelegate << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("Client Cert File") << NoDelegate << "" << "Left");
  connectionsDTableViewHeaders->append(QStringList() << tr("Password") << StaticFunctions::DelegateTypePassword() << "" << "Left");
  listConnections = new DTableView(connectionsDTableViewHeaders);
//  connect(listConnections, SIGNAL(loadStarted(QString,uint,double)), parentWidget, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
//  connect(listConnections, SIGNAL(loadFinished(QString,uint,double)), parentWidget, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
//  connect(listConnections, SIGNAL(loadProgress(QString,uint,double)), parentWidget, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
  connect(listConnections, SIGNAL(itemChanged(QStandardItem*)), this, SLOT(connectioItemChangedSlot(QStandardItem*)));
  connectionsData = new QList<QStringList>();
  fillModelData();
  dStackedWidget->addWidget(listConnections, DIcon::WalletOpen(), tr("Connections Settings"));
  menu = new QMenu(this);
  deleteCurrentLineAction = new QAction(this);
  deleteCurrentLineAction->setIcon(DIcon::Trash());
  connect(deleteCurrentLineAction, SIGNAL(triggered()), this, SLOT(deleteCurrentLineActionTriggered()));
  menu->addAction(deleteCurrentLineAction);

  //File Associations
//  fileAssociations = new FileAssociations;
//  dStackedWidget->addWidget(fileAssociations, DIcon::Database(), tr("File Associations"));

  //Application Theme
  applicationTheme = new ApplicationTheme;
  dStackedWidget->addWidget(applicationTheme, DIcon::Preferences(), tr("Application Theme"));

  //Code Snippets
  codeSnippets = new CodeSnippets;
  dStackedWidget->addWidget(codeSnippets, DIcon::Database(), tr("Code Snippets"));

  mainVLayout->addWidget(dStackedWidget);
  QWidget *widMain = new QWidget;
  widMain->setLayout(mainVLayout);
  retranslateUI();
  setWidget(widMain);
}

void Preferences::fillModelData()
{
  connectionsData->clear();
  QStringList connections = settings.allKeys("ServerConnections");
  int valueToRemove = connections.indexOf("StorePassword");
  if (valueToRemove >= 0)
    connections.removeAt(valueToRemove);
  valueToRemove = connections.indexOf("SortConnectionList");
  if (valueToRemove >= 0)
    connections.removeAt(valueToRemove);
  QMap<QString, QVariant> connectionsSettings;
  foreach (QString connection, connections) {
    connectionsSettings = settings.value("ServerConnections/" + connection).toMap();

    connectionsData->append(QStringList() << ""
                            << connectionsSettings.value("Name").toString()
                            << connectionsSettings.value("ConnectionType").toString()
                            << connectionsSettings.value("User").toString()
                            << connectionsSettings.value("Host").toString()
                            << connectionsSettings.value("Port").toString()
                            << connectionsSettings.value("Database").toString()
                            << connectionsSettings.value("ConnectionsCount").toString()
                            << connectionsSettings.value("Collation").toString()
                            << connectionsSettings.value("UseSSL").toString()
                            << connectionsSettings.value("KeyFile").toString()
                            << connectionsSettings.value("CertFile").toString()
                            << connectionsSettings.value("Password").toString()
                            );
  }
  listConnections->setModelData(connectionsData, false);
}

void Preferences::retranslateUI()
{
  setWindowTitle(tr("Preferences"));
  setObjectName(windowTitle());
  dTitleLabel->setText(windowTitle());
  dTitleLabel->setToolTip(dTitleLabel->windowTitle());
  QLabel *label;
  label = qobject_cast<QLabel *>(styleFLayout->labelForField(stylesCombo));
  label->setText(tr("Select a style:"));
  label = qobject_cast<QLabel *>(styleFLayout->labelForField(iconsStylesCombo));
  label->setText(tr("Select an icon style:"));
  checkBoxRememberWindows->setText(tr("Remember opened windows"));
  checkBoxRememberWindows->setToolTip(checkBoxRememberWindows->text());
  checkBoxOpenLastFile->setText(tr("Open last file used in the modules"));
  checkBoxOpenLastFile->setToolTip(checkBoxOpenLastFile->text());
  checkBoxSaveQueryBeforeExecution->setText(tr("Save Queries before execution?"));
  styleGroupBox->setTitle(tr("Appilcation Style"));
  tabSizeSpinBox->setSuffix(" " + tr("Spaces"));
  label = qobject_cast<QLabel *>(textEditorFLayout->labelForField(tabSizeSpinBox));
  label->setText(tr("Tab size:"));
  label->setToolTip(label->text());
  checkBoxAutomaticIndentation->setText(tr("Enable automatic indentation."));
  checkBoxAutomaticIndentation->setToolTip(checkBoxAutomaticIndentation->text());
  checkBoxCleanwhiteSpaces->setText(tr("Clean white spaces at the end of the line."));
  checkBoxCleanwhiteSpaces->setToolTip(checkBoxCleanwhiteSpaces->text());
  checkBoxSaveABackupFile->setText(tr("Save a backup copy before save a file."));
  //checkBoxShowTabsAndSpaces->setText(tr("Show Tabs and Spaces."));
  checkBoxEnableQueryLog->setText(tr("Enable query log"));
  checkBoxEnableErrorLog->setText(tr("Enable error log"));
  checkBoxUppercaseKeywordsAndReservedWords->setText(tr("Uppercase keywords and reserverd words"));
  licenseTemplate->retranslateUI();
  networkSettings->retranslateUI();
//  fileAssociations->retranslateUI();
  checkBoxAutoreconnect->setText(tr("Use automatic reconnection"));
  checkBoxAutoreconnect->setToolTip(checkBoxAutoreconnect->text());
  checkBoxSelectOnPaste->setText(tr("Select the text after Paste operation."));
  checkBoxSelectOnPaste->setToolTip(checkBoxSelectOnPaste->text());
  checkBoxShowTrayIconMessges->setText(tr("Show messages on Tray Icon"));
  checkBoxShowTrayIconMessges->setToolTip(checkBoxShowTrayIconMessges->text());
  applicationTheme->retranslateUI();
  restoreBackgroudImageToolButton->setToolTip(tr("Restore default value"));
  label = qobject_cast<QLabel *>(styleFLayout->labelForField(remoteConnextionForLogginComboBox));
  label->setText(tr("Use a database connection to store the Query Log:"));
  deleteCurrentLineAction->setText(tr("Delete line: %1"));
  deleteCurrentLineAction->setToolTip(deleteCurrentLineAction->text());
  showStartupTips->setText(tr("Show Startup Tips"));
  showStartupTips->setToolTip(showStartupTips->text());
}

void Preferences::checkBoxCleanwhiteSpacesValueChanged(int value)
{
  settings.setValue("TextEditor/CleanWhiteSpaces", value);
}

void Preferences::checkBoxSaveABackupFileValueChanged(int value)
{
  settings.setValue("TextEditor/CreateBackupFile", value);
}

void Preferences::connectioItemChangedSlot(QStandardItem *item)
{
  //0 - Connection type
  //1 - User
  //2 - Host
  //3 - Port
  //4 - Database
  //5 - Conexion count -- No parsed but keeped the space.
  //6 - Collation
  //7 - UseSSL
  //8 - KeyFile
  //9 - CertFile
  //10 - Password
  //11 - Connection name

  settings.beginGroup("ServerConnections");
//  if ((item->data(Qt::UserRole).toString() != item->data(Qt::DisplayRole).toString()) && (item->column() == 0))
//    settings.remove(item->data(Qt::UserRole).toString());
  settings.setValue(listConnections->indexData(item->row(), 0, Qt::EditRole).toString()
                    , QString("%1:%2@%3:%4/%5 Count:%6 Collation:%7 UseSSL:%8 KeyFile:%9 CertFile:%10 Password:%11")
                    .arg(listConnections->indexData(item->row(), 1, Qt::EditRole).toString()
                    , listConnections->indexData(item->row(), 2, Qt::EditRole).toString()
                    , listConnections->indexData(item->row(), 3, Qt::EditRole).toString()
                    , listConnections->indexData(item->row(), 4, Qt::EditRole).toString()
                    , listConnections->indexData(item->row(), 5, Qt::EditRole).toString()
                    , listConnections->indexData(item->row(), 6, Qt::EditRole).toString()
                    , listConnections->indexData(item->row(), 7, Qt::EditRole).toString()
                    , listConnections->indexData(item->row(), 8, Qt::EditRole).toString()
                    , listConnections->indexData(item->row(), 9, Qt::EditRole).toString()
                    , listConnections->indexData(item->row(), 10, Qt::EditRole).toString()
                    , StaticFunctions::password(listConnections->indexData(item->row(), 11, Qt::EditRole).toString(), true)));
  settings.endGroup();
  fillModelData();
  listConnections->setModelData(connectionsData, false);
}

void Preferences::checkBoxSaveQueryBeforeExecutionStateChanged()
{
  settings.setValue("SaveQueriesBeforeExecution", checkBoxSaveQueryBeforeExecution->isChecked());
}

//void Preferences::checkBoxShowTabsAndSpacesValueChanged(int value)
//{
//  settings.setValue("TextEditor/ShowTabsAndSpaces", value);
//}

void Preferences::checkBoxEnableQueryLogValueChanged(int value)
{
  settings.setValue("GeneralSettings/EnableQueryLog", value);
  remoteConnextionForLogginComboBox->setEnabled(value);
  if (value == Qt::Checked)
    this->serverConnection->openTheDQueryLogger();
}

void Preferences::checkBoxEnableErrorLogValueChanged(int value)
{
  settings.setValue("GeneralSettings/EnableErrorLog", value);
}

void Preferences::checkBoxShowTrayIconMessgesValueChanged(int value)
{
  settings.setValue("GeneralSettings/ShowTrayIconMessges", value);
}

void Preferences::fileSelectorBackgroundImageSlot()
{
  settings.setValue("GeneralSettings/BackgroundImage", fileSelectorBackgroundImage->getFileName());
}

void Preferences::checkBoxAutoreconnectValueChanged(int value)
{
  settings.setValue("DBMS/Reconnect", value);
}

void Preferences::checkBoxSelectOnPasteValueChanged(int value)
{
  settings.setValue("TextEditor/SelectOnPaste", value);
}

void Preferences::restoreBackgroudImageToolButtonClicked()
{
  fileSelectorBackgroundImage->setFileName("Random");
  fileSelectorBackgroundImageSlot();
}

void Preferences::backgroudImageHelpToolButtonClicked()
{
  QMessageBox::information(this, fileSelectorBackgroundImage->getText()
                           , tr("Select a file to use or type Random to use a Calíope specific image."));
}

void Preferences::remoteConnextionForLogginComboBoxCurrentIndexChanged(int value)
{
  settings.setValue("GeneralSettings/QueryLogOnRemoteDatabase", remoteConnextionForLogginComboBox->itemText(value));
}

void Preferences::checkBoxUppercaseKeywordsAndReservedWordsStateChanged()
{
  settings.setValue("GeneralSettings/UppercaseKeywordsAndReservedWords", checkBoxUppercaseKeywordsAndReservedWords->isChecked());
}

void Preferences::deleteCurrentLineActionTriggered()
{
  settings.remove("ServerConnections/" + listConnections->getRow(listConnections->currentIndexItem().row()).at(0)->text());
  fillModelData();
}

void Preferences::showStartupTipsValueChanged(int value)
{
  settings.setValue("GeneralSettings/ShowStartupTips", value);
}

void Preferences::contextMenuEvent(QContextMenuEvent *event)
{
  deleteCurrentLineAction->setText(tr("Delete line: %1").arg(listConnections->currentIndexItem().row() + 1));
  deleteCurrentLineAction->setToolTip(deleteCurrentLineAction->text());
  menu->exec(event->globalPos());
}

void Preferences::checkBoxAutomaticIndentationValueChanged(int value)
{
  settings.setValue("TextEditor/AutomaticIndentation", value);
}

void Preferences::tabSizeSpinBoxValueChanged(int value)
{
  settings.setValue("TextEditor/TabSize", value);
}

void Preferences::checkBoxOpenLastFileStateChanged()
{
  settings.setValue("GeneralSettings/OpenLastFile", checkBoxOpenLastFile->isChecked());
  if (!checkBoxOpenLastFile->isChecked()) {
    settings.setValue("GeneralSettings/LastMariaDBFile", "");
  }
}

void Preferences::checkBoxRememberWindowsStateChanged()
{
  settings.setValue("GeneralSettings/RememberOpenedWindows", checkBoxRememberWindows->isChecked());
}

void Preferences::stylesComboCurrentIndexChanged(const int index)
{
  qApp->setStyle(stylesCombo->itemText(index));
  settings.setValue("GeneralSettings/Style", stylesCombo->itemText(index));
}

void Preferences::iconsStylesComboCurrentIndexChanged(const int index)
{
  settings.setValue("GeneralSettings/IconsStyle", iconsStylesCombo->itemText(index));
}
