/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QVBoxLayout>
#include <QGroupBox>
#include <QPushButton>
#include <QCheckBox>
#include <QRadioButton>
#include <QTreeWidget>
#include <QTimer>
#include <QApplication>
#include <QMessageBox>
#include <QSplitter>

#include "objectmigration.h"
#include "dtitlelabel.h"
#include "basetexteditor.h"
#include "connectdialog.h"
#include "dicon.h"

#include "qdebug.h"

ObjectMigration::ObjectMigration(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;

  secondaryServerConnection = new DBMS(false);
  //connect(secondaryServerConnection, SIGNAL(statusBarMessage(QString)), this, SLOT(statusBarMessageSlot(QString)));

  setWindowIcon(DIcon::GoNext());
  QWidget *widMain = new QWidget;
  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  dTitleLabel = new DTitleLabel;
  mainVLayout->addWidget(dTitleLabel);
  groupBoxAction = new QGroupBox(this);
  QHBoxLayout *thirdLayout = new QHBoxLayout;
  groupBoxAction->setLayout(thirdLayout);
  mainVLayout->addWidget(groupBoxAction);

  migratePushButton = new QPushButton;
  connect(migratePushButton, SIGNAL(clicked()), this, SLOT(migratePushButtonSlot()));
  migratePushButton->setIcon(DIcon::GoNext());
  thirdLayout->addWidget(migratePushButton);
  optionDROP = new QCheckBox;
  thirdLayout->addWidget(optionDROP);
  optionExportData = new QCheckBox;
  thirdLayout->addWidget(optionExportData);
  optionFOREIGN_KEY_CHECKS = new QCheckBox;
  thirdLayout->addWidget(optionFOREIGN_KEY_CHECKS);
  optionPreview = new QCheckBox;
  thirdLayout->addWidget(optionPreview);
  stopMigrationPushButton = new QPushButton;
  connect(stopMigrationPushButton, SIGNAL(clicked()), this, SLOT(stopMigrationPushButtonSlot()));
  stopMigrationPushButton->setIcon(DIcon::Stop());
  QFrame* separatorFrame = new QFrame();
  separatorFrame->setFrameShape(QFrame::VLine);
  thirdLayout->addWidget(separatorFrame);
  thirdLayout->addWidget(stopMigrationPushButton);

  thirdLayout->addStretch();
  QSplitter *mainSplitter = new QSplitter(Qt::Horizontal);
  QSizePolicy sizePolicy2 = mainSplitter->sizePolicy();
  sizePolicy2.setVerticalPolicy(QSizePolicy::Expanding);
  mainSplitter->setSizePolicy(sizePolicy2);
  objectsListWidget = new QTreeWidget;
  connect(objectsListWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(itemActivatedSlot(QTreeWidgetItem*,int)));
  connect(objectsListWidget, SIGNAL(itemChanged(QTreeWidgetItem*,int)), this, SLOT(itemActivatedSlot(QTreeWidgetItem*,int)));
  mainSplitter->addWidget(objectsListWidget);
//  mainVLayout->addLayout(secondLayout);
  resultEditor = new BaseTextEditor(EditorTypes::NoEditor);
  resultEditor->setMaximumBlockCount(5000);
  resultEditor->setWordWrapMode(QTextOption::NoWrap);
  mainSplitter->addWidget(resultEditor);
  mainVLayout->addWidget(mainSplitter);

  retranslateUI();
  widMain->setLayout(mainVLayout);
  setWidget(widMain);
  statementsToExecute = QStringList();
  QTimer::singleShot(0, this, SLOT(fillDatabasesSlot()));
}

void ObjectMigration::retranslateUI()
{
  setWindowTitle(tr("Object Migration"));
  setObjectName(windowTitle());
  dTitleLabel->setText(windowTitle());
  dTitleLabel->setToolTip(dTitleLabel->text());
  groupBoxAction->setTitle(tr("Options"));
  groupBoxAction->setToolTip(groupBoxAction->title());
  objectsListWidget->setWindowTitle(tr("Objects"));
  objectsListWidget->setToolTip(objectsListWidget->windowTitle());
  objectsListWidget->setHeaderLabel(objectsListWidget->windowTitle());
  optionDROP->setText(tr("Replace on destination"));
  optionDROP->setToolTip(optionDROP->text());
  migratePushButton->setText(tr("Migrate"));
  migratePushButton->setToolTip(migratePushButton->text());
  optionExportData->setText(tr("Export data"));
  optionExportData->setToolTip(optionExportData->text());
  optionFOREIGN_KEY_CHECKS->setText(tr("Skip Foreign Key checks"));
  optionFOREIGN_KEY_CHECKS->setToolTip(optionFOREIGN_KEY_CHECKS->text());
  optionPreview->setText(tr("Preview"));
  optionPreview->setToolTip(optionPreview->text());
  stopMigrationPushButton->setText(tr("Stop migration"));
  stopMigrationPushButton->setToolTip(stopMigrationPushButton->text());
}

void ObjectMigration::fillDatabasesSlot()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  databases.clear();
  foreach (QString database, serverConnection->getDatabases(true)) {
    QTreeWidgetItem *item = new QTreeWidgetItem((QTreeWidget*)0, QStringList(database), ItemTypes::Database);
    item->setIcon(0, DIcon::Database());
    item->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    item->setCheckState(0, Qt::Unchecked);
    databases.append(item);
  }
  objectsListWidget->clear();
  objectsListWidget->insertTopLevelItems(0, databases);
  objectsListWidget->resizeColumnToContents(0);
  QApplication::restoreOverrideCursor();
}

void ObjectMigration::itemActivatedSlot(QTreeWidgetItem *item, int column)
{
  QTreeWidgetItem *itemChild;
  if (item->type() == ItemTypes::Database) {
    QApplication::setOverrideCursor(Qt::WaitCursor);
    emit loadProgress(0);
    foreach (QTreeWidgetItem *tableItem, databases)
      if (tableItem->parent() == item)
        tableItem->setCheckState(column, item->checkState(column));
    if (item->childCount() == 0) {
      foreach (QString tables, serverConnection->tables()->list(item->text(0))) {
        itemChild = new QTreeWidgetItem(item, QStringList(tables), ItemTypes::Table);
        itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
        itemChild->setIcon(0, DIcon::DatabaseTable());
        itemChild->setCheckState(0, Qt::Checked);
        databases.append(itemChild);
      }
      emit loadProgress(16);
      foreach (QString tables, serverConnection->views()->list(item->text(0))) {
        itemChild = new QTreeWidgetItem(item, QStringList(tables), ItemTypes::View);
        itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
        itemChild->setIcon(0, DIcon::DatabaseView());
        itemChild->setCheckState(0, Qt::Checked);
        databases.append(itemChild);
      }
      emit loadProgress(32);
      foreach (QString tables, serverConnection->events()->list(item->text(0))) {
        itemChild = new QTreeWidgetItem(item, QStringList(tables), ItemTypes::Event);
        itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
        itemChild->setIcon(0, DIcon::DatabaseEvent());
        itemChild->setCheckState(0, Qt::Checked);
        databases.append(itemChild);
      }
      emit loadProgress(48);
      foreach (QString tables, serverConnection->functions()->list(item->text(0))) {
        itemChild = new QTreeWidgetItem(item, QStringList(tables), ItemTypes::Routine);
        itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
        itemChild->setIcon(0, DIcon::QueryExecution());
        itemChild->setCheckState(0, Qt::Checked);
        databases.append(itemChild);
      }
      emit loadProgress(64);
      foreach (QString tables, serverConnection->procedures()->list(item->text(0))) {
        itemChild = new QTreeWidgetItem(item, QStringList(tables), ItemTypes::Procedure);
        itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
        itemChild->setIcon(0, DIcon::QueryExecution());
        itemChild->setCheckState(0, Qt::Checked);
        databases.append(itemChild);
      }
      emit loadProgress(80);
      foreach (QString tables, serverConnection->triggers()->list(item->text(0))) {
        itemChild = new QTreeWidgetItem(item, QStringList(tables), ItemTypes::Trigger);
        itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
        itemChild->setIcon(0, DIcon::Database());
        itemChild->setCheckState(0, Qt::Checked);
        databases.append(itemChild);
      }
      emit loadProgress(100);
    }
    QApplication::restoreOverrideCursor();
  }
}

void ObjectMigration::migratePushButtonSlot()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  emit loadProgress(0);
  statementsToExecute.clear();
  counter = 0;
  int counter2 = 0;
  if (optionFOREIGN_KEY_CHECKS->isChecked())
    statementsToExecute.append("SET FOREIGN_KEY_CHECKS := 0");
  foreach (QTreeWidgetItem *item, databases) {
    if (item->checkState(0) == Qt::Checked) {
      switch (item->type()) {
      case ItemTypes::Database:
        if (optionDROP->isChecked())
          statementsToExecute.append("DROP DATABASE IF EXISTS " + item->text(0));
        statementsToExecute.append(serverConnection->databases()->getDefinition(item->text(0)));
        break;
      case ItemTypes::Table:
        if (optionDROP->isChecked())
          statementsToExecute.append("DROP TABLE IF EXISTS " + item->text(0));
        statementsToExecute.append("USE " + item->text(0).split(".").at(0));
        statementsToExecute.append(serverConnection->tables()->getDefinition(item->text(0)));
        if (optionExportData->isChecked())
          statementsToExecute.append(serverConnection->tables()->getTableDataToInsert(item->text(0)));
        break;
      case ItemTypes::View:
        if (optionDROP->isChecked())
          statementsToExecute.append("DROP VIEW IF EXISTS " + item->text(0));
        statementsToExecute.append("USE " + item->text(0).split(".").at(0));
        statementsToExecute.append(serverConnection->views()->getDefinition(item->text(0)));
        break;
      case ItemTypes::Event:
        if (optionDROP->isChecked())
          statementsToExecute.append("DROP EVENT IF EXISTS " + item->text(0));
        statementsToExecute.append("USE " + item->text(0).split(".").at(0));
        statementsToExecute.append(serverConnection->events()->getDefinition(item->text(0)));
        break;
      case ItemTypes::Routine:
        if (optionDROP->isChecked())
          statementsToExecute.append("DROP FUNCTION|PROCEDURE IF EXISTS " + item->text(0));
        statementsToExecute.append("USE " + item->text(0).split(".").at(0));
        statementsToExecute.append(serverConnection->functions()->getDefinition(item->text(0)));
        break;
      case ItemTypes::Procedure:
        if (optionDROP->isChecked())
          statementsToExecute.append("DROP PROCEDURE IF EXISTS " + item->text(0));
        statementsToExecute.append("USE " + item->text(0).split(".").at(0));
        statementsToExecute.append(serverConnection->procedures()->getDefinition(item->text(0)));
        break;
      case ItemTypes::Trigger:
        if (optionDROP->isChecked())
          statementsToExecute.append("DROP TRIGGER IF EXISTS " + item->text(0));
        statementsToExecute.append("USE " + item->text(0).split(".").at(0));
        statementsToExecute.append(serverConnection->triggers()->getDefinition(item->text(0)));
        break;
      default:
        break;
      }
    }
    emit loadProgress((int) (counter2 * 100 / databases.count()));
    counter2++;
  }
  if (optionFOREIGN_KEY_CHECKS->isChecked())
    statementsToExecute.append("SET FOREIGN_KEY_CHECKS := 1");
  emit loadProgress(100);
  QApplication::restoreOverrideCursor();

  ConnectDialog *connectFrom = new ConnectDialog(secondaryServerConnection);
  if (!optionPreview->isChecked()) {
    if (connectFrom->exec() == QDialog::Accepted) {
      if (!connectFrom->getConnectionPerformed()) {
        if (secondaryServerConnection->isOpened())
          secondaryServerConnection->close();
        secondaryServerConnection->setConnectionParameters(connectFrom->getValues());
        if (!secondaryServerConnection->open())
          QMessageBox::critical(this, tr("Cannot connect to the server"), secondaryServerConnection->lastError());
      }
    }
  } else {
    QMessageBox::information(this, tr("Preview option is enabled"), tr("Preview option is enabled. Disable it to migrate the objects."));
  }
  resultEditor->clear();
  delete(connectFrom);
  QTimer::singleShot(0, this, SLOT(statementsToExecuteSlot()));
}

void ObjectMigration::statementsToExecuteSlot()
{
  if (statementsToExecute.count() > counter) {
    if (optionPreview->isChecked())
      resultEditor->appendPlainText(statementsToExecute.at(counter));
    else
      resultEditor->appendPlainText(secondaryServerConnection->outputAsVV(statementsToExecute.at(counter)));
    emit loadProgress((int) (counter * 100 / statementsToExecute.count()));
    counter++;
    QTimer::singleShot(0, this, SLOT(statementsToExecuteSlot()));
  } else {
    emit loadProgress(100);
  }
}

void ObjectMigration::stopMigrationPushButtonSlot()
{
  counter = statementsToExecute.count() + 1;
  emit loadProgress(0);
}
