/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef NETWORKSETTINGS_H
#define NETWORKSETTINGS_H

#include <QWidget>

#include "dsettings.h"

class QLineEdit;
class QCheckBox;
class QSpinBox;
class QPushButton;
class DMessageLabel;
class QFormLayout;

/**
 * @brief The NetworkSettings class allows configuration of application network proxies.
 *
 * This widget provides a form interface for the user to input proxy server details
 * (host, port, user, and password) and toggle proxy usage on or off. It persists
 * these settings using the DSettings class.
 */
class NetworkSettings : public QWidget
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the NetworkSettings widget.
   *
   * Initializes the layout and loads any previously saved proxy configurations.
   */
  NetworkSettings();

  /**
   * @brief Applies the configured proxy settings to the application.
   *
   * Configures the global Qt network proxy based on the current state of the widgets.
   */
  void setProxySettings();

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels and button text to reflect the current application language.
   */
  void retranslateUI();

private:
  QLineEdit *serverLineEdit;
  QLineEdit *userLineEdit;
  QLineEdit *passwordLineEdit;
  QCheckBox *checkBoxUseProxy;
  QSpinBox *spinBoxPort;
  QPushButton *applyProxySettings;
  DMessageLabel *dMessageLabel;
  DSettings settings;
  QFormLayout *formLayout;

private slots:
  /**
   * @brief Handles changes to the proxy server address input.
   * @param proxy The new server address text.
   */
  void proxyChangedSlot(const QString proxy);

  /**
   * @brief Handles the toggling of the "Use Proxy" checkbox.
   *
   * Enables or disables the input fields based on the checkbox state.
   */
  void checkBoxUseProxyStateChanged();

  /**
   * @brief Handles changes to the proxy port.
   * @param value The new port number.
   */
  void portSpinBoxValueChanged(int value);

  /**
   * @brief Saves the current proxy configuration.
   *
   * Typically connected to the "Apply" button.
   */
  void proxySettings();

  /**
   * @brief Handles changes to the proxy username input.
   * @param user The new username text.
   */
  void userChangedSlot(const QString user);

  /**
   * @brief Handles changes to the proxy password input.
   * @param password The new password text.
   */
  void passwordChangedSlot(const QString password);
};

#endif // NETWORKSETTINGS_H
