/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <QMainWindow>
#include <QSystemTrayIcon>
#include <QUrl>
#include <QProcess>
#include <QNetworkAccessManager>

#include "dbms.h"
#include "dsettings.h"

class QLabel;
class ConnectDialog;
class QMdiArea;
class ServerInformation;
class QTreeWidget;
class Catalogs;
class QTreeWidgetItem;
class SQLQuery;
class QSignalMapper;
class DWebView;
class MariaDBHelp;
class ProcessList;
class TableMaintenance;
class Result;
class Backup;
class RestoreBackupFile;
class Users;
class QActionGroup;
class QTranslator;
class Preferences;
class QMdiSubWindow;
class TextEditor;
class DStatusBar;
class ObjectsDiagram;
class Projects;
class QSystemTrayIcon;
class DatabaseComparision;
class QProgressDialog;
class DMdiSubWindow;
class DQueryLog;
class DReportViewer;
class ObjectMigration;
class Backup;
class GitLabFileSearcher;
class DHelpViewer;
class DMonitoring;
class QDialog;
class QFileSystemWatcher;
class BaseTextEditor;
class DLoadData;
class DDashboard;
class DTour;

/**
 * @brief The MainWindow class is the central hub of the application.
 *
 * This class inherits from QMainWindow and manages the Multiple Document Interface (MDI)
 * area. It acts as the parent for all sub-windows (editors, catalogs, tools) and
 * handles global application actions, menus, toolbars, and the primary database connection.
 */
class MainWindow : public QMainWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the MainWindow.
   */
  MainWindow();

  /**
   * @brief Destructor.
   */
  ~MainWindow();

  /**
   * @brief Pointer to the main database management system connection.
   */
  DBMS *serverConnection;

  /**
   * @brief Custom status bar for displaying progress and messages.
   */
  DStatusBar *dStatusBar;

  /**
   * @brief Constant exit code used to signal an application reboot.
   */
  static int const EXIT_CODE_REBOOT = -123456789;

protected:
  /**
   * @brief Reimplements the closeEvent.
   *
   * Checks for unsaved changes or active processes before closing the application.
   * @param event The close event parameters.
   */
  void closeEvent(QCloseEvent *event);

  /**
   * @brief Reimplements the resizeEvent.
   *
   * Handles dynamic scaling of background images or UI elements when the window size changes.
   * @param resizeEvent The resize event parameters.
   */
  void resizeEvent(QResizeEvent *resizeEvent);
//  void moveEvent(QMoveEvent *event);

private slots:
  /**
   * @brief Displays the "About MariaDB GUI" information dialog.
   */
  void aboutMariaDBGUIActionTriggered();

  /**
   * @brief Opens the connection dialog to establish a link to a server.
   */
  void connectToServerActionTriggered();

  /**
   * @brief Opens the server information view.
   */
  void serverInformationActionTriggered();

  /**
   * @brief Opens the user administration interface.
   */
  void userAdministrationActionTriggered();

  /**
   * @brief Closes the currently active database connection.
   */
  void closeCurrentConnectionActionTriggered();

  /**
   * @brief Opens the database catalogs (schema browser).
   */
  void catalogsActionTriggered();

  /**
   * @brief Opens the MariaDB help documentation.
   */
  void mysqlHelpActionTriggered();

  /**
   * @brief Opens the server process list viewer.
   */
  void processActionTriggered();

  /**
   * @brief Opens the table maintenance tool (check, repair, optimize).
   */
  void tableMaintenanceActionTriggered();

  /**
   * @brief Opens the database backup tool.
   */
  void backupActionTriggered();

  /**
   * @brief Opens the database restore tool.
   */
  void restoreActionTriggered();

  /**
   * @brief Opens the application preferences dialog.
   */
  void preferencesActionTriggered();

  /**
   * @brief Opens a new SQL script editor.
   */
  void sqlScriptActionTriggered();

  /**
   * @brief Opens the query builder/execution window.
   */
  void queryActionTriggered();

  /**
   * @brief Opens a new empty query window.
   */
  void newEmtyQueryActionTriggered();

  /**
   * @brief Updates the "Window" menu list with active sub-windows.
   */
  void updateWindowMenu();

  /**
   * @brief Sets the active sub-window based on the menu selection.
   * @param window The window object mapped to the action.
   */
  void setActiveSubWindowMap(QObject *window);

  /**
   * @brief Switches the application language.
   * @param action The menu action containing the language selection.
   */
  void swithLanguage(QAction *action);

  /**
   * @brief Slot called when an MDI sub-window is activated.
   * @param window The newly active sub-window.
   */
  void mdiSubWindowActivatedSlot(QMdiSubWindow * window);

  /**
   * @brief Triggers the server shutdown command.
   */
  void shutdownServerActionTriggered();

  /**
   * @brief Populates the "Change Database" menu with available schemas.
   */
  void updateChangeDatabaseMenu();

  /**
   * @brief Slot to handle database change requests.
   */
  void changeDatabaseSlot();

  /**
   * @brief Changes the active database.
   * @param database The name of the target database.
   */
  void changeDatabaseSlot(QString database);

  /**
   * @brief Populates the "Change Charset" menu.
   */
  void updateChangeCharsetMenu();

  /**
   * @brief Changes the connection character set.
   * @param charset The name of the charset.
   */
  void changeCharsetSlot(QString charset);

  /**
   * @brief Toggles full-screen mode.
   * @param fullScreen True to enable full screen, false to exit.
   */
  void toggleFullScreenActionTriggered(const bool fullScreen);

  /**
   * @brief Displays the error log viewer.
   */
  void showErrorLogActionTriggered();

  /**
   * @brief Handles cleanup when the error log viewer is closed.
   */
  void showErrorLogActionClosed();

  /**
   * @brief Opens the Entity-Relationship (ER) objects diagram.
   */
  void objectsDiagramActionTriggered();

  /**
   * @brief Opens a file dialog to load an external file.
   */
  void openFileActionTriggered();

  /**
   * @brief Populates the "Recent Files" menu before it is shown.
   */
  void openRecentFilesMenuAboutToShowSlot();

  /**
   * @brief Opens a file selected from the recent files list.
   * @param action The action representing the file.
   */
  void recentFilesActionGroupTriggered(QAction *action);

  /**
   * @brief Opens the standalone text editor.
   */
  void textEditorActionTriggered();

  /**
   * @brief Sends a command to stop the replication slave.
   */
  void stopReplicationSlaveActionTriggered();

  /**
   * @brief Sends a command to start the replication slave.
   */
  void startReplicationSlaveActionTriggered();

  /**
   * @brief Reboots the replication slave process.
   */
  void rebootReplicationSlaveActionTriggered();

  /**
   * @brief Displays replication status information.
   */
  void replicationInformationActionTriggered();

  /**
   * @brief Flushes privileges on the server.
   */
  void maintenanceFlushPrivilegesActionTriggered();

  /**
   * @brief Flushes the host cache on the server.
   */
  void maintenanceFlushHostsActionTriggered();

  /**
   * @brief Purges binary logs on the server.
   */
  void maintenancePurgeBinaryLogsActionTriggered();

  /**
   * @brief Checks the time difference between client and server.
   */
  void maintenanceTimeDifferenceActionTriggered();

  /**
   * @brief Displays the slow query log.
   */
  void showSlowLogActionTriggered();

  /**
   * @brief Toggles read-only mode for the catalogs viewer.
   * @param readOnly True to enable read-only safety.
   */
  void catalogsReadOnlyActionSlot(bool readOnly);

  /**
   * @brief Shows the recent files list.
   */
  void viewRecentFilesActionTriggered();

  /**
   * @brief Clears the history of recent URLs/files.
   */
  void clearRecentURLsActionTriggered();

  /**
   * @brief Resets the replication slave configuration.
   */
  void maintenanceResetSlaveActionTriggered();

  /**
   * @brief Flushes the relay logs.
   */
  void maintenanceFlushRelayLogsActionTriggered();

  /**
   * @brief Opens the database comparison tool.
   */
  void databaseComparisionActionTriggered();

  /**
   * @brief Updates table row counts periodically.
   */
  void tableCountSlot();

  /**
   * @brief Prepares the connection menu before showing.
   */
  void connectionMenuAboutToShowSlot();

  /**
   * @brief Prepares the recent connections menu before showing.
   */
  void openRecentConnectionMenuAboutToShowSlot();

  /**
   * @brief Initiates a connection from the recent list.
   * @param action The action containing connection details.
   */
  void openRecentConnectionActionGroupTriggered(QAction *action);

  /**
   * @brief Opens the query execution history log.
   */
  void viewQueryLogActionSlot();

  /**
   * @brief Updates the database information menu.
   */
  void updatedatabaseInformationMenu();

  /**
   * @brief Displays information for a specific database.
   * @param database The name of the database.
   */
  void databaseInformationSlot(QString database);

  /**
   * @brief Takes a snapshot of the current state/screen.
   */
  void takeASnapShotActionTriggered();

  /**
   * @brief Views the source code of the current web view page.
   * @param pageSource The HTML source string.
   */
  void viewDWebViewPageSource(QString pageSource);

  /**
   * @brief Generates a report on server information.
   */
  void actionReportServerInformationTriggered();

  /**
   * @brief Generates a report on HDD usage.
   */
  void actionReportHDDUsageTriggered();

  /**
   * @brief Generates a report on executed queries.
   */
  void actionReportExecutedQueriesTriggered();

  /**
   * @brief Generates a report on data sent/received traffic.
   */
  void actionReportDataSentReceivedTriggered();

  /**
   * @brief Opens the interface to create a custom report.
   */
  void actionCreateCustomReportTriggered();

  /**
   * @brief Prepares the reports menu before showing.
   */
  void menuReportsAboutToShowSlot();

  /**
   * @brief Opens a specific custom report.
   * @param report The name or ID of the report.
   */
  void openCustomReport(QString report);

  /**
   * @brief Generates a report on slow queries.
   */
  void actionReportSlowQueriesTriggered();

  /**
   * @brief Opens the feedback mechanism.
   */
  void tellUsYourCommentsActionTriggered();

  /**
   * @brief Opens the project website in a browser.
   */
  void openProjectSiteActionTriggered();

  /**
   * @brief Stops all configured replication slaves.
   */
  void stopAllReplicationSlavesActionTriggered();

  /**
   * @brief Starts all configured replication slaves.
   */
  void startAllReplicationSlavesActionTriggered();

  /**
   * @brief Handles UI updates after a successful reconnection.
   */
  void reconnectionPerformedSlot();

  /**
   * @brief Decrements the counter for open SQL windows.
   */
  void decrementSQLWindowCounter();

  /**
   * @brief Decrements the counter for open Query windows.
   */
  void decrementSQLQueryWindowCounter();

  /**
   * @brief Decrements the counter for open Text Editor windows.
   */
  void decrementTextWindowCounter();

  /**
   * @brief Exports application settings to a file.
   */
  void exportSettingsActionTriggered();

  /**
   * @brief Imports application settings from a file.
   */
  void importSettingsActionTriggered();

  /**
   * @brief Triggers "Save All" for open documents.
   */
  void saveAllActionTriggered();

  /**
   * @brief Opens the table migration tool.
   */
  void migrateTableActionTriggered();

  /**
   * @brief Displays database metadata.
   */
  void databaseMetadataActionTriggered();

  /**
   * @brief Prepares the replication menu before showing.
   */
  void replicationMenuAboutToShowSlot();

  /**
   * @brief Starts a specific replication connection.
   * @param connectionName The name of the connection.
   */
  void replicationStartConnection(QString connectionName);

  /**
   * @brief Stops a specific replication connection.
   * @param connectionName The name of the connection.
   */
  void replicationStopConnection(QString connectionName);

  /**
   * @brief Restarts a specific replication connection.
   * @param connectionName The name of the connection.
   */
  void replicationRestartConnection(QString connectionName);

  /**
   * @brief Resets a specific replication connection.
   * @param connectionName The name of the connection.
   */
  void replicationResetConnection(QString connectionName);

  /**
   * @brief Opens the object migration tool.
   */
  void objectMigrationActionTriggered();

  /**
   * @brief Opens the GitLab file searcher tool.
   */
  void gitLabFileSearcherActionTriggered();

  /**
   * @brief Opens the MariaDB Explorer help.
   */
  void mariadbExplorerHelpActionTriggered();

  /**
   * @brief Opens the Doxygen source code documentation.
   */
  void mariadbExplorerHelpSourcecodeActionDoxygenTriggered();

  /**
   * @brief Opens the QDoc source code documentation.
   */
  void mariadbExplorerHelpSourceCodeActionQDocTriggered();

  /**
   * @brief Opens the system monitoring dashboard.
   */
  void monitoringActionTriggered();

  /**
   * @brief Slot called when a watched file changes on disk.
   * @param path The path of the modified file.
   */
  void fileSystemWatcherFileChanged(QString path);

  /**
   * @brief Opens the data loading tool (Load Data Infile).
   */
  void dLoadDataActionTriggered();

  /**
   * @brief Opens the graphical dashboard.
   */
  void dDashboardActionTriggered();

  /**
   * @brief Triggers the Calíope plugin action.
   */
  void caliopePluginActionTriggered();

  /**
   * @brief Checks for application updates.
   */
  void checkForUpdatesActionTriggered();

  /**
   * @brief Reads standard error output from the updater process.
   */
  void readyReadStandardErrorCaliopeUpdater();

  /**
   * @brief Handles the completion of the updater process.
   * @param exitCode The process exit code.
   * @param exitStatus The process exit status.
   */
  void finishedCaliopeUpdater(int exitCode, QProcess::ExitStatus exitStatus);

  /**
   * @brief Starts the interactive application tour.
   */
  void takeATourActionTriggered();

  /**
   * @brief Shows the startup tips dialog.
   */
  void showStartupTipsSlot();

public slots:
  /**
   * @brief Displays a message on the status bar or system tray.
   * @param message The message text.
   * @param icon The icon type.
   * @param timeout Duration to display.
   */
  void statusBarMessage(const QString &message, QSystemTrayIcon::MessageIcon icon = QSystemTrayIcon::Information, int timeout = 0);

  /**
   * @brief Updates the status bar with a progress message.
   * @param message The task description.
   * @param timeout Expected duration.
   * @param progress Percentage complete (0-100).
   */
  void statusBarProgressMessageSlot(const QString &message, const unsigned int timeout = 0, const double progress = 0);

  /**
   * @brief Shows a temporary popup message on the status bar.
   * @param message The message text.
   * @param timeout Duration in milliseconds.
   */
  void statusBarMessagePopup(const QString &message, const int timeout = 0);

  /**
   * @brief Adds a sub-window to the MDI area.
   * @param window The DMdiSubWindow instance to add.
   */
  void addSubWindow(DMdiSubWindow *window);

  /**
   * @brief Opens a file in the appropriate editor.
   * @param fileName The path to the file.
   * @param line The line number to scroll to (default 1).
   */
  void openFile(QString fileName = QString(), unsigned int line = 1);

  /**
   * @brief Updates the main progress bar.
   * @param progress The value (0-100).
   */
  void setProgressBarProgressSlot(const double progress);

  /**
   * @brief Opens a result tab displaying table data.
   * @param table The table name.
   * @param database The database name.
   * @param where Optional WHERE clause filter.
   */
  void showResultTab(QString table, QString database = QString(), QString where = QString());

  /**
   * @brief Updates the main window title.
   */
  void setTitle();

  /**
   * @brief Initiates the application reboot sequence.
   */
  void rebootSlot();

  /**
   * @brief Sets the application cursor to busy/normal state.
   * @param busy True for busy cursor, false for normal.
   */
  void setBusy(bool busy = false);

  /**
   * @brief Deletes a specific query element and closes associated views.
   * @param queryElement The identifier of the query element.
   */
  void deleteAndCloseQuery(QString queryElement);

signals:
  /**
   * @brief Emitted when the active database changes.
   */
  void databaseChanged();

  /**
   * @brief Emitted when a save action is requested globally.
   */
  void save();

private:
  DSettings settings;

  /**
   * @brief Checks if it is safe to close the application (unsaved changes).
   * @return True if safe to close, false otherwise.
   */
  bool okToClose();

  QLabel *messageLabel;
  QMdiArea *mdiMain;
  ServerInformation *serverInformationTab;
  Catalogs *catalogs;
  SQLQuery *query;
  DWebView *dWebView;
  MariaDBHelp *mariadbHelp;
  ProcessList *processList;
  TextEditor *sqlEditor;
  TableMaintenance *tableMaintenance;
  Result *result;
  Backup *backup;
  RestoreBackupFile *restore;
  Users *userAdministration;
  Preferences *preferences;
  QAction *exitAction;
  QAction *rebootAction;
  QAction *aboutQtAction;
  QAction *aboutMariaDBGUIAction;
  QAction *connectToServerAction;
  QAction *serverInformationAction;
  QAction *userAdministrationAction;
  QAction *backupAction;
  QAction *restoreAction;
  QAction *catalogsAction;
  QAction *queryAction;
  QAction *newEmtyQueryAction;
  QAction *tableMaintenanceAction;
  QAction *mysqlScriptAction;
  QAction *mariadbHelpAction;
  QAction *preferencesAction;
  QAction *processAction;
  QAction *showFileToolBarAction;
  QAction *closeCurrentConnectionAction;
  QAction *showCurrent;
  QAction *cascadeSubWindowsAction;
  QAction *activateNextSubWindowAction;
  QAction *activatePreviousSubWindowAction;
  QAction *closeActiveSubWindowAction;
  QAction *closeAllSubWindowsAction;
  QAction *tileSubWindowsAction;
  QAction *objectsDiagramAction;
  QMenu *fileMenu;
  QMenu *projectMenu;
  QMenu *helpMenu;
  QMenu *connectionMenu;
  QMenu *languageMenu;
  QMenu *changeDatabaseMenu;
  QMenu *changeCharsetMenu;
  QActionGroup *languageActionGroup;
  QMenu *viewMenu;
  QMenu *windowMenu;
  QSignalMapper *windowMapper;
  QSignalMapper *databasesMapper;
  QSignalMapper *charsetMapper;
  QToolBar *fileToolbar;

  /**
   * @brief Retranslates the UI components.
   */
  void retranslateUI();

  QTranslator *mainTranslator;
  ObjectsDiagram *objectsDiagram;
  QAction *shutdownServerAction;
  Projects *projects;
  QMenu *editorsMenu;
  int sqlWindowCounter;
  int sqlQueryWindowCounter;
  int textWindowCounter;
  QAction *toggleFullScreenAction;
  QAction *openFileAction;
  QMenu *openRecentFilesMenu;
  QActionGroup *recentFilesActionGroup;
  TextEditor *textEditor;
  QAction *textEditorAction;

  /**
   * @brief Initializes default application settings on first run.
   */
  void createInitialSettings();

  QMenu *replicationMenu;
  QAction *stopReplicationSlaveAction;
  QAction *startReplicationSlaveAction;
  QAction *rebootReplicationSlaveAction;
  QAction *replicationInformationAction;
  QMenu *maintenanceMenu;

  /**
   * @brief Enables or disables connection-dependent menus.
   * @param enabled True to enable, false to disable.
   */
  void enabledDisableConnectionMenus(bool enabled);

  QAction *maintenanceFlushPrivilegesAction;
  QAction *maintenanceFlushHostsAction;
  QAction *maintenancePurgeBinaryLogsAction;
  QAction *maintenanceTimeDifferenceAction;
  QAction *showSlowLogAction;
  QSystemTrayIcon *trayIcon;
  QMenu *trayIconMenu;
  bool trayIconEnabled;

  /**
   * @brief Sets up the system tray icon and menu.
   */
  void createSystemaTrayIcon();

  QToolBar *catalogsToolbar;
  QAction *catalogsReadOnlyAction;
  QAction *catalogsRefreshAction;
  QAction *catalogsNewDatabaseAction;
  QAction *catalogsDeleteDatabaseAction;
  QAction *viewRecentFilesAction;
  QAction *clearRecentURLsAction;
  QAction *resetSlaveAction;
  QAction *maintenanceFlushRelayLogsAction;
  QAction *databaseComparisionAction;
  DatabaseComparision *databaseComparision;
  QProgressDialog *migrationProgressDialog;
  DBMS *secondaryServerConnection;
  QTimer *timerTableCount;
  QMenu *openRecentConnectionMenu;
  QActionGroup *openRecentConnectionActionGroup;
  QAction *catalogsPopulateTableAction;

  /**
   * @brief Helper to execute a database change command.
   * @param database The target database.
   * @return True on success.
   */
  bool changeDatabase(QString database);

  QAction *viewQueryLogAction;
  DQueryLog *dQueryLog;
  QMenu *databaseInformationMenu;
  QSignalMapper *databaseInformationMapper;
  QAction *takeASnapShotAction;
  QMenu *menuReports;
  QAction *actionReportServerInformation;
  DReportViewer *dReportViewerServerInformation;
  DReportViewer *dReportViewerHDDUsage;
  QAction *actionReportHDDUsage;
  DReportViewer *dReportViewerExecutedQueries;
  QAction *actionReportExecutedQueries;
  DReportViewer *dReportViewerDataSentReceived;
  QAction *actionReportDataSentReceived;
  QAction *actionCreateCustomReport;
  QSignalMapper *customReportMapper;
  DReportViewer *dReportViewerCustomReport;
  QAction *actionReportSlowQueries;
  DReportViewer *dReportViewerSlowQueries;
  QAction *tellUsYourCommentsAction;
  QAction *openProjectSiteAction;
  QAction *stopAllReplicationSlavesAction;
  QAction *startAllReplicationSlavesAction;

  /**
   * @brief Creates a new web view window.
   * @param title Window title.
   * @param url URL to load.
   * @return Pointer to the DWebView.
   */
  DWebView *newDWebView(QString title, QUrl url = QUrl());

  /**
   * @brief Helper to add a generic widget as a sub-window.
   */
  void addSubWindow(QWidget *window);

  QAction *exportSettingsAction;
  QAction *importSettingsAction;
  QAction *saveAllAction;
  QAction *migrateTableAction;
  QAction *databaseMetadataAction;
  QSignalMapper *replicationStartMapper;
  QSignalMapper *replicationStopMapper;
  QSignalMapper *replicationResetMapper;
  QSignalMapper *replicationRestartMapper;
  QAction *objectMigrationAction;
  ObjectMigration *objectMigration;
  GitLabFileSearcher *gitLabFileSearcher;
  QAction *gitLabFileSearcherAction;
  QAction *mariadbExplorerHelpAction;
  DHelpViewer *dHelpViewer;
  QAction *mariadbExplorerHelpSourceCodeActionDoxygen;
  DHelpViewer *dHelpViewerSourceCode;
  QAction *mariadbExplorerHelpSourceCodeActionQDoc;
  QAction *monitoringAction;
  DMonitoring *dMonitoring;
  QAction *showErrorLogAction;
  QDialog *errorLogDialog;
  QFileSystemWatcher *fileSystemWatcher;
  BaseTextEditor *errorLogBaseTextEditor;
  MYSQL *restoreConnection;
  DLoadData *dLoadData;
  QAction *dLoadDataAction;
  QAction *dDashboardAction;
  DDashboard *dDashboard;
  QAction *caliopePluginAction;
  QLabel *busyLabelMovie;
  QDialog *busyDialog;
  QMovie *runningSnakeMovie;

  /**
   * @brief Retrieves the application version number.
   * @return The version string.
   */
  QString getVersionNumber();

  QAction *checkForUpdatesAction;
  QProcess *processCaliopeUpdater;
  QAction *takeATourAction;
  DTour *dTour;
  QList<QAction *> listOfMainActions;
  QNetworkAccessManager manager;

  /**
   * @brief Increments and manages the query window counters.
   * @param sqlQueryWindowCounter The current count.
   * @param decrementOnClose Whether to decrement when closed.
   */
  void addQueryWindow(int sqlQueryWindowCounter, bool decrementOnClose = true);

  /**
   * @brief Creates all QActions for the application.
   */
  void createActions();

  /**
   * @brief Initializes the main menu bar.
   */
  void createMenus();

  /**
   * @brief Initializes the main toolbars.
   */
  void createToolBar();

  /**
   * @brief Populates the Window menu.
   */
  void fillWindowMenu();

  /**
   * @brief Shows the welcome message/screen.
   */
  void showWelcomeMessage();

  /**
   * @brief Tasks to perform after a new connection is established.
   */
  void newConnectionPerformed();

  /**
   * @brief Prepares HTML documentation for display.
   * @param filelistPath Path to the file list.
   * @param filePath Path to the specific file.
   * @return The resolved URL.
   */
  QUrl prepareHTMLDocumentation(QString filelistPath, QString filePath);
};

#endif // MAINWINDOW_H
