/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QTreeWidget>
#include <QMdiArea>
#include <QAction>
#include <QToolBar>
#include <QMenu>
#include <QMenuBar>
#include <QMessageBox>
#include <QMdiSubWindow>
#include <QSignalMapper>
#include <QCoreApplication>
#include <QTranslator>
#include <QFile>
#include <QApplication>
#include <QDir>
#include <QTime>
#include <QFileDialog>
#include <QVBoxLayout>
#include <QDialogButtonBox>
#include <QDateTimeEdit>
#include <QCloseEvent>
#include <QProgressDialog>
#include <QTimer>
#include <QImageWriter>
#include <QWhatsThis>
#include <QDesktopServices>
#include <QInputDialog>
#include <QSplashScreen>
#include <QPainter>
#include <QRandomGenerator>
#include <QActionGroup>
#include <QFormLayout>
#include <QFileSystemWatcher>
#include <QDockWidget>
#include <QGraphicsColorizeEffect>
#include <QPushButton>

#include "mainwindow.h"
#include "connectdialog.h"
#include "serverInformation.h"
#include "catalogs.h"
#include "dstatusbar.h"
#include "mariadbhelp.h"
#include "processlist.h"
#include "sqlquery.h"
#include "tablemaintenance.h"
#include "result.h"
#include "backup.h"
#include "restorebackupfile.h"
#include "users.h"
#include "preferences.h"
#include "objectsdiagram.h"
#include "networksettings.h"
#include "projects.h"
#include "staticfunctions.h"
#include "texteditor.h"
#include "dtitlelabel.h"
#include "databasecomparision.h"
#include "dmdisubwindow.h"
#include "dquerylog.h"
#include "basetexteditor.h"
#include "dreportviewer.h"
#include "objectmigration.h"
#include "dlistwidgetdialog.h"
#include "dicon.h"
#include "backup.h"
#include "gitlabfilesearcher.h"
#include "dhelpviewer.h"
#include "dmonitoring.h"
#include "dloaddata.h"
#include "ddashboard.h"
#include "dsoundeffect.h"
#include "dsettings.h"
#include "dtour.h"
#include "dstartuptips.h"

#include "QDebug"

/*!
  \class MainWindow
  \brief MainWindow::MainWindow.
  \inmodule MainModule
 */
MainWindow::MainWindow()
{
  busyDialog = new QDialog(this);
  busyDialog->setModal(true);
//  busyDialog->setWindowFlags(Qt::Window | Qt::FramelessWindowHint);
  busyLabelMovie = new QLabel(busyDialog);
  runningSnakeMovie = new QMovie(DIcon::Loading());
  busyLabelMovie->setMovie(runningSnakeMovie);
  runningSnakeMovie->setScaledSize(QSize(100, 100));
  QVBoxLayout *busyDialogVLayout = new QVBoxLayout;
  busyDialogVLayout->addWidget(busyLabelMovie);
  busyDialog->setLayout(busyDialogVLayout);
  QSplashScreen *splash = new QSplashScreen(QPixmap(DIcon::RandomCaliopeAt512px()));
  splash->show();
  splash->showMessage(tr("Initial settings"));

  QCoreApplication::setApplicationName("Calíope");
  QCoreApplication::setOrganizationDomain("caliope.org");
  QCoreApplication::setOrganizationName("DVC Software");
  qApp->setProperty("SessionId", QDateTime::currentDateTime());

  splash->showMessage(tr("Creating temporary files"));
  QDir tempDirectory(QDir::tempPath());
  tempDirectory.removeRecursively();
  tempDirectory.mkpath(StaticFunctions::workingDirectory());
//#if defined(Q_OS_WIN)
//  QFile::copy(":/images/ico/Calíope.ico", "Calíope.ico");
//  QFile::setPermissions("Calíope.ico", QFileDevice::ReadOwner | QFileDevice::WriteOwner);

//  QFile::copy(":/mariadb/lib/libmariadb.dll", "libmariadb.dll");
//#elif defined(Q_OS_LINUX)
//  QFile::copy(":/mariadb/lib/libmariadb.so", StaticFunctions::workingDirectory() + "libmariadb.so");
//#elif defined(Q_OS_MAC)
//  QFile::copy(":/mariadb/lib/libmariadb.so", StaticFunctions::workingDirectory() + "libmariadb.so");
//#endif
  QFile::copy(":/documentation/en_US/caliope.qhc", StaticFunctions::workingDirectory() + "caliope.qhc");
  QFile::copy(":/documentation/en_US/caliope.qch", StaticFunctions::workingDirectory() + "caliope.qch");
  QFile::copy(":/documentation/en_US/caliope.qhp", StaticFunctions::workingDirectory() + "caliope.qhp");

  QFile::copy(":/documentation/source_code/Doxygen/caliope_source_code.qhc", StaticFunctions::workingDirectory() + "caliope_source_code.qhc");
  QFile::copy(":/documentation/source_code/Doxygen/caliope_source_code.qch", StaticFunctions::workingDirectory() + "caliope_source_code.qch");
  QFile::copy(":/documentation/source_code/Doxygen/caliope_source_code.qhp", StaticFunctions::workingDirectory() + "caliope_source_code.qhp");

  QFile::copy(":/sounds/Alarm.wav", StaticFunctions::workingDirectory() + "Alarm.wav");
  QFile::copy(":/sounds/Close.wav", StaticFunctions::workingDirectory() + "Close.wav");
  QFile::copy(":/sounds/Error_1.wav", StaticFunctions::workingDirectory() + "Error_1.wav");
  QFile::copy(":/sounds/Error_2.wav", StaticFunctions::workingDirectory() + "Error_2.wav");
  QFile::copy(":/sounds/Exit.wav", StaticFunctions::workingDirectory() + "Exit.wav");
  QFile::copy(":/sounds/Open.wav", StaticFunctions::workingDirectory() + "Open.wav");
  QFile::copy(":/sounds/Refresh.wav", StaticFunctions::workingDirectory() + "Refresh.wav");
  QFile::copy(":/sounds/Run_1.wav", StaticFunctions::workingDirectory() + "Run_1.wav");
  QFile::copy(":/sounds/Run_2.wav", StaticFunctions::workingDirectory() + "Run_2.wav");
  QFile::copy(":/sounds/Screenshot.wav", StaticFunctions::workingDirectory() + "Screenshot.wav");

  qApp->setProperty("ErrorLogFile", QDir::currentPath() + QDir::separator() + QCoreApplication::applicationName() + ".err");
  splash->showMessage(tr("Creating internal objects"));
  serverConnection = new DBMS;
  connect(serverConnection, SIGNAL(databaseChanged()), this, SLOT(changeDatabaseSlot()));
  connect(serverConnection, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  connect(serverConnection, SIGNAL(statusBarMessage(QString)), this, SLOT(statusBarMessage(QString)));
  connect(serverConnection, SIGNAL(reconnectionPerformed()), this, SLOT(reconnectionPerformedSlot()));

  projects = new Projects;
  connect(projects, SIGNAL(updateTitle()), this, SLOT(setTitle()));
  connect(projects, SIGNAL(statusBarMessage(QString)), this, SLOT(statusBarMessage(QString)));
  connect(projects, SIGNAL(addSubWindow(DMdiSubWindow*)), this, SLOT(addSubWindow(DMdiSubWindow*)));
  connect(projects, SIGNAL(openFile(QString,uint)), this, SLOT(openFile(QString,uint)));
  connect(projects, SIGNAL(statusBarMessagePopup(QString)), this, SLOT(statusBarMessagePopup(QString)));

  DSettings settings;
  /*Setting the style interfiers with the mouse, very strange*/
  QFile styleFile(":/styles/mainStyle.css");
  if (!styleFile.open(QFile::ReadOnly | QFile::Text))
    statusBarMessage(tr("Cannot read file %1:\n%2.").arg(":/styles/mainStyle.css", styleFile.errorString()), QSystemTrayIcon::Critical);
  qApp->setStyle(settings.value("GeneralSettings/Style", QApplication::style()->objectName()).toString());
  qApp->setStyleSheet(QString::fromUtf8(styleFile.readAll()));
  styleFile.close();
  setGeometry(settings.value("MainWindow/Geometry", QRect(100, 100, 600, 400)).toRect());
  if (settings.value("MainWindow/Maximized", false).toBool())
    setWindowState(Qt::WindowMaximized);
  setWindowTitle(QCoreApplication::applicationName());
  setObjectName(windowTitle());
  setAttribute(Qt::WA_DeleteOnClose);
  setAttribute(Qt::WA_CustomWhatsThis);
  setWindowIcon(DIcon::Database());

  if (settings.value("GeneralSettings/ApplicationFirstRun", true).toBool()
      &&
      QMessageBox::question(this, tr("Send information to Calíope"),
                                  tr("Would you share the following anonymous information with caliope.org? The purpose is to know how much our application is used.")
                                     + "<BR /><BR />"
                                     + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm")
                                     + " "
                                     + QSysInfo::prettyProductName()
                            ) == QMessageBox::Yes
      ) {
    QNetworkRequest request;
    request.setUrl(QUrl(QString("https://caliope.org/usage/?UsageInfo=%1,%2 <BR />")
                        .arg(QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm")
                             , QSysInfo::prettyProductName())
                        ));
    manager.get(request);
  }
  settings.setValue("GeneralSettings/ApplicationFirstRun", false);

  mdiMain = new QMdiArea;
  mdiMain->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
  mdiMain->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
  connect(mdiMain, SIGNAL(subWindowActivated(QMdiSubWindow*)), this, SLOT(mdiSubWindowActivatedSlot(QMdiSubWindow*)));
  setCentralWidget(mdiMain);

  splash->showMessage(tr("Creating more internal objects"));
  createActions();
  createMenus();
  createSystemaTrayIcon();
  updateWindowMenu();
  dStatusBar = new DStatusBar;
  setStatusBar(dStatusBar);
  createToolBar();

  windowMapper = new QSignalMapper(this);
  connect(windowMapper, SIGNAL(mappedObject(QObject*)), this, SLOT(setActiveSubWindowMap(QObject*)));
  databasesMapper = new QSignalMapper(this);
  connect(databasesMapper, SIGNAL(mappedString(QString)), this, SLOT(changeDatabaseSlot(QString)));
  charsetMapper = new QSignalMapper(this);
  connect(charsetMapper, SIGNAL(mappedString(QString)), this, SLOT(changeCharsetSlot(QString)));
  databaseInformationMapper = new QSignalMapper(this);
  connect(databaseInformationMapper, SIGNAL(mappedString(QString)), this, SLOT(databaseInformationSlot(QString)));
  customReportMapper = new QSignalMapper(this);
  connect(customReportMapper, SIGNAL(mappedString(QString)), this, SLOT(openCustomReport(QString)));

  replicationStartMapper = new QSignalMapper(this);
  connect(replicationStartMapper, SIGNAL(mappedString(QString)), this, SLOT(replicationStartConnection(QString)));
  replicationStopMapper = new QSignalMapper(this);
  connect(replicationStopMapper, SIGNAL(mappedString(QString)), this, SLOT(replicationStopConnection(QString)));
  replicationResetMapper = new QSignalMapper(this);
  connect(replicationResetMapper, SIGNAL(mappedString(QString)), this, SLOT(replicationResetConnection(QString)));
  replicationRestartMapper = new QSignalMapper(this);
  connect(replicationRestartMapper, SIGNAL(mappedString(QString)), this, SLOT(replicationRestartConnection(QString)));

  connect(showFileToolBarAction, SIGNAL(triggered(bool)), fileToolbar, SLOT(setVisible(bool)));

  splash->showMessage(tr("Setting the languaje"));
  qApp->setProperty("ApplicationLanguage", settings.value("GeneralSettings/Language", "en/US"));
  mainTranslator = new QTranslator(this);
  if (settings.value("GeneralSettings/Language", "NoDefaultValueThisTime") != "en/US") {
    QAction *languageAction = new QAction("", this);
    languageAction->setData(settings.value("GeneralSettings/Language", "en/US"));
    swithLanguage(languageAction);
  } else {
    retranslateUI();
  }

  splash->showMessage(tr("Reading application arguments"));
  QStringList applicationArguments = QApplication::arguments();
  if (applicationArguments.count() == 1) {
    connectToServerActionTriggered();
  } else {
    QRegularExpression regularExpression;
    regularExpression.setPattern("Monitoring");
    regularExpression.setPatternOptions(QRegularExpression::CaseInsensitiveOption);
    qsizetype position;
    position = applicationArguments.indexOf(regularExpression);
    if (position >= 0) {
      monitoringActionTriggered();
    }
    regularExpression.setPattern("Geometry=\\d+x\\d+x\\d+x\\d+");
    position = applicationArguments.indexOf(regularExpression);
    if (position >= 0) {
      QStringList rect = applicationArguments.at(position).split("=").at(1).split("x", Qt::KeepEmptyParts, Qt::CaseInsensitive);
      setGeometry(QRect(rect.at(0).toInt(), rect.at(1).toInt(), rect.at(2).toInt(), rect.at(3).toInt()));
    }
    regularExpression.setPattern("Connection=.*");
    position = applicationArguments.indexOf(regularExpression);
    if (position >= 0) {
      QMap<QString, QVariant> params = settings.value("ServerConnections/" + applicationArguments.at(position).split("=").at(1)).toMap();
      if (params.value("Name") != applicationArguments.at(position).split("=").at(1))
        QMessageBox::critical(this, tr("Cannot connect to the server"), applicationArguments.at(position).split("=").at(1) + tr(" is not a valid connection name"));
      serverConnection->setConnectionParameters(params);
      if (params.value("ConnectionType") == "--")
        serverConnection->setDBMSType(StaticFunctions::Undefined);
      if (params.value("ConnectionType")  == "MySQL")
        serverConnection->setDBMSType(StaticFunctions::MySQL);
      if (params.value("ConnectionType")  == "MariaDB")
        serverConnection->setDBMSType(StaticFunctions::MariaDB);
      if (!serverConnection->open())
        QMessageBox::critical(this, tr("Cannot connect to the server"), serverConnection->lastError());
      if (serverConnection->isOpened())
        newConnectionPerformed();
    }
  }

  createInitialSettings();

  NetworkSettings networkSettings;
  networkSettings.setProxySettings();

  showWelcomeMessage();

  if (settings.value("GeneralSettings/RememberOpenedWindows", false).toBool()) {
    QApplication::setOverrideCursor(Qt::WaitCursor);
    foreach (QString window, settings.value("GeneralSettings/OpenedWindows-" + qApp->property("ConnectionName").toString(), QStringList()).toStringList()) {
      if (window == tr("Server Information"))
        serverInformationActionTriggered();
      if (window == tr("User Administration"))
        userAdministrationActionTriggered();
      if (window == tr("Backup"))
        backupActionTriggered();
      if (window == tr("Monitoring"))
        monitoringActionTriggered();
      if (window == tr("Restore Backup File"))
        restoreActionTriggered();
      if (window.startsWith(tr("Catalogs")))
        catalogsActionTriggered();
      if (window == tr("Table Maintenance"))
        tableMaintenanceActionTriggered();
      if (window == tr("Process List"))
        processActionTriggered();
      if (window == tr("Load Data"))
        dLoadDataActionTriggered();
      if (window == tr("Query Log")) {
        viewQueryLogActionSlot();
        continue;
      }
      if (window == tr("Object Migration"))
        objectMigrationActionTriggered();
//      if (window.startsWith(tr("SQL Query")) || window.endsWith(".sql", Qt::CaseInsensitive))
//        sqlScriptActionTriggered();
      if (window.startsWith(tr("Query")))
        queryActionTriggered();
      if (window == tr("Database Comparision"))
        databaseComparisionActionTriggered();
      if (window == tr("GitLab File Searcher"))
        gitLabFileSearcherActionTriggered();
      if (window == tr("Dashboard"))
        dDashboardActionTriggered();
    }
    QApplication::restoreOverrideCursor();
  }

  foreach (QString arg, QApplication::arguments()) {
    QFile file(arg);
    if (file.exists()) {
      QFileInfo fileInfo(file.fileName());
      QString fileCompleteSuffix = fileInfo.completeSuffix().toLower();
      if (fileCompleteSuffix == "sql") {
        sqlScriptActionTriggered();
        sqlEditor->openFile(fileInfo.absoluteFilePath());
      }
    }
  }
  splash->close();
//  retranslateUI();
  if (settings.value("GeneralSettings/ShowStartupTips", true).toBool()) {
    showStartupTipsSlot();
  }
  serverConnection->logApplicationStarted();
  DSoundEffect::Open();
}

MainWindow::~MainWindow()
{
  QDir tempDirectory(StaticFunctions::workingDirectory());
  tempDirectory.removeRecursively();
  DSoundEffect::Close();
}

void MainWindow::statusBarProgressMessageSlot(const QString &message, const unsigned int timeout, const double progress)
{
  statusBarMessage(message, QSystemTrayIcon::Information, timeout);
  dStatusBar->setProgress(progress);
}

/*!
 \fn void MainWindow::toggleFullScreenActionTriggered(const bool fullScreen)
 \brief toggleFullScreenActionTriggered
 */
void MainWindow::toggleFullScreenActionTriggered(const bool fullScreen)
{
  if (fullScreen)
    setWindowState(windowState() | Qt::WindowFullScreen);
  else
    setWindowState(windowState() & ~Qt::WindowFullScreen);
}

void MainWindow::showErrorLogActionTriggered()
{
  fileSystemWatcher = new QFileSystemWatcher;
  connect(fileSystemWatcher, SIGNAL(fileChanged(QString)),this,SLOT(fileSystemWatcherFileChanged(QString)));
  fileSystemWatcher->addPath(qApp->property("ErrorLogFile").toString());
  errorLogDialog = new QDialog(this);
  errorLogDialog->setGeometry(settings.value("ErrorLogDialog/Geometry", QRect(100, 100, 600, 400)).toRect());
  connect(errorLogDialog, SIGNAL(rejected()), this, SLOT(showErrorLogActionClosed()));
  QVBoxLayout *verticalLayout = new QVBoxLayout;
  verticalLayout->addWidget(new DTitleLabel(tr("Calíope error log")));
  errorLogBaseTextEditor = new BaseTextEditor(EditorTypes::NoEditor);
  fileSystemWatcherFileChanged(qApp->property("ErrorLogFile").toString());
  verticalLayout->addWidget(errorLogBaseTextEditor);
  QLineEdit *lineEditFile = new QLineEdit;
  lineEditFile->setText(qApp->property("ErrorLogFile").toString());
  QFormLayout *formLayout = new QFormLayout;
  formLayout->addRow(tr("File:"), lineEditFile);
  verticalLayout->addLayout(formLayout);
  errorLogDialog->setLayout(verticalLayout);
//  errorLogDialog->show();
  QDockWidget *errorLogDockWidget = new QDockWidget(this);
  errorLogDockWidget->setWidget(errorLogDialog);
  errorLogDockWidget->setFloating(true);
  errorLogDockWidget->show();
}

void MainWindow::showErrorLogActionClosed()
{
  settings.setValue("ErrorLogDialog/Geometry", errorLogDialog->geometry());
}

void MainWindow::addSubWindow(DMdiSubWindow *window)
{
  mdiMain->addSubWindow(window);
  window->showMaximized();
  dStatusBar->resetStatusBar();
}

void MainWindow::addSubWindow(QWidget *window)
{
  mdiMain->addSubWindow(window);
  window->showMaximized();
  dStatusBar->resetStatusBar();
}

QString MainWindow::getVersionNumber()
{
  return QString("Caliope-%1-%2").arg(qVersion(), "20251201");
}

//void MainWindow::openFile(QString fileName, unsigned int line)
//{
//}

void MainWindow::mdiSubWindowActivatedSlot(QMdiSubWindow *window)
{
  dStatusBar->resetStatusBar();
  if (window == 0) {
    catalogsToolbar->hide();
    return;
  }
  if (window->windowTitle().startsWith(tr("Catalogs")))
    catalogsToolbar->show();
  else
    catalogsToolbar->hide();
//  if (window->windowTitle().startsWith(tr("Query")))
//    sqlQueryToolbar->show();
//  else
//    sqlQueryToolbar->hide();
}

void MainWindow::showWelcomeMessage()
{
  QStringList texts;
  texts << tr("In God we trust")
        << tr("In a world without walls and fences, who needs windows and gates?")
        << tr("Wellcome!!! Have a nice day.")
        << tr("MariaDB is great!!!")
        << tr("Qt is great!!!")
        << tr("Open space has more fresh air than closed Windows")
        << tr("Microsoft gives you \"Windows\", Linux a \"home\"!")
        << tr("I'd rather be at $HOME")
        << tr("Windows assumes the user is an idiot. Linux demands proof");
  QRandomGenerator random(QTime::currentTime().second());
  dStatusBar->showFancyMessage(texts[(int) random.bounded(32767) % texts.size()]);
}

void MainWindow::newConnectionPerformed()
{
  qApp->setProperty("ConnectionName", serverConnection->getConnectionName());
  statusBarMessage(tr("Connected successfully to: %1").arg(serverConnection->getHostName()));
  enabledDisableConnectionMenus(true);
  setTitle();
  sqlWindowCounter = 0;
  sqlQueryWindowCounter = 0;
  textWindowCounter = 0;
}

//QUrl MainWindow::prepareHTMLDocumentation(QString filelistPath, QString filePath)
//{
//  QDir tempDir(QDir::tempPath() + "/html");
//  tempDir.removeRecursively();
//  tempDir.mkpath(QDir::tempPath() + "/html/search");
//  QFile fileList(filelistPath);
//  fileList.open(QIODevice::ReadOnly | QIODevice::Text);
//  QTextStream stream(&fileList);
//  while (!stream.atEnd()) {
//    QString fileName(stream.readLine());
//    QFile fileToOpen(filePath + fileName);
//    if (!fileToOpen.open(QIODevice::ReadOnly | QIODevice::Text))
//      qDebug() << fileName << fileToOpen.error();
//    //Work arround since fileToOpen.copy is not coping HTML file correctly.
//    if (fileName.endsWith(".html") || fileName.endsWith(".css") || fileName.endsWith(".js")) {
//      QFile fileToSave(QDir::tempPath() + fileName);
//      if (!fileToSave.open(QIODevice::WriteOnly | QIODevice::Text))
//        qDebug() << fileName << fileToOpen.error();
//      QTextStream out(&fileToSave);
//      out << fileToOpen.readAll();
//      fileToSave.close();
//      fileToOpen.close();
//    } else {
//      if (!fileToOpen.copy(QDir::tempPath() + fileName))
//        qDebug() << fileName << fileToOpen.error();
//    }
//  }
//  return QUrl("file://" + tempDir.absolutePath() + "/index.html");
//}

bool MainWindow::okToClose()
{
  foreach (QMdiSubWindow *subWindow, mdiMain->subWindowList()) {
    if (subWindow->inherits("TextEditor")) {
      TextEditor *wid = qobject_cast<TextEditor *>(subWindow);
      if (!wid->okToClose())
        return false;
    }
  }
  return true;
}

void MainWindow::closeEvent(QCloseEvent *event)
{
  if (okToClose()) {
    settings.setValue("MainWindow/Geometry", geometry());
    settings.setValue("MainWindow/Maximized", isMaximized());
    if (settings.value("GeneralSettings/RememberOpenedWindows", false).toBool()) {
      QStringList list;
      foreach (QMdiSubWindow *window, mdiMain->subWindowList(QMdiArea::ActivationHistoryOrder))
        list.append(window->windowTitle());
      settings.setValue("GeneralSettings/OpenedWindows-" + qApp->property("ConnectionName").toString(), list);
    }
    QFile file(settings.fileName());
    QFile backFile(settings.fileName() + ".back");
    backFile.remove();
    if (!file.copy(settings.fileName() + ".back"))
      statusBarMessage(tr("Error on saving application settings backup file. ") + settings.fileName() + ".back");
    serverConnection->close();
    event->accept();
  } else {
    event->ignore();
  }
}

void MainWindow::resizeEvent(QResizeEvent *resizeEvent)
{
//Commented due to an error on windows: mdiMain->size()
  QImage newBackground(mdiMain->size(), QImage::Format_ARGB32_Premultiplied);
  QPainter painter(&newBackground);
  painter.fillRect(newBackground.rect(), QColor(98, 114, 127));
  QSettings setting;
  QString image(setting.value("GeneralSettings/BackgroundImage", DIcon::RandomCaliopeAt512px()).toString());
  QImage originalImage(image == "Random" ? DIcon::RandomCaliopeAt512px() : image);
  QRect scaledRect = originalImage.rect();
  scaledRect.moveCenter(newBackground.rect().center());
  painter.drawImage(scaledRect, originalImage);
  mdiMain->setBackground(newBackground);
  QMainWindow::resizeEvent(resizeEvent);
}

//void MainWindow::moveEvent(QMoveEvent *event)
//{
//  QWidget::moveEvent(event);
//  if (dTour)
//    dTour->moveWindow();
//}

void MainWindow::createActions()
{
  takeATourAction = new QAction(this);
  takeATourAction->setIcon(QIcon(DIcon::RandomCaliopeAt512px()));
  connect(takeATourAction, SIGNAL(triggered()), this, SLOT(takeATourActionTriggered()));

  exitAction = new QAction(this);
  exitAction->setIcon(DIcon::SystemShutdown());
  exitAction->setShortcut(QKeySequence::Quit);
  connect(exitAction, SIGNAL(triggered()), this, SLOT(close()));

  rebootAction = new QAction(this);
  rebootAction->setIcon(DIcon::Reboot());
  connect(rebootAction, SIGNAL(triggered()), this, SLOT(rebootSlot()));

  aboutQtAction = new QAction(this);
  aboutQtAction->setIcon(DIcon::QtLogo());
  connect(aboutQtAction, SIGNAL(triggered()), qApp, SLOT(aboutQt()));

  aboutMariaDBGUIAction = new QAction(this);
  aboutMariaDBGUIAction->setIcon(QIcon(DIcon::RandomCaliopeAt512px()));
  connect(aboutMariaDBGUIAction, SIGNAL(triggered()), this, SLOT(aboutMariaDBGUIActionTriggered()));

  caliopePluginAction = new QAction(this);
  caliopePluginAction->setIcon(QIcon(DIcon::RandomCaliopeAt512px()));
  connect(caliopePluginAction, SIGNAL(triggered()), this, SLOT(caliopePluginActionTriggered()));

  checkForUpdatesAction = new QAction(this);
  checkForUpdatesAction->setIcon(QIcon(DIcon::RandomCaliopeAt512px()));
  connect(checkForUpdatesAction, SIGNAL(triggered()), this, SLOT(checkForUpdatesActionTriggered()));

  connectToServerAction = new QAction(this);
  connectToServerAction->setIcon(DIcon::DatabaseServer());
  connect(connectToServerAction, SIGNAL(triggered()), this, SLOT(connectToServerActionTriggered()));

  serverInformationAction = new QAction(this);
  serverInformationAction->setIcon(DIcon::Information());
  serverInformationAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_I));
  connect(serverInformationAction, SIGNAL(triggered()), this, SLOT(serverInformationActionTriggered()));

  userAdministrationAction = new QAction(this);
  userAdministrationAction->setIcon(DIcon::DatabaseUser());
  userAdministrationAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_U));
  connect(userAdministrationAction, SIGNAL(triggered()), this, SLOT(userAdministrationActionTriggered()));

  backupAction = new QAction(this);
  backupAction->setIcon(DIcon::DatabaseBackup());
  backupAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_R));
  connect(backupAction, SIGNAL(triggered()), this, SLOT(backupActionTriggered()));

  monitoringAction = new QAction(this);
  monitoringAction->setIcon(DIcon::DatabaseMonitorig());
  monitoringAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_S));
  connect(monitoringAction, SIGNAL(triggered()), this, SLOT(monitoringActionTriggered()));

  restoreAction = new QAction(this);
  restoreAction->setIcon(DIcon::DatabaseRestore());
  restoreAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_J));
  connect(restoreAction, SIGNAL(triggered()), this, SLOT(restoreActionTriggered()));

  dLoadDataAction = new QAction(this);
  dLoadDataAction->setIcon(DIcon::LoadData());
  dLoadDataAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_L));
  connect(dLoadDataAction, SIGNAL(triggered()), this, SLOT(dLoadDataActionTriggered()));

  catalogsAction = new QAction(this);
  catalogsAction->setIcon(DIcon::Database());
  catalogsAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_C));
  connect(catalogsAction, SIGNAL(triggered()), this, SLOT(catalogsActionTriggered()));

  queryAction = new QAction(this);
  queryAction->setIcon(DIcon::DatabaseQuery());
  queryAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_Q));
  connect(queryAction, SIGNAL(triggered()), this, SLOT(queryActionTriggered()));

  newEmtyQueryAction = new QAction(this);
  newEmtyQueryAction->setIcon(DIcon::QueryExecution());
  connect(newEmtyQueryAction, SIGNAL(triggered()), this, SLOT(newEmtyQueryActionTriggered()));

  tableMaintenanceAction = new QAction(this);
  tableMaintenanceAction->setIcon(DIcon::DatabaseTableStatus());
  tableMaintenanceAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_M));
  connect(tableMaintenanceAction, SIGNAL(triggered()), this, SLOT(tableMaintenanceActionTriggered()));

  mysqlScriptAction = new QAction(this);
  mysqlScriptAction->setIcon(DIcon::Database());
  connect(mysqlScriptAction, SIGNAL(triggered()), this, SLOT(sqlScriptActionTriggered()));

  mariadbHelpAction = new QAction(this);
  mariadbHelpAction->setIcon(DIcon::MariaDBSeal());
  connect(mariadbHelpAction, SIGNAL(triggered()), this, SLOT(mysqlHelpActionTriggered()));

  mariadbExplorerHelpAction = new QAction(this);
  mariadbExplorerHelpAction->setIcon(DIcon::Help());
  connect(mariadbExplorerHelpAction, SIGNAL(triggered()), this, SLOT(mariadbExplorerHelpActionTriggered()));

  mariadbExplorerHelpSourceCodeActionDoxygen = new QAction(this);
  mariadbExplorerHelpSourceCodeActionDoxygen->setIcon(DIcon::Help());
  connect(mariadbExplorerHelpSourceCodeActionDoxygen, SIGNAL(triggered()), this, SLOT(mariadbExplorerHelpSourcecodeActionDoxygenTriggered()));

  mariadbExplorerHelpSourceCodeActionQDoc = new QAction(this);
  mariadbExplorerHelpSourceCodeActionQDoc->setIcon(DIcon::Help());
  connect(mariadbExplorerHelpSourceCodeActionQDoc, SIGNAL(triggered()), this, SLOT(mariadbExplorerHelpSourceCodeActionQDocTriggered()));

  viewQueryLogAction = new QAction(this);
  viewQueryLogAction->setIcon(DIcon::DatabaseQueryLog());
  connect(viewQueryLogAction, SIGNAL(triggered()), this, SLOT(viewQueryLogActionSlot()));

  preferencesAction = new QAction(this);
  preferencesAction->setIcon(DIcon::Preferences());
  preferencesAction->setShortcut(QKeySequence::Preferences);
  connect(preferencesAction, SIGNAL(triggered()), this, SLOT(preferencesActionTriggered()));

  processAction = new QAction(this);
  processAction->setIcon(DIcon::DatabaseProcessList());
  processAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_P));
  connect(processAction, SIGNAL(triggered()), this, SLOT(processActionTriggered()));

  showFileToolBarAction = new QAction(this);
  showFileToolBarAction->setCheckable(true);
  showFileToolBarAction->setChecked(true);

  closeCurrentConnectionAction = new QAction(this);
  closeCurrentConnectionAction->setIcon(DIcon::Close());
  closeCurrentConnectionAction->setDisabled(true);
  connect(closeCurrentConnectionAction, SIGNAL(triggered()), this, SLOT(closeCurrentConnectionActionTriggered()));

  cascadeSubWindowsAction = new QAction(this);
  cascadeSubWindowsAction->setIcon(DIcon::Preferences());
  connect(cascadeSubWindowsAction, SIGNAL(triggered()), mdiMain, SLOT(cascadeSubWindows()));

  activateNextSubWindowAction = new QAction(this);
  activateNextSubWindowAction->setIcon(DIcon::GoNext());
  activateNextSubWindowAction->setShortcut(QKeySequence::NextChild);
  connect(activateNextSubWindowAction, SIGNAL(triggered()), mdiMain, SLOT(activateNextSubWindow()));

  activatePreviousSubWindowAction = new QAction(this);
  activatePreviousSubWindowAction->setIcon(DIcon::GoPrevious());
  activatePreviousSubWindowAction->setShortcut(QKeySequence::PreviousChild);
  connect(activatePreviousSubWindowAction, SIGNAL(triggered()), mdiMain, SLOT(activatePreviousSubWindow()));

  closeActiveSubWindowAction = new QAction(this);
  closeActiveSubWindowAction->setIcon(DIcon::DocumentClose());
  connect(closeActiveSubWindowAction, SIGNAL(triggered()), mdiMain, SLOT(closeActiveSubWindow()));

  closeAllSubWindowsAction = new QAction(this);
  closeAllSubWindowsAction->setIcon(DIcon::WindowClose());
  connect(closeAllSubWindowsAction, SIGNAL(triggered()), mdiMain, SLOT(closeAllSubWindows()));

  tileSubWindowsAction = new QAction(this);
  tileSubWindowsAction->setIcon(DIcon::ViewSortAscending());
  connect(tileSubWindowsAction, SIGNAL(triggered()), mdiMain, SLOT(tileSubWindows()));

  shutdownServerAction = new QAction(this);
  shutdownServerAction->setIcon(DIcon::SystemShutdown());
  shutdownServerAction->setDisabled(true);
  connect(shutdownServerAction, SIGNAL(triggered()), this, SLOT(shutdownServerActionTriggered()));

  toggleFullScreenAction = new QAction(this);
  toggleFullScreenAction->setIcon(DIcon::ViewFullscreen());
  toggleFullScreenAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_F10));
  toggleFullScreenAction->setCheckable(true);
  connect(toggleFullScreenAction, SIGNAL(triggered(bool)), this, SLOT(toggleFullScreenActionTriggered(bool)));

  showErrorLogAction = new QAction(this);
  showErrorLogAction->setIcon(DIcon::Error());
  connect(showErrorLogAction, SIGNAL(triggered()), this, SLOT(showErrorLogActionTriggered()));

  objectsDiagramAction = new QAction(this);
  objectsDiagramAction->setIcon(DIcon::GraphicsCharts());
  connect(objectsDiagramAction, SIGNAL(triggered()), this, SLOT(objectsDiagramActionTriggered()));

  openFileAction = new QAction(this);
  openFileAction->setIcon(DIcon::DocumentOpen());
  connect(openFileAction, SIGNAL(triggered()), this, SLOT(openFileActionTriggered()));

  recentFilesActionGroup = new QActionGroup(this);
  connect(recentFilesActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(recentFilesActionGroupTriggered(QAction*)));

  textEditorAction = new QAction(this);
  textEditorAction->setIcon(DIcon::DocumentNew());
  connect(textEditorAction, SIGNAL(triggered()), this, SLOT(textEditorActionTriggered()));

  stopReplicationSlaveAction = new QAction(this);
  stopReplicationSlaveAction->setIcon(DIcon::Stop());
  connect(stopReplicationSlaveAction, SIGNAL(triggered()), this, SLOT(stopReplicationSlaveActionTriggered()));

  resetSlaveAction = new QAction(this);
  resetSlaveAction->setIcon(DIcon::Reboot());
  connect(resetSlaveAction, SIGNAL(triggered()), this, SLOT(maintenanceResetSlaveActionTriggered()));

  startReplicationSlaveAction = new QAction(this);
  startReplicationSlaveAction->setIcon(DIcon::SystemShutdown());
  connect(startReplicationSlaveAction, SIGNAL(triggered()), this, SLOT(startReplicationSlaveActionTriggered()));

  rebootReplicationSlaveAction = new QAction(this);
  rebootReplicationSlaveAction->setIcon(DIcon::Reboot());
  connect(rebootReplicationSlaveAction, SIGNAL(triggered()), this, SLOT(rebootReplicationSlaveActionTriggered()));

  replicationInformationAction = new QAction(this);
  replicationInformationAction->setIcon(DIcon::DatabaseReplication());
  connect(replicationInformationAction, SIGNAL(triggered()), this, SLOT(replicationInformationActionTriggered()));

  maintenanceFlushPrivilegesAction = new QAction(DIcon::QueryExecution(), "FLUSH PRIVILEGES", this);
  connect(maintenanceFlushPrivilegesAction, SIGNAL(triggered()), this, SLOT(maintenanceFlushPrivilegesActionTriggered()));

  maintenanceFlushHostsAction = new QAction(DIcon::QueryExecution(), "FLUSH HOSTS", this);
  connect(maintenanceFlushHostsAction, SIGNAL(triggered()), this, SLOT(maintenanceFlushHostsActionTriggered()));

  maintenancePurgeBinaryLogsAction = new QAction(DIcon::QueryExecution(), "PURGE BINARY LOGS BEFORE NOW()", this);
  connect(maintenancePurgeBinaryLogsAction, SIGNAL(triggered()), this, SLOT(maintenancePurgeBinaryLogsActionTriggered()));

  maintenanceFlushRelayLogsAction = new QAction(DIcon::QueryExecution(), "FLUSH RELAY LOGS", this);
  connect(maintenanceFlushRelayLogsAction, SIGNAL(triggered()), this, SLOT(maintenanceFlushRelayLogsActionTriggered()));

  databaseMetadataAction = new QAction(this);
  databaseMetadataAction->setIcon(DIcon::PDF());
  connect(databaseMetadataAction, SIGNAL(triggered()), this, SLOT(databaseMetadataActionTriggered()));

  maintenanceTimeDifferenceAction = new QAction(this);
  maintenanceTimeDifferenceAction->setIcon(DIcon::QueryExecution());
  connect(maintenanceTimeDifferenceAction, SIGNAL(triggered()), this, SLOT(maintenanceTimeDifferenceActionTriggered()));

  showSlowLogAction = new QAction(this);
  showSlowLogAction->setIcon(DIcon::QueryExecution());
  connect(showSlowLogAction, SIGNAL(triggered()), this, SLOT(showSlowLogActionTriggered()));

  //Catalogs' Actions
  catalogsRefreshAction = new QAction(this);
  catalogsRefreshAction->setIcon(DIcon::Refresh());

  catalogsReadOnlyAction = new QAction(this);
  catalogsReadOnlyAction->setIcon(DIcon::LockQueryExecution());
  catalogsReadOnlyAction->setCheckable(true);
  catalogsReadOnlyAction->setChecked(true);

  catalogsNewDatabaseAction = new QAction(this);
  catalogsNewDatabaseAction->setIcon(DIcon::Database());
  catalogsNewDatabaseAction->setEnabled(false);

  catalogsDeleteDatabaseAction = new QAction(this);
  catalogsDeleteDatabaseAction->setIcon(DIcon::Delete());
  catalogsDeleteDatabaseAction->setEnabled(false);

  catalogsPopulateTableAction = new QAction(this);
  catalogsPopulateTableAction->setIcon(DIcon::GoDown());
  catalogsPopulateTableAction->setEnabled(false);

  viewRecentFilesAction = new QAction(this);
  viewRecentFilesAction->setIcon(DIcon::DocumentOpenRecent());
  viewRecentFilesAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_R));
  connect(viewRecentFilesAction, SIGNAL(triggered()), this, SLOT(viewRecentFilesActionTriggered()));

  clearRecentURLsAction = new QAction(this);
  clearRecentURLsAction->setIcon(DIcon::Clear());
  connect(clearRecentURLsAction, SIGNAL(triggered()), this, SLOT(clearRecentURLsActionTriggered()));

  databaseComparisionAction = new QAction(this);
  databaseComparisionAction->setIcon(DIcon::DatabaseComparison());
  databaseComparisionAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_D));
  connect(databaseComparisionAction, SIGNAL(triggered()), this, SLOT(databaseComparisionActionTriggered()));

  openRecentConnectionActionGroup = new QActionGroup(this);
  connect(openRecentConnectionActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(openRecentConnectionActionGroupTriggered(QAction*)));

  takeASnapShotAction = new QAction(this);
  takeASnapShotAction->setIcon(DIcon::Screenshot());
  connect(takeASnapShotAction, SIGNAL(triggered()), this, SLOT(takeASnapShotActionTriggered()));

  actionReportServerInformation = new QAction(this);
  actionReportServerInformation->setIcon(DIcon::TextEditor());
  connect(actionReportServerInformation, SIGNAL(triggered()), this, SLOT(actionReportServerInformationTriggered()));

  actionReportHDDUsage = new QAction(this);
  actionReportHDDUsage->setIcon(DIcon::GraphicsCharts());
  connect(actionReportHDDUsage, SIGNAL(triggered()), this, SLOT(actionReportHDDUsageTriggered()));

  actionReportExecutedQueries = new QAction(this);
  actionReportExecutedQueries->setIcon(DIcon::GraphicsCharts());
  connect(actionReportExecutedQueries, SIGNAL(triggered()), this, SLOT(actionReportExecutedQueriesTriggered()));

  actionReportDataSentReceived = new QAction(this);
  actionReportDataSentReceived->setIcon(DIcon::GraphicsCharts());
  connect(actionReportDataSentReceived, SIGNAL(triggered()), this, SLOT(actionReportDataSentReceivedTriggered()));

  actionCreateCustomReport = new QAction(this);
  actionCreateCustomReport->setIcon(DIcon::DocumentNew());
  connect(actionCreateCustomReport, SIGNAL(triggered()), this, SLOT(actionCreateCustomReportTriggered()));

  actionReportSlowQueries = new QAction(this);
  actionReportSlowQueries->setIcon(DIcon::TextEditor());
  connect(actionReportSlowQueries, SIGNAL(triggered()), this, SLOT(actionReportSlowQueriesTriggered()));

  tellUsYourCommentsAction = new QAction(this);
  connect(tellUsYourCommentsAction, SIGNAL(triggered()), this, SLOT(tellUsYourCommentsActionTriggered()));

  openProjectSiteAction = new QAction(this);
  connect(openProjectSiteAction, SIGNAL(triggered()), this, SLOT(openProjectSiteActionTriggered()));

  // We have to create Multimaster replication actions although they won't be used, otherwise we have to change the translation logic
  stopAllReplicationSlavesAction = new QAction(this);
  stopAllReplicationSlavesAction->setIcon(DIcon::Stop());
  connect(stopAllReplicationSlavesAction, SIGNAL(triggered()), this, SLOT(stopAllReplicationSlavesActionTriggered()));

  startAllReplicationSlavesAction = new QAction(this);
  startAllReplicationSlavesAction->setIcon(DIcon::SystemShutdown());
  connect(startAllReplicationSlavesAction, SIGNAL(triggered()), this, SLOT(startAllReplicationSlavesActionTriggered()));

  exportSettingsAction = new QAction(this);
  exportSettingsAction->setIcon(DIcon::Export());
  connect(exportSettingsAction, SIGNAL(triggered()), this, SLOT(exportSettingsActionTriggered()));

  importSettingsAction = new QAction(this);
  importSettingsAction->setIcon(DIcon::Import());
  connect(importSettingsAction, SIGNAL(triggered()), this, SLOT(importSettingsActionTriggered()));

  saveAllAction = new QAction(this);
  saveAllAction->setShortcut(QKeySequence(Qt::CTRL | Qt::SHIFT | Qt::Key_S));
  connect(saveAllAction, SIGNAL(triggered()), this, SLOT(saveAllActionTriggered()));

  migrateTableAction = new QAction(this);
  migrateTableAction->setIcon(DIcon::DatabaseMigration());
  migrateTableAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_T));
  connect(migrateTableAction, SIGNAL(triggered()), this, SLOT(migrateTableActionTriggered()));

  objectMigrationAction = new QAction(this);
  objectMigrationAction->setIcon(DIcon::DatabaseMigration());
  objectMigrationAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_B));
  connect(objectMigrationAction, SIGNAL(triggered()), this, SLOT(objectMigrationActionTriggered()));

  gitLabFileSearcherAction = new QAction(this);
  gitLabFileSearcherAction->setIcon(DIcon::GitLab());
  gitLabFileSearcherAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_G));
  connect(gitLabFileSearcherAction, SIGNAL(triggered()), this, SLOT(gitLabFileSearcherActionTriggered()));

  dDashboardAction = new QAction(this);
  dDashboardAction->setIcon(DIcon::Dashboard());
  dDashboardAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_A));
  connect(dDashboardAction, SIGNAL(triggered()), this, SLOT(dDashboardActionTriggered()));

}

void MainWindow::objectsDiagramActionTriggered()
{
  if (serverConnection->testOpened()) {
    objectsDiagram = mdiMain->findChild<ObjectsDiagram *>(tr("Objects Diagram"));
    if (!objectsDiagram) {
      objectsDiagram = new ObjectsDiagram;
      addSubWindow(objectsDiagram);
    }
    mdiMain->setActiveSubWindow(objectsDiagram);
  }
}

void MainWindow::openFileActionTriggered()
{
  openFile();
}

void MainWindow::openRecentFilesMenuAboutToShowSlot()
{
  openRecentFilesMenu->clear();
  const int limit = 5;
  QStringList fileList = settings.value("RecentMariaDBFiles/Files", QStringList()).toStringList();
  fileList.sort();
  for(int counter = 0; counter < (fileList.size() / 2); counter++) fileList.swapItemsAt(counter, fileList.size() - (1 + counter));
  int fileCount = 0;
  foreach (QString file, fileList)
    if (!file.isEmpty() && fileCount++ < limit)
      recentFilesActionGroup->addAction(openRecentFilesMenu->addAction(StaticFunctions::iconFromFileName(file), file));
}

void MainWindow::recentFilesActionGroupTriggered(QAction *action)
{
  openFile(action->text());
}

void MainWindow::openFile(QString fileName, unsigned int line)
{
  if (fileName.isEmpty()) {
    QFileDialog fileDialog;
    fileDialog.setDirectory(QDir::home());
    fileName = fileDialog.getOpenFileName(0, tr("Open a file"), QDir::home().absolutePath());
  }
  QApplication::setOverrideCursor(Qt::WaitCursor);
  switch(StaticFunctions::findEditorType(fileName)) {
  case EditorTypes::SQLQuery:
    sqlScriptActionTriggered();
    sqlEditor->openFile(fileName);
    if (line > 1)
      sqlEditor->gotoLine(line);
    break;
  case EditorTypes::NoEditor:
    textEditorActionTriggered();
    textEditor->openFile(fileName);
    if (line > 1)
      textEditor->gotoLine(line);
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
    break;
  // default: Q_ASSERT(false);
  }
  QApplication::restoreOverrideCursor();
}

void MainWindow::setProgressBarProgressSlot(const double progress)
{
  dStatusBar->setProgress(progress);
}

void MainWindow::textEditorActionTriggered()
{
//  if (optionsDock->isVisible())
//    optionsDock->hide();
  textEditor = new TextEditor(projects, this->serverConnection, EditorTypes::NoEditor, ++textWindowCounter);
  connect(textEditor, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  connect(textEditor, SIGNAL(updatePrositionViewer(int,int,int)), dStatusBar, SLOT(setPrositionViewer(int,int,int)));
  connect(textEditor, SIGNAL(windowClosed()), this, SLOT(decrementTextWindowCounter()));
  connect(this, SIGNAL(save()), textEditor, SLOT(saveFileActionTriggered()));
  addSubWindow(textEditor);
}

void MainWindow::stopReplicationSlaveActionTriggered()
{
  serverConnection->replication()->stopSlave();
  statusBarMessage(tr("The replication slave has been stoped."));
}

void MainWindow::startReplicationSlaveActionTriggered()
{
  serverConnection->replication()->startSlave();
  statusBarMessage(tr("The replication slave has been started."));
}

void MainWindow::rebootReplicationSlaveActionTriggered()
{
  serverConnection->replication()->rebootSlave();
  statusBarMessage(tr("The replication slave has been rebooted."));
}

void MainWindow::replicationInformationActionTriggered()
{
  serverInformationActionTriggered();
  serverInformationTab->setCurrentTab(1);
}

void MainWindow::maintenanceFlushPrivilegesActionTriggered()
{
  serverConnection->flushPrivileges();
  statusBarMessage(tr("Privileges flushed."));
}

void MainWindow::maintenanceFlushHostsActionTriggered()
{
  serverConnection->flushHosts();
  statusBarMessage(tr("Hosts flushed."));
}

void MainWindow::maintenancePurgeBinaryLogsActionTriggered()
{
  serverConnection->replication()->purgeBinaryLogs();
  statusBarMessage(tr("Logs purged."));
}

void MainWindow::maintenanceTimeDifferenceActionTriggered()
{
  QMessageBox::information(this, tr("Time Difference")
                           , tr("Seconds of difference: %1").arg(serverConnection->runQuerySingleColumn("SELECT UNIX_TIMESTAMP()").at(0).toLongLong()
                                                                 - (QDateTime::currentMSecsSinceEpoch() / 1000)));
}

void MainWindow::showSlowLogActionTriggered()
{
  QDialog *dialog = new QDialog(this);
  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->addWidget(new DTitleLabel(tr("Select a date")));
  QDateTimeEdit *dateTimeEdit = new QDateTimeEdit(QDateTime::currentDateTime());
  dateTimeEdit->setCalendarPopup(true);
  dateTimeEdit->setDisplayFormat("yyyy-MMM-dd hh:mm:ss");
  mainVLayout->addWidget(dateTimeEdit);
  QDialogButtonBox *buttonBox = new QDialogButtonBox(QDialogButtonBox::Cancel | QDialogButtonBox::Ok);
  connect(buttonBox, SIGNAL(rejected()), dialog, SLOT(reject()));
  connect(buttonBox, SIGNAL(accepted()), dialog, SLOT(accept()));
  mainVLayout->addWidget(buttonBox);
  dialog->setLayout(mainVLayout);
  if (dialog->exec() == QDialog::Accepted)
    showResultTab("slow_log", "mysql", "`start_time` >= '" + dateTimeEdit->dateTime().toString("yyyy-MM-dd hh:mm:ss") + "'");
}

void MainWindow::catalogsReadOnlyActionSlot(bool readOnly)
{
  catalogsNewDatabaseAction->setEnabled(!readOnly);
  catalogsDeleteDatabaseAction->setEnabled(!readOnly);
  catalogsPopulateTableAction->setEnabled(!readOnly);
  catalogs->setReadOnly(readOnly);
}

void MainWindow::viewRecentFilesActionTriggered()
{
  QPoint point = pos();
  point.setX(point.x() + (width() / 2) - (openRecentFilesMenu->width() / 2));
  point.setY(point.y() + (height() / 2) - (openRecentFilesMenu->height() / 2));
  openRecentFilesMenu->popup(point);
}

void MainWindow::clearRecentURLsActionTriggered()
{
  settings.remove("Internet/VisitedURLs");
}

void MainWindow::maintenanceResetSlaveActionTriggered()
{
  serverConnection->replication()->resetSlave();
  statusBarMessage(tr("The slave has been reset."));
}

void MainWindow::maintenanceFlushRelayLogsActionTriggered()
{
  serverConnection->replication()->flushRelayLogs();
  statusBarMessage(tr("Logs flushed."));
}

void MainWindow::databaseComparisionActionTriggered()
{
  if (serverConnection->testOpened()) {
    databaseComparision = mdiMain->findChild<DatabaseComparision *>(tr("Database Comparision"));
    if (!databaseComparision) {
      databaseComparision = new DatabaseComparision(this->serverConnection);
      connect(databaseComparision, SIGNAL(statusBarMessage(QString)), this, SLOT(statusBarMessage(QString)));
      connect(databaseComparision, SIGNAL(statusBarProgressMessage(QString,uint,double)), this, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
      addSubWindow(databaseComparision);
    }
    mdiMain->setActiveSubWindow(databaseComparision);
  }
}

void MainWindow::tableCountSlot()
{
  statusBarMessage(tr("Table count: %1").arg(secondaryServerConnection->database(secondaryServerConnection->getDatabase())->tableCount()));
}

void MainWindow::connectionMenuAboutToShowSlot()
{
  bool enabled = serverConnection->isOpened();
  closeCurrentConnectionAction->setEnabled(enabled);
  changeCharsetMenu->setEnabled(enabled);
  changeDatabaseMenu->setEnabled(enabled);
  replicationMenu->setEnabled(enabled);
  maintenanceMenu->setEnabled(enabled);
  shutdownServerAction->setEnabled(enabled);
  menuReports->setEnabled(enabled);
  databaseMetadataAction->setEnabled(enabled);
  viewQueryLogAction->setEnabled(this->serverConnection->queryLogEnabled());
}

void MainWindow::openRecentConnectionMenuAboutToShowSlot()
{
  openRecentConnectionMenu->clear();
  QStringList connections = settings.allKeys("ServerConnections");

  int valueToRemove = connections.indexOf("StorePassword");
  if (valueToRemove >= 0)
    connections.removeAt(valueToRemove);
  valueToRemove = connections.indexOf("SortConnectionList");
  if (valueToRemove >= 0)
    connections.removeAt(valueToRemove);
  connections.sort();
  foreach (QString connection, connections)
   if (!connection.isEmpty())
     openRecentConnectionActionGroup->addAction(openRecentConnectionMenu->addAction(connection));
}

void MainWindow::openRecentConnectionActionGroupTriggered(QAction *action)
{
  QMap<QString, QVariant> params = settings.value("ServerConnections/" + action->text()).toMap();
  if (serverConnection->isOpened())
    serverConnection->close();
  serverConnection->setConnectionParameters(params);
  if (params.value("ConnectionType") == "--")
    serverConnection->setDBMSType(StaticFunctions::Undefined);
  if (params.value("ConnectionType")  == "MySQL")
    serverConnection->setDBMSType(StaticFunctions::MySQL);
  if (params.value("ConnectionType")  == "MariaDB")
    serverConnection->setDBMSType(StaticFunctions::MariaDB);
  if (!serverConnection->open())
    QMessageBox::critical(this, tr("Cannot connect to the server"), serverConnection->lastError());
  if (serverConnection->isOpened())
    newConnectionPerformed();
}

void MainWindow::viewQueryLogActionSlot()
{
  dQueryLog = mdiMain->findChild<DQueryLog *>(tr("Query Log"));
  if (!dQueryLog) {
    dQueryLog = new DQueryLog(serverConnection);
    addSubWindow(dQueryLog);
  }
  mdiMain->setActiveSubWindow(dQueryLog);
}

void MainWindow::updatedatabaseInformationMenu()
{
  databaseInformationMenu->clear();
  foreach (QString database, serverConnection->getDatabases()) {
    QAction *action = databaseInformationMenu->addAction(database);
    action->setIcon(DIcon::Database());
    connect(action, SIGNAL(triggered()), databaseInformationMapper, SLOT(map()));
    databaseInformationMapper->setMapping(action, database);
  }
}

void MainWindow::databaseInformationSlot(QString database)
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QString information = "";
  foreach (QString info, serverConnection->database(database)->info())
    information += info + "\n";
  QApplication::restoreOverrideCursor();
  QMessageBox::information(this, tr("Database information"), information);
}

void MainWindow::takeASnapShotActionTriggered()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFileDialog fileDialog;
  fileDialog.setDirectory(QDir::home());
  QString imageFormats;
  foreach(QString format, QImageWriter::supportedImageFormats())
    imageFormats += "*." + format + " ";
  QString file = fileDialog.getSaveFileName(this, tr("Save to Image"), settings.value("GeneralSettings/LastFileImg", QDir::home().absolutePath()).toString(), tr("Image files (%1)").arg(imageFormats));
  settings.setValue("GeneralSettings/LastFileImg", fileDialog.directory().filePath(file));
  QPixmap pixmap(this->size());
  this->render(&pixmap);
  pixmap.save(file, "PNG", 5);
  QApplication::restoreOverrideCursor();
  statusBarMessage(tr("File saved at: %1").arg(file));
  DSoundEffect::Screenshot();
}

void MainWindow::viewDWebViewPageSource(QString pageSource)
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  textEditor = new TextEditor(projects, this->serverConnection, EditorTypes::NoEditor, ++textWindowCounter);
  connect(textEditor, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  connect(textEditor, SIGNAL(updatePrositionViewer(int,int,int)), dStatusBar, SLOT(setPrositionViewer(int,int,int)));
  connect(textEditor, SIGNAL(windowClosed()), this, SLOT(decrementTextWindowCounter()));
  connect(this, SIGNAL(save()), textEditor, SLOT(saveFileActionTriggered()));
  addSubWindow(textEditor);
  textEditor->textEditor->setPlainText(pageSource);
  QApplication::restoreOverrideCursor();
}

void MainWindow::actionReportServerInformationTriggered()
{
  dReportViewerServerInformation = new DReportViewer(serverConnection, tr("Server Information"), ReportTypes::PlainText);
  connect(dReportViewerServerInformation, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  dReportViewerServerInformation->setSqlQuery(StaticFunctions::serverInformationQuery());
  addSubWindow(dReportViewerServerInformation);
  dReportViewerServerInformation->showReportData();
}

void MainWindow::actionReportHDDUsageTriggered()
{
  dReportViewerHDDUsage = new DReportViewer(serverConnection, tr("HDD Usage Graphics"), ReportTypes::PieChart, "MB");
  connect(dReportViewerHDDUsage, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  dReportViewerHDDUsage->setSqlQuery("SELECT `TABLE_SCHEMA`, CAST(SUM(`DATA_LENGTH` + `INDEX_LENGTH`) / 1024 / 1024 AS UNSIGNED) AS `Total` FROM `information_schema`.`TABLES` WHERE `TABLE_TYPE` IN ('BASE TABLE', 'SYSTEM VERSIONED') AND `ENGINE` NOT IN ('FEDERATED', 'MEMORY') GROUP BY `TABLE_SCHEMA` ORDER BY `Total` DESC");
  addSubWindow(dReportViewerHDDUsage);
  dReportViewerHDDUsage->showReportData();
}

void MainWindow::actionReportExecutedQueriesTriggered()
{
  dReportViewerExecutedQueries = new DReportViewer(serverConnection, tr("Executed Queries"), ReportTypes::PieChart, "queries");
  connect(dReportViewerExecutedQueries, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  dReportViewerExecutedQueries->setSqlQuery("SELECT `VARIABLE_NAME`, `VARIABLE_VALUE` FROM " + serverConnection->getGlobalStatusTable() + " WHERE `VARIABLE_NAME` IN ('COM_DELETE', 'COM_INSERT', 'COM_SELECT', 'COM_UPDATE', 'COM_ROLLBACK') ORDER BY CAST(`VARIABLE_VALUE` AS UNSIGNED) DESC");
  addSubWindow(dReportViewerExecutedQueries);
  dReportViewerExecutedQueries->showReportData();
}

void MainWindow::actionReportDataSentReceivedTriggered()
{
  dReportViewerDataSentReceived = new DReportViewer(serverConnection, tr("Data Sent/Received"), ReportTypes::PieChart, "MB");
  connect(dReportViewerDataSentReceived, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  dReportViewerDataSentReceived->setSqlQuery("SELECT `VARIABLE_NAME`, FORMAT(`VARIABLE_VALUE` / 1024 / 1024, 2) FROM " + serverConnection->getGlobalStatusTable() + " WHERE `VARIABLE_NAME` IN ('BYTES_RECEIVED', 'BYTES_SENT')");
  addSubWindow(dReportViewerDataSentReceived);
  dReportViewerDataSentReceived->showReportData();
}

void MainWindow::actionCreateCustomReportTriggered()
{
  DReportViewer *dReportViewer = new DReportViewer(0, tr("Custom reports"), ReportTypes::None);
  dReportViewer->addCustomReport();
}

void MainWindow::menuReportsAboutToShowSlot()
{
  menuReports->clear();
  menuReports->addAction(actionReportServerInformation);
  menuReports->addAction(actionReportSlowQueries);
  menuReports->addAction(actionReportHDDUsage);
  menuReports->addAction(actionReportDataSentReceived);
  menuReports->addAction(actionReportExecutedQueries);
  menuReports->addSeparator()->setText(tr("Custom reports"));

  QStringList reports = settings.allKeys("CustomReports");
  int valueToRemove = reports.indexOf("LastFileImg");
  if (valueToRemove >= 0)
      reports.removeAt(valueToRemove);
  ReportTypes::ReportType reportType;
  foreach (QString report, reports) {
    reportType =  (ReportTypes::ReportType) settings.value("CustomReports/" + report).toString().split("#", Qt::SkipEmptyParts).at(1).split(":").at(1).toInt();
    QAction *action = menuReports->addAction((reportType == ReportTypes::PlainText ? DIcon::TextEditor() : DIcon::GraphicsCharts()), report);
    connect(action, SIGNAL(triggered()), customReportMapper, SLOT(map()));
    customReportMapper->setMapping(action, report);
  }

  connectionMenu->addSeparator();
  menuReports->addAction(actionCreateCustomReport);
}

void MainWindow::openCustomReport(QString report)
{
  QStringList data = settings.value("CustomReports/" + report).toString().split("#", Qt::SkipEmptyParts);
  dReportViewerCustomReport = new DReportViewer(serverConnection, report, (ReportTypes::ReportType) data.at(1).split(":").at(1).toInt(), data.at(0).split(":").at(1));
  connect(dReportViewerCustomReport, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  dReportViewerCustomReport->setSqlQuery(data.at(2).split(":").at(1));
  addSubWindow(dReportViewerCustomReport);
  dReportViewerCustomReport->showReportData();
}

void MainWindow::actionReportSlowQueriesTriggered()
{
  dReportViewerSlowQueries = new DReportViewer(serverConnection, tr("Slow Queries"), ReportTypes::PlainText);
  connect(dReportViewerSlowQueries, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  dReportViewerSlowQueries->setSqlQuery(StaticFunctions::slowQueriesQuery());
  addSubWindow(dReportViewerSlowQueries);
  dReportViewerSlowQueries->showReportData();
}

void MainWindow::tellUsYourCommentsActionTriggered()
{
  QDesktopServices::openUrl(QUrl("mailto:caliope@caliope.org?subject=Comments on Calíope&body=Comments", QUrl::TolerantMode));
}

void MainWindow::openProjectSiteActionTriggered()
{
  QDesktopServices::openUrl(QUrl("https://caliope.org", QUrl::TolerantMode));
}

void MainWindow::stopAllReplicationSlavesActionTriggered()
{
  serverConnection->replication()->stopAllSlaves();
  statusBarMessage(tr("All the replication slaves have been stoped."));
}

void MainWindow::startAllReplicationSlavesActionTriggered()
{
  serverConnection->replication()->startAllSlaves();
  statusBarMessage(tr("All the replication slaves hava been started."));
}

void MainWindow::reconnectionPerformedSlot()
{
  qDebug() << "Reconnection performed at: " << QTime::currentTime();
  changeDatabase(serverConnection->getDatabase());
}

void MainWindow::decrementSQLWindowCounter()
{
  --sqlWindowCounter;
}

void MainWindow::decrementSQLQueryWindowCounter()
{
  --sqlQueryWindowCounter;
}

void MainWindow::decrementTextWindowCounter()
{
  --textWindowCounter;
}

void MainWindow::exportSettingsActionTriggered()
{
  QString fileName(QFileDialog::getSaveFileName(this, tr("Save file"), QDir::home().absolutePath() + "/Calíope.ini", tr("Settings files (*.ini)")));
  QFile::copy(settings.fileName(), fileName);
  statusBarMessage(tr("File saved at: %1").arg(fileName), QSystemTrayIcon::Information);
}

void MainWindow::importSettingsActionTriggered()
{
  QString fileName(QFileDialog::getOpenFileName(this, tr("Save file"), QDir::home().absolutePath() + "/Calíope.ini", tr("Settings files (*.ini)")));
  if (QFile::moveToTrash(settings.fileName()))
    statusBarMessage(tr("File %1 moved to trash").arg(settings.fileName()), QSystemTrayIcon::Information);
  else
    statusBarMessage(tr("File %1 NOT moved to trash").arg(settings.fileName()), QSystemTrayIcon::Information);
  if (QFile::copy(fileName, settings.fileName()))
    statusBarMessage(tr("File saved at: %1").arg(settings.fileName()), QSystemTrayIcon::Information);
  else
    statusBarMessage(tr("File NOT saved at: %1").arg(settings.fileName()), QSystemTrayIcon::Information);
}

void MainWindow::saveAllActionTriggered()
{
  emit save();
  statusBarMessage(tr("All editors saved"), QSystemTrayIcon::Information);
}

void MainWindow::migrateTableActionTriggered()
{
  bool restoreConnectionOpened;
  secondaryServerConnection = new DBMS(false);
  ConnectDialog *connectFrom = new ConnectDialog(secondaryServerConnection);
  connect(secondaryServerConnection, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  connect(secondaryServerConnection, SIGNAL(statusBarMessage(QString)), this, SLOT(statusBarMessage(QString)));
  if (connectFrom->exec() == QDialog::Accepted) {
    secondaryServerConnection->setUserName(connectFrom->getValues().value("User").toString());
    secondaryServerConnection->setHostName(connectFrom->getValues().value("Host").toString());
    secondaryServerConnection->setPassword(StaticFunctions::password(connectFrom->getValues().value("Password").toString()));
    secondaryServerConnection->setDatabase(connectFrom->getValues().value("Database").toString());
    secondaryServerConnection->setPort(connectFrom->getValues().value("Port").toUInt());
    secondaryServerConnection->setCharacterSet(connectFrom->getValues().value("Collation").toString().split("|").at(0));
    secondaryServerConnection->setCollation(connectFrom->getValues().value("Collation").toString().split("|").at(1));
    secondaryServerConnection->setUseSSL(connectFrom->getValues().value("UseSSL").toInt());
    secondaryServerConnection->setKeyFile(connectFrom->getValues().value("KeyFile").toString());
    secondaryServerConnection->setCertFile(connectFrom->getValues().value("CertFile").toString());
    secondaryServerConnection->setConnectionName(connectFrom->getValues().value("Name").toString());

    restoreConnection = mysql_init(NULL);
    if (!restoreConnection)
      QMessageBox::warning(0, tr("Library initialization"), tr("Error on library initialization for the database restore connection."));

    if (this->serverConnection->getUseSSL())
      mysql_ssl_set(restoreConnection, this->serverConnection->getKeyFile().toUtf8().data()
                    , this->serverConnection->getCertFile().toUtf8().data(), NULL, NULL, NULL);
    restoreConnectionOpened = mysql_real_connect(restoreConnection, secondaryServerConnection->getHostName().toUtf8().data()
                                                 , secondaryServerConnection->getUserName().toUtf8().data()
                                                 , secondaryServerConnection->getPassword().toUtf8().data()
                                                 , secondaryServerConnection->getDatabase().isEmpty() ? NULL : secondaryServerConnection->getDatabase().toUtf8().data()
                                                                                                        , secondaryServerConnection->getPort()
                                                 , NULL, 0);
    if (this->serverConnection->isOpened() && restoreConnectionOpened && (serverConnection->operator !=(secondaryServerConnection))) {
      DListWidgetDialog *dListWidgetDialog = new DListWidgetDialog(serverConnection->database()->getTables());
      if (dListWidgetDialog->exec() == QDialog::Accepted) {
        timerTableCount = new QTimer(this);
        migrationProgressDialog = new QProgressDialog(tr("Migrating data"), tr("Abort"), 0, 0, this);
        migrationProgressDialog->setWindowModality(Qt::WindowModal);
        migrationProgressDialog->show();
        QString statement;
        statement = "SET FOREIGN_KEY_CHECKS := 0;";
        if (mysql_real_query(restoreConnection, statement.toStdString().c_str(), statement.toStdString().size()) != 0) {
          qDebug() << QString("%1 %2 %3").arg(tr("Error:")).arg(mysql_errno(restoreConnection)).arg(mysql_error(restoreConnection));
        }
        statement = "SET SQL_MODE := 'ALLOW_INVALID_DATES';";
        if (mysql_real_query(restoreConnection, statement.toStdString().c_str(), statement.toStdString().size()) != 0) {
          qDebug() << QString("%1 %2 %3").arg(tr("Error:")).arg(mysql_errno(restoreConnection)).arg(mysql_error(restoreConnection));
        }
        foreach (QString table, dListWidgetDialog->selectedItems()) {
          statement = "DROP TABLE IF EXISTS `" + secondaryServerConnection->getDatabase() + "`.`" + table + "`";
          if (mysql_real_query(restoreConnection, statement.toStdString().c_str(), statement.toStdString().size()) != 0) {
            qDebug() << QString("%1 %2 %3").arg(tr("Error:")).arg(mysql_errno(restoreConnection)).arg(mysql_error(restoreConnection));
          }
          statement = serverConnection->table(table, serverConnection->getDatabase())->getDefinition(false);
          if (mysql_real_query(restoreConnection, statement.toStdString().c_str(), statement.toStdString().size()) != 0) {
            qDebug() << QString("%1 %2 %3").arg(tr("Error:")).arg(mysql_errno(restoreConnection)).arg(mysql_error(restoreConnection));
          }
          foreach(QString insertStatement, serverConnection->tables()->getTableDataToInsert("`" + serverConnection->getDatabase() + "`.`" + table + "`", false)) {
            if (mysql_real_query(restoreConnection, insertStatement.toStdString().c_str(), insertStatement.toStdString().size()) != 0) {
              qDebug() << QString("%1 %2 %3").arg(tr("Error:")).arg(mysql_errno(restoreConnection)).arg(mysql_error(restoreConnection));
            }
          }
        }
        migrationProgressDialog->close();
      }
      mysql_close(restoreConnection);
    } else {
      QMessageBox::information(this, tr("Table migration"), tr("Both connections are the same."));
    }
  }
  delete(connectFrom);
}

void MainWindow::databaseMetadataActionTriggered()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFileDialog fileDialog;
  fileDialog.setDirectory(QDir::home());
  QString file = fileDialog.getSaveFileName(this, tr("Save to Pdf"), settings.value("GeneralSettings/LastFilePdf", QDir::home().absolutePath()).toString(), tr("Pdf & Ps files (*.pdf *.ps)"));
  settings.setValue("GeneralSettings/LastFilePdf", fileDialog.directory().filePath(file));
  QPrinter printer(QPrinter::HighResolution);
  printer.setOutputFileName(file);
  printer.setOutputFormat(file.endsWith(".pdf") ? QPrinter::PdfFormat : QPrinter::NativeFormat);

  QTextDocument doc;
  doc.setHtml(serverConnection->outputAsHTML("SELECT `a`.`TABLE_SCHEMA` AS `TABLE_SCHEMA`, `a`.`TABLE_NAME` AS `TABLE_NAME`, `a`.`COLUMN_NAME` AS `COLUMN_NAME`, `a`.`ORDINAL_POSITION` AS `ORDINAL_POSITION`, `a`.`COLUMN_DEFAULT` AS `COLUMN_DEFAULT`, `a`.`IS_NULLABLE` AS `IS_NULLABLE`, `a`.`COLLATION_NAME` AS `COLLATION_NAME`, `a`.`COLUMN_TYPE` AS `COLUMN_TYPE`, `a`.`COLUMN_KEY` AS `COLUMN_KEY`, `a`.`EXTRA` AS `EXTRA`, `a`.`COLUMN_COMMENT` AS `COLUMN_COMMENT` FROM `information_schema`.`COLUMNS` `a`, `information_schema`.`TABLES` `b` WHERE `a`.`TABLE_SCHEMA` = `b`.`TABLE_SCHEMA` AND `a`.`TABLE_NAME` = `b`.`TABLE_NAME` AND `b`.`TABLE_TYPE` IN ('BASE TABLE', 'SYSTEM VERSIONED') AND `b`.`TABLE_NAME` LIKE 'SEVRI%' ORDER BY `a`.`TABLE_SCHEMA`, `a`.`TABLE_NAME`,`a`.`ORDINAL_POSITION`"));
  doc.print(&printer);

  QApplication::restoreOverrideCursor();
  statusBarMessage(tr("File saved at: %1").arg(file));
}

void MainWindow::replicationMenuAboutToShowSlot()
{
  replicationMenu->clear();
  switch(serverConnection->getDBMSType()) {
  case StaticFunctions::MySQL:
    replicationMenu->addAction(startReplicationSlaveAction);
    replicationMenu->addAction(stopReplicationSlaveAction);
    replicationMenu->addAction(resetSlaveAction);
    replicationMenu->addAction(rebootReplicationSlaveAction);
    break;
  case StaticFunctions::MariaDB:
    foreach (QString masterConnection, serverConnection->replication()->getSlavesNames()) {
      if (!masterConnection.isEmpty()) {
        replicationMenu->addSeparator()->setText(masterConnection);
        QAction *actionStop = replicationMenu->addAction(tr("Stop %1").arg(masterConnection));
        actionStop->setIcon(DIcon::Stop());
        connect(actionStop, SIGNAL(triggered()), replicationStopMapper, SLOT(map()));
        replicationStopMapper->setMapping(actionStop, masterConnection);

        QAction *actionStart = replicationMenu->addAction(tr("Start %1").arg(masterConnection));
        actionStart->setIcon(DIcon::SystemShutdown());
        connect(actionStart, SIGNAL(triggered()), replicationStartMapper, SLOT(map()));
        replicationStartMapper->setMapping(actionStart, masterConnection);

        QAction *actionReset = replicationMenu->addAction(tr("Reset %1").arg(masterConnection));
        actionReset->setIcon(DIcon::Reboot());
        connect(actionReset, SIGNAL(triggered()), replicationResetMapper, SLOT(map()));
        replicationResetMapper->setMapping(actionReset, masterConnection);

        QAction *actionRestart = replicationMenu->addAction(tr("Restart %1").arg(masterConnection));
        actionRestart->setIcon(DIcon::Reboot());
        connect(actionRestart, SIGNAL(triggered()), replicationStartMapper, SLOT(map()));
        replicationStartMapper->setMapping(actionRestart, masterConnection);
      }
    }
    replicationMenu->addSeparator()->setText(tr("All connections"));
    replicationMenu->addAction(startAllReplicationSlavesAction);
    replicationMenu->addAction(stopAllReplicationSlavesAction);
    break;
  case StaticFunctions::Undefined:
  default:
    break;
  }
  replicationMenu->addSeparator();
  replicationMenu->addAction(replicationInformationAction);
  replicationMenu->addAction(maintenancePurgeBinaryLogsAction);
  replicationMenu->addAction(maintenanceFlushRelayLogsAction);
}

void MainWindow::replicationStartConnection(QString connectionName)
{
  serverConnection->replication()->startSlave(connectionName);
}

void MainWindow::replicationStopConnection(QString connectionName)
{
  serverConnection->replication()->stopSlave(connectionName);
}

void MainWindow::replicationRestartConnection(QString connectionName)
{
  serverConnection->replication()->rebootSlave(connectionName);
}

void MainWindow::replicationResetConnection(QString connectionName)
{
  serverConnection->replication()->resetSlave(connectionName);
}

void MainWindow::objectMigrationActionTriggered()
{
  if (serverConnection->testOpened()) {
    objectMigration = mdiMain->findChild<ObjectMigration *>(tr("Object Migration"));
    if (!objectMigration) {
      objectMigration = new ObjectMigration(this->serverConnection);
      connect(objectMigration, SIGNAL(loadProgress(int)), dStatusBar, SLOT(setProgress(int)));
      addSubWindow(objectMigration);
    }
    mdiMain->setActiveSubWindow(objectMigration);
  }
}

void MainWindow::gitLabFileSearcherActionTriggered()
{
//  if (serverConnection->testOpened()) {
    gitLabFileSearcher = mdiMain->findChild<GitLabFileSearcher *>(tr("GitLab File Searcher"));
    if (!gitLabFileSearcher) {
      gitLabFileSearcher = new GitLabFileSearcher();
      connect(gitLabFileSearcher, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
      addSubWindow(gitLabFileSearcher);
    }
    mdiMain->setActiveSubWindow(gitLabFileSearcher);
    //  }
}

void MainWindow::mariadbExplorerHelpActionTriggered()
{
  dHelpViewer = mdiMain->findChild<DHelpViewer *>(tr("Calíope Help"));
  if (!dHelpViewer) {
    dHelpViewer = new DHelpViewer(
          StaticFunctions::workingDirectory() + "caliope.qhc"
          , StaticFunctions::workingDirectory() + "caliope.qhp"
          );
    addSubWindow(dHelpViewer);
  }
  mdiMain->setActiveSubWindow(dHelpViewer);
}

void MainWindow::mariadbExplorerHelpSourcecodeActionDoxygenTriggered()
{
  dHelpViewerSourceCode = mdiMain->findChild<DHelpViewer *>(tr("Calíope Source Code Help"));
  if (!dHelpViewerSourceCode) {
    dHelpViewerSourceCode = new DHelpViewer(
          StaticFunctions::workingDirectory() + "caliope_source_code.qhc"
          , StaticFunctions::workingDirectory() + "caliope_source_code.qhp"
          );
    addSubWindow(dHelpViewerSourceCode);
  }
  mdiMain->setActiveSubWindow(dHelpViewerSourceCode);
}

void MainWindow::mariadbExplorerHelpSourceCodeActionQDocTriggered()
{
//  dHelpViewerSourceCode = mdiMain->findChild<DHelpViewer *>(tr("Calíope Source Code Help"));
//  if (!dHelpViewerSourceCode) {
//    QString extraPath;
//#ifdef QT_DEBUG
//    extraPath = "/bin";
//#endif
//    dHelpViewerSourceCode = new DHelpViewer(
//          qApp->applicationDirPath() + extraPath + "/documentation/source_code/QDoc/Calíope_source_code.qhc"
//          , qApp->applicationDirPath() + extraPath + "/documentation/source_code/QDoc/Calíope_source_code.qhp"
//          );
//    addSubWindow(dHelpViewerSourceCode);
//  }
  //  mdiMain->setActiveSubWindow(dHelpViewerSourceCode);
}

void MainWindow::monitoringActionTriggered()
{
  dMonitoring = mdiMain->findChild<DMonitoring *>(tr("Monitoring"));
  if (!dMonitoring) {
    dMonitoring = new DMonitoring();
    connect(dMonitoring, SIGNAL(loadProgress(int)), dStatusBar, SLOT(setProgress(int)));
    connect(dMonitoring, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
    addSubWindow(dMonitoring);
  }
  mdiMain->setActiveSubWindow(dMonitoring);
}

void MainWindow::fileSystemWatcherFileChanged(QString path)
{
  QFile fileToOpen(path);
  bool opened = fileToOpen.open(QFile::ReadOnly | QFile::Text);
  if (!opened) {
    return;
  }
  QTextStream textStream(&fileToOpen);
  textStream.seek(fileToOpen.size());
  while (textStream.pos() > 0 && textStream.read(1) != "\n") {
    textStream.seek(textStream.pos() - 2);
  }
  errorLogBaseTextEditor->appendPlainText(textStream.readLine());
  errorLogBaseTextEditor->gotoLine(errorLogBaseTextEditor->document()->blockCount() - 1, true);
  fileToOpen.close();
}

void MainWindow::dLoadDataActionTriggered()
{
  if (serverConnection->testOpened()) {
    dLoadData = mdiMain->findChild<DLoadData *>(tr("Load Data"));
    if (!dLoadData) {
      dLoadData = new DLoadData(serverConnection);
      addSubWindow(dLoadData);
    }
    mdiMain->setActiveSubWindow(dLoadData);
  }
}

void MainWindow::dDashboardActionTriggered()
{
  if (serverConnection->testOpened()) {
    dDashboard = mdiMain->findChild<DDashboard *>(tr("Dashboard"));
    if (!dDashboard) {
      dDashboard = new DDashboard(this->serverConnection);
//      connect(dDashboard, SIGNAL(loadProgress(int)), dStatusBar, SLOT(setProgress(int)));
      addSubWindow(dDashboard);
    }
    mdiMain->setActiveSubWindow(dDashboard);
  }
}

void MainWindow::caliopePluginActionTriggered()
{
  QString file;
  QString extension;
  if (serverConnection->runQuerySingleColumn("SELECT `VARIABLE_VALUE` FROM "
                                             "`information_schema`.`GLOBAL_VARIABLES`"
                                             " WHERE `VARIABLE_NAME` = 'VERSION_COMPILE_OS'")
      .at(0).contains("linux", Qt::CaseInsensitive)) {
    file = ":/Plugin/Caliope_Plugin.so";
    extension = "*.so";
  }
  QFileInfo fileInfo(file);
  QString fileName = QFileDialog::getSaveFileName(this, tr("Save file"), fileInfo.filePath(), extension);
  if (fileName.isEmpty())
    return;
  QFile::copy(file, fileName);
  QString message = tr("Place the file %1 into the folder %2 of the server %3"
                       ).arg(fileName
                             , serverConnection->runQuerySingleColumn("SELECT `VARIABLE_VALUE` FROM "
                                                                      "`information_schema`.`GLOBAL_VARIABLES`"
                                                                      " WHERE `VARIABLE_NAME` = 'PLUGIN_DIR'").at(0)
                             , serverConnection->getHostName()
                      );
  QMessageBox messageBox;
  QString message2 = tr("Calíope_Plugin is a MariaDB pluging that provides some basic User Defined Functions (UDF)"
                        " to use along Calíope. For example it provides two functions to monitor the Operative System"
                        " memory so the a pie chart can be created in the Dashboard. After the file Caliope_Plugin is placed in the correct"
                        " folder in the MariaDB server you need to execte the following statements:%1"
                        ).arg("\n\n"
                              "DROP FUNCTION IF EXISTS `Caliope_OS_Total_Memory`;\n"
                              "CREATE FUNCTION `Caliope_OS_Total_Memory` RETURNS INT SONAME 'Caliope_Plugin.so';\n"
                              "SELECT `Caliope_OS_Total_Memory`();\n"
                              "\n\n"
                              "DROP FUNCTION IF EXISTS `Caliope_OS_Free_Memory`;\n"
                              "CREATE FUNCTION `Caliope_OS_Free_Memory` RETURNS INT SONAME 'Caliope_Plugin.so';\n"
                              "SELECT `Caliope_OS_Free_Memory`();"
                              "\n\n"
                              "DROP FUNCTION IF EXISTS `Caliope_OS_LastLoadAverage`;\n"
                              "CREATE FUNCTION `Caliope_OS_LastLoadAverage` RETURNS STRING SONAME 'Caliope_Plugin.so';\n"
                              "SELECT `Caliope_OS_LastLoadAverage`();"
                              "\n\n"
                              "DROP FUNCTION IF EXISTS `Caliope_OS_CPUTotal`;\n"
                              "CREATE FUNCTION `Caliope_OS_CPUTotal` RETURNS INT SONAME 'Caliope_Plugin.so';\n"
                              "SELECT `Caliope_OS_CPUTotal`();"
                              "\n");
  messageBox.setText(message);
  messageBox.setDetailedText(message
                             + "\n\n"
                             + message2);
  messageBox.exec();
}

void MainWindow::checkForUpdatesActionTriggered()
{
  QStringList arguments;
  arguments << getVersionNumber();
  processCaliopeUpdater = new QProcess(this);
  connect(processCaliopeUpdater, SIGNAL(readyReadStandardError()), this, SLOT(readyReadStandardErrorCaliopeUpdater()));
  connect(processCaliopeUpdater, SIGNAL(finished(int,QProcess::ExitStatus)), this, SLOT(finishedCaliopeUpdater(int,QProcess::ExitStatus)));
  processCaliopeUpdater->startDetached(QCoreApplication::applicationDirPath() + "/" + "CaliopeUpdater", arguments);
//  processCaliopeUpdater->waitForFinished();
  close();
}

void MainWindow::readyReadStandardErrorCaliopeUpdater()
{
  qDebug() << processCaliopeUpdater->readAllStandardError();
}

void MainWindow::finishedCaliopeUpdater(int exitCode, QProcess::ExitStatus exitStatus)
{
  Q_UNUSED(exitCode);
  Q_UNUSED(exitStatus);
}

void MainWindow::takeATourActionTriggered()
{
  QMap<QString, uint> sizes;
  sizes.insert("MainMenuHeight", menuBar()->height());
  sizes.insert("MainToolbarHeight", fileToolbar->height() * 1.6);
  sizes.insert("StatusbarHeight", dStatusBar->height());
  dTour = new DTour(this, sizes);
  dTour->show();
}

void MainWindow::showStartupTipsSlot()
{
  DStartupTips *tips = new DStartupTips();
  tips->show();
}

void MainWindow::shutdownServerActionTriggered()
{
  if (serverConnection->shutdown())
    statusBarMessage(tr("The database server has been shutdown."));
}

void MainWindow::retranslateUI()
{
  //createActions
  takeATourAction->setText(tr("Take a tour"));
  takeATourAction->setStatusTip(takeATourAction->text());

  exitAction->setText(tr("&Quit"));
  exitAction->setStatusTip(tr("Exits the application."));

  rebootAction->setText(tr("&Reboot"));
  rebootAction->setStatusTip(tr("Reboots the application."));

  aboutQtAction->setText(tr("About &Qt"));
  aboutQtAction->setStatusTip(tr("Shows the About dialog for Qt."));

  aboutMariaDBGUIAction->setText(tr("About Calíope"));
  aboutMariaDBGUIAction->setStatusTip(tr("Shows the About dialog for Calíope."));

  caliopePluginAction->setText(tr("Calíope plugin"));
  caliopePluginAction->setStatusTip(caliopePluginAction->text());

  checkForUpdatesAction->setText(tr("Check for updates"));
  checkForUpdatesAction->setStatusTip(checkForUpdatesAction->text());

  connectToServerAction->setText(tr("Connect to Server..."));
  connectToServerAction->setStatusTip(tr("Connect to a Database Server."));

  serverInformationAction->setText(tr("Server Information"));
  serverInformationAction->setStatusTip(serverInformationAction->text());

  userAdministrationAction->setText(tr("User Administration"));
  userAdministrationAction->setStatusTip(userAdministrationAction->text());

  backupAction->setText(tr("Backup"));
  backupAction->setStatusTip(backupAction->text());

  monitoringAction->setText(tr("Monitoring"));
  monitoringAction->setStatusTip(monitoringAction->text());

  restoreAction->setText(tr("Restore Backup"));
  restoreAction->setStatusTip(restoreAction->text());

  dLoadDataAction->setText(tr("Load Data"));
  dLoadDataAction->setStatusTip(dLoadDataAction->text());

  catalogsAction->setText(tr("Catalogs"));
  catalogsAction->setStatusTip(catalogsAction->text());

  queryAction->setText(tr("Query"));
  queryAction->setStatusTip(queryAction->text());

  newEmtyQueryAction->setText(tr("New empty Query"));
  newEmtyQueryAction->setStatusTip(newEmtyQueryAction->text());

  tableMaintenanceAction->setText(tr("Table Maintenance"));
  tableMaintenanceAction->setStatusTip(tableMaintenanceAction->text());

  mysqlScriptAction->setText(tr("SQL Script"));
  mysqlScriptAction->setStatusTip(mysqlScriptAction->text());

  mariadbHelpAction->setText(tr("MariaDB Help"));
  mariadbHelpAction->setStatusTip(mariadbHelpAction->text());

  mariadbExplorerHelpAction->setText(tr("Calíope Help"));
  mariadbExplorerHelpAction->setStatusTip(mariadbExplorerHelpAction->text());

  mariadbExplorerHelpSourceCodeActionDoxygen->setText(tr("Calíope Source Code Help") + " (Doxygen)");
  mariadbExplorerHelpSourceCodeActionDoxygen->setStatusTip(mariadbExplorerHelpSourceCodeActionDoxygen->text());

  mariadbExplorerHelpSourceCodeActionQDoc->setText(tr("Calíope Source Code Help") + " (QDoc)");
  mariadbExplorerHelpSourceCodeActionQDoc->setStatusTip(mariadbExplorerHelpSourceCodeActionQDoc->text());

  viewQueryLogAction->setText(tr("View query log"));
  viewQueryLogAction->setStatusTip(viewQueryLogAction->text());

  preferencesAction->setText(tr("Preferences"));
  preferencesAction->setStatusTip(preferencesAction->text());

  processAction->setText(tr("Processes"));

  showFileToolBarAction->setText(tr("&File Toolbar"));
  showFileToolBarAction->setStatusTip(showFileToolBarAction->text());

  closeCurrentConnectionAction->setText(tr("&Close connection"));
  closeCurrentConnectionAction->setStatusTip(tr("Close current database connection"));

  cascadeSubWindowsAction->setText(tr("Cascade windows"));
  cascadeSubWindowsAction->setStatusTip(cascadeSubWindowsAction->text());

  activateNextSubWindowAction->setText(tr("Next window"));
  activateNextSubWindowAction->setStatusTip(activateNextSubWindowAction->text());

  activatePreviousSubWindowAction->setText(tr("Previous window"));
  activatePreviousSubWindowAction->setStatusTip(activatePreviousSubWindowAction->text());

  closeActiveSubWindowAction->setText(tr("Close current window"));
  closeActiveSubWindowAction->setStatusTip(closeActiveSubWindowAction->text());

  closeAllSubWindowsAction->setText(tr("Close all windows"));
  closeAllSubWindowsAction->setStatusTip(closeAllSubWindowsAction->text());

  tileSubWindowsAction->setText(tr("Tile windows"));
  tileSubWindowsAction->setStatusTip(tileSubWindowsAction->text());

  shutdownServerAction->setText(tr("Shutdown Database Server"));
  shutdownServerAction->setStatusTip(shutdownServerAction->text());

  toggleFullScreenAction->setText(tr("Full screen"));
  toggleFullScreenAction->setStatusTip(toggleFullScreenAction->text());

  showErrorLogAction->setText(tr("Show error log"));
  showErrorLogAction->setStatusTip(showErrorLogAction->text());

  objectsDiagramAction->setText(tr("Objects diagram"));
  objectsDiagramAction->setStatusTip(objectsDiagramAction->text());

  openFileAction->setText(tr("Open file"));
  openFileAction->setShortcut(QKeySequence(Qt::CTRL | Qt::ALT | Qt::Key_O));
  openFileAction->setStatusTip(openFileAction->text());

  textEditorAction->setText(tr("Text Editor"));
  textEditorAction->setStatusTip(textEditorAction->text());

  stopReplicationSlaveAction->setText(tr("Stop Slave"));
  stopReplicationSlaveAction->setStatusTip(stopReplicationSlaveAction->text());

  resetSlaveAction->setText(tr("Reset Slave"));
  resetSlaveAction->setStatusTip(resetSlaveAction->text());

  startReplicationSlaveAction->setText(tr("Start Slave"));
  startReplicationSlaveAction->setStatusTip(startReplicationSlaveAction->text());

  rebootReplicationSlaveAction->setText(tr("Restart Slave"));
  rebootReplicationSlaveAction->setStatusTip(rebootReplicationSlaveAction->text());

  replicationInformationAction->setText(tr("Replication Information"));
  replicationInformationAction->setStatusTip(replicationInformationAction->text());

  maintenanceFlushPrivilegesAction->setStatusTip(maintenanceFlushPrivilegesAction->text());

  maintenancePurgeBinaryLogsAction->setStatusTip(maintenancePurgeBinaryLogsAction->text());

  maintenanceFlushHostsAction->setStatusTip(maintenanceFlushHostsAction->text());

  maintenanceTimeDifferenceAction->setText(tr("Time Difference"));
  maintenanceTimeDifferenceAction->setStatusTip(maintenanceTimeDifferenceAction->text());

  showSlowLogAction->setText(tr("Show the Slow Log"));
  showSlowLogAction->setStatusTip(showSlowLogAction->text());

  //Create Menus
  editorsMenu->setTitle(tr("Editors"));
//  projectFilesMenu->setTitle(tr("Project files"));
  fileMenu->setTitle(tr("&File"));
  openRecentFilesMenu->setTitle(tr("Recent files"));
  projectMenu->setTitle(tr("&Projects"));
  connectionMenu->setTitle(tr("&Connection"));
  changeCharsetMenu->setTitle(tr("Change character set"));
  changeDatabaseMenu->setTitle(tr("Change database"));
  replicationMenu->setTitle(tr("Replication"));
  maintenanceMenu->setTitle(tr("Maintenance"));
  viewMenu->setTitle(tr("&View"));
  languageMenu->setTitle(tr("&Languaje"));
  windowMenu->setTitle(tr("&Window"));
  helpMenu->setTitle(tr("&Help"));
  openRecentConnectionMenu->setTitle(tr("Recent connections"));
  databaseInformationMenu->setTitle(tr("Database information"));
  menuReports->setTitle(tr("Reports"));

  //Create Toobar
//  projectFilesPushButton->setText(tr("Project files"));

  viewRecentFilesAction->setText(tr("Select recent file..."));
  viewRecentFilesAction->setStatusTip(viewRecentFilesAction->text());

  catalogsRefreshAction->setText(tr("Refresh"));
  catalogsRefreshAction->setStatusTip(catalogsRefreshAction->text());

  catalogsReadOnlyAction->setText(tr("Read only"));
  catalogsReadOnlyAction->setStatusTip(catalogsReadOnlyAction->text());

  catalogsNewDatabaseAction->setText(tr("Create Database"));
  catalogsNewDatabaseAction->setStatusTip(catalogsNewDatabaseAction->text());

  catalogsDeleteDatabaseAction->setText(tr("Delete Database"));
  catalogsDeleteDatabaseAction->setStatusTip(catalogsDeleteDatabaseAction->text());

  catalogsPopulateTableAction->setText(tr("Populate table"));
  catalogsPopulateTableAction->setStatusTip(catalogsPopulateTableAction->text());

  viewRecentFilesAction->setText(tr("Select recent file..."));
  viewRecentFilesAction->setStatusTip(viewRecentFilesAction->text());

  clearRecentURLsAction->setText(tr("Clear recent files"));
  clearRecentURLsAction->setStatusTip(clearRecentURLsAction->text());

  databaseComparisionAction->setText(tr("Database comparison"));
  databaseComparisionAction->setStatusTip(databaseComparisionAction->text());

  takeASnapShotAction->setText(tr("Take a snapshot"));
  takeASnapShotAction->setStatusTip(takeASnapShotAction->text());

  actionReportServerInformation->setText(tr("Server information"));
  actionReportServerInformation->setStatusTip(actionReportServerInformation->text());

  actionReportHDDUsage->setText(tr("HDD Usage Graphics"));
  actionReportHDDUsage->setStatusTip(actionReportHDDUsage->text());

  actionReportExecutedQueries->setText(tr("Executed Queries"));
  actionReportExecutedQueries->setStatusTip(actionReportExecutedQueries->text());

  actionReportDataSentReceived->setText(tr("Data Sent/Received"));
  actionReportDataSentReceived->setStatusTip(actionReportDataSentReceived->text());

  actionCreateCustomReport->setText(tr("Create a custom report"));
  actionCreateCustomReport->setStatusTip(actionCreateCustomReport->text());

  actionReportSlowQueries->setText(tr("Slow Queries"));
  actionReportSlowQueries->setStatusTip(actionReportSlowQueries->text());

  tellUsYourCommentsAction->setText(tr("Tell us your comments"));
  tellUsYourCommentsAction->setStatusTip(tellUsYourCommentsAction->text());

  openProjectSiteAction->setText(tr("Open project web site"));
  openProjectSiteAction->setStatusTip(openProjectSiteAction->text());

  stopAllReplicationSlavesAction->setText(tr("Stop all Slaves"));
  stopAllReplicationSlavesAction->setStatusTip(stopAllReplicationSlavesAction->text());

  startAllReplicationSlavesAction->setText(tr("Start all Slaves"));
  startAllReplicationSlavesAction->setStatusTip(startAllReplicationSlavesAction->text());

  exportSettingsAction->setText(tr("Export settings file"));
  exportSettingsAction->setStatusTip(exportSettingsAction->text());

  importSettingsAction->setText(tr("Import settings file"));
  importSettingsAction->setStatusTip(importSettingsAction->text());

  saveAllAction->setText(tr("Save all"));
  saveAllAction->setStatusTip(saveAllAction->text());

  migrateTableAction->setText(tr("Migrate tables"));
  migrateTableAction->setStatusTip(migrateTableAction->text());

  databaseMetadataAction->setText(tr("Database metadata"));
  databaseMetadataAction->setStatusTip(databaseMetadataAction->text());

  objectMigrationAction->setText(tr("Object Migration"));
  objectMigrationAction->setStatusTip(objectMigrationAction->text());

  gitLabFileSearcherAction->setText(tr("GitLab File Searcher"));
  gitLabFileSearcherAction->setStatusTip(gitLabFileSearcherAction->text());

  dDashboardAction->setText(tr("Dashboard"));
  dDashboardAction->setStatusTip(dDashboardAction->text());
}

void MainWindow::createInitialSettings()
{
  DSettings settings;
  if (!settings.value("FileAssociations/SQLFiles", false).toBool())
    settings.setValue("FileAssociations/SQLFiles", "sql");
  if (!settings.value("TextEditor/DefaultPointSize", false).toBool())
    settings.setValue("TextEditor/DefaultPointSize", 11);
}

void MainWindow::enabledDisableConnectionMenus(bool enabled)
{
  closeCurrentConnectionAction->setEnabled(enabled);
  changeDatabaseMenu->setEnabled(enabled);
  changeCharsetMenu->setEnabled(enabled);
  shutdownServerAction->setEnabled(enabled);
  replicationMenu->setEnabled(enabled);
  maintenanceMenu->setEnabled(enabled);
  databaseInformationMenu->setEnabled(enabled);
  menuReports->setEnabled(enabled);
  databaseMetadataAction->setEnabled(enabled);
}

void MainWindow::createSystemaTrayIcon()
{
  trayIconEnabled = false;
  if (QSystemTrayIcon::isSystemTrayAvailable()) {
    trayIconMenu = new QMenu(this);
    trayIconMenu->addAction(rebootAction);
    trayIconMenu->addAction(exitAction);
    trayIcon = new QSystemTrayIcon(this);
    trayIcon->setIcon(windowIcon());
    trayIcon->setToolTip(QCoreApplication::applicationName());
    trayIcon->setContextMenu(trayIconMenu);
    trayIconEnabled = true;
    trayIcon->show();
  }
}

void MainWindow::preferencesActionTriggered()
{
  preferences = mdiMain->findChild<Preferences *>(tr("Preferences"));
  if (!preferences) {
    preferences = new Preferences(serverConnection);
    addSubWindow(preferences);
  }
  mdiMain->setActiveSubWindow(preferences);
}

void MainWindow::userAdministrationActionTriggered()
{
  if (serverConnection->testOpened()) {
    userAdministration = mdiMain->findChild<Users *>(tr("User Administration"));
    if (!userAdministration) {
      userAdministration = new Users(this->serverConnection);
      addSubWindow(userAdministration);
    }
    mdiMain->setActiveSubWindow(userAdministration);
  }
}

void MainWindow::restoreActionTriggered()
{
  if (serverConnection->testOpened()) {
    restore = mdiMain->findChild<RestoreBackupFile *>(tr("Restore Backup File"));
    if (!restore) {
      restore = new RestoreBackupFile(serverConnection);
      addSubWindow(restore);
    }
    mdiMain->setActiveSubWindow(restore);
  }
}

void MainWindow::backupActionTriggered()
{
  if (serverConnection->testOpened()) {
    backup = mdiMain->findChild<Backup *>(tr("Backup"));
    if (!backup) {
      backup = new Backup(serverConnection);
      connect(backup, SIGNAL(loadProgress(int)), dStatusBar, SLOT(setProgress(int)));
      connect(backup, SIGNAL(emitMessage(QString,uint,double)), this, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
      addSubWindow(backup);
    }
    mdiMain->setActiveSubWindow(backup);
  }
}

void MainWindow::sqlScriptActionTriggered()
{
  sqlEditor = new TextEditor(projects, this->serverConnection, EditorTypes::SQLQuery, ++sqlWindowCounter);
  connect(sqlEditor, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
  connect(sqlEditor, SIGNAL(updatePrositionViewer(int,int,int)), dStatusBar, SLOT(setPrositionViewer(int,int,int)));
  connect(sqlEditor, SIGNAL(windowClosed()), this, SLOT(decrementSQLWindowCounter()));
  connect(this, SIGNAL(save()), sqlEditor, SLOT(saveFileActionTriggered()));
  addSubWindow(sqlEditor);
}

void MainWindow::processActionTriggered()
{
  if (serverConnection->testOpened()) {
    processList = mdiMain->findChild<ProcessList *>(tr("Process List"));
    if (!processList) {
      processList = new ProcessList(this->serverConnection);
      addSubWindow(processList);
    }
    mdiMain->setActiveSubWindow(processList);
  }
}

void MainWindow::mysqlHelpActionTriggered()
{
  if (serverConnection->testOpened()) {
    mariadbHelp = mdiMain->findChild<MariaDBHelp *>(tr("MySQL On-Line Help"));
    if (!mariadbHelp) {
      mariadbHelp = new MariaDBHelp(this->serverConnection);
      addSubWindow(mariadbHelp);
    }
    mdiMain->setActiveSubWindow(mariadbHelp);
  }
}

void MainWindow::addQueryWindow(int sqlQueryWindowCounter, bool decrementOnClose)
{
  query = new SQLQuery(projects, this->serverConnection, sqlQueryWindowCounter);
  connect(query, SIGNAL(executionStarted(double)), this, SLOT(setProgressBarProgressSlot(double)));
  connect(query, SIGNAL(executionProgress(double)), this, SLOT(setProgressBarProgressSlot(double)));
  connect(query, SIGNAL(executionFinished(double)) , this, SLOT(setProgressBarProgressSlot(double)));
  connect(query, SIGNAL(showResultTab(QString,QString,QString)), this, SLOT(showResultTab(QString,QString,QString)));
  connect(query, SIGNAL(statusBarMessage(QString)), this, SLOT(statusBarMessage(QString)));
  if (decrementOnClose) {
    connect(query, SIGNAL(windowClosed()), this, SLOT(decrementSQLQueryWindowCounter()));
  }
  connect(query, SIGNAL(updatePrositionViewer(int,int,int)), dStatusBar, SLOT(setPrositionViewer(int,int,int)));
  connect(query, SIGNAL(busy(bool)), this, SLOT(setBusy(bool)));
  connect(query, SIGNAL(deleteAndClose(QString)), this, SLOT(deleteAndCloseQuery(QString)));
  addSubWindow(query);
}

void MainWindow::queryActionTriggered()
{
  if (serverConnection->testOpened()) {
    query = mdiMain->findChild<SQLQuery *>("SQLQuery");
    if (!query) {
      addQueryWindow(++sqlQueryWindowCounter);
    } else {
      if (mdiMain->activeSubWindow()->objectName() == "SQLQuery") {
        addQueryWindow(++sqlQueryWindowCounter);
      }
    }
    mdiMain->setActiveSubWindow(query);
  }
}

void MainWindow::newEmtyQueryActionTriggered()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QString data = "0";
  int counter = 0;
  while (!data.isEmpty()) {
    data = settings.value("SQLQuery/LastQuery-" + qApp->property("ConnectionName").toString() + "-" + tr("Query %1").arg(++counter), "").toString();
    if (data.isEmpty()) {
      addQueryWindow(counter, false);
      QApplication::restoreOverrideCursor();
      break;
    }
  }
  QApplication::restoreOverrideCursor();
}

void MainWindow::serverInformationActionTriggered()
{
  if (serverConnection->testOpened()) {
    serverInformationTab = mdiMain->findChild<ServerInformation *>(tr("Server Information"));
    if (!serverInformationTab) {
      serverInformationTab = new ServerInformation(this->serverConnection);
      addSubWindow(serverInformationTab);
    }
    mdiMain->setActiveSubWindow(serverInformationTab);
  }
}

void MainWindow::catalogsActionTriggered()
{
  if (serverConnection->testOpened()) {
    catalogs = mdiMain->findChild<Catalogs *>(tr("Catalogs"));
    if (!catalogs) {
      catalogs = new Catalogs(this->serverConnection);
      connect(catalogsNewDatabaseAction, SIGNAL(triggered()), catalogs, SLOT(newDatabaseActionTriggered()));
      connect(catalogsReadOnlyAction, SIGNAL(toggled(bool)), this, SLOT(catalogsReadOnlyActionSlot(bool)));
      connect(catalogsDeleteDatabaseAction, SIGNAL(triggered()), catalogs, SLOT(deleteDatabaseActionTriggered()));
      connect(catalogsPopulateTableAction, SIGNAL(triggered()), catalogs, SLOT(populateTableActionTriggered()));
      connect(catalogsRefreshAction, SIGNAL(triggered()), catalogs, SLOT(updateObjectInfo()));
      connect(this, SIGNAL(databaseChanged()), catalogs, SLOT(updateObjectInfo()));
      connect(catalogs, SIGNAL(loadStarted(QString,uint,double)), this, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
      connect(catalogs, SIGNAL(loadFinished(QString,uint,double)), this, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
      connect(catalogs, SIGNAL(loadProgress(QString,uint,double)), this, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
      connect(catalogs, SIGNAL(statusBarProgressMessage(QString,uint,double)), this, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
      connect(catalogs, SIGNAL(statusBarMessagePopup(QString)), this, SLOT(statusBarMessagePopup(QString)));
      connect(catalogs, SIGNAL(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)), this, SLOT(statusBarMessage(QString,QSystemTrayIcon::MessageIcon,int)));
      addSubWindow(catalogs);
    }
    mdiMain->setActiveSubWindow(catalogs);
  }
}

void MainWindow::connectToServerActionTriggered()
{
  ConnectDialog *connectFrom = new ConnectDialog(this->serverConnection);
  QMap<QString, QVariant> params;
  if (connectFrom->exec() == QDialog::Accepted) {
    if (!connectFrom->getConnectionPerformed()) {
      if (serverConnection->isOpened())
        serverConnection->close();
      params = connectFrom->getValues();
      serverConnection->setConnectionParameters(params);
      if (params.value("ConnectionType") == "--")
        serverConnection->setDBMSType(StaticFunctions::Undefined);
      if (params.value("ConnectionType")  == "MySQL")
        serverConnection->setDBMSType(StaticFunctions::MySQL);
      if (params.value("ConnectionType")  == "MariaDB")
        serverConnection->setDBMSType(StaticFunctions::MariaDB);

      if (!serverConnection->open())
        QMessageBox::critical(this, tr("Cannot connect to the server"), serverConnection->lastError());
    }
  }
  if (serverConnection->isOpened())
    newConnectionPerformed();
}

void MainWindow::tableMaintenanceActionTriggered()
{
//  if (serverConnection->testOpened()) {
//    tableMaintenance = mdiMain->findChild<TableMaintenance *>(tr("Table Maintenance"));
//    if (!tableMaintenance) {
//      tableMaintenance = new TableMaintenance(this);
//      mdiMain->addSubWindow(tableMaintenance);
//      tableMaintenance->showMaximized();
//      dStatusBar->resetStatusBar();
//    }
//  }
  if (serverConnection->testOpened()) {
//    optionsDock->hide();
    tableMaintenance = mdiMain->findChild<TableMaintenance *>(tr("Table Maintenance"));
    if (!tableMaintenance) {
      tableMaintenance = new TableMaintenance(this->serverConnection);
      connect(tableMaintenance, SIGNAL(loadProgress(int)), dStatusBar, SLOT(setProgress(int)));
      addSubWindow(tableMaintenance);
    }
    mdiMain->setActiveSubWindow(tableMaintenance);
  }
}

void MainWindow::showResultTab(QString table, QString database, QString where)
{
  if (serverConnection->testOpened()) {
    result = mdiMain->findChild<Result *>(tr("Result"));
    if (!result) {
//      if (optionsDock->isVisible())
//        optionsDock->hide();
      result = new Result(this->serverConnection, table, database, where);
      connect(result, SIGNAL(statusBarProgressMessage(QString,uint,double)), this, SLOT(statusBarProgressMessageSlot(QString,uint,double)));
      addSubWindow(result);
    }
    result->setDatabase(database.isEmpty() ? serverConnection->getDatabase() : database);
    result->setTable(table);
    result->setWhere(where);
    result->updateResult();
    result->showMaximized();
  }
}

void MainWindow::setTitle()
{
  setWindowTitle(
        (serverConnection->isOpened() ? (serverConnection->getConnectionName() + " - ") : "")
        + "[" + projects->name() + "]"
        + (serverConnection->isOpened() ? (" " + serverConnection->getConnectionString()) : "")
        );
}

void MainWindow::rebootSlot()
{
  qDebug() << "Performing application reboot...";
  qApp->exit(MainWindow::EXIT_CODE_REBOOT);
}

void MainWindow::setBusy(bool busy)
{
  if (busy) {
    busyDialog->show();
    runningSnakeMovie->start();
  } else {
    runningSnakeMovie->stop();
    busyDialog->close();
  }
}

void MainWindow::deleteAndCloseQuery(QString queryElement)
{
  settings.remove(queryElement);
}

//Projects *MainWindow::getCurrentProject()
//{
//  return projects;
//}

bool MainWindow::changeDatabase(QString database)
{
  if (database != serverConnection->getDatabase()) {
    serverConnection->changeDatabase(database);
    setTitle();
    return true;
  }
  return false;
}

//void MainWindow::optionsTreeViewItemClicked(QTreeWidgetItem *item, int column)
//{
//  switch (item->type()) {
//  case ItemTypes::Database:
//    changeDatabase(item->text(column).split(".").at(0));
//    catalogs->updateObjectInfo();
//    break;
//  case ItemTypes::Table:
//    catalogs->selectTable(item->text(column).split(".").at(0), item->text(column).split(".").at(1));
//    break;
//  case ItemTypes::User:
//    userAdministration->userInfo(item->text(column));
//    break;
//  case ItemTypes::UserHost:
//    userAdministration->userInfo(item->parent()->text(column));
//    break;
//  // default: Q_ASSERT(false);
//  }
//}

//void MainWindow::optionsDoubleClickedSlot(QTreeWidgetItem *item, int column)
//{
//  if (item->type() == ItemTypes::Table)
//    showResultTab(item->text(column).split(".").at(1));
//}

//void MainWindow::selectOptionsItem(QString item)
//{
//  optionsTreeView->clearSelection();
//  optionsTreeView->setCurrentItem(optionsTreeView->findItems(serverConnection->getDatabase() + "." + item
//                                                             , Qt::MatchRecursive | Qt::MatchExactly).at(0)
//                                  , 0, QItemSelectionModel::Select);
//}

//void MainWindow::fillOptions(ItemTypes::ItemType type)
//{
//  int maxWidth = 0;
//  QList<QTreeWidgetItem *> items;
//  optionsTreeView->clear();
//  optionsTreeView->setColumnCount(1);

//  switch (type) {
//  case ItemTypes::CatalogsTab:
//    optionsTreeView->setWindowTitle(tr("Databases"));
//    optionsTreeView->setHeaderLabel(optionsTreeView->windowTitle());
//    foreach (QString database, serverConnection->getDatabases()) {
//      if (maxWidth < QString(database).length())
//        maxWidth = QString(database).length();
//      QTreeWidgetItem *item = new QTreeWidgetItem((QTreeWidget*)0, QStringList(database), ItemTypes::Database);
//      item->setIcon(0, DIcon::Database());
//      items.append(item);
//      foreach (QString table, serverConnection->database(database)->getTables()) {
//        if (maxWidth < QString(table).length())
//          maxWidth = QString(table).length();
//        QTreeWidgetItem *childItem = new QTreeWidgetItem(item, QStringList(database + "." + table), ItemTypes::Table);
//        childItem->setIcon(0, DIcon::DatabaseTable());
//        items.append(childItem);
//      }
//    }
//    break;
//  case ItemTypes::UsersTab:
////    optionsTreeView->setWindowTitle(tr("Users"));
////    optionsTreeView->setHeaderLabel(optionsTreeView->windowTitle());
////    rows = serverConnection->getUsers();
////    for (int counter = 0; counter < rows.count(); counter++) {
////      rowsData = serverConnection->getUserHosts(rows.at(counter));
////      if (maxWidth < QString(rows.at(counter)).length())
////        maxWidth = QString(rows.at(counter)).length();
////      item = new QTreeWidgetItem((QTreeWidget*)0, QStringList(rows.at(counter)), ItemTypes::User);
////      item->setIcon(0, DIcon::Database());
////      items.append(item);
////      for (int counter2 = 0; counter2 < rowsData.count(); counter2++) {
////        itemChild = new QTreeWidgetItem(item, QStringList(rowsData.at(counter2)), ItemTypes::UserHost);
////        itemChild->setIcon(0, DIcon::Database());
////        items.append(itemChild);
////      }
////    }
//    break;
//  // default: Q_ASSERT(false);
//  }
//  optionsTreeView->resizeColumnToContents(0);
//  optionsTreeView->insertTopLevelItems(0, items);
//  optionsTreeView->setMaximumWidth(maxWidth * 10);
//}

void MainWindow::createToolBar()
{
  fileToolbar = addToolBar(tr("Main Toolbar"));
  fileToolbar->setIconSize(QSize(24, 24));
  fileToolbar->addActions(listOfMainActions);
  fileToolbar->addSeparator();
  fileToolbar->addWidget(projects->getProjectFilesPushButton());

  //Catalogs Toolbar
  catalogsToolbar = addToolBar(tr("Catalogs Toolbar"));
  catalogsToolbar->addAction(catalogsReadOnlyAction);
  catalogsToolbar->addSeparator();
  catalogsToolbar->addAction(catalogsRefreshAction);
  catalogsToolbar->addSeparator();
//  catalogsToolbar->addAction(newObjectAction);
//  catalogsToolbar->addAction(editObjectAction);
//  catalogsToolbar->addAction(deleteObjectAction);
//  catalogsToolbar->addSeparator();
  catalogsToolbar->addAction(catalogsNewDatabaseAction);
  catalogsToolbar->addAction(catalogsDeleteDatabaseAction);
  catalogsToolbar->addAction(catalogsPopulateTableAction);
  catalogsToolbar->setIconSize(QSize(24, 24));
  catalogsToolbar->hide();
}

void MainWindow::closeCurrentConnectionActionTriggered()
{
  okToClose();
  setWindowTitle(QCoreApplication::applicationName());
  serverConnection->close();
  enabledDisableConnectionMenus(false);
}

void MainWindow::createMenus()
{
  editorsMenu = new QMenu(this);
  editorsMenu->setIcon(DIcon::TextEditor());
  editorsMenu->addAction(mysqlScriptAction);
  editorsMenu->addAction(textEditorAction);

  fileMenu = menuBar()->addMenu(tr("&File"));
  fileMenu->addMenu(editorsMenu);
  fileMenu->addSeparator();
  fileMenu->addSeparator();
  fileMenu->addAction(openFileAction);
  openRecentFilesMenu = new QMenu(this);
  openRecentFilesMenu->setIcon(DIcon::DocumentOpenRecent());
  connect(openRecentFilesMenu, SIGNAL(aboutToShow()), this, SLOT(openRecentFilesMenuAboutToShowSlot()));
  fileMenu->addMenu(openRecentFilesMenu);
  fileMenu->addAction(viewRecentFilesAction);
  fileMenu->addSeparator();
  fileMenu->addAction(saveAllAction);
  fileMenu->addSeparator();
  fileMenu->addAction(exportSettingsAction);
  fileMenu->addAction(importSettingsAction);
  fileMenu->addSeparator();

  listOfMainActions.append(serverInformationAction);
  listOfMainActions.append(userAdministrationAction);
  listOfMainActions.append(backupAction);
  listOfMainActions.append(restoreAction);
  listOfMainActions.append(dLoadDataAction);
  listOfMainActions.append(catalogsAction);
  listOfMainActions.append(queryAction);
  listOfMainActions.append(newEmtyQueryAction);
  listOfMainActions.append(tableMaintenanceAction);
  listOfMainActions.append(databaseComparisionAction);
  listOfMainActions.append(migrateTableAction);
  listOfMainActions.append(preferencesAction);
  listOfMainActions.append(processAction);
  listOfMainActions.append(objectMigrationAction);
  listOfMainActions.append(gitLabFileSearcherAction);
  listOfMainActions.append(monitoringAction);
  listOfMainActions.append(dDashboardAction);
//  listOfMainActions.append(objectsDiagramAction);

  fileMenu->addActions(listOfMainActions);

  fileMenu->addSeparator();
  fileMenu->addAction(rebootAction);
  fileMenu->addAction(exitAction);

  projectMenu = menuBar()->addMenu("");
  projects->buildMenu(projectMenu);

  connectionMenu = menuBar()->addMenu("");
  connect(connectionMenu, SIGNAL(aboutToShow()), this, SLOT(connectionMenuAboutToShowSlot()));
  connectionMenu->addAction(connectToServerAction);
  connectionMenu->addAction(closeCurrentConnectionAction);
  connectionMenu->addSeparator();

  openRecentConnectionMenu = new QMenu(this);
  openRecentConnectionMenu->setIcon(DIcon::DocumentOpenRecent());
  connect(openRecentConnectionMenu, SIGNAL(aboutToShow()), this, SLOT(openRecentConnectionMenuAboutToShowSlot()));
  connectionMenu->addMenu(openRecentConnectionMenu);
  connectionMenu->addSeparator();

  changeCharsetMenu = new QMenu(this);
  changeCharsetMenu->setIcon(DIcon::CharacterSet());
  changeCharsetMenu->setDisabled(true);
  connect(changeCharsetMenu, SIGNAL(aboutToShow()), this, SLOT(updateChangeCharsetMenu()));
  connectionMenu->addMenu(changeCharsetMenu);
  changeDatabaseMenu = new QMenu(this);
  changeDatabaseMenu->setIcon(DIcon::Database());
  changeDatabaseMenu->setDisabled(true);
  connect(changeDatabaseMenu, SIGNAL(aboutToShow()), this, SLOT(updateChangeDatabaseMenu()));
  connectionMenu->addMenu(changeDatabaseMenu);

  databaseInformationMenu = new QMenu(this);
  databaseInformationMenu->setIcon(DIcon::Information());
  databaseInformationMenu->setDisabled(true);
  connect(databaseInformationMenu, SIGNAL(aboutToShow()), this, SLOT(updatedatabaseInformationMenu()));
  connectionMenu->addMenu(databaseInformationMenu);

  connectionMenu->addSeparator();
  replicationMenu = new QMenu(this);
  replicationMenu->setIcon(DIcon::DatabaseReplication());
  replicationMenu->setDisabled(true);
  connect(replicationMenu, SIGNAL(aboutToShow()), this, SLOT(replicationMenuAboutToShowSlot()));
  connectionMenu->addMenu(replicationMenu);

  maintenanceMenu = new QMenu(this);
  maintenanceMenu->setIcon(DIcon::DatabaseMaintenance());
  maintenanceMenu->setDisabled(true);
  maintenanceMenu->addSeparator()->setText(tr("General"));
  maintenanceMenu->addAction(maintenanceTimeDifferenceAction);
  maintenanceMenu->addAction(maintenanceFlushPrivilegesAction);
  maintenanceMenu->addAction(maintenanceFlushHostsAction);
  maintenanceMenu->addSeparator()->setText(tr("Replication"));
  maintenanceMenu->addAction(maintenancePurgeBinaryLogsAction);
  maintenanceMenu->addAction(maintenanceFlushRelayLogsAction);
  maintenanceMenu->addSeparator()->setText(tr("Logs"));
  maintenanceMenu->addAction(showSlowLogAction);
  connectionMenu->addMenu(maintenanceMenu);

  connectionMenu->addSeparator();
  connectionMenu->addAction(shutdownServerAction);
  connectionMenu->addSeparator();
  connectionMenu->addAction(viewQueryLogAction);

  menuReports = new QMenu(this);
  menuReports->setIcon(DIcon::GraphicsCharts());
  connect(menuReports, SIGNAL(aboutToShow()), this, SLOT(menuReportsAboutToShowSlot()));
  menuReports->setDisabled(true);
  connectionMenu->addSeparator();
  connectionMenu->addMenu(menuReports);

  connectionMenu->addSeparator();
  connectionMenu->addAction(databaseMetadataAction);

  viewMenu = menuBar()->addMenu("");
  viewMenu->addAction(toggleFullScreenAction);
  viewMenu->addAction(showFileToolBarAction);
  viewMenu->addAction(showErrorLogAction);

  languageMenu = menuBar()->addMenu(DIcon::AccessoriesCharacterMap(), "");
  languageActionGroup = new QActionGroup(this);
  connect(languageActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(swithLanguage(QAction*)));

  QAction *languageAction = new QAction("English - U.S.A.", this);
  languageAction->setCheckable(true);
  languageAction->setData("en/US");
  languageAction->setChecked(settings.value("GeneralSettings/Language", "en") == languageAction->data().toString() ? true : false);
  languageAction->setIcon(DIcon::FlagUS());
  languageMenu->addAction(languageAction);
  languageActionGroup->addAction(languageAction);
  languageAction = new QAction("Español - Costa Rica", this);
  languageAction->setCheckable(true);
  languageAction->setData("es/CR");
  languageAction->setChecked(settings.value("GeneralSettings/Language", "en") == languageAction->data().toString() ? true : false);
  languageAction->setIcon(DIcon::FlagCR());
  languageMenu->addAction(languageAction);
  languageActionGroup->addAction(languageAction);

  windowMenu = menuBar()->addMenu("");
  connect(windowMenu, SIGNAL(aboutToShow()), this, SLOT(updateWindowMenu()));

  helpMenu = menuBar()->addMenu(DIcon::Help(), "");
  helpMenu->addAction(mariadbHelpAction);
  helpMenu->addAction(mariadbExplorerHelpAction);
  helpMenu->addAction(mariadbExplorerHelpSourceCodeActionDoxygen);
//  helpMenu->addAction(mariadbExplorerHelpSourceCodeActionQDoc);
  helpMenu->addSeparator();
  helpMenu->addAction(aboutQtAction);
  helpMenu->addAction(aboutMariaDBGUIAction);
  helpMenu->addAction(takeATourAction);
  helpMenu->addAction(QWhatsThis::createAction(this));
  helpMenu->addSeparator();
  helpMenu->addAction(tellUsYourCommentsAction);
  helpMenu->addAction(openProjectSiteAction);
  helpMenu->addSeparator();
  helpMenu->addAction(caliopePluginAction);
  helpMenu->addSeparator();
  helpMenu->addAction(checkForUpdatesAction);
}

void MainWindow::swithLanguage(QAction *action)
{
  qApp->setProperty("ApplicationLanguage", action->data().toString());
  ///Remember add entry in StaticFunctions::currentLocale()
  QStringList languageData = action->data().toString().split("/");
  bool translatable = true;
  if (action->data().toString() == "es/CR")
    translatable = mainTranslator->load("Caliope_" + languageData.at(0) + "_" + languageData.at(1).toLower(), ":/translations");
  if (action->data().toString() == "en/US") {
    if (mainTranslator->load("", "")) {
      statusBarMessage(tr("Loaded default languaje."));
    }
  }
  if (translatable) {
    qApp->installTranslator(mainTranslator);
    retranslateUI();
    foreach (QMdiSubWindow *subWindow, mdiMain->subWindowList())
      if (subWindow->inherits("DMdiSubWindow")) {
        DMdiSubWindow *wid = qobject_cast<DMdiSubWindow *>(subWindow);
        wid->retranslateUI();
      }
    projects->retranslateUI();
  } else {
    statusBarMessage(tr("Cannot load the translation."), QSystemTrayIcon::Critical);
    action->setChecked(false);
  }
  settings.setValue("GeneralSettings/Language", action->data().toString());
}

/*!
  \fn void MainWindow::statusBarMessage(const QString &message, QSystemTrayIcon::MessageIcon icon, int timeout)
  \brief MainWindow::statusBarMessage
 */
void MainWindow::statusBarMessage(const QString &message, QSystemTrayIcon::MessageIcon icon, int timeout)
{
  if (message.isEmpty())
    return;
  dStatusBar->showFancyMessage(message, timeout);
  if (trayIconEnabled && settings.value("GeneralSettings/ShowTrayIconMessges", false).toBool())
    trayIcon->showMessage(QCoreApplication::applicationName(), message, icon, (timeout == 0 ? 2000 : 0) * 2);
}

void MainWindow::statusBarMessagePopup(const QString &message, int timeout)
{
  dStatusBar->showMessagePopup(message, timeout);
}

void MainWindow::aboutMariaDBGUIActionTriggered()
{
  QMessageBox::information(0, tr("About Calíope"),
                           "Calíope.\n"
                           "Copyright (c) 2008-2026  David Villalobos Cambronero.\n\n"

                           "This program is free software: you can redistribute it and/or modify"
                           "it under the terms of the GNU General Public License as published by"
                           "the Free Software Foundation, either version 3 of the License, or"
                           "(at your option) any later version.\n\n"

                           "This program is distributed in the hope that it will be useful,"
                           "but WITHOUT ANY WARRANTY; without even the implied warranty of"
                           "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the"
                           "GNU General Public License for more details.\n\n"

                           "You should have received a copy of the GNU General Public License"
                           "along with this program. If not, see <http://www.gnu.org/licenses/>.\n\n"

                           "Current version: " + getVersionNumber());
}

void MainWindow::updateWindowMenu()
{
  windowMenu->clear();
  windowMenu->addAction(cascadeSubWindowsAction);
  windowMenu->addAction(tileSubWindowsAction);
  windowMenu->addAction(activateNextSubWindowAction);
  windowMenu->addAction(activatePreviousSubWindowAction);
  windowMenu->addAction(closeActiveSubWindowAction);
  windowMenu->addAction(closeAllSubWindowsAction);
  windowMenu->addSeparator()->setText(tr("Windows"));
  foreach (QMdiSubWindow *subWindow, mdiMain->subWindowList()) {
    QAction *action = windowMenu->addAction(subWindow->windowTitle());
    action->setCheckable(true);
    // The following line was commented because there are moments when
    // mdiMain->currentSubWindow() == 0, so, windowTitle() cannot be reached
    // It happends on SQLQuery Windowsz
    if (mdiMain->currentSubWindow() != 0)
      action->setChecked(action->text() == mdiMain->currentSubWindow()->windowTitle());
    connect(action, SIGNAL(triggered()), windowMapper, SLOT(map()));
    windowMapper->setMapping(action, subWindow);
  }
  windowMenu->addSeparator()->setText(tr("Windows options"));
  windowMenu->addAction(takeASnapShotAction);
}

void MainWindow::setActiveSubWindowMap(QObject  *window)
{
  mdiMain->setActiveSubWindow(qobject_cast<QMdiSubWindow *>(window));
}

void MainWindow::updateChangeDatabaseMenu()
{
  changeDatabaseMenu->clear();
  QString currentDatabase = serverConnection->getDatabase();
  foreach (QString database, serverConnection->getDatabases()) {
    QAction *action = changeDatabaseMenu->addAction(database);
    action->setCheckable(true);
    action->setChecked(action->text() == currentDatabase);
    action->setIcon(DIcon::Database());
    connect(action, SIGNAL(triggered()), databasesMapper, SLOT(map()));
    databasesMapper->setMapping(action, database);
  }
}

void MainWindow::changeDatabaseSlot()
{
  //serverConnection->changeDatabase(database);
  foreach (QMdiSubWindow *subWindow, mdiMain->subWindowList())
    if (subWindow->inherits("SQLQuery")) {
      SQLQuery *wid = qobject_cast<SQLQuery *>(subWindow);
      wid->fillMariaDBSymbols();
    }
  setTitle();
  emit databaseChanged();
}

void MainWindow::changeDatabaseSlot(QString database)
{
  serverConnection->changeDatabase(database);
  changeDatabaseSlot();
}

void MainWindow::updateChangeCharsetMenu()
{
  changeCharsetMenu->clear();
  QList<QStringList> *rows = serverConnection->getCharacterSets();
  QString charSet(serverConnection->getCharacterSet());
  charSet = charSet.toLower().replace("-", "");
  for (int row = 0; row < rows->count(); row++) {
    QAction *action = changeCharsetMenu->addAction(rows->at(row).at(0) + " | " + rows->at(row).at(1));
    action->setCheckable(true);
    action->setChecked(rows->at(row).at(0).toLower() == charSet);
    connect(action, SIGNAL(triggered()), charsetMapper, SLOT(map()));
    charsetMapper->setMapping(action, rows->at(row).at(0));
  }
}

void MainWindow::changeCharsetSlot(QString charset)
{
  serverConnection->setCharacterSet(charset);
  statusBarMessage(tr("Character set changed to: %1").arg(serverConnection->getCharacterSet()));
}
