/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef LICENSETEMPLATE_H
#define LICENSETEMPLATE_H

#include <QDialog>

#include "dsettings.h"

class QPlainTextEdit;
class QDialogButtonBox;
class DFileSelector;

/**
 * @brief The LicenseTemplate class provides a dialog to configure the source code license header.
 *
 * This class allows the user to define a custom license text (e.g., GPL, MIT) that
 * can be automatically prepended to new files created within the application.
 * It supports loading the text from an external file or manual entry.
 */
class LicenseTemplate : public QDialog
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the LicenseTemplate dialog.
   *
   * Initializes the user interface and loads the currently saved license text.
   */
  LicenseTemplate();

  /**
   * @brief Retrieves the currently configured license text.
   *
   * This static method allows other parts of the application to fetch the
   * license header without needing to instantiate the dialog.
   *
   * @return A QString containing the license text.
   */
  static QString License();

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels and button text to reflect the current application language.
   */
  void retranslateUI();

private:
  DSettings settings;
  QPlainTextEdit *licensePlainTextEdit;
  QDialogButtonBox *buttonBox;
  DFileSelector *dFileSelector;

private slots:
  /**
   * @brief Saves the current text in the editor to persistent settings.
   */
  void saveLicenseTemplateText();

  /**
   * @brief Displays help information regarding license templates.
   */
  void helpLicenseTemplate();

  /**
   * @brief Loads the content of the file selected in the DFileSelector.
   *
   * Reads the file and populates the text editor with its contents.
   */
  void fileSelectorSlot();

  /**
   * @brief Validates the content of the text editor.
   *
   * Enables or disables the save/OK button based on whether the text is empty.
   */
  void validateText();
};

#endif // LICENSETEMPLATE_H
