/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef GITLABFILESEARCHER_H
#define GITLABFILESEARCHER_H

#include <QtNetwork>
#include <QSystemTrayIcon>

#include "dmdisubwindow.h"
#include "dsettings.h"

class DTitleLabel;
class DMessageLabel;
class QGroupBox;
class QLabel;
class QFormLayout;
class QLineEdit;
class DFileSelector;
class QPushButton;
class DTableView;
//class QNetworkAccessManager;
class BaseTextEditor;
class QSpinBox;

/**
 * @brief The GitLabFileSearcher class provides an interface for interacting with GitLab repositories.
 *
 * This class inherits from DMdiSubWindow and allows the user to configure GitLab API access
 * (using a bearer token), search for files within a specific project using keywords,
 * and download specific files to a local working directory.
 */
class GitLabFileSearcher : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the GitLabFileSearcher window.
   */
  GitLabFileSearcher();

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels, buttons, and tooltips to match the current application language.
   */
  void retranslateUI();

private slots:
  /**
   * @brief Slot to save or validate the Bearer Token when editing finishes.
   */
  void bearerTokenLineEditEditingFinished();

  /**
   * @brief Slot to save or validate the Repository API URL when editing finishes.
   */
  void repositoryAPILineEditEditingFinished();

  /**
   * @brief Slot to save or validate the local working directory path when editing finishes.
   */
  void workingDirectoryLineEditEditingFinished();

  /**
   * @brief Slot to save the search keyword when editing finishes.
   */
  void keywordToSearchLineEditEditingFinished();

  /**
   * @brief Initiates the download process for the configured files.
   */
  void downloadFilesPushButtonClicked();

  /**
   * @brief Executes the search query against the GitLab API.
   */
  void searchPushButtonClicked();

  /**
   * @brief Updates the stored Project ID when the spinbox value changes.
   * @param value The new project ID.
   */
  void gitLabProyectIdValueChanged(int value);

  /**
   * @brief Slot to handle changes to a specific file URL input.
   */
  void gitLabURLFileLineEditEditingFinished();

  /**
   * @brief Adds a specific file configuration to the list of files to be managed.
   */
  void addGitFilePushButtonClicked();

//  void gitLabFilesDTableViewRowDoubleClicked(QList<QVariant> row);

  /**
   * @brief Removes the selected file from the management list.
   */
  void deleteFileActionSlot();

  /**
   * @brief Saves the current list of GitLab files to persistent settings.
   */
  void updateGitLabFileSearcherFilesSettings();

  /**
   * @brief Handles the completion of a general network request (e.g., Search).
   * @param reply The network reply containing data or error information.
   */
  void downloadFinished(QNetworkReply *reply);

//  void searchResultsDTableViewRowClicked(QList<QVariant> row);

  /**
   * @brief Handles clicks on the search results table.
   *
   * Typically used to preview or select a file found via the search API.
   * @param index The index of the item clicked.
   */
  void searchResultsDTableViewClickedSlot(const QModelIndex &index);

  /**
   * @brief Opens the configured local working directory in the system file explorer.
   */
  void openWorkingDirectoryPushButtonClicked();

  /**
   * @brief Exports the current configuration (API keys, file lists) to a file.
   */
  void exportConfigutationPushButtonClicked();

  /**
   * @brief Imports a configuration from a file.
   */
  void importConfigutationPushButtonClicked();

  /**
   * @brief Handles the completion of a file download request.
   * @param reply The network reply containing the file content.
   */
  void downloadFinished2(QNetworkReply *reply);

private:
  DTitleLabel *dTitleLabel;
  DMessageLabel *dMessageLabel;
  QGroupBox *infoGroupBox;
  QFormLayout  *reposiroryInfoFormLayout;
  QLabel *repositoryAPILabel;
  QLineEdit *repositoryAPILineEdit;
  QLabel *bearerTokenLabel;
  QLineEdit *bearerTokenLineEdit;
  DSettings settings;
  DFileSelector *workingDirectoryLineEdit;
  QLabel *keywordToSearchLabel;
  QLineEdit *keywordToSearchLineEdit;
  QPushButton *downloadFilesPushButton;
  QPushButton *searchPushButton;
  QGroupBox *filesGroupBox;
  QFormLayout *filesFormLayout;
  QLabel *gitLabProyectIdLabel;
  QSpinBox *gitLabProyectIdLineEdit;
  QLabel *gitLabURLFileLabel;
  QLineEdit *gitLabURLFileLineEdit;
  DTableView *gitLabFilesDTableView;
  QPushButton *addGitFilePushButton;
  QList<QStringList> *gitLabFilesModel;
  QMenu *menu;
  QAction *deleteFileAction;

  /**
   * @brief Helper function to execute a download request.
   * @param localFilename The name of the file to save locally.
   */
  void doDownload(QString localFilename);

  QNetworkAccessManager manager;
//  QVector<QNetworkReply *> currentDownloads;
  DTableView *searchResultsDTableView;
  QList<QStringList> *searchResultsModel;
  BaseTextEditor *searchedFile;
  QString lastOpenedFile;

  /**
   * @brief Populates the table view with the list of configured files.
   */
  void fillGitLabFilesDTableView();

  QLabel *searchResultsLabel;
  QLabel *gitLabFilesLabel;
  QPushButton *openWorkingDirectoryPushButton;
  bool fileDownLoadFinished;
  QPushButton *exportConfigutationPushButton;
  QPushButton *importConfigutationPushButton;
  QNetworkAccessManager manager2;
  bool fileDownLoadFinished2;

protected:
  /**
   * @brief Handles context menu events (right-click) for the window.
   * @param event The context menu event parameters.
   */
  void contextMenuEvent(QContextMenuEvent *event);

signals:
  /**
   * @brief Emitted to display a message on the main application status bar.
   * @param message The message text.
   * @param icon The icon type (Information, Warning, etc.).
   * @param timeout The duration to show the message.
   */
  void statusBarMessage(const QString &message, QSystemTrayIcon::MessageIcon icon = QSystemTrayIcon::Information, int timeout = 0);
};

#endif // GITLABFILESEARCHER_H
