/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QVBoxLayout>
#include <QGroupBox>
#include <QFormLayout>
#include <QLineEdit>
#include <QPushButton>
#include <QAbstractItemModel>
#include <QMenu>
#include <QContextMenuEvent>
#include <QApplication>
#include <QSplitter>
#include <QMessageBox>
#include <QStandardItem>
#include <QDesktopServices>
#include <QDomDocument>
#include <QFileDialog>
#include <QDomElement>
#include <QSpinBox>

#include "gitlabfilesearcher.h"
#include "dicon.h"
#include "dtitlelabel.h"
#include "dmessagelabel.h"
#include "dfileselector.h"
#include "dtableview.h"
#include "staticfunctions.h"
#include "basetexteditor.h"

#include <QDebug>

GitLabFileSearcher::GitLabFileSearcher()
{
  setWindowIcon(DIcon::GitLab());
  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  dTitleLabel = new DTitleLabel;
  mainVLayout->addWidget(dTitleLabel);
  dMessageLabel = new DMessageLabel(MessageTypes::Information);
  mainVLayout->addWidget(dMessageLabel);
  infoGroupBox = new QGroupBox;
  reposiroryInfoFormLayout = new QFormLayout;
  infoGroupBox->setLayout(reposiroryInfoFormLayout);

  repositoryAPILabel = new QLabel;
  repositoryAPILineEdit = new QLineEdit;
  QHBoxLayout *hBoxLayout1 = new QHBoxLayout;
  hBoxLayout1->addWidget(repositoryAPILabel);
  hBoxLayout1->addWidget(repositoryAPILineEdit);
  repositoryAPILineEdit->setText(settings.value("GitLabFileSearcher/RepositoryAPI", QString()).toString());
  connect(repositoryAPILineEdit, SIGNAL(editingFinished()), this, SLOT(repositoryAPILineEditEditingFinished()));

  QSizePolicy sizePolicy = repositoryAPILineEdit->sizePolicy();
  sizePolicy.setHorizontalStretch(0);
  repositoryAPILineEdit->setSizePolicy(sizePolicy);

  bearerTokenLabel = new QLabel;
  bearerTokenLineEdit = new QLineEdit;
  hBoxLayout1->addWidget(bearerTokenLabel);
  hBoxLayout1->addWidget(bearerTokenLineEdit);
  reposiroryInfoFormLayout->addRow(hBoxLayout1);
  bearerTokenLineEdit->setText(settings.value("GitLabFileSearcher/BearerToken", QString()).toString());
  connect(bearerTokenLineEdit, SIGNAL(editingFinished()), this, SLOT(bearerTokenLineEditEditingFinished()));

  workingDirectoryLineEdit = new DFileSelector(DFileSelectorContexts::Directory);
  QHBoxLayout *hBoxLayout2 = new QHBoxLayout;
  hBoxLayout2->addWidget(workingDirectoryLineEdit);
  workingDirectoryLineEdit->setFileName(settings.value("GitLabFileSearcher/WorkingDirectory", QString()).toString());
  connect(workingDirectoryLineEdit, SIGNAL(changed()), this, SLOT(workingDirectoryLineEditEditingFinished()));

  openWorkingDirectoryPushButton = new QPushButton;
  openWorkingDirectoryPushButton->setIcon(DIcon::DocumentOpen());
  connect(openWorkingDirectoryPushButton, SIGNAL(clicked()), this, SLOT(openWorkingDirectoryPushButtonClicked()));
  hBoxLayout2->addWidget(openWorkingDirectoryPushButton);

  keywordToSearchLabel = new QLabel;
  keywordToSearchLineEdit = new QLineEdit;
  hBoxLayout2->addWidget(keywordToSearchLabel);
  hBoxLayout2->addWidget(keywordToSearchLineEdit);
  reposiroryInfoFormLayout->addRow(hBoxLayout2);
  keywordToSearchLineEdit->setText(settings.value("GitLabFileSearcher/KeywordToSearch", QString()).toString());
  connect(keywordToSearchLineEdit, SIGNAL(editingFinished()), this, SLOT(keywordToSearchLineEditEditingFinished()));
  connect(keywordToSearchLineEdit, SIGNAL(returnPressed()), this, SLOT(searchPushButtonClicked()));

  downloadFilesPushButton = new QPushButton;
  downloadFilesPushButton->setIcon(DIcon::EmblemDownloads());
  connect(downloadFilesPushButton, SIGNAL(clicked()), this, SLOT(downloadFilesPushButtonClicked()));
  searchPushButton = new QPushButton;
  searchPushButton->setIcon(DIcon::Find());
  QHBoxLayout *hBoxLayout4 = new QHBoxLayout;
  connect(searchPushButton, SIGNAL(clicked()), this, SLOT(searchPushButtonClicked()));
  hBoxLayout4->addWidget(downloadFilesPushButton);
  hBoxLayout4->addWidget(searchPushButton);
  exportConfigutationPushButton = new QPushButton;
  exportConfigutationPushButton->setIcon(DIcon::Export());
  connect(exportConfigutationPushButton, SIGNAL(clicked()), this, SLOT(exportConfigutationPushButtonClicked()));
  hBoxLayout4->addWidget(exportConfigutationPushButton);
  importConfigutationPushButton = new QPushButton;
  importConfigutationPushButton->setIcon(DIcon::Import());
  connect(importConfigutationPushButton, SIGNAL(clicked()), this, SLOT(importConfigutationPushButtonClicked()));
  hBoxLayout4->addWidget(importConfigutationPushButton);
  reposiroryInfoFormLayout->addRow(hBoxLayout4);

  QFrame* sepatatorFrame = new QFrame();
  sepatatorFrame->setFrameShape(QFrame::HLine);
  reposiroryInfoFormLayout->addRow(sepatatorFrame);

  filesGroupBox = new QGroupBox;
  filesFormLayout = new QFormLayout;
  filesGroupBox->setLayout(filesFormLayout);
  gitLabProyectIdLabel = new QLabel;
  gitLabProyectIdLineEdit = new QSpinBox;
  gitLabProyectIdLineEdit->setRange(0, 2147483647);
  QHBoxLayout *hBoxLayout3 = new QHBoxLayout;
  hBoxLayout3->addWidget(gitLabProyectIdLabel);
  hBoxLayout3->addWidget(gitLabProyectIdLineEdit);
  gitLabProyectIdLineEdit->setValue(settings.value("GitLabFileSearcher/GitLabProyectId", QString()).toInt());
  connect(gitLabProyectIdLineEdit, SIGNAL(valueChanged(int)), this, SLOT(gitLabProyectIdValueChanged(int)));
  gitLabURLFileLabel = new QLabel;
  gitLabURLFileLineEdit = new QLineEdit;
  hBoxLayout3->addWidget(gitLabURLFileLabel);
  hBoxLayout3->addWidget(gitLabURLFileLineEdit);
  gitLabURLFileLineEdit->setText(settings.value("GitLabFileSearcher/LastFileAdded", QString()).toString());
  connect(gitLabURLFileLineEdit, SIGNAL(editingFinished()), this, SLOT(gitLabURLFileLineEditEditingFinished()));
  addGitFilePushButton = new QPushButton;
  connect(addGitFilePushButton, SIGNAL(clicked()), this, SLOT(addGitFilePushButtonClicked()));
  hBoxLayout3->addWidget(addGitFilePushButton);
  filesFormLayout->addRow(hBoxLayout3);
  QList<QStringList> *gitLabFilesDTableViewHeaders = new QList<QStringList>;
  gitLabFilesDTableViewHeaders->append(QStringList() << tr("Proyect Id") << StaticFunctions::DelegateTypeNoDelegate() << "" << "Left");
  gitLabFilesDTableViewHeaders->append(QStringList() << tr("Proyect name") << StaticFunctions::DelegateTypeNoDelegate() << "" << "Left");
  gitLabFilesDTableViewHeaders->append(QStringList() << tr("File name") << StaticFunctions::DelegateTypeNoDelegate() << "" << "Left");
  gitLabFilesDTableViewHeaders->append(QStringList() << tr("File URL") << StaticFunctions::DelegateTypeNoDelegate() << "" << "Left");
  gitLabFilesDTableViewHeaders->append(QStringList() << tr("Local filename") << StaticFunctions::DelegateTypeNoDelegate() << "" << "Left");
  gitLabFilesDTableView = new DTableView(gitLabFilesDTableViewHeaders);
  gitLabFilesModel = new QList<QStringList>();
  fillGitLabFilesDTableView();
  QList<QStringList> *searchResultsDTableViewViewHeaders = new QList<QStringList>;
  searchResultsDTableViewViewHeaders->append(QStringList() << tr("Local filename") << StaticFunctions::DelegateTypeNoDelegate() << "" << "Left");
  searchResultsDTableViewViewHeaders->append(QStringList() << tr("Line number") << StaticFunctions::DelegateTypeNoDelegate() << "" << "Left");
  searchResultsDTableView = new DTableView(searchResultsDTableViewViewHeaders);
  connect(searchResultsDTableView, SIGNAL(clicked(QModelIndex)), this, SLOT(searchResultsDTableViewClickedSlot(QModelIndex)));
  searchResultsModel = new QList<QStringList>();
  QVBoxLayout *searchResultsVLayout = new QVBoxLayout;
  QWidget *searchResultsWidget = new QWidget;
  searchResultsWidget->setContentsMargins(0, 0, 0, 0);
  searchResultsWidget->setLayout(searchResultsVLayout);
  searchResultsLabel = new QLabel;
  searchResultsVLayout->addWidget(searchResultsLabel);
  searchResultsVLayout->addWidget(searchResultsDTableView);
  QVBoxLayout *gitLabFilesVLayout = new QVBoxLayout;
  QWidget *gitLabFilesWidget = new QWidget;
  gitLabFilesWidget->setContentsMargins(0, 0, 0, 0);
  gitLabFilesWidget->setLayout(gitLabFilesVLayout);
  gitLabFilesLabel = new QLabel;
  gitLabFilesVLayout->addWidget(gitLabFilesLabel);
  gitLabFilesVLayout->addWidget(gitLabFilesDTableView);
  QSplitter *splitter = new QSplitter;
  splitter->addWidget(gitLabFilesWidget);
  splitter->addWidget(searchResultsWidget);
  filesFormLayout->addRow(splitter);
  reposiroryInfoFormLayout->addRow(filesGroupBox);

  QFrame* sepatatorFrame2 = new QFrame();
  sepatatorFrame2->setFrameShape(QFrame::HLine);
  reposiroryInfoFormLayout->addRow(sepatatorFrame2);

  searchedFile = new BaseTextEditor(EditorTypes::SQLQuery);
  reposiroryInfoFormLayout->addRow(searchedFile);

  mainVLayout->addWidget(infoGroupBox);
  retranslateUI();

  menu = new QMenu(this);
  deleteFileAction = new QAction;
  connect(deleteFileAction, SIGNAL(triggered()), this, SLOT(deleteFileActionSlot()));
  menu->addAction(deleteFileAction);

  QWidget *widMain = new QWidget;
  widMain->setLayout(mainVLayout);
  setWidget(widMain);

  fileDownLoadFinished = true;
  connect(&manager, &QNetworkAccessManager::finished, this, &GitLabFileSearcher::downloadFinished);
  connect(&manager2, &QNetworkAccessManager::finished, this, &GitLabFileSearcher::downloadFinished2);
}

void GitLabFileSearcher::retranslateUI()
{
  setWindowTitle(tr("GitLab File Searcher"));
  setObjectName(windowTitle());
  dTitleLabel->setText(windowTitle());
  dTitleLabel->setToolTip(dTitleLabel->text());
  dMessageLabel->setText(tr("GitLab File Searcher is an option to search within the files of various repositories in GitLab, it makes use of the API through a Bearer token. We are aware that it is a very specific option and that it is not useful in all cases, but in the end, enjoy it."));
  infoGroupBox->setTitle(tr("Repository information"));
  repositoryAPILabel->setText(tr("Repository API"));
  repositoryAPILabel->setToolTip(repositoryAPILabel->text());
  bearerTokenLabel->setText(tr("Bearer token"));
  bearerTokenLabel->setToolTip(bearerTokenLabel->text());
  workingDirectoryLineEdit->setText(tr("Working directory"));
  keywordToSearchLabel->setText(tr("Keyword to search"));
  keywordToSearchLabel->setToolTip(keywordToSearchLabel->text());
  downloadFilesPushButton->setText(tr("Download files"));
  downloadFilesPushButton->setToolTip(downloadFilesPushButton->text());
  searchPushButton->setText(tr("Search in files"));
  searchPushButton->setToolTip(searchPushButton->text());
  filesGroupBox->setTitle(tr("Files"));
  gitLabProyectIdLabel->setText(tr("GitLab project identifier"));
  gitLabProyectIdLabel->setToolTip(gitLabProyectIdLabel->text());
  gitLabURLFileLabel->setText(tr("File URL"));
  gitLabURLFileLabel->setToolTip(gitLabURLFileLabel->text());
  addGitFilePushButton->setText(tr("Add file"));
  addGitFilePushButton->setToolTip(addGitFilePushButton->text());
  gitLabFilesLabel->setText(tr("GitLab files"));
  gitLabFilesLabel->setToolTip(gitLabFilesLabel->text());
  searchResultsLabel->setText(tr("Search results"));
  searchResultsLabel->setToolTip(searchResultsLabel->text());
  openWorkingDirectoryPushButton->setText(tr("Open working directory"));
  openWorkingDirectoryPushButton->setToolTip(openWorkingDirectoryPushButton->text());
  exportConfigutationPushButton->setText(tr("Export configuration"));
  exportConfigutationPushButton->setToolTip(exportConfigutationPushButton->text());
  importConfigutationPushButton->setText(tr("Import configuration"));
  importConfigutationPushButton->setToolTip(importConfigutationPushButton->text());
}

void GitLabFileSearcher::bearerTokenLineEditEditingFinished()
{
  settings.setValue("GitLabFileSearcher/BearerToken", bearerTokenLineEdit->text());
}

void GitLabFileSearcher::repositoryAPILineEditEditingFinished()
{
  if (repositoryAPILineEdit->text().startsWith("http")) {
    settings.setValue("GitLabFileSearcher/RepositoryAPI", repositoryAPILineEdit->text());
  } else {
    QMessageBox::warning(this, tr("Repository API"), repositoryAPILineEdit->text() + tr(" is not a valid URL."));
    repositoryAPILineEdit->setFocus();
    repositoryAPILineEdit->selectAll();
  }
}

void GitLabFileSearcher::workingDirectoryLineEditEditingFinished()
{
  settings.setValue("GitLabFileSearcher/WorkingDirectory", workingDirectoryLineEdit->getFileName());
}

void GitLabFileSearcher::keywordToSearchLineEditEditingFinished()
{
  settings.setValue("GitLabFileSearcher/KeywordToSearch", keywordToSearchLineEdit->text());
}

void GitLabFileSearcher::downloadFilesPushButtonClicked()
{
  for (int counter = 0; counter < gitLabFilesModel->count(); counter++)
    doDownload(gitLabFilesModel->at(counter).at(5));
  QMessageBox::information(this, tr("File download"), tr("All file have been downloaded"));
  emit statusBarMessage(tr("All file have been downloaded"));
}

void GitLabFileSearcher::searchPushButtonClicked()
{
  QFile *inputFile;
  searchResultsModel->clear();
  searchResultsDTableView->setModelData(searchResultsModel, true);
  unsigned int lineCounter = 0;
  for (int counter = 0; counter < gitLabFilesModel->count(); counter++) {
    inputFile = new QFile(workingDirectoryLineEdit->getFileName() + QDir::separator() + gitLabFilesModel->at(counter).at(5));
    if (!inputFile->exists()) {
      QApplication::restoreOverrideCursor();
      if (QMessageBox::question(this, tr("File does no exists"),
                                tr("The file ") + gitLabFilesModel->at(counter).at(5) + tr(" does no exits.")
                                + "<br />" +
                                tr("Do you want to download it?")) == QMessageBox::Yes) {
        doDownload(gitLabFilesModel->at(counter).at(5));
        while (!fileDownLoadFinished)
          QCoreApplication::processEvents(QEventLoop::AllEvents, 100);
      }
    }
    inputFile->close();
    QFile inputFile2(workingDirectoryLineEdit->getFileName() + QDir::separator() + gitLabFilesModel->at(counter).at(5));
    bool opened = inputFile2.open(QIODevice::ReadOnly);
    Q_UNUSED(opened);
    if (!inputFile2.isOpen()) {
      QMessageBox::information(this, tr("File can not be opened"), workingDirectoryLineEdit->getFileName()
                               + QDir::separator() + gitLabFilesModel->at(counter).at(5)
                               + "<br />");
      QApplication::restoreOverrideCursor();
      return;
    }
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QTextStream stream(&inputFile2);
    lineCounter = 0;
    for (QString line = stream.readLine(); !line.isNull(); line = stream.readLine()) {
      lineCounter++;
      if (line.contains(keywordToSearchLineEdit->text())) {
        searchResultsModel->append(QStringList()
                                 << DIcon::ImportIconPath()
                                 << gitLabFilesModel->at(counter).at(5)
                                 << QString("%1").arg((lineCounter)));
        searchResultsDTableView->setModelData(searchResultsModel, true);
      }
    }
    QApplication::restoreOverrideCursor();
  }
  emit statusBarMessage(tr("Search finished"));
}

void GitLabFileSearcher::gitLabProyectIdValueChanged(int value)
{
  settings.setValue("GitLabFileSearcher/GitLabProyectId", value);
}

void GitLabFileSearcher::gitLabURLFileLineEditEditingFinished()
{
  settings.setValue("GitLabFileSearcher/LastFileAdded", gitLabURLFileLineEdit->text());
}

void GitLabFileSearcher::addGitFilePushButtonClicked()
{
  QString urlAPI = repositoryAPILineEdit->text()
      + "/projects/" + QString::number(gitLabProyectIdLineEdit->value())
      + "/?ref=simple=yes&private=true&private_token=" + bearerTokenLineEdit->text();
  QNetworkRequest request((QUrl(urlAPI)));
  manager2.get(request);

  fileDownLoadFinished2 = false;
  while (!fileDownLoadFinished2)
    QCoreApplication::processEvents(QEventLoop::AllEvents, 100);

  QString projectName;
  QFile file(workingDirectoryLineEdit->getFileName() + QDir::separator() + "TemporaryFile");
  if (file.open(QIODevice::ReadOnly)) {
    QJsonDocument jsonDocument = QJsonDocument::fromJson(file.readAll());
    QJsonObject jsonObject = jsonDocument.object();
    projectName = jsonObject.value(QString("name")).toString();
  } else {
    projectName = "NoName";
    QMessageBox::information(this, tr("File can not be opened"), workingDirectoryLineEdit->getFileName() + QDir::separator() + "TemporaryFile");
  }
  gitLabFilesModel->append(QStringList()
                           << DIcon::FolderIconPath()
                           << QString::number(gitLabProyectIdLineEdit->value())
                           << projectName
                           << gitLabURLFileLineEdit->text().trimmed().split(QDir::separator()).last()
                           << gitLabURLFileLineEdit->text().trimmed()
                           << StaticFunctions::randomName((20)) + "." + gitLabURLFileLineEdit->text().split(".").last().trimmed());
  gitLabFilesDTableView->setModelData(gitLabFilesModel, true);
  updateGitLabFileSearcherFilesSettings();
  emit statusBarMessage(tr("File added"));
}

//void GitLabFileSearcher::gitLabFilesDTableViewRowDoubleClicked(QList<QVariant> row)
//{
//  gitLabProyectIdLineEdit->setText(row.at(1).toString());
//  gitLabURLFileLineEdit->setText(row.at(2).toString());
//}

void GitLabFileSearcher::deleteFileActionSlot()
{
  QFile file (workingDirectoryLineEdit->getFileName() + QDir::separator() + gitLabFilesDTableView->indexData(gitLabFilesDTableView->currentIndexItem().row(), 4, Qt::DisplayRole).toString());
  file.remove();
  gitLabFilesModel->removeAll(QStringList() << DIcon::FolderIconPath()
                              << gitLabFilesDTableView->indexData(gitLabFilesDTableView->currentIndexItem().row(), 0, Qt::DisplayRole).toString()
                              << gitLabFilesDTableView->indexData(gitLabFilesDTableView->currentIndexItem().row(), 1, Qt::DisplayRole).toString()
                              << gitLabFilesDTableView->indexData(gitLabFilesDTableView->currentIndexItem().row(), 2, Qt::DisplayRole).toString()
                              << gitLabFilesDTableView->indexData(gitLabFilesDTableView->currentIndexItem().row(), 3, Qt::DisplayRole).toString()
                              << gitLabFilesDTableView->indexData(gitLabFilesDTableView->currentIndexItem().row(), 4, Qt::DisplayRole).toString()
                              );

  gitLabFilesDTableView->setModelData(gitLabFilesModel, true);
  updateGitLabFileSearcherFilesSettings();
  emit statusBarMessage(tr("File deleted"));
}

void GitLabFileSearcher::updateGitLabFileSearcherFilesSettings()
{
  settings.remove("GitLabFileSearcherFiles");
  for (int counter = 0; counter < gitLabFilesModel->count(); counter++) {
    settings.setValue(QString("GitLabFileSearcherFiles/%1").arg(counter), QString("%1^%2^%3^%4^%5")
                      .arg(gitLabFilesModel->at(counter).at(1)
                           , gitLabFilesModel->at(counter).at(2)
                           , gitLabFilesModel->at(counter).at(3)
                           , gitLabFilesModel->at(counter).at(4)
                           , gitLabFilesModel->at(counter).at(5)));
  }
}

void GitLabFileSearcher::downloadFinished(QNetworkReply *reply)
{
  QFile file(workingDirectoryLineEdit->getFileName() + QDir::separator() + reply->url().toString().split("&fileName=").at(1));
  if (file.open(QIODevice::WriteOnly)) {
    file.write(reply->readAll());
    file.close();
    fileDownLoadFinished = true;
  } else {
    QMessageBox::information(this, tr("File can not be opened"), workingDirectoryLineEdit->getFileName() + QDir::separator() + reply->url().toString().split("&fileName=").at(1));
  }
  emit statusBarMessage(workingDirectoryLineEdit->getFileName() + QDir::separator() + reply->url().toString().split("&fileName=").at(1) + tr(" downloaded"));
}

void GitLabFileSearcher::searchResultsDTableViewClickedSlot(const QModelIndex &index)
{
  if (lastOpenedFile != workingDirectoryLineEdit->getFileName() + QDir::separator() + searchResultsModel->at(index.row()).at(1)) {
    lastOpenedFile = workingDirectoryLineEdit->getFileName() + QDir::separator() + searchResultsModel->at(index.row()).at(1);
    QFile inputFile(lastOpenedFile);
    bool opened = inputFile.open(QIODevice::ReadOnly);
    Q_UNUSED(opened);
    if (!inputFile.isOpen())
      return;
    searchedFile->setPlainText(inputFile.readAll());
    inputFile.close();
  }
  searchedFile->gotoLine(searchResultsModel->at(index.row()).at(2).toUInt() - 1);
  QStandardItem *item;
  QColor whiteColor = QColor(255, 255, 255);
  QColor grayColor = QColor(242, 242, 242);
  for (int counter = 0; counter < gitLabFilesModel->count(); counter++) {
    gitLabFilesDTableView->setCurrentIndex(gitLabFilesDTableView->model()->index(counter, 0));
    item = gitLabFilesDTableView->currentItem();
    ((counter % 2) != 0) ? item->setBackground(QBrush(grayColor, Qt::SolidPattern)) : item->setBackground(QBrush(whiteColor, Qt::SolidPattern));
    gitLabFilesDTableView->setCurrentIndex(gitLabFilesDTableView->model()->index(counter, 1));
    item = gitLabFilesDTableView->currentItem();
    ((counter % 2) != 0) ? item->setBackground(QBrush(grayColor, Qt::SolidPattern)) : item->setBackground(QBrush(whiteColor, Qt::SolidPattern));
    gitLabFilesDTableView->setCurrentIndex(gitLabFilesDTableView->model()->index(counter, 2));
    item = gitLabFilesDTableView->currentItem();
    ((counter % 2) != 0) ? item->setBackground(QBrush(grayColor, Qt::SolidPattern)) : item->setBackground(QBrush(whiteColor, Qt::SolidPattern));
  }
  for (int counter = 0; counter < gitLabFilesModel->count(); counter++) {
    if (gitLabFilesModel->at(counter).at(5) == searchResultsModel->at(index.row()).at(1)) {
      gitLabFilesDTableView->setCurrentIndex(gitLabFilesDTableView->model()->index(counter, 0));
      item = gitLabFilesDTableView->currentItem();
      item->setBackground(QBrush(QColor(114, 188, 212), Qt::SolidPattern));
      gitLabFilesDTableView->setCurrentIndex(gitLabFilesDTableView->model()->index(counter, 1));
      item = gitLabFilesDTableView->currentItem();
      item->setBackground(QBrush(QColor(114, 188, 212), Qt::SolidPattern));
      gitLabFilesDTableView->setCurrentIndex(gitLabFilesDTableView->model()->index(counter, 3));
      item = gitLabFilesDTableView->currentItem();
      item->setBackground(QBrush(QColor(114, 188, 212), Qt::SolidPattern));
      gitLabFilesDTableView->setCurrentIndex(gitLabFilesDTableView->model()->index(counter, 4));
      item = gitLabFilesDTableView->currentItem();
      item->setBackground(QBrush(QColor(114, 188, 212), Qt::SolidPattern));
      gitLabFilesDTableView->setCurrentIndex(gitLabFilesDTableView->model()->index(counter, 2));
      item = gitLabFilesDTableView->currentItem();
      item->setBackground(QBrush(QColor(114, 188, 212), Qt::SolidPattern));
    }
  }
  searchResultsDTableView->setFocus();
}

void GitLabFileSearcher::openWorkingDirectoryPushButtonClicked()
{
  QDesktopServices::openUrl(workingDirectoryLineEdit->getFileName());
}

void GitLabFileSearcher::exportConfigutationPushButtonClicked()
{
  QDomDocument xmlDocument;
  QFile fileName(QFileDialog::getSaveFileName(this, tr("Export configuration"), "", tr("XML files (*.xml)")));
  QDomElement rootElement = xmlDocument.createElement("Configurations");
  xmlDocument.appendChild(rootElement);
  if (!fileName.open(QIODevice::WriteOnly | QIODevice::Text)) {
    QMessageBox::warning(this, tr("Export configuration"), fileName.errorString());
    return;
  }
  QDomElement repositoryAPI = xmlDocument.createElement("RepositoryAPI");
  repositoryAPI.appendChild(xmlDocument.createTextNode(repositoryAPILineEdit->text()));
  rootElement.appendChild(repositoryAPI);
  QDomElement bearerToken = xmlDocument.createElement("BearerToken");
  bearerToken.appendChild(xmlDocument.createTextNode(bearerTokenLineEdit->text()));
  rootElement.appendChild(bearerToken);
  QDomElement workingDirectory = xmlDocument.createElement("WorkingDirectory");
  workingDirectory.appendChild(xmlDocument.createTextNode(workingDirectoryLineEdit->getFileName()));
  rootElement.appendChild(workingDirectory);
  QDomElement keywordToSearch = xmlDocument.createElement("KeywordToSearch");
  keywordToSearch.appendChild(xmlDocument.createTextNode(keywordToSearchLineEdit->text()));
  rootElement.appendChild(keywordToSearch);

  QDomElement gitLabFiles = xmlDocument.createElement("GitLabFiles");
  QDomElement gitLabFile;
  QDomElement gitLabProyectId;
  QDomElement gitLabProyectName;
  QDomElement gitLabFileName;
  QDomElement fileURL;
  for (int counter = 0; counter < gitLabFilesModel->count(); counter++) {
    gitLabFile = xmlDocument.createElement("GitLabFile");
    gitLabFile.setAttribute("LocalFilename", gitLabFilesModel->at(counter).at(5));
    gitLabFiles.appendChild(gitLabFile);
    gitLabProyectId = xmlDocument.createElement("GitLabProyectId");
    gitLabProyectId.appendChild(xmlDocument.createTextNode(gitLabFilesModel->at(counter).at(1)));
    gitLabFile.appendChild(gitLabProyectId);
    gitLabProyectName = xmlDocument.createElement("GitLabProyectName");
    gitLabProyectName.appendChild(xmlDocument.createTextNode(gitLabFilesModel->at(counter).at(2)));
    gitLabFile.appendChild(gitLabProyectName);
    gitLabFileName = xmlDocument.createElement("FileName");
    gitLabFileName.appendChild(xmlDocument.createTextNode(gitLabFilesModel->at(counter).at(3)));
    gitLabFile.appendChild(gitLabFileName);
    fileURL = xmlDocument.createElement("FileURL");
    fileURL.appendChild(xmlDocument.createTextNode(gitLabFilesModel->at(counter).at(4)));
    gitLabFile.appendChild(fileURL);
  }
  rootElement.appendChild(gitLabFiles);

  QTextStream out(&fileName);
  out << xmlDocument.toString();
  fileName.flush();
  fileName.close();
}

void GitLabFileSearcher::importConfigutationPushButtonClicked()
{
  QDomDocument xmlDocument;
  QFile fileName(QFileDialog::getOpenFileName(this, tr("Import configuration"), "", tr("XML files (*.xml)")));
  if (!fileName.open(QIODevice::ReadOnly | QIODevice::Text)) {
    QMessageBox::warning(this, tr("Import configuration"), fileName.errorString());
    return;
  }
  xmlDocument.setContent(&fileName);
  fileName.close();
  repositoryAPILineEdit->setText(xmlDocument.firstChildElement("Configurations").firstChildElement("RepositoryAPI").text());
  repositoryAPILineEditEditingFinished();
  bearerTokenLineEdit->setText(xmlDocument.firstChildElement("Configurations").firstChildElement("BearerToken").text());
  bearerTokenLineEditEditingFinished();
  workingDirectoryLineEdit->setFileName(xmlDocument.firstChildElement("Configurations").firstChildElement("WorkingDirectory").text());
  workingDirectoryLineEditEditingFinished();
  keywordToSearchLineEdit->setText(xmlDocument.firstChildElement("Configurations").firstChildElement("KeywordToSearch").text());
  keywordToSearchLineEditEditingFinished();
  QDomNodeList gitLabFileNodes = xmlDocument.firstChildElement("Configurations").firstChildElement("GitLabFiles").elementsByTagName("GitLabFile");
  QDomElement gitLabFileDomElement;
  gitLabFilesModel->clear();
  for(int counter = 0; counter < gitLabFileNodes.count(); counter++) {
    gitLabFileDomElement = gitLabFileNodes.at(counter).toElement();
    gitLabFilesModel->append(QStringList()
                             << DIcon::FolderIconPath()
                             << gitLabFileDomElement.firstChildElement("GitLabProyectId").text()
                             << gitLabFileDomElement.firstChildElement("GitLabProyectName").text()
                             << gitLabFileDomElement.firstChildElement("FileName").text()
                             << gitLabFileDomElement.firstChildElement("FileURL").text()
                             << gitLabFileDomElement.attribute("LocalFilename"));
  }
  gitLabFilesDTableView->setModelData(gitLabFilesModel, true);
  updateGitLabFileSearcherFilesSettings();
}

void GitLabFileSearcher::downloadFinished2(QNetworkReply *reply)
{
  QFile::remove(workingDirectoryLineEdit->getFileName() + QDir::separator() + "TemporaryFile");
  QFile file(workingDirectoryLineEdit->getFileName() + QDir::separator() + "TemporaryFile");
  if (file.open(QIODevice::WriteOnly)) {
    file.write(reply->readAll());
    file.close();
  } else {
    QMessageBox::information(this, tr("File can not be opened"), workingDirectoryLineEdit->getFileName() + QDir::separator() + "TemporaryFile");
  }
  fileDownLoadFinished2 = true;
}

//void GitLabFileSearcher::searchResultsDTableViewRowClicked(QList<QVariant> row)
//{
//  QFile inputFile(workingDirectoryLineEdit->getFileName() + QDir::separator() + row.at(1).toString());
//  inputFile.open(QIODevice::ReadOnly);
//  if (!inputFile.isOpen())
//    return;
//  searchedFile->setPlainText(inputFile.readAll());
//  inputFile.close();
//  searchedFile->gotoLine(row.at(1).toUInt());
//}

void GitLabFileSearcher::contextMenuEvent(QContextMenuEvent *event)
{
  deleteFileAction->setText(tr("Delete file"));
  menu->exec(event->globalPos());
}

void GitLabFileSearcher::doDownload(QString localFilename)
{
  //  "https://git.utn.ac.cr/api/v4/projects/127/repository/files/BaseDeDatos%2FRegistroUniversitario-Tablas.sql/raw?ref=desarrollo&private_token=DtTN3WUeJ9dLg_B9agNu"
  QStringList fileInfo;
  QString fileName;
  QString branchName;
  QString urlAPI;
  int lineNumber = 0;

  for (int counter = 0; counter < gitLabFilesModel->count(); counter++)
    if (gitLabFilesModel->at(counter).at(5) == localFilename)
      lineNumber = counter;

  fileInfo = gitLabFilesModel->at(lineNumber).at(4).split("blob");
  fileInfo = fileInfo.at(1).split(QDir::separator(), Qt::SkipEmptyParts);
  branchName = fileInfo.takeFirst();
  fileName = fileInfo.takeLast();
  urlAPI = repositoryAPILineEdit->text()
      + "/projects/" + gitLabFilesModel->at(lineNumber).at(1)
      + "/repository/files/" + fileInfo.join("%2F") + "%2F"
                + fileName + "/raw?red¿f=" + branchName
      + "&private_token=" + bearerTokenLineEdit->text()
      + "&fileName=" + gitLabFilesModel->at(lineNumber).at(5);

  fileDownLoadFinished = false;
  QNetworkRequest request((QUrl(urlAPI)));
  manager.get(request);
}

void GitLabFileSearcher::fillGitLabFilesDTableView()
{
  QStringList files = settings.allKeys("GitLabFileSearcherFiles");
  QStringList details;
  foreach (QString file, files) {
    details = settings.value("GitLabFileSearcherFiles/" + file, QString()).toString().split("^");
    gitLabFilesModel->append(QStringList() << DIcon::FolderIconPath() << details.at(0) << details.at(1) << details.at(2) << details.at(3) << details.at(4));
    gitLabFilesDTableView->setModelData(gitLabFilesModel, true);
  }
}
