/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef GENERALHIGHLIGHTER_H
#define GENERALHIGHLIGHTER_H

#include <QSyntaxHighlighter>
#include <QTextBlockUserData>
#include <QRegularExpression>

#include "editortypes.h"
#include "dsettings.h"

class QTextDocument;

/**
 * @brief Structure to hold information about brackets or parentheses.
 *
 * This structure is used to store the character (e.g., '{', '(') and its
 * relative position within a text block to facilitate parenthesis matching.
 */
struct BracketInfo
{
  char character;
  int position;
};

/**
 * @brief The GeneralHighlighter class provides syntax highlighting for various editor modes.
 *
 * Inheriting from QSyntaxHighlighter, this class applies color and font formatting
 * to the text document based on the specific editor type (SQL, Diff, SVN Log, etc.).
 * It handles keyword highlighting, comments, strings, and search pattern highlighting.
 */
class GeneralHighlighter : public QSyntaxHighlighter
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the GeneralHighlighter.
   * @param parent The text document to which highlighting will be applied.
   * @param type The context type of the editor (e.g., SQLQuery, Diff).
   */
  GeneralHighlighter(QTextDocument *parent = 0, EditorTypes::EditorType type = EditorTypes::NoEditor);

  /**
   * @brief Sets the pattern to highlight for "Find" operations.
   * @param exp The string or regular expression to highlight as a search result.
   */
  void setFindPattern(const QString &exp);

protected:
  /**
   * @brief Applies highlighting rules to a specific block of text.
   *
   * This is the main entry point called by the Qt framework when the document changes.
   * @param text The text content of the current block.
   */
  void highlightBlock(const QString &text);

private:
  struct HighlightingRule
  {
    QRegularExpression pattern;
    QTextCharFormat format;
  };
  QVector<HighlightingRule> thirdHighlightingRules;
  QVector<HighlightingRule> secondHighlightingRules;
  QVector<HighlightingRule> firstHighlightingRules;
  HighlightingRule rule;
  HighlightingRule findRule;

  /**
   * @brief Helper to highlight text based on a specific rule.
   * @param rule The rule containing the pattern and format.
   * @param text The text to process.
   */
  void highlightProcedure(HighlightingRule rule, const QString &text);

  /**
   * @brief Initializes rules for SQL syntax highlighting.
   */
  void sqlHighlightRules();

  /**
   * @brief Initializes rules for Diff file highlighting.
   */
  void diffHighlightRules();

  /**
   * @brief Initializes rules for SVN log highlighting.
   */
  void svnHighlightRules();

  /**
   * @brief Initializes rules for Commit message highlighting.
   */
  void commitHighlightRules();

  enum States {Normal, Commenting};
  QTextCharFormat commentFormat;

  /**
   * @brief Handles highlighting for multi-line comment blocks.
   * @param text The text block content.
   * @param startOfComment The pattern indicating the start of a comment.
   * @param endOfComment The pattern indicating the end of a comment.
   */
  void multiLineComments(const QString &text, const QString &startOfComment, const QString &endOfComment);

  EditorTypes::EditorType editorType;
  QTextCharFormat normalFormat;
  QTextDocument *document;
  DSettings settings;
  QRegularExpressionMatch regularExpressionMatch;
  QRegularExpressionMatchIterator regularExpressionMatchIterator;
};

/**
 * @brief Custom user data associated with text blocks.
 *
 * This class stores information about brackets found within a text block.
 * It is used by the editor to identify matching parentheses across the document.
 */
class TextBlockData : public QTextBlockUserData
{
public:
  /**
   * @brief Constructs a TextBlockData object.
   */
  TextBlockData();

  /**
   * @brief Retrieves the list of brackets found in the block.
   * @return A vector of pointers to BracketInfo structures.
   */
  QVector<BracketInfo *> brackets();

  /**
   * @brief Adds a bracket definition to the storage.
   * @param info Pointer to the BracketInfo structure to insert.
   */
  void insert(BracketInfo *info);

private:
  QVector<BracketInfo *> vectorBrackets;
};
#endif // GENERALHIGHLIGHTER_H
