/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef FRAMINGTEXTHANDLER_H
#define FRAMINGTEXTHANDLER_H

#include <QObject>
#include <QPainter>
#include <QTextDocument>
#include <QTextObjectInterface>

/**
 * @brief The FramingTextHandler class draws a visual frame or placeholder for text objects.
 *
 * This class implements the QTextObjectInterface to handle the custom rendering
 * of specific text regions. It is typically used to represent folded code blocks,
 * displaying a placeholder (e.g., " ... ) ") instead of the full content.
 */
class FramingTextHandler: public QObject, public QTextObjectInterface
{
  Q_OBJECT
  Q_INTERFACES(QTextObjectInterface)

public:
  /**
   * @brief Constructs the FramingTextHandler.
   */
  explicit FramingTextHandler();

  /**
   * @brief Calculates the size of the framing object.
   *
   * Called by the Qt text engine to determine the dimensions required to
   * display the placeholder text.
   *
   * @param doc Pointer to the text document.
   * @param posInDocument The character position of the object.
   * @param format The format object containing properties for this text object.
   * @return The size (width and height) of the drawn object.
   */
  QSizeF intrinsicSize(QTextDocument *doc, int posInDocument, const QTextFormat &format);

  /**
   * @brief Draws the framing object.
   *
   * Renders the placeholder text (or frame) onto the document layout using the painter.
   *
   * @param painter The painter used for drawing.
   * @param rect The target rectangle where the object should be drawn.
   * @param doc Pointer to the text document.
   * @param posInDocument The character position of the object.
   * @param format The format object containing properties.
   */
  void drawObject(QPainter *painter, const QRectF &rect, QTextDocument *doc, int posInDocument, const QTextFormat &format);

  /**
   * @brief Unique type identifier for this custom text object.
   *
   * Defined as QTextFormat::UserObject + 81 to avoid collisions with other custom types.
   */
  static const int UserDefinedTextFormatObjectType = QTextFormat::UserObject + 81;

  /**
   * @brief Unique property identifier for this custom format.
   */
  static const int UserDefinedCharFormatPropertyId = 81;

  /**
   * @brief The text displayed when the object is drawn.
   *
   * Defaults to " ... ) ", commonly used to indicate a collapsed code block ending in a parenthesis.
   */
  QString placeHolderText = " ... ) ";
};

#endif // FRAMINGTEXTHANDLER_H
