/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef FOLDINGTEXTHANDLER_H
#define FOLDINGTEXTHANDLER_H

#include <QObject>
#include <QPainter>
#include <QTextDocument>
#include <QTextObjectInterface>

/**
 * @brief The FoldingTextHandler class renders custom text objects for code folding.
 *
 * This class implements the QTextObjectInterface to define how "folded" or
 * "collapsed" text regions are visually represented within a QTextDocument.
 * It handles the calculation of the visual replacement's size and its painting logic.
 */
class FoldingTextHandler: public QObject, public QTextObjectInterface
{
  Q_OBJECT
  Q_INTERFACES(QTextObjectInterface)

public:
  /**
   * @brief Constructs the FoldingTextHandler.
   */
  explicit FoldingTextHandler();

  /**
   * @brief Calculates the size of the custom text object.
   *
   * This method is called by the Qt text engine to determine how much space
   * needs to be reserved for the folded text marker.
   *
   * @param doc Pointer to the document containing the object.
   * @param posInDocument The character position of the object in the document.
   * @param format The formatting information associated with the object.
   * @return A QSizeF representing the width and height of the object.
   */
  QSizeF intrinsicSize(QTextDocument *doc, int posInDocument, const QTextFormat &format);

  /**
   * @brief Draws the custom text object.
   *
   * This method performs the actual rendering of the folded text marker (e.g., an ellipsis or symbol)
   * onto the document layout.
   *
   * @param painter The painter used to draw the object.
   * @param rect The rectangle within which the object should be drawn.
   * @param doc Pointer to the document.
   * @param posInDocument The character position of the object.
   * @param format The formatting information.
   */
  void drawObject(QPainter *painter, const QRectF &rect, QTextDocument *doc, int posInDocument, const QTextFormat &format);

  /**
   * @brief Unique identifier for the user-defined text format object type.
   *
   * Calculated as QTextFormat::UserObject + 2 to avoid conflicts with other custom objects.
   */
  static const int UserDefinedTextFormatObjectType = QTextFormat::UserObject + 2;

  /**
   * @brief Unique identifier for the user-defined character format property.
   */
  static const int UserDefinedCharFormatPropertyId = 2;
};

#endif // FOLDINGTEXTHANDLER_H
