/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef FINDREPLACE_H
#define FINDREPLACE_H

#include <QToolBar>
#include <QTextDocument>

#include "staticfunctions.h"
#include "dsettings.h"

class QLineEdit;
class QAction;
class TextEditor;
class QCompleter;

/**
 * @brief The FindReplace class provides a search and replace toolbar.
 *
 * This class inherits from QToolBar and provides a UI for searching text,
 * replacing occurrences, and highlighting matches within an associated TextEditor.
 * It handles search history and various search options (case sensitivity, regex, etc.).
 */
class FindReplace : public QToolBar
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the FindReplace toolbar.
   * @param document Pointer to the TextEditor widget where operations will be performed.
   */
  FindReplace(TextEditor *document);

private slots:
  /**
   * @brief Hides the toolbar.
   */
  void closeToolBar();

  /**
   * @brief Executes the search logic.
   * @param text The string to search for. If empty, uses the text from the line edit.
   * @param direction The direction of the search (Forward or Backward).
   * @return True if the text was found, false otherwise.
   */
  bool find(QString text = QString(), StaticFunctions::findFlags direction = StaticFunctions::FindForward);

  /**
   * @brief Searches for the next occurrence of the text (forward).
   */
  void findNext();

  /**
   * @brief Searches for the previous occurrence of the text (backward).
   */
  void findPrev();

  /**
   * @brief Replaces the currently selected match or the next occurrence.
   */
  void replaceActionTriggerd();

  /**
   * @brief Replaces all occurrences of the search text with the replacement text.
   */
  void replaceAllActionTriggerd();

  /**
   * @brief Highlights all occurrences of the search text in the document.
   */
  void highlightAllActionTriggerd();

private:
  TextEditor *plainTextEdit;
  QLineEdit *findLineEdit;
  QLineEdit *replaceLineEdit;
  QAction *caseSensitiveAction;
  QAction *entireWordAction;
  QAction *regExpAction;
  QCompleter *completer;
  DSettings settings;

  /**
   * @brief Saves the current search string to history/settings for auto-completion.
   */
  void SaveSearchedStrings();

protected:
  /**
   * @brief Handles the widget show event.
   *
   * Used to set focus to the find line edit when the toolbar appears.
   * @param event The show event parameters.
   */
  void showEvent(QShowEvent *event);

  /**
   * @brief Handles key press events.
   *
   * Used to capture keys like Escape (to close) or Enter (to search).
   * @param event The key event parameters.
   */
  void keyPressEvent(QKeyEvent *event);

  /**
   * @brief Filters events from watched objects.
   *
   * Used to intercept key events from child widgets (like the line edits).
   * @param obj The object receiving the event.
   * @param event The event being processed.
   * @return True if the event was filtered/handled, false otherwise.
   */
  bool eventFilter(QObject *obj, QEvent *event);
};

#endif // FINDREPLACE_H
